<?php
include 'db.php';

ini_set('session.gc_maxlifetime', 31536000);
ini_set('session.cookie_lifetime', 31536000);
session_set_cookie_params([
    'lifetime' => 31536000,
    'path' => '/',
    'domain' => $_SERVER['HTTP_HOST'],
    'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
    'httponly' => true,
    'samesite' => 'Lax'
]);
session_start();

$error = '';
$success = '';
$saved_email = '';

if (isset($_SESSION['resident_id'])) {
  header('Location: resident-dashboard.php');
  exit;
}

if (isset($_GET['guest']) && $_GET['guest'] == '1') {
  $_SESSION['guest_mode'] = true;
  header('Location: resident-dashboard.php');
  exit;
}

if (isset($_GET['logout']) && $_GET['logout'] == '1') {
  $success = 'Vous avez été déconnecté avec succès.';
}

if (isset($_GET['guest_logout']) && $_GET['guest_logout'] == '1') {
  $success = 'Session invité terminée.';
}

if (isset($_COOKIE['remember_token']) && !isset($_SESSION['resident_id'])) {
  $remember_token = $_COOKIE['remember_token'];
  $stmt = $conn->prepare("SELECT id, first_name, email FROM residents WHERE remember_token = ?");
  $stmt->bind_param("s", $remember_token);
  $stmt->execute();
  $result = $stmt->get_result();
  
  if ($user = $result->fetch_assoc()) {
      $_SESSION['resident_id'] = $user['id'];
      $_SESSION['resident_name'] = $user['first_name'];
      $saved_email = $user['email'];
      session_regenerate_id(true);
      header('Location: resident-dashboard.php');
      exit;
  } else {
      $secure = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on';
      $domain = $_SERVER['HTTP_HOST'];
      
      setcookie('remember_token', '', time() - 3600, '/', $domain, $secure, true);
      setcookie('resident_email', '', time() - 3600, '/', $domain, $secure, false);
  }
}

if (isset($_COOKIE['resident_email'])) {
  $saved_email = htmlspecialchars($_COOKIE['resident_email']);
}

if ($_POST) {
  $action = $_POST['action'];
  
  if ($action === 'forgot_password') {
      $email = trim($_POST['email']);
      
      if (empty($email)) {
          $error = 'Email requis';
      } else {
          $stmt = $conn->prepare("SELECT id, first_name FROM residents WHERE email = ?");
          $stmt->bind_param("s", $email);
          $stmt->execute();
          $result = $stmt->get_result();
          
          if ($user = $result->fetch_assoc()) {
              $reset_token = bin2hex(random_bytes(32));
              $expires = date('Y-m-d H:i:s', time() + (1 * 60 * 60));
              
              $stmt = $conn->prepare("UPDATE residents SET reset_token = ?, reset_expires = ? WHERE id = ?");
              $stmt->bind_param("ssi", $reset_token, $expires, $user['id']);
              $stmt->execute();
              
              $reset_link = "https://" . $_SERVER['HTTP_HOST'] . "/reset_password.php?token=" . $reset_token;
              
              $to = $email;
              $subject = "Réinitialisation de votre mot de passe - Vert Chasseur";
              $message = "
              <html>
              <body>
                  <h2>Bonjour " . htmlspecialchars($user['first_name']) . ",</h2>
                  <p>Tu as demandé la réinitialisation de ton mot de passe.</p>
                  <p>Clique sur le lien ci-dessous pour créer un nouveau mot de passe :</p>
                  <p><a href='" . $reset_link . "' style='background-color: #292524; color: white; padding: 12px 24px; text-decoration: none; border-radius: 8px; display: inline-block;'>Réinitialiser mon mot de passe</a></p>
                  <p>Ce lien est valide pendant 1 heure.</p>
                  <p>Si tu n'as pas demandé cette réinitialisation, ignore cet email.</p>
                  <br>
                  <p>L'équipe Vert Chasseur</p>
              </body>
              </html>
              ";
              
              $headers = "MIME-Version: 1.0" . "\r\n";
              $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
              $headers .= 'From: no-reply@imators.systems' . "\r\n";
              
              if (mail($to, $subject, $message, $headers)) {
                  $success = 'Email de réinitialisation envoyé ! Vérifie ta boîte mail.';
              } else {
                  $error = 'Erreur lors de l\'envoi de l\'email';
              }
          } else {
              $success = 'Si un compte existe avec cet email, tu recevras un lien de réinitialisation.';
          }
      }
  }
  elseif ($action === 'login') {
      $email = trim($_POST['email']);
      $password = $_POST['password'];
      
      $stmt = $conn->prepare("SELECT id, password, first_name FROM residents WHERE email = ?");
      $stmt->bind_param("s", $email);
      $stmt->execute();
      $result = $stmt->get_result();
      
      if ($user = $result->fetch_assoc()) {
          if (password_verify($password, $user['password'])) {
              $_SESSION['resident_id'] = $user['id'];
              $_SESSION['resident_name'] = $user['first_name'];
              session_regenerate_id(true);
              
              $remember_token = bin2hex(random_bytes(32));
              $expires = time() + (365 * 24 * 60 * 60);
              
              $stmt = $conn->prepare("UPDATE residents SET remember_token = ? WHERE id = ?");
              $stmt->bind_param("si", $remember_token, $user['id']);
              $stmt->execute();
              
              $secure = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on';
              $domain = $_SERVER['HTTP_HOST'];
              
              setcookie('remember_token', $remember_token, $expires, '/', $domain, $secure, true);
              setcookie('resident_email', $email, $expires, '/', $domain, $secure, false);
              
              header('Location: resident-dashboard.php');
              exit;
          } else {
              $error = 'Email ou mot de passe incorrect';
          }
      } else {
          $error = 'Email ou mot de passe incorrect';
      }
  } 
  elseif ($action === 'register') {
      $first_name = trim($_POST['first_name']);
      $last_name = trim($_POST['last_name']);
      $email = trim($_POST['email']);
      $password = $_POST['password'];
      $address = trim($_POST['address']);
      $captcha_answer = isset($_POST['captcha_answer']) ? (int)$_POST['captcha_answer'] : 0;
      $captcha_expected = isset($_POST['captcha_expected']) ? (int)$_POST['captcha_expected'] : 0;
      $captcha_timestamp = isset($_POST['captcha_timestamp']) ? (int)$_POST['captcha_timestamp'] : 0;
      
      $time_taken = time() - $captcha_timestamp;
      
      if (empty($first_name) || empty($last_name) || empty($email) || empty($password) || empty($address)) {
          $error = 'Tous les champs sont requis';
      } elseif ($captcha_answer !== $captcha_expected) {
          $error = 'Vérification anti-bot incorrecte';
      } elseif ($time_taken < 3) {
          $error = 'Action trop rapide, veuillez réessayer';
      } elseif ($time_taken > 300) {
          $error = 'Session expirée, veuillez recommencer';
      } elseif (strlen($password) < 6) {
          $error = 'Le mot de passe doit contenir au moins 6 caractères';
      } elseif (!isset($_POST['accept_terms'])) {
          $error = 'Tu dois accepter les conditions d\'utilisation et la politique de confidentialité';
      } else {
          $stmt = $conn->prepare("SELECT id FROM residents WHERE email = ?");
          $stmt->bind_param("s", $email);
          $stmt->execute();
          
          if ($stmt->get_result()->num_rows > 0) {
              $error = 'Un compte existe déjà avec cet email';
          } else {
              $hashed_password = password_hash($password, PASSWORD_DEFAULT);
              
              $stmt = $conn->prepare("INSERT INTO residents (first_name, last_name, email, password, address, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
              $stmt->bind_param("sssss", $first_name, $last_name, $email, $hashed_password, $address);
              
              if ($stmt->execute()) {
                  $success = 'Compte créé avec succès ! Tu peux maintenant te connecter.';
              } else {
                  $error = 'Erreur lors de la création du compte';
              }
          }
      }
  }
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
   <meta charset="UTF-8">
   <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
   <title>Compte Résidant - Vert Chasseur</title>
   <link rel="icon" type="image/png" href="logo_new.png">
   <script src="https://cdn.tailwindcss.com"></script>
   <script>
       tailwind.config = {
           darkMode: 'class',
           theme: {
               extend: {}
           }
       }
   </script>
   <link rel="preconnect" href="https://fonts.googleapis.com">
   <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
   <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
   <link href="https://fonts.googleapis.com/css2?family=Princess+Sofia&display=swap" rel="stylesheet">
   <style>
       @media (min-width: 768px) {
           body {
               background-image: url('font-homepage.png');
               background-size: cover;
               background-position: center;
               background-attachment: fixed;
               background-repeat: no-repeat;
           }
       }
       @media (max-width: 767px) {
           body {
               background-color: #ffffff !important;
               background-image: none !important;
           }
           .dark body {
               background-color: #000000 !important;
               background-image: none !important;
           }
       }
       body {
           font-family: 'Inter', sans-serif;
       }
       .txt-chasseur {
           font-family: "Princess Sofia", serif;
           font-weight: 400;
       }
       
       .step-indicator {
           display: flex;
           align-items: center;
           justify-content: center;
           gap: 6px;
           margin-bottom: 32px;
       }
       
       .step-dot {
           width: 8px;
           height: 8px;
           border-radius: 50%;
           transition: all 0.3s ease;
           background: rgba(0, 0, 0, 0.1);
       }
       
       .dark .step-dot {
           background: rgba(255, 255, 255, 0.1);
       }
       
       .step-dot.active {
           width: 24px;
           border-radius: 4px;
           background: #292524;
       }
       
       .dark .step-dot.active {
           background: white;
       }
       
       .step-dot.completed {
           background: #16a34a;
       }
       
       .dark .step-dot.completed {
           background: #22c55e;
       }
       
       .step-content {
           display: none;
           opacity: 0;
           transform: translateX(20px);
       }
       
       .step-content.active {
           display: block;
           animation: slideIn 0.4s ease forwards;
       }
       
       .step-content.slide-out-left {
           animation: slideOutLeft 0.4s ease forwards;
       }
       
       .step-content.slide-out-right {
           animation: slideOutRight 0.4s ease forwards;
       }
       
       @keyframes slideIn {
           from {
               opacity: 0;
               transform: translateX(20px);
           }
           to {
               opacity: 1;
               transform: translateX(0);
           }
       }
       
       @keyframes slideOutLeft {
           from {
               opacity: 1;
               transform: translateX(0);
           }
           to {
               opacity: 0;
               transform: translateX(-20px);
           }
       }
       
       @keyframes slideOutRight {
           from {
               opacity: 1;
               transform: translateX(0);
           }
           to {
               opacity: 0;
               transform: translateX(20px);
           }
       }
       
       .address-results {
           max-height: 200px;
           overflow-y: auto;
       }
       
       .address-item {
           transition: all 0.2s ease;
       }
       
       .address-item:hover {
           background-color: #f5f5f4;
           transform: translateX(4px);
       }
       
       .dark .address-item:hover {
           background-color: #292524;
       }
       
       .checkbox-custom {
           appearance: none;
           width: 1rem;
           height: 1rem;
           border: 2px solid #d1d5db;
           border-radius: 0.25rem;
           background-color: white;
           cursor: pointer;
           position: relative;
           transition: all 0.2s ease;
       }
       
       .checkbox-custom:checked {
           background-color: #292524;
           border-color: #292524;
       }
       
       .checkbox-custom:checked::after {
           content: '';
           position: absolute;
           top: 0px;
           left: 3px;
           width: 4px;
           height: 8px;
           border: 2px solid white;
           border-top: 0;
           border-left: 0;
           transform: rotate(45deg);
       }
       
       .dark .checkbox-custom {
           background-color: #1f2937;
           border-color: #4b5563;
       }
       
       .dark .checkbox-custom:checked {
           background-color: white;
           border-color: white;
       }
       
       .dark .checkbox-custom:checked::after {
           border-color: black;
       }
       
       .logo-light {
           display: block;
       }
       
       .logo-dark {
           display: none;
       }
       
       .dark .logo-light {
           display: none;
       }
       
       .dark .logo-dark {
           display: block;
       }

       @media (max-width: 767px) {
           .mobile-layout {
               display: grid;
               grid-template-rows: 1fr auto;
               min-height: 100vh;
               min-height: -webkit-fill-available;
           }
           
           .mobile-content {
               padding: 2rem 1rem;
               padding-bottom: 100px;
               overflow-y: auto;
           }
           
           .mobile-bottom {
               position: fixed;
               bottom: 0;
               left: 0;
               right: 0;
               background: rgba(255, 255, 255, 0.98);
               backdrop-filter: blur(20px);
               border-top: 1px solid rgba(0, 0, 0, 0.05);
               padding: 1rem;
               z-index: 40;
           }
           
           .dark .mobile-bottom {
               background: rgba(0, 0, 0, 0.98);
               border-top-color: rgba(255, 255, 255, 0.05);
           }
       }

       .spinner {
           border: 2px solid rgba(255, 255, 255, 0.3);
           border-top: 2px solid white;
           border-radius: 50%;
           width: 16px;
           height: 16px;
           animation: spin 0.8s linear infinite;
       }

       .dark .spinner {
           border: 2px solid rgba(0, 0, 0, 0.3);
           border-top: 2px solid black;
       }

       @keyframes spin {
           0% { transform: rotate(0deg); }
           100% { transform: rotate(360deg); }
       }

       .puzzle-container {
           display: grid;
           grid-template-columns: repeat(4, 1fr);
           gap: 8px;
           margin: 24px 0;
       }

       .puzzle-piece {
           aspect-ratio: 1;
           border-radius: 12px;
           border: 2px solid rgba(0, 0, 0, 0.1);
           display: flex;
           align-items: center;
           justify-content: center;
           font-size: 24px;
           font-weight: 600;
           cursor: pointer;
           transition: all 0.3s ease;
           background: white;
           user-select: none;
       }

       .dark .puzzle-piece {
           background: #1f2937;
           border-color: rgba(255, 255, 255, 0.1);
       }

       .puzzle-piece:hover {
           transform: scale(1.05);
           border-color: #292524;
       }

       .dark .puzzle-piece:hover {
           border-color: white;
       }

       .puzzle-piece.selected {
           background: #292524;
           color: white;
           border-color: #292524;
           transform: scale(0.95);
       }

       .dark .puzzle-piece.selected {
           background: white;
           color: black;
           border-color: white;
       }

       .puzzle-piece.correct {
           background: #16a34a;
           color: white;
           border-color: #16a34a;
           pointer-events: none;
           animation: correctPulse 0.5s ease;
       }

       .dark .puzzle-piece.correct {
           background: #22c55e;
           color: black;
           border-color: #22c55e;
       }

       @keyframes correctPulse {
           0%, 100% { transform: scale(1); }
           50% { transform: scale(1.1); }
       }

       .puzzle-instruction {
           text-align: center;
           font-size: 14px;
           color: #57534e;
           margin-bottom: 16px;
           font-weight: 500;
       }

       .dark .puzzle-instruction {
           color: #a8a29e;
       }
   </style>
</head>
<body class="bg-white dark:bg-black transition-colors duration-300">
   <script>
       if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
           document.documentElement.classList.add('dark');
       } else {
           document.documentElement.classList.remove('dark');
       }
       
       window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
           if (!localStorage.theme) {
               if (e.matches) {
                   document.documentElement.classList.add('dark');
               } else {
                   document.documentElement.classList.remove('dark');
               }
           }
       });
   </script>

<div class="min-h-screen flex items-center justify-center p-4 mobile-layout">
   <div class="w-full max-w-md mobile-content">
       <div class="mb-8 text-center">
           <h1 class="text-3xl md:text-4xl font-light text-stone-800 dark:text-white mb-2">
               <span class="txt-chasseur">Bienvenue à Vert Chasseur</span>
           </h1>
           <p class="text-stone-600 dark:text-stone-400 text-xs">Découvre l'application officielle du quartier où le bonheur est à toute heure</p>
       </div>

       <?php if ($error): ?>
       <div class="mb-4 p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-xl">
           <p class="text-red-600 dark:text-red-400 text-sm"><?= htmlspecialchars($error) ?></p>
       </div>
       <?php endif; ?>
       
       <?php if ($success): ?>
       <div class="mb-4 p-3 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-xl">
           <p class="text-green-600 dark:text-green-400 text-sm"><?= htmlspecialchars($success) ?></p>
       </div>
       <?php endif; ?>

       <div class="mb-6 hidden md:block">
           <div class="flex bg-stone-100 dark:bg-stone-800 rounded-xl p-1">
               <button id="login-tab" onclick="toggleForm('login')" class="flex-1 py-2 text-sm font-medium transition-all rounded-lg bg-white dark:bg-black text-stone-900 dark:text-white shadow-sm">
                   Connexion
               </button>
               <button id="register-tab" onclick="toggleForm('register')" class="flex-1 py-2 text-sm font-medium transition-all rounded-lg text-stone-600 dark:text-stone-400">
                   Créer un compte
               </button>
           </div>
       </div>

       <form id="login-form" method="POST" class="space-y-4">
           <input type="hidden" name="action" value="login">
           
           <input type="email" name="email" required placeholder="Email" value="<?= $saved_email ?>"
                  class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-base">
           
           <input type="password" name="password" required placeholder="Mot de passe" 
                  class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-base">
           
           <div class="text-right">
               <button type="button" onclick="openForgotModal()" class="text-stone-600 dark:text-stone-400 hover:text-stone-800 dark:hover:text-stone-200 text-xs transition-colors">
                   Mot de passe oublié ?
               </button>
           </div>
           
           <button type="submit" 
                   class="w-full bg-stone-900 dark:bg-white text-white dark:text-black py-3 rounded-xl font-medium hover:opacity-90 transition-all text-sm">
               Se connecter
           </button>
           
           <div class="text-center pt-4 border-t border-stone-100 dark:border-stone-800">
               <a href="?guest=1" class="inline-flex items-center text-stone-600 dark:text-stone-400 hover:text-stone-800 dark:hover:text-stone-200 transition-colors text-xs">
                   <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                       <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                   </svg>
                   Continuer sans compte
               </a>
           </div>
       </form>

       <form id="register-form" method="POST" class="hidden">
           <input type="hidden" name="action" value="register">
           <input type="hidden" id="captcha-answer" name="captcha_answer" value="">
           <input type="hidden" id="captcha-expected" name="captcha_expected" value="">
           <input type="hidden" id="captcha-timestamp" name="captcha_timestamp" value="">
           
           <div class="step-indicator">
               <div class="step-dot active" id="step-dot-1"></div>
               <div class="step-dot" id="step-dot-2"></div>
               <div class="step-dot" id="step-dot-3"></div>
               <div class="step-dot" id="step-dot-4"></div>
               <div class="step-dot" id="step-dot-5"></div>
           </div>

           <div id="step-1" class="step-content active">
               <div class="mb-6 text-center">
                   <h3 class="text-xl font-semibold text-stone-800 dark:text-white mb-2">Ravi de te rencontrer</h3>
                   <p class="text-stone-600 dark:text-stone-400 text-sm">Commence par nous dire ton nom</p>
               </div>
               
               <div class="space-y-3">
                   <input type="text" id="first_name" name="first_name" required placeholder="Prénom" 
                          class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-base">
                   <input type="text" id="last_name" name="last_name" required placeholder="Nom" 
                          class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-base">
               </div>
               
               <button type="button" onclick="nextStep(2)" class="w-full bg-stone-900 dark:bg-white text-white dark:text-black py-3 rounded-xl font-medium hover:opacity-90 transition-all text-sm mt-6">
                   Continuer
               </button>
           </div>

           <div id="step-2" class="step-content">
               <div class="mb-6 text-center">
                   <h3 class="text-xl font-semibold text-stone-800 dark:text-white mb-2">Ton adresse email</h3>
                   <p class="text-stone-600 dark:text-stone-400 text-sm">Nous l'utiliserons pour sécuriser ton compte</p>
               </div>
               
               <div class="space-y-3">
                   <input type="email" id="email" name="email" required placeholder="exemple@email.com" 
                          class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-base">
               </div>
               
               <div class="flex gap-3 mt-6">
                   <button type="button" onclick="prevStep(1)" class="flex-1 py-3 rounded-xl border border-stone-200 dark:border-stone-700 text-stone-600 dark:text-stone-400 hover:bg-stone-50 dark:hover:bg-stone-800 transition-all text-sm font-medium">
                       Retour
                   </button>
                   <button type="button" onclick="nextStep(3)" class="flex-1 bg-stone-900 dark:bg-white text-white dark:text-black py-3 rounded-xl font-medium hover:opacity-90 transition-all text-sm">
                       Continuer
                   </button>
               </div>
           </div>

           <div id="step-3" class="step-content">
               <div class="mb-6 text-center">
                   <h3 class="text-xl font-semibold text-stone-800 dark:text-white mb-2">Ton adresse</h3>
                   <p class="text-stone-600 dark:text-stone-400 text-sm">Trouve ton domicile à Bruxelles</p>
               </div>
               
               <div class="space-y-3 relative">
                   <input type="text" id="address-input" name="address" required placeholder="Commence à taper ton adresse..." 
                          class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-base">
                   
                   <div id="address-results" class="hidden absolute top-full left-0 right-0 mt-1 bg-white dark:bg-stone-900 border border-stone-200 dark:border-stone-700 rounded-xl shadow-xl z-50 address-results">
                   </div>
               </div>
               
               <div class="flex gap-3 mt-6">
                   <button type="button" onclick="prevStep(2)" class="flex-1 py-3 rounded-xl border border-stone-200 dark:border-stone-700 text-stone-600 dark:text-stone-400 hover:bg-stone-50 dark:hover:bg-stone-800 transition-all text-sm font-medium">
                       Retour
                   </button>
                   <button type="button" onclick="nextStep(4)" class="flex-1 bg-stone-900 dark:bg-white text-white dark:text-black py-3 rounded-xl font-medium hover:opacity-90 transition-all text-sm">
                       Continuer
                   </button>
               </div>
           </div>

           <div id="step-4" class="step-content">
               <div class="mb-6 text-center">
                   <h3 class="text-xl font-semibold text-stone-800 dark:text-white mb-2">Protège ton compte</h3>
                   <p class="text-stone-600 dark:text-stone-400 text-sm">Choisis un mot de passe sécurisé</p>
               </div>
               
               <div class="space-y-4">
                   <input type="password" id="password" name="password" required minlength="6" placeholder="Mot de passe (min. 6 caractères)" 
                          class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-base">
                   
                   <div class="flex items-start space-x-2 py-1">
                       <input type="checkbox" id="accept-terms" name="accept_terms" required class="checkbox-custom mt-0.5 flex-shrink-0">
                       <label for="accept-terms" class="text-xs text-stone-600 dark:text-stone-400 leading-4 cursor-pointer">
                           J'accepte les <a href="https://aktascorp.com/vertchasseur-terms/" target="_blank" class="text-stone-900 dark:text-white hover:underline font-medium">conditions d'utilisation</a> et la <a href="https://aktascorp.com/privacy" target="_blank" class="text-stone-900 dark:text-white hover:underline font-medium">politique de confidentialité</a>
                       </label>
                   </div>
               </div>
               
               <div class="flex gap-3 mt-6">
                   <button type="button" onclick="prevStep(3)" class="flex-1 py-3 rounded-xl border border-stone-200 dark:border-stone-700 text-stone-600 dark:text-stone-400 hover:bg-stone-50 dark:hover:bg-stone-800 transition-all text-sm font-medium">
                       Retour
                   </button>
                   <button type="button" onclick="nextStep(5)" class="flex-1 bg-stone-900 dark:bg-white text-white dark:text-black py-3 rounded-xl font-medium hover:opacity-90 transition-all text-sm">
                       Continuer
                   </button>
               </div>
           </div>

           <div id="step-5" class="step-content">
               <div class="mb-6 text-center">
                   <h3 class="text-xl font-semibold text-stone-800 dark:text-white mb-2">Vérifie que tu es humain</h3>
                   <p class="text-stone-600 dark:text-stone-400 text-sm">Sélectionne les nombres dans l'ordre croissant</p>
               </div>
               
               <div class="puzzle-instruction" id="puzzle-instruction">
                   Tape sur les nombres du plus petit au plus grand
               </div>
               
               <div class="puzzle-container" id="puzzle-container">
               </div>
               
               <div class="flex gap-3 mt-6">
                   <button type="button" onclick="prevStep(4)" class="flex-1 py-3 rounded-xl border border-stone-200 dark:border-stone-700 text-stone-600 dark:text-stone-400 hover:bg-stone-50 dark:hover:bg-stone-800 transition-all text-sm font-medium">
                       Retour
                   </button>
                   <button type="submit" id="final-submit" disabled class="flex-1 bg-stone-900 dark:bg-white text-white dark:text-black py-3 rounded-xl font-medium hover:opacity-90 transition-all text-sm opacity-50 cursor-not-allowed">
                       Créer mon compte
                   </button>
               </div>
           </div>
           
           <div class="text-center pt-4 border-t border-stone-100 dark:border-stone-800 mt-4">
               <a href="?guest=1" class="inline-flex items-center text-stone-600 dark:text-stone-400 hover:text-stone-800 dark:hover:text-stone-200 transition-colors text-xs">
                   <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                       <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                   </svg>
                   Continuer sans compte
               </a>
           </div>
       </form>

       <div class="mt-8 text-center">
           <img src="logo_new.png" alt="Vert Chasseur" class="w-6 h-6 mx-auto opacity-50 logo-light">
           <img src="logo_dark.png" alt="Vert Chasseur" class="w-6 h-6 mx-auto opacity-70 logo-dark">
       </div>
   </div>

   <div class="mobile-bottom md:hidden">
       <div class="flex bg-stone-100 dark:bg-stone-800 rounded-xl p-1">
           <button id="login-tab-mobile" onclick="toggleFormMobile('login')" class="flex-1 py-2.5 text-sm font-medium transition-all rounded-lg bg-white dark:bg-black text-stone-900 dark:text-white shadow-sm">
               Connexion
           </button>
           <button id="register-tab-mobile" onclick="toggleFormMobile('register')" class="flex-1 py-2.5 text-sm font-medium transition-all rounded-lg text-stone-600 dark:text-stone-400">
               Créer un compte
           </button>
       </div>
   </div>
</div>

<div id="forgot-modal" class="fixed inset-0 bg-black/50 hidden items-center justify-center p-4 z-50">
   <div class="bg-white dark:bg-stone-900 rounded-2xl p-6 w-full max-w-sm">
       <div class="mb-4">
           <h2 class="text-xl font-light text-stone-800 dark:text-white mb-1">Mot de passe oublié</h2>
           <p class="text-stone-600 dark:text-stone-400 text-xs">Entre ton email pour recevoir un lien de réinitialisation</p>
       </div>
       
       <form id="forgot-password-form" method="POST" class="space-y-4">
           <input type="hidden" name="action" value="forgot_password">
           
           <input type="email" name="email" required placeholder="Ton email" 
                  class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-base">
           
           <div class="flex gap-3">
               <button type="button" onclick="closeForgotModal()" 
                       class="flex-1 py-3 rounded-xl border border-stone-200 dark:border-stone-700 text-stone-600 dark:text-stone-400 hover:bg-stone-50 dark:hover:bg-stone-800 transition-all text-sm">
                   Annuler
               </button>
               <button type="submit" id="forgot-submit-btn"
                       class="flex-1 bg-stone-900 dark:bg-white text-white dark:text-black py-3 rounded-xl font-medium hover:opacity-90 transition-all text-sm flex items-center justify-center">
                   <span id="forgot-btn-text">Envoyer</span>
                   <div id="forgot-btn-spinner" class="spinner hidden ml-2"></div>
               </button>
           </div>
       </form>
   </div>
</div>

<script>
   let currentStep = 1;
   let puzzleNumbers = [];
   let correctOrder = [];
   let userOrder = [];
   let puzzleTimestamp = 0;

   function toggleForm(type) {
       const loginForm = document.getElementById('login-form');
       const registerForm = document.getElementById('register-form');
       const loginTab = document.getElementById('login-tab');
       const registerTab = document.getElementById('register-tab');
       const loginTabMobile = document.getElementById('login-tab-mobile');
       const registerTabMobile = document.getElementById('register-tab-mobile');
       
       if (type === 'login') {
           loginForm.classList.remove('hidden');
           registerForm.classList.add('hidden');
           
           loginTab.classList.add('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
           loginTab.classList.remove('text-stone-600', 'dark:text-stone-400');
           registerTab.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
           registerTab.classList.add('text-stone-600', 'dark:text-stone-400');
           
           loginTabMobile.classList.add('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
           loginTabMobile.classList.remove('text-stone-600', 'dark:text-stone-400');
           registerTabMobile.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
           registerTabMobile.classList.add('text-stone-600', 'dark:text-stone-400');
       } else {
           loginForm.classList.add('hidden');
           registerForm.classList.remove('hidden');
           resetSteps();
           
           registerTab.classList.add('bg-white','dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
           registerTab.classList.remove('text-stone-600', 'dark:text-stone-400');
           loginTab.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
           loginTab.classList.add('text-stone-600', 'dark:text-stone-400');
           
           registerTabMobile.classList.add('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
           registerTabMobile.classList.remove('text-stone-600', 'dark:text-stone-400');
           loginTabMobile.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
           loginTabMobile.classList.add('text-stone-600', 'dark:text-stone-400');
       }
   }

   function toggleFormMobile(type) {
      const loginForm = document.getElementById('login-form');
      const registerForm = document.getElementById('register-form');
      const loginTab = document.getElementById('login-tab-mobile');
      const registerTab = document.getElementById('register-tab-mobile');
      const loginTabDesktop = document.getElementById('login-tab');
      const registerTabDesktop = document.getElementById('register-tab');
      
      if (type === 'login') {
          loginForm.classList.remove('hidden');
          registerForm.classList.add('hidden');
          
          loginTab.classList.add('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
          loginTab.classList.remove('text-stone-600', 'dark:text-stone-400');
          registerTab.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
          registerTab.classList.add('text-stone-600', 'dark:text-stone-400');
          
          loginTabDesktop.classList.add('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
          loginTabDesktop.classList.remove('text-stone-600', 'dark:text-stone-400');
          registerTabDesktop.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
          registerTabDesktop.classList.add('text-stone-600', 'dark:text-stone-400');
      } else {
          loginForm.classList.add('hidden');
          registerForm.classList.remove('hidden');
          resetSteps();
          
          registerTab.classList.add('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
          registerTab.classList.remove('text-stone-600', 'dark:text-stone-400');
          loginTab.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
          loginTab.classList.add('text-stone-600', 'dark:text-stone-400');
          
          registerTabDesktop.classList.add('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
          registerTabDesktop.classList.remove('text-stone-600', 'dark:text-stone-400');
          loginTabDesktop.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
          loginTabDesktop.classList.add('text-stone-600', 'dark:text-stone-400');
      }
  }

   function nextStep(step) {
       if (!validateStep(currentStep)) {
           return;
       }
       
       const currentStepEl = document.getElementById(`step-${currentStep}`);
       const nextStepEl = document.getElementById(`step-${step}`);
       
       currentStepEl.classList.add('slide-out-left');
       
       setTimeout(() => {
           currentStepEl.classList.remove('active', 'slide-out-left');
           nextStepEl.classList.add('active');
           
           if (step === 5) {
               initPuzzle();
           }
           
           updateStepIndicators(step);
           currentStep = step;
       }, 400);
   }

   function prevStep(step) {
       const currentStepEl = document.getElementById(`step-${currentStep}`);
       const prevStepEl = document.getElementById(`step-${step}`);
       
       currentStepEl.classList.add('slide-out-right');
       
       setTimeout(() => {
           currentStepEl.classList.remove('active', 'slide-out-right');
           prevStepEl.classList.add('active');
           
           updateStepIndicators(step);
           currentStep = step;
       }, 400);
   }

   function updateStepIndicators(step) {
       for (let i = 1; i <= 5; i++) {
           const stepDot = document.getElementById(`step-dot-${i}`);
           
           stepDot.classList.remove('active', 'completed');
           
           if (i < step) {
               stepDot.classList.add('completed');
           } else if (i === step) {
               stepDot.classList.add('active');
           }
       }
   }

   function validateStep(step) {
       if (step === 1) {
           const firstName = document.getElementById('first_name').value.trim();
           const lastName = document.getElementById('last_name').value.trim();
           
           if (!firstName || !lastName) {
               alert('Merci de remplir ton prénom et nom');
               return false;
           }
       } else if (step === 2) {
           const email = document.getElementById('email').value.trim();
           
           if (!email || !email.includes('@')) {
               alert('Merci de saisir un email valide');
               return false;
           }
       } else if (step === 3) {
           const address = document.getElementById('address-input').value.trim();
           
           if (!address) {
               alert('Merci de saisir ton adresse');
               return false;
           }
       } else if (step === 4) {
           const password = document.getElementById('password').value;
           const acceptTerms = document.getElementById('accept-terms').checked;
           
           if (password.length < 6) {
               alert('Le mot de passe doit contenir au moins 6 caractères');
               return false;
           }
           
           if (!acceptTerms) {
               alert('Tu dois accepter les conditions d\'utilisation');
               return false;
           }
       }
       
       return true;
   }

   function resetSteps() {
       currentStep = 1;
       
       for (let i = 1; i <= 5; i++) {
           const stepEl = document.getElementById(`step-${i}`);
           stepEl.classList.remove('active', 'slide-out-left', 'slide-out-right');
       }
       
       document.getElementById('step-1').classList.add('active');
       updateStepIndicators(1);
   }

   function initPuzzle() {
       puzzleTimestamp = Math.floor(Date.now() / 1000);
       document.getElementById('captcha-timestamp').value = puzzleTimestamp;
       
       puzzleNumbers = [];
       while (puzzleNumbers.length < 8) {
           const num = Math.floor(Math.random() * 50) + 1;
           if (!puzzleNumbers.includes(num)) {
               puzzleNumbers.push(num);
           }
       }
       
       correctOrder = [...puzzleNumbers].sort((a, b) => a - b);
       userOrder = [];
       
       const container = document.getElementById('puzzle-container');
       container.innerHTML = '';
       
       puzzleNumbers.forEach((num, index) => {
           const piece = document.createElement('div');
           piece.className = 'puzzle-piece';
           piece.textContent = num;
           piece.dataset.value = num;
           piece.dataset.index = index;
           piece.onclick = () => selectPuzzlePiece(piece, num);
           container.appendChild(piece);
       });
       
       const expectedSum = correctOrder.reduce((sum, num) => sum + num, 0);
       document.getElementById('captcha-expected').value = expectedSum;
       
       document.getElementById('final-submit').disabled = true;
       document.getElementById('final-submit').classList.add('opacity-50', 'cursor-not-allowed');
   }

   function selectPuzzlePiece(element, value) {
       if (element.classList.contains('correct')) {
           return;
       }
       
       const expectedNext = correctOrder[userOrder.length];
       
       if (value === expectedNext) {
           element.classList.add('correct');
           userOrder.push(value);
           
           if (userOrder.length === correctOrder.length) {
               setTimeout(() => {
                   const sum = userOrder.reduce((sum, num) => sum + num, 0);
                   document.getElementById('captcha-answer').value = sum;
                   
                   document.getElementById('final-submit').disabled = false;
                   document.getElementById('final-submit').classList.remove('opacity-50', 'cursor-not-allowed');
                   
                   document.getElementById('puzzle-instruction').textContent = '✓ Vérification réussie !';
                   document.getElementById('puzzle-instruction').style.color = '#16a34a';
               }, 300);
           }
       } else {
           element.classList.add('selected');
           setTimeout(() => {
               element.classList.remove('selected');
           }, 500);
           
           const allPieces = document.querySelectorAll('.puzzle-piece');
           allPieces.forEach(p => {
               p.classList.remove('correct');
           });
           
           userOrder = [];
           document.getElementById('puzzle-instruction').textContent = 'Oups ! Recommence du début';
           document.getElementById('puzzle-instruction').style.color = '#dc2626';
           
           setTimeout(() => {
               document.getElementById('puzzle-instruction').textContent = 'Tape sur les nombres du plus petit au plus grand';
               document.getElementById('puzzle-instruction').style.color = '';
           }, 2000);
       }
   }

   function openForgotModal() {
       document.getElementById('forgot-modal').classList.remove('hidden');
       document.getElementById('forgot-modal').classList.add('flex');
   }

   function closeForgotModal() {
       document.getElementById('forgot-modal').classList.add('hidden');
       document.getElementById('forgot-modal').classList.remove('flex');
   }

   document.getElementById('forgot-password-form').addEventListener('submit', function() {
       const submitBtn = document.getElementById('forgot-submit-btn');
       const btnText = document.getElementById('forgot-btn-text');
       const btnSpinner = document.getElementById('forgot-btn-spinner');
       
       submitBtn.disabled = true;
       btnText.textContent = 'Envoi en cours';
       btnSpinner.classList.remove('hidden');
   });

   let searchTimeout;

   document.addEventListener('DOMContentLoaded', function() {
       const addressInput = document.getElementById('address-input');
       const addressResults = document.getElementById('address-results');

       if (addressInput) {
           addressInput.addEventListener('input', function() {
               const query = this.value.trim();
               
               clearTimeout(searchTimeout);
               
               if (query.length < 3) {
                   addressResults.classList.add('hidden');
                   return;
               }

               searchTimeout = setTimeout(() => {
                   searchAddresses(query);
               }, 500);
           });

           document.addEventListener('click', function(e) {
               if (!addressInput.contains(e.target) && !addressResults.contains(e.target)) {
                   addressResults.classList.add('hidden');
               }
           });
       }
   });

   async function searchAddresses(query) {
       const addressResults = document.getElementById('address-results');
       
       try {
           addressResults.innerHTML = '<div class="p-3 text-center text-stone-500 dark:text-stone-400 text-xs">Recherche...</div>';
           addressResults.classList.remove('hidden');

           const searchQuery = `${query}, Brussels, Belgium`;
           const response = await fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(searchQuery)}&countrycodes=be&limit=6&addressdetails=1`);
           
           if (!response.ok) {
               throw new Error('Erreur de recherche');
           }

           const data = await response.json();
           
           if (data && data.length > 0) {
               const brusselsResults = data.filter(item => 
                   item.display_name.toLowerCase().includes('brussels') || 
                   item.display_name.toLowerCase().includes('bruxelles') ||
                   item.display_name.toLowerCase().includes('brussel') ||
                   item.display_name.toLowerCase().includes('uccle') ||
                   item.display_name.toLowerCase().includes('ixelles') ||
                   item.display_name.toLowerCase().includes('schaerbeek') ||
                   item.display_name.toLowerCase().includes('etterbeek') ||
                   item.display_name.toLowerCase().includes('saint-gilles') ||
                   item.display_name.toLowerCase().includes('forest') ||
                   item.display_name.toLowerCase().includes('anderlecht') ||
                   item.display_name.toLowerCase().includes('molenbeek') ||
                   item.display_name.toLowerCase().includes('woluwe') ||
                   item.display_name.toLowerCase().includes('auderghem')
               );
               
               if (brusselsResults.length > 0) {
                   addressResults.innerHTML = brusselsResults.map(item => {
                       const displayName = item.display_name.split(',').slice(0, 2).join(',');
                       return `<div class="address-item p-3 cursor-pointer text-stone-900 dark:text-white text-xs border-b border-stone-100 dark:border-stone-800 last:border-b-0" onclick="selectAddress('${displayName.replace(/'/g, "\\'")}')">
                           <div class="font-medium">${displayName}</div>
                       </div>`
                   }).join('');
               } else {
                   addressResults.innerHTML = '<div class="p-3 text-center text-amber-600 dark:text-amber-400 text-xs">Aucune adresse à Bruxelles</div>';
               }
           } else {
               addressResults.innerHTML = '<div class="p-3 text-center text-stone-500 dark:text-stone-400 text-xs">Aucune adresse trouvée</div>';
           }
       } catch (error) {
           addressResults.innerHTML = '<div class="p-3 text-center text-red-500 text-xs">Erreur lors de la recherche</div>';
       }
   }

   function selectAddress(address) {
       document.getElementById('address-input').value = address;
       document.getElementById('address-results').classList.add('hidden');
   }
</script>

</body>
</html>