<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Livraison Express - Vert Chasseur</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Princess+Sofia&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {
                    colors: {
                        dark: {
                            100: '#1e1e1e',
                            200: '#252525',
                            300: '#2d2d2d',
                            400: '#353535',
                            500: '#3a3a3a',
                        }
                    }
                }
            }
        }
    </script>
    <style>
        body {
            font-family: 'Poppins', sans-serif;
            letter-spacing: -0.01em;
        }
        .txt-chasseur {
            font-family: "Princess Sofia", serif;
            font-weight: 400;
            font-style: normal;
        }
        .step-card {
            transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            transform: translateX(100%);
            opacity: 0;
        }
        .step-card.active {
            transform: translateX(0);
            opacity: 1;
        }
        .step-card.prev {
            transform: translateX(-100%);
        }
        .progress-dot {
            transition: all 0.3s ease;
        }
        .progress-dot.active {
            background: black;
            transform: scale(1.2);
        }
        @media (prefers-color-scheme: dark) {
            body { background: black; color: white; }
            .bg-white { background: black !important; }
            .text-black { color: white !important; }
            .text-gray-600 { color: #9ca3af !important; }
            .text-gray-800 { color: white !important; }
            .text-gray-700 { color: #d1d5db !important; }
            .text-gray-500 { color: #6b7280 !important; }
            .border-gray-200 { border-color: #374151 !important; }
            .border-gray-300 { border-color: #4b5563 !important; }
            .bg-gray-50 { background: #111827 !important; }
            .bg-gray-100 { background: #1f2937 !important; }
            .hover\\:bg-gray-50:hover { background: #374151 !important; }
            .hover\\:bg-gray-100:hover { background: #374151 !important; }
            input, select { background: #1f2937 !important; color: white !important; border-color: #374151 !important; }
            .bg-green-50 { background: #064e3b !important; }
            .bg-red-50 { background: #7f1d1d !important; }
            .progress-dot.active { background: white !important; }
        }
    </style>
</head>
<body class="bg-white dark:bg-black text-black dark:text-white transition-colors duration-300">
    <script>
        if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            document.documentElement.classList.add('dark');
        }
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', e => {
            if (e.matches) {
                document.documentElement.classList.add('dark');
            } else {
                document.documentElement.classList.remove('dark');
            }
        });
    </script>

    <main class="container mx-auto px-6 py-12 max-w-4xl">
        <header class="text-center mb-12">
            <p class="text-gray-500 dark:text-gray-400 text-sm tracking-wider uppercase mb-3">Uccle · Bruxelles</p>
            <h1 class="text-4xl md:text-5xl font-light mb-5 text-black dark:text-white">
                <span class="txt-chasseur">Livraison Express</span>
            </h1>
            <p class="text-lg text-gray-600 dark:text-gray-300 leading-relaxed max-w-2xl mx-auto">
                Commande dans ton quartier, livré en 30 minutes
            </p>
        </header>

        <div class="flex justify-center items-center space-x-4 mb-12">
            <div id="progress-1" class="progress-dot w-3 h-3 rounded-full bg-black dark:bg-white active"></div>
            <div class="w-8 h-0.5 bg-gray-200 dark:bg-gray-700"></div>
            <div id="progress-2" class="progress-dot w-3 h-3 rounded-full bg-gray-200 dark:bg-gray-700"></div>
            <div class="w-8 h-0.5 bg-gray-200 dark:bg-gray-700"></div>
            <div id="progress-3" class="progress-dot w-3 h-3 rounded-full bg-gray-200 dark:bg-gray-700"></div>
            <div class="w-8 h-0.5 bg-gray-200 dark:bg-gray-700"></div>
            <div id="progress-4" class="progress-dot w-3 h-3 rounded-full bg-gray-200 dark:bg-gray-700"></div>
            <div class="w-8 h-0.5 bg-gray-200 dark:bg-gray-700"></div>
            <div id="progress-5" class="progress-dot w-3 h-3 rounded-full bg-gray-200 dark:bg-gray-700"></div>
        </div>

        <div class="relative min-h-[500px] mb-20">
            <div id="step-1" class="step-card active absolute inset-0">
                <div class="bg-white dark:bg-dark-200 rounded-3xl p-8 shadow-lg border border-gray-200 dark:border-gray-700">
                    <div class="text-center mb-8">
                        <div class="w-16 h-16 bg-black dark:bg-white rounded-2xl mx-auto mb-4 flex items-center justify-center">
                            <i class="fas fa-clock text-white dark:text-black text-2xl"></i>
                        </div>
                        <h2 class="text-2xl font-medium text-black dark:text-white mb-2">Quand veux-tu être livré ?</h2>
                        <p class="text-gray-600 dark:text-gray-300">Choisis ton créneau de livraison</p>
                    </div>
                    
                    <div class="space-y-6 max-w-md mx-auto">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">Date de livraison</label>
                            <input type="date" id="delivery-date" class="w-full p-4 border border-gray-200 dark:border-gray-700 rounded-2xl bg-white dark:bg-dark-300 text-black dark:text-white focus:ring-2 focus:ring-black focus:border-transparent transition-all">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">Créneau horaire</label>
                            <select id="time-slot" class="w-full p-4 border border-gray-200 dark:border-gray-700 rounded-2xl bg-white dark:bg-dark-300 text-black dark:text-white focus:ring-2 focus:ring-black focus:border-transparent transition-all">
                                <option value="">Sélectionner un créneau</option>
                            </select>
                        </div>
                        
                        <div class="bg-gray-50 dark:bg-dark-300 p-4 rounded-2xl border border-gray-200 dark:border-gray-700">
                            <h3 class="font-medium text-black dark:text-white mb-2">📅 Nos horaires</h3>
                            <ul class="text-sm text-gray-600 dark:text-gray-300 space-y-1">
                                <li>• Lundi-Vendredi : 12h40-13h40, 15h20-17h00</li>
                                <li>• Vendredi soir : jusqu'à 19h00</li>
                                <li>• Weekend : 14h20-17h50</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>

            <div id="step-2" class="step-card absolute inset-0">
                <div class="bg-white dark:bg-dark-200 rounded-3xl p-8 shadow-lg border border-gray-200 dark:border-gray-700">
                    <div class="text-center mb-8">
                        <div class="w-16 h-16 bg-black dark:bg-white rounded-2xl mx-auto mb-4 flex items-center justify-center">
                            <i class="fas fa-map-marker-alt text-white dark:text-black text-2xl"></i>
                        </div>
                        <h2 class="text-2xl font-medium text-black dark:text-white mb-2">Où te livrer ?</h2>
                        <p class="text-gray-600 dark:text-gray-300">Zones disponibles : Vert Chasseur et Fort Jaco</p>
                    </div>
                    
                    <div class="space-y-6 max-w-md mx-auto">
                        <button onclick="useLocation()" class="w-full p-4 bg-black dark:bg-white text-white dark:text-black rounded-2xl font-medium hover:bg-gray-800 dark:hover:bg-gray-200 transition-all">
                            <i class="fas fa-location-arrow mr-2"></i>
                            Utiliser ma position actuelle
                        </button>
                        
                        <div class="text-center text-gray-500 dark:text-gray-400">ou</div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">Saisir ton adresse</label>
                            <input type="text" 
                                   id="address" 
                                   placeholder="Ex: Chaussée d'Alsemberg 123, Uccle"
                                   class="w-full p-4 border border-gray-200 dark:border-gray-700 rounded-2xl bg-white dark:bg-dark-300 text-black dark:text-white focus:ring-2 focus:ring-black focus:border-transparent transition-all">
                            <button onclick="validateAddress()" class="w-full mt-3 p-3 bg-gray-100 dark:bg-gray-700 text-black dark:text-white rounded-xl font-medium hover:bg-gray-200 dark:hover:bg-gray-600 transition-all">
                                <i class="fas fa-check mr-2"></i>Vérifier cette adresse
                            </button>
                        </div>
                        
                        <div id="zone-status" class="hidden p-4 rounded-2xl text-center font-medium"></div>
                    </div>
                </div>
            </div>

            <div id="step-3" class="step-card absolute inset-0">
                <div class="bg-white dark:bg-dark-200 rounded-3xl p-8 shadow-lg border border-gray-200 dark:border-gray-700">
                    <div class="text-center mb-8">
                        <div class="w-16 h-16 bg-black dark:bg-white rounded-2xl mx-auto mb-4 flex items-center justify-center">
                            <i class="fas fa-store text-white dark:text-black text-2xl"></i>
                        </div>
                        <h2 class="text-2xl font-medium text-black dark:text-white mb-2">Choisis ton commerce</h2>
                        <p class="text-gray-600 dark:text-gray-300">Sélectionne où tu veux commander</p>
                    </div>
                    
                    <div class="grid gap-6 max-w-3xl mx-auto md:grid-cols-2">
                        <div class="commerce-card bg-white dark:bg-dark-300 rounded-2xl p-6 border-2 border-gray-200 dark:border-gray-700 cursor-pointer transition-all hover:shadow-lg" onclick="selectCommerce('carrefour')">
                            <div class="flex items-center space-x-4 mb-4">
                                <div class="w-14 h-14 bg-gray-100 dark:bg-gray-800 rounded-2xl flex items-center justify-center">
                                    <i class="fas fa-shopping-cart text-black dark:text-white text-xl"></i>
                                </div>
                                <div>
                                    <h3 class="font-semibold text-black dark:text-white text-lg">Carrefour Market</h3>
                                    <p class="text-sm text-gray-600 dark:text-gray-400">Supermarché</p>
                                </div>
                            </div>
                            <div class="space-y-2 text-sm">
                                <p class="text-gray-600 dark:text-gray-400"><i class="fas fa-map-marker-alt mr-2"></i>Vert Chasseur</p>
                                <p class="text-gray-600 dark:text-gray-400"><i class="fas fa-clock mr-2"></i>Livraison en 25-30min</p>
                                <p class="text-gray-600 dark:text-gray-400"><i class="fas fa-box mr-2"></i>Alimentaire & Courses</p>
                            </div>
                        </div>
                        
                        <div class="commerce-card bg-white dark:bg-dark-300 rounded-2xl p-6 border-2 border-gray-200 dark:border-gray-700 cursor-pointer transition-all hover:shadow-lg" onclick="selectCommerce('pharmacie')">
                            <div class="flex items-center space-x-4 mb-4">
                                <div class="w-14 h-14 bg-gray-100 dark:bg-gray-800 rounded-2xl flex items-center justify-center">
                                    <i class="fas fa-pills text-black dark:text-white text-xl"></i>
                                </div>
                                <div>
                                    <h3 class="font-semibold text-black dark:text-white text-lg">Pharmacie Plus</h3>
                                    <p class="text-sm text-gray-600 dark:text-gray-400">Pharmacie & Parapharmacie</p>
                                </div>
                            </div>
                            <div class="space-y-2 text-sm">
                                <p class="text-gray-600 dark:text-gray-400"><i class="fas fa-map-marker-alt mr-2"></i>Fort Jaco</p>
                                <p class="text-gray-600 dark:text-gray-400"><i class="fas fa-clock mr-2"></i>Livraison en 15-20min</p>
                                <p class="text-gray-600 dark:text-gray-400"><i class="fas fa-heartbeat mr-2"></i>Santé & Bien-être</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div id="step-4" class="step-card absolute inset-0">
                <div class="bg-white dark:bg-dark-200 rounded-3xl p-8 shadow-lg border border-gray-200 dark:border-gray-700">
                    <div class="text-center mb-8">
                        <div class="w-16 h-16 bg-black dark:bg-white rounded-2xl mx-auto mb-4 flex items-center justify-center">
                            <i class="fas fa-shopping-bag text-white dark:text-black text-2xl"></i>
                        </div>
                        <h2 class="text-2xl font-medium text-black dark:text-white mb-2" id="commerce-title">Que veux-tu commander ?</h2>
                        <p class="text-gray-600 dark:text-gray-300" id="commerce-subtitle">Sélectionne tes articles</p>
                    </div>
                    
                    <div class="max-w-4xl mx-auto">
                        <div id="catalog" class="grid gap-4 md:grid-cols-2 lg:grid-cols-3 max-h-64 overflow-y-auto mb-6"></div>
                        
                        <div id="cart-summary" class="bg-gray-50 dark:bg-dark-300 p-6 rounded-2xl border border-gray-200 dark:border-gray-700">
                            <div class="flex justify-between items-center">
                                <div>
                                    <h3 class="font-semibold text-black dark:text-white">Ton panier</h3>
                                    <p id="cart-count" class="text-sm text-gray-600 dark:text-gray-400">0 articles</p>
                                </div>
                                <div class="text-right">
                                    <p class="text-lg font-bold text-black dark:text-white" id="cart-total">0,00€</p>
                                    <p class="text-xs text-gray-500 dark:text-gray-400">+ frais de livraison</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div id="step-5" class="step-card absolute inset-0">
                <div class="bg-white dark:bg-dark-200 rounded-3xl p-8 shadow-lg border border-gray-200 dark:border-gray-700 h-full flex items-center justify-center">
                    <div class="text-center">
                        <div class="w-20 h-20 border-4 border-black dark:border-white border-t-transparent rounded-full animate-spin mx-auto mb-6"></div>
                        <h2 class="text-2xl font-medium text-black dark:text-white mb-4">Recherche d'un livreur...</h2>
                        <p class="text-gray-600 dark:text-gray-300 mb-2">Nous trouvons le meilleur livreur disponible pour toi</p>
                        <div class="flex justify-center space-x-1 mt-4">
                            <div class="w-2 h-2 bg-black dark:bg-white rounded-full animate-bounce" style="animation-delay: 0ms"></div>
                            <div class="w-2 h-2 bg-black dark:bg-white rounded-full animate-bounce" style="animation-delay: 150ms"></div>
                            <div class="w-2 h-2 bg-black dark:bg-white rounded-full animate-bounce" style="animation-delay: 300ms"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="flex justify-center space-x-4">
            <button onclick="goBack()" id="back-btn" class="hidden px-6 py-3 border-2 border-gray-300 dark:border-gray-600 text-gray-700 dark:text-gray-300 rounded-2xl font-medium hover:border-gray-400 dark:hover:border-gray-500 transition-all">
                <i class="fas fa-arrow-left mr-2"></i>Retour
            </button>
            <button onclick="nextStep()" id="next-btn" class="px-8 py-3 bg-black dark:bg-white text-white dark:text-black rounded-2xl font-medium hover:bg-gray-800 dark:hover:bg-gray-200 transition-all">
                Continuer <i class="fas fa-arrow-right ml-2"></i>
            </button>
            <button onclick="finalizeOrder()" id="finalize-btn" class="hidden px-8 py-3 bg-green-600 text-white rounded-2xl font-medium hover:bg-green-700 transition-all">
                <i class="fas fa-phone mr-2"></i>Finaliser la commande
            </button>
        </div>
    </main>

    <footer class="w-full py-8 mt-16 border-t border-gray-200 dark:border-gray-700">
        <p class="text-xs text-gray-500 dark:text-gray-400 text-center max-w-3xl mx-auto px-4">
            © 2025 Vert Chasseur · Livraison Express · 
            <a href="https://aktascorp.com" class="underline hover:text-gray-600 dark:hover:text-gray-300">aktascorp</a> member
        </p>
    </footer>

    <script>
        let currentStep = 1;
        let selectedCommerce = null;
        let addressValid = false;
        let cart = [];

        const deliveryZones = ['Vert Chasseur', 'Fort Jaco'];

        const commerceCatalogs = {
            carrefour: [
                { id: 1, name: "Pain de mie complet", price: 2.85, category: "Boulangerie", icon: "🍞" },
                { id: 2, name: "Lait demi-écrémé 1L", price: 1.35, category: "Produits frais", icon: "🥛" },
                { id: 3, name: "Œufs bio x12", price: 4.20, category: "Produits frais", icon: "🥚" },
                { id: 4, name: "Bananes bio 1kg", price: 2.65, category: "Fruits", icon: "🍌" },
                { id: 5, name: "Pommes Gala 1kg", price: 3.45, category: "Fruits", icon: "🍎" },
                { id: 6, name: "Coca-Cola 1.5L", price: 2.95, category: "Boissons", icon: "🥤" },
                { id: 7, name: "Pâtes Barilla 500g", price: 1.75, category: "Épicerie", icon: "🍝" },
                { id: 8, name: "Riz basmati 1kg", price: 3.20, category: "Épicerie", icon: "🍚" },
                { id: 9, name: "Yaourt nature x8", price: 3.80, category: "Produits frais", icon: "🥛" },
                { id: 10, name: "Fromage râpé 200g", price: 4.15, category: "Produits frais", icon: "🧀" },
                { id: 11, name: "Jambon de Paris 4tr", price: 3.95, category: "Charcuterie", icon: "🥓" },
                { id: 12, name: "Eau Evian 6x1.5L", price: 4.85, category: "Boissons", icon: "💧" }
            ],
            pharmacie: [
                { id: 21, name: "Paracétamol 1g", price: 3.75, category: "Médicaments", icon: "💊" },
                { id: 22, name: "Vitamine C effervescent", price: 9.50, category: "Vitamines", icon: "🍊" },
                { id: 23, name: "Pansements assortis", price: 4.85, category: "Soins", icon: "🩹" },
                { id: 24, name: "Dentifrice Sensodyne", price: 4.20, category: "Hygiène", icon: "🦷" },
                { id: 25, name: "Shampoing L'Oréal", price: 7.95, category: "Hygiène", icon: "🧴" },
                { id: 26, name: "Crème hydratante Nivea", price: 8.50, category: "Cosmétiques", icon: "🧴" },
                { id: 27, name: "Spray gorge", price: 6.25, category: "Médicaments", icon: "💊" },
                { id: 28, name: "Compléments magnésium", price: 12.90, category: "Vitamines", icon: "💊" }
            ]
        };

        document.addEventListener('DOMContentLoaded', function() {
            initializeStep1();
            updateUI();
        });

        function updateProgress(step) {
            for(let i = 1; i <= 5; i++) {
                const dot = document.getElementById(`progress-${i}`);
                if(i <= step) {
                    dot.classList.add('active');
                    dot.classList.remove('bg-gray-200', 'dark:bg-gray-700');
                    dot.classList.add('bg-black', 'dark:bg-white');
                } else {
                    dot.classList.remove('active', 'bg-black', 'dark:bg-white');
                    dot.classList.add('bg-gray-200', 'dark:bg-gray-700');
                }
            }
        }

        function updateUI() {
            const backBtn = document.getElementById('back-btn');
            const nextBtn = document.getElementById('next-btn');
            const finalizeBtn = document.getElementById('finalize-btn');
            
            if (currentStep > 1) {
                backBtn.classList.remove('hidden');
            } else {
                backBtn.classList.add('hidden');
            }
            
            let isValid = false;
            let buttonText = 'Continuer';
            
            switch(currentStep) {
                case 1:
                    isValid = validateStep1();
                    break;
                case 2:
                    isValid = addressValid;
                    break;
                case 3:
                    isValid = selectedCommerce !== null;
                    break;
                case 4:
                    isValid = cart.length > 0;
                    buttonText = 'Commander';
                    break;
                case 5:
                    nextBtn.classList.add('hidden');
                    finalizeBtn.classList.remove('hidden');
                    return;
            }
            
            nextBtn.classList.remove('hidden');
            finalizeBtn.classList.add('hidden');
            nextBtn.disabled = !isValid;
            nextBtn.innerHTML = `${buttonText} <i class="fas fa-arrow-right ml-2"></i>`;
            
            if (isValid) {
                nextBtn.className = 'px-8 py-3 bg-black dark:bg-white text-white dark:text-black rounded-2xl font-medium hover:bg-gray-800 dark:hover:bg-gray-200 transition-all';
            } else {
                nextBtn.className = 'px-8 py-3 bg-gray-300 dark:bg-gray-600 text-gray-500 dark:text-gray-400 rounded-2xl font-medium cursor-not-allowed';
            }
        }

        function showStep(step) {
            document.getElementById(`step-${currentStep}`).classList.remove('active');
            document.getElementById(`step-${currentStep}`).classList.add('prev');
            
            setTimeout(() => {
                document.getElementById(`step-${step}`).classList.add('active');
                document.getElementById(`step-${step}`).classList.remove('prev');
                
                if (step === 4) {
                    setTimeout(() => showCatalog(), 200);
                } else if (step === 5) {
                    setTimeout(() => searchDeliveryPerson(), 200);
                }
            }, 200);
            
            currentStep = step;
            updateProgress(step);
            updateUI();
        }

        function nextStep() {
            if (currentStep < 5) {
                showStep(currentStep + 1);
            }
        }

        function goBack() {
            if (currentStep > 1) {
                showStep(currentStep - 1);
            }
        }

        function initializeStep1() {
            const dateInput = document.getElementById('delivery-date');
            const timeSlot = document.getElementById('time-slot');
            const today = new Date();
            
            dateInput.min = today.toISOString().split('T')[0];
            dateInput.value = today.toISOString().split('T')[0];
            
            updateTimeSlots();
            
            dateInput.addEventListener('change', () => {
                updateTimeSlots();
                updateUI();
            });
            
            timeSlot.addEventListener('change', updateUI);
        }

        function updateTimeSlots() {
            const dateInput = document.getElementById('delivery-date');
            const timeSlot = document.getElementById('time-slot');
            const selectedDate = new Date(dateInput.value);
            const dayOfWeek = selectedDate.getDay();
            
            timeSlot.innerHTML = '<option value="">Sélectionner un créneau</option>';
            
            if (dayOfWeek >= 1 && dayOfWeek <= 5) {
                timeSlot.innerHTML += '<option value="12:40-13:40">🕐 12h40 - 13h40 (Pause déjeuner)</option>';
                timeSlot.innerHTML += '<option value="15:20-17:00">🕒 15h20 - 17h00 (Après-midi)</option>';
                if (dayOfWeek === 5) {
                    timeSlot.innerHTML += '<option value="17:00-19:00">🌆 17h00 - 19h00 (Vendredi soir)</option>';
               }
           } else {
               timeSlot.innerHTML += '<option value="14:20-17:50">🌞 14h20 - 17h50 (Weekend)</option>';
           }
       }

       function validateStep1() {
           return document.getElementById('delivery-date').value && document.getElementById('time-slot').value;
       }

       function useLocation() {
           if (navigator.geolocation) {
               navigator.geolocation.getCurrentPosition(function(position) {
                   document.getElementById('address').value = 'Ma position GPS';
                   validateAddressInZone();
               }, function(error) {
                   alert('Impossible d\'obtenir ta position.');
               });
           } else {
               alert('Géolocalisation non supportée.');
           }
       }

       function validateAddress() {
           const address = document.getElementById('address').value.trim();
           const statusDiv = document.getElementById('zone-status');
           
           if (!address) {
               statusDiv.classList.add('hidden');
               addressValid = false;
               updateUI();
               return;
           }
           
           const isInZone = address.toLowerCase().includes('vert chasseur') || 
                          address.toLowerCase().includes('fort jaco') || 
                          address.toLowerCase().includes('uccle') ||
                          address.toLowerCase().includes('alsemberg') ||
                          address.toLowerCase().includes('position gps');
           
           if (isInZone) {
               statusDiv.className = 'p-4 rounded-2xl text-center font-medium bg-green-50 dark:bg-green-900 text-green-800 dark:text-green-200 border border-green-200 dark:border-green-700';
               statusDiv.innerHTML = `
                   <div class="flex items-center justify-center space-x-2">
                       <i class="fas fa-check-circle"></i>
                       <span>✅ Adresse dans notre zone de livraison</span>
                   </div>
               `;
               addressValid = true;
           } else {
               statusDiv.className = 'p-4 rounded-2xl text-center font-medium bg-red-50 dark:bg-red-900 text-red-800 dark:text-red-200 border border-red-200 dark:border-red-700';
               statusDiv.innerHTML = `
                   <div class="flex items-center justify-center space-x-2">
                       <i class="fas fa-times-circle"></i>
                       <span>❌ Adresse hors zone de livraison</span>
                   </div>
               `;
               addressValid = false;
           }
           
           statusDiv.classList.remove('hidden');
           updateUI();
       }

       function selectCommerce(commerceId) {
           document.querySelectorAll('.commerce-card').forEach(card => {
               card.classList.remove('border-black', 'dark:border-white', 'bg-gray-50', 'dark:bg-dark-400');
               card.classList.add('border-gray-200', 'dark:border-gray-700');
           });
           
           event.currentTarget.classList.remove('border-gray-200', 'dark:border-gray-700');
           event.currentTarget.classList.add('border-black', 'dark:border-white', 'bg-gray-50', 'dark:bg-dark-400');
           
           selectedCommerce = commerceId;
           updateUI();
       }

       function showCatalog() {
           const catalogDiv = document.getElementById('catalog');
           const titleDiv = document.getElementById('commerce-title');
           const subtitleDiv = document.getElementById('commerce-subtitle');
           
           if (selectedCommerce === 'carrefour') {
               titleDiv.textContent = 'Carrefour Market';
               subtitleDiv.textContent = 'Sélectionne tes courses';
           } else {
               titleDiv.textContent = 'Pharmacie Plus';
               subtitleDiv.textContent = 'Sélectionne tes articles';
           }
           
           const products = commerceCatalogs[selectedCommerce];
           let html = '';
           
           products.forEach(product => {
               const quantity = cart.find(item => item.id === product.id)?.quantity || 0;
               
               html += `
                   <div class="bg-white dark:bg-dark-300 p-4 rounded-2xl border border-gray-200 dark:border-gray-700 transition-all hover:shadow-md">
                       <div class="flex items-center justify-between mb-3">
                           <div class="flex items-center space-x-3">
                               <div class="text-2xl">${product.icon}</div>
                               <div class="flex-1">
                                   <h4 class="font-medium text-black dark:text-white text-sm">${product.name}</h4>
                                   <p class="text-xs text-gray-500 dark:text-gray-400">${product.category}</p>
                               </div>
                           </div>
                           <div class="text-right">
                               <p class="font-bold text-black dark:text-white">${product.price.toFixed(2)}€</p>
                           </div>
                       </div>
                       <div class="flex items-center justify-between">
                           <div class="flex items-center space-x-2">
                               <button onclick="removeFromCart(${product.id})" class="w-8 h-8 bg-gray-100 dark:bg-gray-700 text-black dark:text-white rounded-full flex items-center justify-center text-sm hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors ${quantity > 0 ? '' : 'opacity-30'}">
                                   <i class="fas fa-minus"></i>
                               </button>
                               <span class="w-8 text-center font-medium text-black dark:text-white">${quantity}</span>
                               <button onclick="addToCart(${product.id})" class="w-8 h-8 bg-black dark:bg-white text-white dark:text-black rounded-full flex items-center justify-center text-sm hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors">
                                   <i class="fas fa-plus"></i>
                               </button>
                           </div>
                       </div>
                   </div>
               `;
           });
           
           catalogDiv.innerHTML = html;
           updateCartSummary();
       }

       function addToCart(productId) {
           const totalItems = cart.reduce((sum, item) => sum + item.quantity, 0);
           if (totalItems >= 25) {
               alert('Maximum 25 articles par commande');
               return;
           }
           
           const existingItem = cart.find(item => item.id === productId);
           
           if (existingItem) {
               existingItem.quantity++;
           } else {
               cart.push({ id: productId, quantity: 1 });
           }
           
           showCatalog();
           updateUI();
       }

       function removeFromCart(productId) {
           const existingItem = cart.find(item => item.id === productId);
           
           if (existingItem && existingItem.quantity > 0) {
               existingItem.quantity--;
               if (existingItem.quantity === 0) {
                   cart = cart.filter(item => item.id !== productId);
               }
           }
           
           showCatalog();
           updateUI();
       }

       function updateCartSummary() {
           const totalItems = cart.reduce((sum, item) => sum + item.quantity, 0);
           const allProducts = [...commerceCatalogs.carrefour, ...commerceCatalogs.pharmacie];
           const cartTotal = cart.reduce((sum, cartItem) => {
               const product = allProducts.find(p => p.id === cartItem.id);
               return sum + (product ? product.price * cartItem.quantity : 0);
           }, 0);
           
           document.getElementById('cart-count').textContent = `${totalItems} article${totalItems > 1 ? 's' : ''}`;
           document.getElementById('cart-total').textContent = `${cartTotal.toFixed(2)}€`;
       }

       function searchDeliveryPerson() {
           setTimeout(() => {
               const step5 = document.getElementById('step-5');
               step5.innerHTML = `
                   <div class="bg-white dark:bg-dark-200 rounded-3xl p-8 shadow-lg border border-gray-200 dark:border-gray-700 h-full flex items-center justify-center">
                       <div class="text-center">
                           <div class="w-20 h-20 bg-green-500 rounded-full mx-auto mb-6 flex items-center justify-center">
                               <i class="fas fa-check text-white text-3xl"></i>
                           </div>
                           <h2 class="text-2xl font-medium text-black dark:text-white mb-4">Livreur trouvé !</h2>
                           <p class="text-gray-600 dark:text-gray-300 mb-6">Ahmed va s'occuper de ta livraison</p>
                           
                           <div class="bg-gray-50 dark:bg-dark-300 p-6 rounded-2xl border border-gray-200 dark:border-gray-700 max-w-md mx-auto">
                               <div class="flex items-center space-x-4 mb-4">
                                   <div class="w-16 h-16 bg-gray-300 rounded-full flex items-center justify-center">
                                       <i class="fas fa-user text-gray-600 text-xl"></i>
                                   </div>
                                   <div>
                                       <h3 class="font-bold text-black dark:text-white">Ahmed</h3>
                                       <p class="text-sm text-gray-600 dark:text-gray-400">Livreur Vert Chasseur</p>
                                       <p class="text-sm text-gray-600 dark:text-gray-400">📱 +32 485 123 456</p>
                                   </div>
                               </div>
                               
                               <div class="space-y-2 text-sm text-gray-600 dark:text-gray-300">
                                   <p>• Ahmed va faire tes courses</p>
                                   <p>• Il t'enverra des SMS de suivi</p>
                                   <p>• Paiement par lien Stripe sécurisé</p>
                                   <p>• Frais de livraison : 2,50€</p>
                               </div>
                           </div>
                       </div>
                   </div>
               `;
               updateUI();
           }, 2500);
       }

       function finalizeOrder() {
           const total = cart.reduce((sum, cartItem) => {
               const allProducts = [...commerceCatalogs.carrefour, ...commerceCatalogs.pharmacie];
               const product = allProducts.find(p => p.id === cartItem.id);
               return sum + (product ? product.price * cartItem.quantity : 0);
           }, 0);
           
           const orderId = 'VC' + Date.now().toString().slice(-6);
           
           if (confirm(`Confirmer ta commande ?\n\nTotal : ${total.toFixed(2)}€ + 2,50€ de livraison\nNuméro : ${orderId}`)) {
               window.location.href = 'tel:+32485123456';
               setTimeout(() => {
                   alert(`✅ Commande confirmée !\n\nAhmed va te contacter dans quelques minutes.\nTu recevras un lien de paiement par SMS.`);
               }, 1000);
           }
       }
   </script>
</body>
</html>