<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vert Chasseur - Affichage</title>
    <link rel="icon" type="image/png" href="https://vertchasseur.com/logo_new.png">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@200;300;400;500;600;700&family=Princess+Sofia&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        html, body {
            width: 100%;
            height: 100%;
        }

        body {
            font-family: 'Poppins', sans-serif;
            overflow: hidden;
            transition: background 1s ease, color 1s ease;
        }

        body.light-mode {
            background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 50%, #f5f7fa 100%);
            color: #1a1a1a;
        }

        body.dark-mode {
            background: linear-gradient(135deg, #0f0f0f 0%, #1a1a1a 50%, #0d0d0d 100%);
            color: #f0f0f0;
        }

        .viewport {
            width: 100%;
            height: 100%;
            position: relative;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .scene {
            position: absolute;
            width: 100%;
            height: 100%;
            display: flex;
            justify-content: center;
            align-items: center;
            opacity: 0;
            visibility: hidden;
            transition: opacity 1s cubic-bezier(0.25, 0.46, 0.45, 0.94), visibility 1s;
            z-index: 1;
        }

        .scene.active {
            opacity: 1;
            visibility: visible;
            z-index: 10;
        }

        .scene-content {
            width: 100%;
            height: 100%;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            padding: 100px 30px;
            gap: 60px;
        }

        /* ====== LOGO ====== */
        .logo-fixed {
            position: fixed;
            bottom: 50px;
            left: 50%;
            transform: translateX(-50%);
            height: 95px;
            width: auto;
            z-index: 50;
            opacity: 0.65;
            transition: opacity 0.4s ease;
        }

        .logo-fixed:hover {
            opacity: 1;
        }

        .logo-fixed.hidden {
            display: none;
        }

        /* ====== SCENE 1: WELCOME ====== */
        .welcome-title {
            font-size: clamp(50px, 14vw, 110px);
            font-weight: 200;
            letter-spacing: -0.05em;
            line-height: 1.05;
            text-align: center;
            max-width: 90%;
        }

        .txt-chasseur {
            font-family: 'Princess Sofia', serif;
            color: #16a34a;
            font-weight: 400;
            display: inline-block;
            margin: 0 8px;
        }

        .date-time-wrapper {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 100px;
            flex-wrap: wrap;
        }

        .date-time-item {
            font-size: clamp(18px, 3.5vw, 32px);
            font-weight: 300;
            letter-spacing: -0.01em;
            opacity: 0.7;
            min-width: 320px;
            padding: 20px 40px;
            border-radius: 16px;
            background: rgba(22, 163, 74, 0.08);
            backdrop-filter: blur(10px);
            transition: all 0.3s ease;
        }

        body.dark-mode .date-time-item {
            background: rgba(22, 163, 74, 0.12);
        }

        .date-time-item:hover {
            background: rgba(22, 163, 74, 0.15);
            transform: translateY(-4px);
        }

        @media (max-width: 768px) {
            .date-time-wrapper {
                gap: 40px;
                flex-direction: column;
            }

            .date-time-item {
                min-width: 260px;
            }

            .scene-content {
                padding: 60px 20px;
                gap: 40px;
            }
        }

        /* ====== SCENE 2: WEATHER ====== */
        .weather-header {
            font-size: clamp(38px, 8vw, 56px);
            font-weight: 300;
            letter-spacing: -0.02em;
            text-align: center;
        }

        .weather-card {
            display: grid;
            grid-template-columns: auto 1fr;
            gap: 50px;
            align-items: center;
            padding: 60px;
            border-radius: 32px;
            background: rgba(22, 163, 74, 0.08);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(22, 163, 74, 0.15);
            max-width: 600px;
            width: 100%;
        }

        body.dark-mode .weather-card {
            background: rgba(22, 163, 74, 0.12);
            border: 1px solid rgba(22, 163, 74, 0.2);
        }

        .weather-icon {
            width: 140px;
            height: 140px;
        }

        .weather-main-info {
            display: flex;
            flex-direction: column;
            gap: 16px;
        }

        .weather-temp {
            font-size: clamp(56px, 14vw, 96px);
            font-weight: 200;
            line-height: 1;
            letter-spacing: -0.03em;
        }

        .weather-condition {
            font-size: clamp(20px, 4vw, 32px);
            font-weight: 400;
            opacity: 0.7;
        }

        .weather-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
            gap: 28px;
            padding: 50px;
            border-radius: 28px;
            background: rgba(22, 163, 74, 0.06);
            backdrop-filter: blur(15px);
            max-width: 850px;
            width: 100%;
            margin-top: 30px;
        }

        body.dark-mode .weather-details {
            background: rgba(22, 163, 74, 0.1);
        }

        .detail-box {
            display: flex;
            flex-direction: column;
            gap: 12px;
            padding: 20px;
            border-radius: 16px;
            background: rgba(255, 255, 255, 0.5);
            backdrop-filter: blur(10px);
            transition: all 0.3s ease;
        }

        body.dark-mode .detail-box {
            background: rgba(0, 0, 0, 0.2);
        }

        .detail-box:hover {
            transform: translateY(-6px);
            background: rgba(22, 163, 74, 0.15);
        }

        .detail-label {
            font-size: 12px;
            font-weight: 600;
            letter-spacing: 1px;
            text-transform: uppercase;
            opacity: 0.6;
        }

        .detail-value {
            font-size: clamp(18px, 3vw, 26px);
            font-weight: 600;
            letter-spacing: -0.01em;
        }

        @media (max-width: 768px) {
            .weather-card {
                grid-template-columns: 1fr;
                gap: 30px;
                padding: 40px;
                border-radius: 24px;
            }

            .weather-icon {
                width: 100px;
                height: 100px;
                margin: 0 auto;
            }

            .weather-details {
                grid-template-columns: repeat(2, 1fr);
                gap: 16px;
                padding: 30px;
                margin-top: 20px;
            }

            .detail-box {
                padding: 16px;
            }
        }

        @media (max-width: 480px) {
            .weather-details {
                grid-template-columns: 1fr;
            }
        }

        /* ====== SCENE 3: APP ====== */
        .app-section {
            display: flex;
            flex-direction: column;
            gap: 50px;
            align-items: center;
            max-width: 700px;
        }

        .app-intro {
            display: flex;
            flex-direction: column;
            gap: 25px;
            align-items: center;
            text-align: center;
        }

        .app-subtitle {
            font-size: clamp(14px, 2.5vw, 18px);
            font-weight: 600;
            letter-spacing: 2px;
            text-transform: uppercase;
            color: #16a34a;
        }

        .app-title {
            font-size: clamp(44px, 12vw, 80px);
            font-weight: 200;
            letter-spacing: -0.04em;
            line-height: 1.1;
        }

        .app-text {
            font-size: clamp(16px, 3vw, 24px);
            font-weight: 400;
            opacity: 0.75;
            line-height: 1.6;
        }

        .qr-container {
            display: flex;
            flex-direction: column;
            gap: 30px;
            align-items: center;
            padding: 60px 50px;
            border-radius: 32px;
            background: linear-gradient(135deg, rgba(22, 163, 74, 0.1) 0%, rgba(22, 163, 74, 0.05) 100%);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(22, 163, 74, 0.2);
            max-width: 420px;
            width: 100%;
        }

        body.dark-mode .qr-container {
            background: linear-gradient(135deg, rgba(22, 163, 74, 0.15) 0%, rgba(22, 163, 74, 0.08) 100%);
        }

        .qr-label {
            font-size: 13px;
            font-weight: 600;
            letter-spacing: 1.2px;
            text-transform: uppercase;
            opacity: 0.7;
        }

        .qr-code {
            width: 300px;
            height: 300px;
            background: white;
            padding: 20px;
            border-radius: 20px;
            display: block;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
        }

        body.dark-mode .qr-code {
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
        }

        .qr-note {
            font-size: 12px;
            opacity: 0.65;
            letter-spacing: 0.5px;
        }

        @media (max-width: 768px) {
            .app-section {
                gap: 40px;
            }

            .qr-container {
                padding: 40px 30px;
                gap: 24px;
                border-radius: 24px;
            }

            .qr-code {
                width: 260px;
                height: 260px;
                padding: 16px;
            }

            .scene-content {
                padding: 60px 20px;
                gap: 40px;
            }
        }

        @media (max-width: 480px) {
            .qr-container {
                padding: 32px 20px;
            }

            .qr-code {
                width: 220px;
                height: 220px;
                padding: 12px;
            }

            .app-section {
                max-width: 100%;
            }
        }
    </style>
</head>
<body class="light-mode">

    <div class="viewport">
        <!-- SCENE 1: WELCOME -->
        <div id="scene1" class="scene active">
            <div class="scene-content">
                <h1 class="welcome-title">
                    Bienvenue à <span class="txt-chasseur">Vert Chasseur</span>
                </h1>
                <div class="date-time-wrapper">
                    <div class="date-time-item" id="dateDisplay">Lundi 1 janvier</div>
                    <div class="date-time-item" id="timeDisplay">00:00:00</div>
                </div>
            </div>
        </div>

        <!-- SCENE 2: WEATHER -->
        <div id="scene2" class="scene">
            <div class="scene-content">
                <h2 class="weather-header">Météo à Uccle</h2>
                <div class="weather-card">
                    <img id="weatherIcon" class="weather-icon" src="" alt="Météo">
                    <div class="weather-main-info">
                        <div class="weather-temp" id="weatherTemp">--°C</div>
                        <div class="weather-condition" id="weatherCondition">--</div>
                    </div>
                </div>
                <div class="weather-details">
                    <div class="detail-box">
                        <div class="detail-label">Ressenti</div>
                        <div class="detail-value" id="weatherFeelsLike">--°C</div>
                    </div>
                    <div class="detail-box">
                        <div class="detail-label">Humidité</div>
                        <div class="detail-value" id="weatherHumidity">--%</div>
                    </div>
                    <div class="detail-box">
                        <div class="detail-label">Vent</div>
                        <div class="detail-value" id="weatherWind">-- km/h</div>
                    </div>
                    <div class="detail-box">
                        <div class="detail-label">Pression</div>
                        <div class="detail-value" id="weatherPressure">-- mb</div>
                    </div>
                    <div class="detail-box">
                        <div class="detail-label">Visibilité</div>
                        <div class="detail-value" id="weatherVisibility">-- km</div>
                    </div>
                    <div class="detail-box">
                        <div class="detail-label">Indice UV</div>
                        <div class="detail-value" id="weatherUV">--</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- SCENE 3: APP -->
        <div id="scene3" class="scene">
            <div class="scene-content">
                <div class="app-section">
                    <div class="app-intro">
                        <div class="app-subtitle">Reste connecté</div>
                        <h2 class="app-title">Télécharge Vert Chasseur</h2>
                        <p class="app-text">Accès aux commerces, météo, actualités et bien plus</p>
                    </div>
                    <div class="qr-container">
                        <div class="qr-label">Scanne le QR code</div>
                        <img id="qrCode" class="qr-code" src="" alt="QR Code">
                        <div class="qr-note">iOS et Android</div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <img id="logoLight" class="logo-fixed" src="https://vertchasseur.com/logo_new.png" alt="Logo">
    <img id="logoDark" class="logo-fixed hidden" src="https://vertchasseur.com/logo_dark.png" alt="Logo">

    <script>
        const scenes = [
            { id: 'scene1', duration: 60000 },
            { id: 'scene2', duration: 15000 },
            { id: 'scene3', duration: 10000 }
        ];

        let idx = 0;
        let startTime = 0;
        let timer = null;

        function save() {
            sessionStorage.setItem('vc', JSON.stringify({ idx, startTime }));
        }

        function load() {
            const data = sessionStorage.getItem('vc');
            if (data) {
                const state = JSON.parse(data);
                const elapsed = Date.now() - state.startTime;
                idx = elapsed >= scenes[state.idx].duration ? (state.idx + 1) % scenes.length : state.idx;
                startTime = idx === state.idx ? state.startTime : Date.now();
            } else {
                idx = 0;
                startTime = Date.now();
            }
            save();
        }

        function show() {
            document.querySelectorAll('.scene').forEach((s, i) => s.classList.toggle('active', i === idx));
            if (idx === 1) weather();
        }

        function next() {
            document.getElementById(scenes[idx].id).classList.remove('active');
            idx = (idx + 1) % scenes.length;
            startTime = Date.now();
            save();
            show();
            schedule();
        }

        function schedule() {
            clearTimeout(timer);
            timer = setTimeout(next, scenes[idx].duration);
        }

        function clock() {
            const now = new Date();
            const days = ['dimanche', 'lundi', 'mardi', 'mercredi', 'jeudi', 'vendredi', 'samedi'];
            const months = ['janvier', 'février', 'mars', 'avril', 'mai', 'juin', 'juillet', 'août', 'septembre', 'octobre', 'novembre', 'décembre'];
            const d = days[now.getDay()].charAt(0).toUpperCase() + days[now.getDay()].slice(1);
            const m = months[now.getMonth()];
            document.getElementById('dateDisplay').textContent = `${d} ${now.getDate()} ${m}`;
            document.getElementById('timeDisplay').textContent = `${String(now.getHours()).padStart(2, '0')}:${String(now.getMinutes()).padStart(2, '0')}:${String(now.getSeconds()).padStart(2, '0')}`;
        }

        async function daynight() {
            try {
                const r = await fetch('https://api.sunrise-sunset.org/json?lat=50.8047&lng=4.3827');
                const d = await r.json();
                const now = new Date();
                const sr = new Date(d.results.sunrise);
                const ss = new Date(d.results.sunset);
                const isDay = now >= sr && now < ss;
                document.body.className = isDay ? 'light-mode' : 'dark-mode';
                document.getElementById('logoLight').classList.toggle('hidden', !isDay);
                document.getElementById('logoDark').classList.toggle('hidden', isDay);
            } catch (e) {}
        }

        async function weather() {
            try {
                const r = await fetch('https://api.weatherapi.com/v1/current.json?key=fa1a36a37c3f4e7ab7e233043252305&q=Uccle,Belgium&lang=fr');
                const d = await r.json();
                document.getElementById('weatherIcon').src = `https:${d.current.condition.icon}`;
                document.getElementById('weatherTemp').textContent = `${Math.round(d.current.temp_c)}°C`;
                document.getElementById('weatherCondition').textContent = d.current.condition.text;
                document.getElementById('weatherFeelsLike').textContent = `${Math.round(d.current.feelslike_c)}°C`;
                document.getElementById('weatherHumidity').textContent = `${d.current.humidity}%`;
                document.getElementById('weatherWind').textContent = `${Math.round(d.current.wind_kph)} km/h`;
                document.getElementById('weatherPressure').textContent = `${Math.round(d.current.pressure_mb)} mb`;
                document.getElementById('weatherVisibility').textContent = `${Math.round(d.current.vis_km)} km`;
                document.getElementById('weatherUV').textContent = `${d.current.uv}`;
            } catch (e) {}
        }

        function qr() {
            document.getElementById('qrCode').src = `https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=https://vertchasseur.com/download-app`;
        }

        window.addEventListener('load', () => {
            load();
            show();
            schedule();
            clock();
            daynight();
            qr();
            if (navigator.wakeLock) navigator.wakeLock.request('screen').catch(() => {});
            setInterval(clock, 1000);
            setInterval(daynight, 60000);
            setInterval(weather, 600000);
        });

        document.addEventListener('visibilitychange', () => !document.hidden && (clock(), daynight(), weather()));
    </script>

</body>
</html>