<?php
session_start();
include '../db.php';

if (!isset($_SESSION['merchant_id'])) {
    header('Location: /comm-access');
    exit;
}

$merchant_id = $_SESSION['merchant_id'];
$shop_id = $_SESSION['shop_id'];
$manager_name = $_SESSION['manager_name'];

$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_shop'])) {
    $shop_title = trim($_POST['shop_title']);
    $shop_address = trim($_POST['shop_address']);
    $shop_description = trim($_POST['shop_description']);
    $shop_category = trim($_POST['shop_category']);
    
    if (!empty($shop_title)) {
        $update_shop_stmt = $conn->prepare("UPDATE shop SET title = ?, address_shop = ?, description = ?, category = ? WHERE id = ?");
        $update_shop_stmt->bind_param("ssssi", $shop_title, $shop_address, $shop_description, $shop_category, $shop_id);
        
        if ($update_shop_stmt->execute()) {
            $success_message = 'Informations du commerce mises à jour !';
        } else {
            $error_message = 'Erreur lors de la mise à jour.';
        }
    }
}

$merchant_stmt = $conn->prepare("SELECT mp.*, s.title as shop_name, s.address_shop, s.description, s.category FROM merchant_profiles mp LEFT JOIN shop s ON mp.shop_id = s.id WHERE mp.id = ?");
$merchant_stmt->bind_param("i", $merchant_id);
$merchant_stmt->execute();
$merchant_result = $merchant_stmt->get_result();
$merchant = $merchant_result->fetch_assoc();

$shop_name = $merchant['shop_name'] ?? 'Commerce non configuré';
$shop_address = $merchant['address_shop'] ?? '';
$shop_description = $merchant['description'] ?? '';
$shop_category = $merchant['category'] ?? '';
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Comm Access</title>
    <link rel="icon" type="image/png" href="logo_new.png">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {}
            }
        }
    </script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Poppins', sans-serif;
            letter-spacing: -0.01em;
        }
        
        .help-overlay, .edit-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(8px);
            z-index: 1000;
            display: none;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .help-overlay.show, .edit-overlay.show {
            display: flex;
        }
        
        .help-content, .edit-content {
            background: white;
            max-width: 700px;
            width: 100%;
            border-radius: 16px;
            padding: 32px;
            max-height: 85vh;
            overflow-y: auto;
        }
        
        .dark .help-content, .dark .edit-content {
            background: #1c1917;
        }
    </style>
</head>
<body class="bg-white dark:bg-black text-black dark:text-white transition-colors duration-300">
    <script>
        if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            document.documentElement.classList.add('dark');
        }
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', e => {
            if (e.matches) {
                document.documentElement.classList.add('dark');
            } else {
                document.documentElement.classList.remove('dark');
            }
        });
    </script>

    <div class="help-overlay" id="helpOverlay">
        <div class="help-content">
            <div class="flex justify-between items-start mb-6">
                <h2 class="text-2xl font-light">Guide complet Comm Access</h2>
                <button onclick="closeHelp()" class="text-gray-500 hover:text-black dark:hover:text-white">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                    </svg>
                </button>
            </div>
            
            <div class="space-y-6 text-sm">
                <div class="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4 border border-blue-200 dark:border-blue-800">
                    <h3 class="font-medium mb-2 flex items-center">
                        <svg class="w-5 h-5 mr-2 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        Qu'est-ce que Comm Access ?
                    </h3>
                    <p class="text-gray-600 dark:text-gray-400">Comm Access est votre tableau de bord professionnel pour suivre l'activité de votre commerce au sein du programme Vert Chasseur. Vous pouvez consulter en temps réel les visites des clients membres.</p>
                </div>
                
                <div>
                    <h3 class="font-medium mb-3 flex items-center">
                        <svg class="w-5 h-5 mr-2 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        Données collectées
                    </h3>
                    <div class="space-y-2 text-gray-600 dark:text-gray-400 pl-7">
                        <p>• <strong>Visites clients</strong> : Enregistrées lorsqu'un membre présente sa carte chez vous</p>
                        <p>• <strong>Identité du client</strong> : Prénom et nom uniquement (confidentiel)</p>
                        <p>• <strong>Horodatage</strong> : Date et heure précise de chaque visite</p>
                        <p>• <strong>Aucune donnée personnelle sensible</strong> n'est partagée</p>
                    </div>
                </div>
                
                <div>
                    <h3 class="font-medium mb-3 flex items-center">
                        <svg class="w-5 h-5 mr-2 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
                        </svg>
                        Statistiques principales
                    </h3>
                    <div class="space-y-2 text-gray-600 dark:text-gray-400 pl-7">
                        <p>• <strong>Aujourd'hui</strong> : Nombre de visites depuis minuit avec variation par rapport à hier</p>
                        <p>• <strong>Client phare</strong> : Client le plus fidèle sur les 7 derniers jours</p>
                        <p>• <strong>Cette semaine</strong> : Total des 7 derniers jours</p>
                        <p>• <strong>Ce mois</strong> : Total des 30 derniers jours</p>
                    </div>
                </div>
                
                <div>
                    <h3 class="font-medium mb-3 flex items-center">
                        <svg class="w-5 h-5 mr-2 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 12l3-3 3 3 4-4M8 21l4-4 4 4M3 4h18M4 4h16v12a1 1 0 01-1 1H5a1 1 0 01-1-1V4z"/>
                        </svg>
                        Graphiques et analyses
                    </h3>
                    <div class="space-y-2 text-gray-600 dark:text-gray-400 pl-7">
                        <p>• <strong>Activité horaire</strong> : Visualisez vos heures de pointe aujourd'hui</p>
                        <p>• <strong>Tendance mensuelle</strong> : Évolution sur 30 jours pour anticiper</p>
                        <p>• <strong>Mise à jour automatique</strong> : Données rafraîchies toutes les 2 secondes en arrière-plan</p>
                    </div>
                </div>
                
                <div>
                    <h3 class="font-medium mb-3 flex items-center">
                        <svg class="w-5 h-5 mr-2 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"/>
                        </svg>
                        Visites récentes
                    </h3>
                    <div class="space-y-2 text-gray-600 dark:text-gray-400 pl-7">
                        <p>• Liste en temps réel des 10 derniers clients</p>
                        <p>• Heure exacte de passage</p>
                        <p>• Mise à jour automatique silencieuse</p>
                    </div>
                </div>
                
                <div>
                    <h3 class="font-medium mb-3 flex items-center">
                        <svg class="w-5 h-5 mr-2 text-indigo-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                        </svg>
                        Personnalisation du commerce
                    </h3>
                    <div class="space-y-2 text-gray-600 dark:text-gray-400 pl-7">
                        <p>• Modifiez le nom de votre commerce</p>
                        <p>• Mettez à jour l'adresse</p>
                        <p>• Ajoutez une description</p>
                        <p>• Catégorisez votre activité</p>
                    </div>
                </div>
                
                <div>
                    <h3 class="font-medium mb-3 flex items-center">
                        <svg class="w-5 h-5 mr-2 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"/>
                        </svg>
                        Sécurité et confidentialité
                    </h3>
                    <div class="space-y-2 text-gray-600 dark:text-gray-400 pl-7">
                        <p>• Votre code d'accès est strictement confidentiel</p>
                        <p>• Ne le partagez avec personne</p>
                        <p>• Les données clients sont protégées (RGPD)</p>
                        <p>• En cas de perte : msg@aktascorp.com</p>
                    </div>
                </div>
                
                <div class="bg-green-50 dark:bg-green-900/20 rounded-lg p-4 border border-green-200 dark:border-green-800">
                    <h3 class="font-medium mb-2 flex items-center">
                        <svg class="w-5 h-5 mr-2 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/>
                        </svg>
                        Support technique
                    </h3>
                    <p class="text-gray-600 dark:text-gray-400">Une question ? Un problème ? Contactez-nous à <strong>msg@aktascorp.com</strong> - Réponse sous 24h ouvrées</p>
                </div>
            </div>
        </div>
    </div>

    <div class="edit-overlay" id="editOverlay">
        <div class="edit-content">
            <div class="flex justify-between items-start mb-6">
                <h2 class="text-2xl font-light">Modifier mon commerce</h2>
                <button onclick="closeEdit()" class="text-gray-500 hover:text-black dark:hover:text-white">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                    </svg>
                </button>
            </div>
            
            <?php if ($success_message): ?>
            <div class="mb-4 p-4 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg">
                <p class="text-green-800 dark:text-green-200 text-sm"><?= htmlspecialchars($success_message) ?></p>
            </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
            <div class="mb-4 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                <p class="text-red-800 dark:text-red-200 text-sm"><?= htmlspecialchars($error_message) ?></p>
            </div>
            <?php endif; ?>
            
            <form method="POST" class="space-y-6">
                <div>
                    <label class="block text-sm font-medium mb-2">Nom du commerce *</label>
                    <input 
                        type="text" 
                        name="shop_title" 
                        value="<?= htmlspecialchars($shop_name) ?>" 
                        required
                        class="w-full px-4 py-3 bg-white dark:bg-black border border-stone-300 dark:border-stone-700 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500 dark:focus:ring-green-600 transition-all">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2">Adresse</label>
                    <input 
                        type="text" 
                        name="shop_address" 
                        value="<?= htmlspecialchars($shop_address) ?>"
                        class="w-full px-4 py-3 bg-white dark:bg-black border border-stone-300 dark:border-stone-700 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500 dark:focus:ring-green-600 transition-all">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2">Catégorie</label>
                    <input 
                        type="text" 
                        name="shop_category" 
                        value="<?= htmlspecialchars($shop_category) ?>"
                        placeholder="Ex: Restauration, Commerce, Services..."
                        class="w-full px-4 py-3 bg-white dark:bg-black border border-stone-300 dark:border-stone-700 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500 dark:focus:ring-green-600 transition-all">
                </div>
                
                <div>
                    <label class="block text-sm font-medium mb-2">Description</label>
                    <textarea 
                        name="shop_description" 
                        rows="4"
                        placeholder="Décrivez votre commerce..."
                        class="w-full px-4 py-3 bg-white dark:bg-black border border-stone-300 dark:border-stone-700 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-500 dark:focus:ring-green-600 transition-all"><?= htmlspecialchars($shop_description) ?></textarea>
                </div>
                
                <button 
                    type="submit" 
                    name="update_shop"
                    class="w-full bg-green-600 hover:bg-green-700 text-white font-medium py-3 px-4 rounded-lg transition-colors">
                    Enregistrer les modifications
                </button>
            </form>
        </div>
    </div>

    <nav class="border-b border-stone-200 dark:border-stone-800 bg-white dark:bg-black sticky top-0 z-50">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center h-16">
                <div>
                    <h1 class="text-xl md:text-2xl font-light">Comm Access</h1>
                    <p class="text-xs text-gray-500 dark:text-gray-400 hidden md:block" id="shopNameHeader"><?= htmlspecialchars($shop_name) ?></p>
                </div>
                <div class="flex items-center space-x-4">
                    <button onclick="openHelp()" class="text-sm text-gray-600 dark:text-gray-400 hover:text-black dark:hover:text-white transition-colors flex items-center space-x-1">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        <span class="hidden md:inline">Aide</span>
                    </button>
                    <a href="logout.php" class="text-sm text-gray-600 dark:text-gray-400 hover:text-black dark:hover:text-white transition-colors">
                        Déconnexion
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <main class="min-h-screen px-4 py-8 md:py-12">
        <div class="max-w-7xl mx-auto">
            <div class="mb-8">
                <h2 class="text-3xl md:text-4xl font-light mb-2">Bonjour, <?= htmlspecialchars(explode(' ', $manager_name)[0]) ?></h2>
                <p class="text-gray-600 dark:text-gray-400">Voici l'activité en temps réel de votre commerce</p>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                <div class="bg-white dark:bg-stone-900 rounded-2xl p-6 border border-stone-200 dark:border-stone-800">
                    <div class="flex items-center justify-between mb-4">
                        <h3 class="text-sm font-medium text-gray-600 dark:text-gray-400 uppercase tracking-wider">Aujourd'hui</h3>
                        <span class="text-xs px-2 py-1 rounded-full bg-gray-100 dark:bg-stone-800" id="todayBadge">—</span>
                    </div>
                    <p class="text-4xl md:text-5xl font-light mb-2" id="visitsToday">0</p>
                    <p class="text-sm text-gray-500 dark:text-gray-500">Visites</p>
                </div>

                <div class="bg-white dark:bg-stone-900 rounded-2xl p-6 border border-stone-200 dark:border-stone-800">
                    <h3 class="text-sm font-medium text-gray-600 dark:text-gray-400 uppercase tracking-wider mb-4">Client phare (7j)</h3>
                    <p class="text-2xl md:text-3xl font-light mb-2" id="topClientName">—</p>
                    <p class="text-sm text-gray-500 dark:text-gray-500" id="topClientVisits">Aucune donnée</p>
                </div>

                <div class="bg-white dark:bg-stone-900 rounded-2xl p-6 border border-stone-200 dark:border-stone-800">
                    <h3 class="text-sm font-medium text-gray-600 dark:text-gray-400 uppercase tracking-wider mb-4">Cette semaine</h3>
                    <p class="text-4xl md:text-5xl font-light mb-2" id="visitsWeek">0</p>
                    <p class="text-sm text-gray-500 dark:text-gray-500">Visites (7j)</p>
                </div>

                <div class="bg-white dark:bg-stone-900 rounded-2xl p-6 border border-stone-200 dark:border-stone-800">
                    <h3 class="text-sm font-medium text-gray-600 dark:text-gray-400 uppercase tracking-wider mb-4">Ce mois</h3>
                    <p class="text-4xl md:text-5xl font-light mb-2" id="visitsMonth">0</p>
                    <p class="text-sm text-gray-500 dark:text-gray-500">Visites (30j)</p>
                </div>
            </div>

            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8 mb-8">
                <div class="lg:col-span-2 space-y-8">
                    <div class="bg-white dark:bg-stone-900 rounded-2xl p-6 border border-stone-200 dark:border-stone-800">
                        <h3 class="text-xl font-light mb-6">Activité horaire</h3>
                        <canvas id="hourlyChart"></canvas>
                    </div>

                    <div class="bg-white dark:bg-stone-900 rounded-2xl p-6 border border-stone-200 dark:border-stone-800">
                        <h3 class="text-xl font-light mb-6">Tendance (30 derniers jours)</h3>
                        <canvas id="dailyChart"></canvas>
                    </div>
                </div>

                <div class="space-y-8">
                    <div class="bg-white dark:bg-stone-900 rounded-2xl p-6 border border-stone-200 dark:border-stone-800">
                        <h3 class="text-xl font-light mb-6">Visites récentes</h3>
                        <div id="recentVisits">
                            <p class="text-center text-gray-500 dark:text-gray-500 py-8">Chargement...</p>
                        </div>
                    </div>

                    <div class="bg-white dark:bg-stone-900 rounded-2xl p-6 border border-stone-200 dark:border-stone-800">
                        <div class="flex items-center justify-between mb-6">
                            <h3 class="text-xl font-light">Mon commerce</h3>
                            <button onclick="openEdit()" class="text-sm text-gray-600 dark:text-gray-400 hover:text-black dark:hover:text-white transition-colors">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                                </svg>
                            </button>
                        </div>
                        <div class="space-y-4">
                            <div>
                                <p class="text-xs text-gray-500 dark:text-gray-500 mb-1">Nom</p>
                                <p class="font-medium" id="shopName"><?= htmlspecialchars($shop_name) ?></p>
                            </div>
                            <?php if ($shop_category): ?>
                            <div>
                                <p class="text-xs text-gray-500 dark:text-gray-500 mb-1">Catégorie</p>
                                <p class="font-medium" id="shopCategory"><?= htmlspecialchars($shop_category) ?></p>
                            </div>
                            <?php endif; ?>
                            <?php if ($shop_address): ?>
                            <div>
                                <p class="text-xs text-gray-500 dark:text-gray-500 mb-1">Adresse</p>
                                <p class="font-medium text-sm" id="shopAddress"><?= htmlspecialchars($shop_address) ?></p>
                            </div>
                            <?php endif; ?>
                            <?php if ($shop_description): ?>
                            <div>
                                <p class="text-xs text-gray-500 dark:text-gray-500 mb-1">Description</p>
                                <p class="font-medium text-sm" id="shopDescription"><?= htmlspecialchars($shop_description) ?></p>
                            </div>
                            <?php endif; ?>
                            <div class="pt-4 border-t border-stone-200 dark:border-stone-800">
                                <p class="text-xs text-gray-500 dark:text-gray-500 mb-1">Gérant</p>
                                <p class="font-medium"><?= htmlspecialchars($merchant['manager_name']) ?></p>
                            </div>
                            <div>
                                <p class="text-xs text-gray-500 dark:text-gray-500 mb-1">Email</p>
                                <p class="font-medium text-sm"><?= htmlspecialchars($merchant['email']) ?></p>
                            </div>
                            <div>
                                <p class="text-xs text-gray-500 dark:text-gray-500 mb-1">Code d'accès</p>
                                <p class="font-medium font-mono"><?= htmlspecialchars($merchant['access_code']) ?></p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <script>
        function openHelp() {
            document.getElementById('helpOverlay').classList.add('show');
        }
        
        function closeHelp() {
            document.getElementById('helpOverlay').classList.remove('show');
        }
        
        function openEdit() {
            document.getElementById('editOverlay').classList.add('show');
        }
        
        function closeEdit() {
            document.getElementById('editOverlay').classList.remove('show');
        }
        
        let hourlyChart = null;
        let dailyChart = null;
        
        function initCharts() {
            const isDark = document.documentElement.classList.contains('dark');
            const textColor = isDark ? '#9ca3af' : '#6b7280';
            const gridColor = isDark ? '#292524' : '#f5f5f4';

            const hourlyCtx = document.getElementById('hourlyChart').getContext('2d');
            hourlyChart = new Chart(hourlyCtx, {
                type: 'bar',
                data: {
                    labels: [],
                    datasets: [{
                        data: [],
                        backgroundColor: isDark ? 'rgba(255, 255, 255, 0.9)' : 'rgba(0, 0, 0, 0.9)',
                        borderRadius: 6,
                        barThickness: 16
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: true,
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                color: textColor,
                                stepSize: 1
                            },
                            grid: {
                                color: gridColor,
                                drawBorder: false
                            },
                            border: {
                                display: false
                            }
                        },
                        x: {
                            ticks: {
                                color: textColor
                            },
                            grid: {
                                display: false
                            },
                            border: {
                                display: false
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            backgroundColor: isDark ? '#1c1917' : '#fff',
                            titleColor: isDark ? '#fff' : '#000',
                            bodyColor: isDark ? '#fff' : '#000',
                            borderColor: isDark ? '#292524' : '#e7e5e4',
                            borderWidth: 1
                        }
                    }
                }
            });

            const dailyCtx = document.getElementById('dailyChart').getContext('2d');
            dailyChart = new Chart(dailyCtx, {
                type: 'line',
                data: {
                    labels: [],
                    datasets: [{
                        data: [],
                        borderColor: isDark ? 'rgba(255, 255, 255, 0.9)' : 'rgba(0, 0, 0, 0.9)',
                        backgroundColor: isDark ? 'rgba(255, 255, 255, 0.05)' : 'rgba(0, 0, 0, 0.05)',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.3,
                        pointRadius: 3,
                        pointHoverRadius: 6,
                        pointBackgroundColor: isDark ? '#fff' : '#000',
                        pointBorderWidth: 2,
                        pointBorderColor: isDark ? '#000' : '#fff'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: true,
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                color: textColor,
                                stepSize: 1
                            },
                            grid: {
                                color: gridColor,
                                drawBorder: false
                            },
                            border: {
                                display: false
                            }
                        },
                        x: {
                            ticks: {
                                color: textColor,
                                maxRotation: 45,
                                minRotation: 45
                            },
                            grid: {
                                display: false
                            },
                            border: {
                                display: false
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            backgroundColor: isDark ? '#1c1917' : '#fff',
                            titleColor: isDark ? '#fff' : '#000',
                            bodyColor: isDark ? '#fff' : '#000',
                            borderColor: isDark ? '#292524' : '#e7e5e4',
                            borderWidth: 1
                        }
                    }
                }
            });
        }
        
        async function refreshData() {
            try {
                const response = await fetch('api-dashboard.php');
                const data = await response.json();
                
                document.getElementById('visitsToday').textContent = data.visits_today;
                document.getElementById('visitsWeek').textContent = data.visits_week;
                document.getElementById('visitsMonth').textContent = data.visits_month;
                
                const badge = document.getElementById('todayBadge');
                if (data.change_percent != 0) {
                    badge.textContent = (data.change_percent > 0 ? '+' : '') + data.change_percent.toFixed(0) + '%';
                    badge.className = 'text-xs px-2 py-1 rounded-full ' + 
                        (data.change_percent > 0 ? 
                            'bg-green-100 dark:bg-green-900/20 text-green-600 dark:text-green-400' : 
                            'bg-red-100 dark:bg-red-900/20 text-red-600 dark:text-red-400');
                } else {
                    badge.textContent = '—';
                    badge.className = 'text-xs px-2 py-1 rounded-full bg-gray-100 dark:bg-stone-800';
                }
                
                if (data.top_client) {
                    document.getElementById('topClientName').textContent = data.top_client.name;
                    document.getElementById('topClientVisits').textContent = 
                        data.top_client.count + ' visite' + (data.top_client.count > 1 ? 's' : '');
                } else {
                    document.getElementById('topClientName').textContent = '—';
                    document.getElementById('topClientVisits').textContent = 'Aucune donnée';
                }
                
                if (data.recent_visits.length > 0) {
                    let html = '<div class="space-y-3">';
                    data.recent_visits.forEach(visit => {
                        html += `
                            <div class="flex items-center justify-between py-2 border-b border-stone-200 dark:border-stone-800 last:border-0">
                                <div>
                                    <p class="font-medium text-sm">${visit.name}</p>
                                    <p class="text-xs text-gray-500 dark:text-gray-500">${visit.time}</p>
                                </div>
                                <div class="w-8 h-8 bg-green-100 dark:bg-green-900/20 rounded-full flex items-center justify-center">
                                    <svg class="w-4 h-4 text-green-600 dark:text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                                    </svg>
                                </div>
                            </div>
                        `;
                    });
                    html += '</div>';
                    document.getElementById('recentVisits').innerHTML = html;
                } else {
                    document.getElementById('recentVisits').innerHTML = 
                        '<p class="text-center text-gray-500 dark:text-gray-500 py-8">Aucune visite récente</p>';
                }
                
                if (hourlyChart) {
                    hourlyChart.data.labels = data.hourly_labels;
                    hourlyChart.data.datasets[0].data = data.hourly_data;
                    hourlyChart.update('none');
                }
                
                if (dailyChart) {
                    dailyChart.data.labels = data.daily_labels;
                    dailyChart.data.datasets[0].data = data.daily_data;
                    dailyChart.update('none');
                }
            } catch (error) {
                console.error('Erreur de rafraîchissement:', error);
            }
        }
        
        initCharts();
        refreshData();
        setInterval(refreshData, 2000);
        
        window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', () => {
            setTimeout(() => location.reload(), 100);
        });
        
        <?php if ($success_message): ?>
        setTimeout(() => {
            openEdit();
        }, 100);
        <?php endif; ?>
    </script>
</body>
</html>