<?php
session_start();
include 'db.php';

if (!isset($_SESSION['resident_id'])) {
    header('Location: ../login.php');
    exit;
}

$resident_id = $_SESSION['resident_id'];
$stmt = $conn->prepare("SELECT email FROM residents WHERE id = ?");
$stmt->bind_param("i", $resident_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();

if ($user['email'] !== 'izhakaktas@gmail.com') {
    header('Location: index.php');
    exit;
}

$success_message = '';
$error_message = '';

function detectSpamAccount($first_name, $last_name, $email, $address) {
    $spam_score = 0;
    
    if (strlen($first_name) < 2 || strlen($last_name) < 2) {
        $spam_score += 30;
    }
    
    if (preg_match('/[0-9]{3,}/', $first_name) || preg_match('/[0-9]{3,}/', $last_name)) {
        $spam_score += 25;
    }
    
    if (preg_match('/admin|test|demo|spam|hack|bot|user[0-9]|temp/i', $first_name) || 
        preg_match('/admin|test|demo|spam|hack|bot|user[0-9]|temp/i', $last_name)) {
        $spam_score += 40;
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $spam_score += 50;
    }
    
    if (preg_match('/@(tempmail|guerrillamail|mailinator|throwaway|10minutemail)/i', $email)) {
        $spam_score += 35;
    }
    
    if (empty($address) || strlen($address) < 5) {
        $spam_score += 20;
    }
    
    if (preg_match('/[^a-zA-Z0-9\s\-.,]/i', $first_name) || preg_match('/[^a-zA-Z0-9\s\-.,]/i', $last_name)) {
        $spam_score += 30;
    }
    
    if (strtoupper($first_name) === $first_name && strlen($first_name) > 3) {
        $spam_score += 15;
    }
    
    return min($spam_score, 100);
}

$stats_query = $conn->query("
    SELECT 
        COUNT(*) as total_users,
        SUM(member_ = 1) as premium_users,
        COUNT(DISTINCT last_name) as unique_families,
        DATE(MIN(created_at)) as first_user_date,
        DATE(MAX(created_at)) as last_user_date
    FROM residents
");
$stats = $stats_query->fetch_assoc();
$monthly_revenue = $stats['premium_users'] * 1.99;

$week_ago = date('Y-m-d', strtotime('-7 days'));
$new_users_week = $conn->query("SELECT COUNT(*) as count FROM residents WHERE DATE(created_at) >= '$week_ago'")->fetch_assoc()['count'];

if (isset($_POST['action'])) {
    if ($_POST['action'] == 'update_resident') {
        $update_id = $_POST['resident_id'];
        $first_name = trim($_POST['first_name']);
        $last_name = trim($_POST['last_name']);
        $email = trim($_POST['email']);
        $address = trim($_POST['address']);
        $member = isset($_POST['member']) ? 1 : 0;
        
        $update_stmt = $conn->prepare("UPDATE residents SET first_name = ?, last_name = ?, email = ?, address = ?, member_ = ? WHERE id = ?");
        $update_stmt->bind_param("ssssii", $first_name, $last_name, $email, $address, $member, $update_id);
        
        if ($update_stmt->execute()) {
            $success_message = 'Utilisateur mis à jour avec succès';
        } else {
            $error_message = 'Erreur lors de la mise à jour';
        }
    }
    
    if ($_POST['action'] == 'delete_resident') {
        $delete_id = $_POST['resident_id'];
        
        if ($delete_id != $resident_id) {
            $delete_stmt = $conn->prepare("DELETE FROM residents WHERE id = ?");
            $delete_stmt->bind_param("i", $delete_id);
            
            if ($delete_stmt->execute()) {
                $success_message = 'Utilisateur supprimé avec succès';
            } else {
                $error_message = 'Erreur lors de la suppression';
            }
        } else {
            $error_message = 'Tu ne peux pas supprimer ton propre compte';
        }
    }
    
    if ($_POST['action'] == 'bulk_delete') {
        $ids = isset($_POST['selected_ids']) ? array_map('intval', $_POST['selected_ids']) : [];
        
        if (empty($ids)) {
            $error_message = 'Aucun utilisateur sélectionné';
        } else {
            if (in_array($resident_id, $ids)) {
                $error_message = 'Tu ne peux pas supprimer ton propre compte';
            } else {
                $placeholders = implode(',', array_fill(0, count($ids), '?'));
                $delete_stmt = $conn->prepare("DELETE FROM residents WHERE id IN ($placeholders)");
                $delete_stmt->bind_param(str_repeat('i', count($ids)), ...$ids);
                
                if ($delete_stmt->execute()) {
                    $success_message = count($ids) . ' utilisateur(s) supprimé(s) avec succès';
                } else {
                    $error_message = 'Erreur lors de la suppression';
                }
            }
        }
    }
    
    if ($_POST['action'] == 'bulk_promote') {
        $ids = isset($_POST['selected_ids']) ? array_map('intval', $_POST['selected_ids']) : [];
        
        if (empty($ids)) {
            $error_message = 'Aucun utilisateur sélectionné';
        } else {
            $placeholders = implode(',', array_fill(0, count($ids), '?'));
            $update_stmt = $conn->prepare("UPDATE residents SET member_ = 1 WHERE id IN ($placeholders)");
            $update_stmt->bind_param(str_repeat('i', count($ids)), ...$ids);
            
            if ($update_stmt->execute()) {
                $success_message = count($ids) . ' utilisateur(s) promu(s) en Premium avec succès';
            } else {
                $error_message = 'Erreur lors de la promotion';
            }
        }
    }
    
    if ($_POST['action'] == 'bulk_demote') {
        $ids = isset($_POST['selected_ids']) ? array_map('intval', $_POST['selected_ids']) : [];
        
        if (empty($ids)) {
            $error_message = 'Aucun utilisateur sélectionné';
        } else {
            if (in_array($resident_id, $ids)) {
                $error_message = 'Tu ne peux pas rétrograder ton propre compte';
            } else {
                $placeholders = implode(',', array_fill(0, count($ids), '?'));
                $update_stmt = $conn->prepare("UPDATE residents SET member_ = 0 WHERE id IN ($placeholders)");
                $update_stmt->bind_param(str_repeat('i', count($ids)), ...$ids);
                
                if ($update_stmt->execute()) {
                    $success_message = count($ids) . ' utilisateur(s) rétrogradé(s) en Standard avec succès';
                } else {
                    $error_message = 'Erreur lors de la rétrogradation';
                }
            }
        }
    }
}

$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'all';
$spam_filter = isset($_GET['spam_filter']) ? $_GET['spam_filter'] : 'none';

$query = "SELECT * FROM residents WHERE 1=1";

if ($search) {
    $query .= " AND (first_name LIKE ? OR last_name LIKE ? OR email LIKE ? OR address LIKE ?)";
}

if ($filter == 'premium') {
    $query .= " AND member_ = 1";
} elseif ($filter == 'standard') {
    $query .= " AND member_ = 0";
}

$query .= " ORDER BY created_at DESC";

$stmt = $conn->prepare($query);

if ($search) {
    $search_param = "%$search%";
    $stmt->bind_param("ssss", $search_param, $search_param, $search_param, $search_param);
}

$stmt->execute();
$all_residents = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

$residents_with_spam = [];
foreach ($all_residents as $resident) {
    $spam_score = detectSpamAccount($resident['first_name'], $resident['last_name'], $resident['email'], $resident['address']);
    $resident['spam_score'] = $spam_score;
    $residents_with_spam[] = $resident;
}

if ($spam_filter == 'suspicious') {
    $residents_with_spam = array_filter($residents_with_spam, function($r) {
        return $r['spam_score'] >= 50;
    });
} elseif ($spam_filter == 'high') {
    $residents_with_spam = array_filter($residents_with_spam, function($r) {
        return $r['spam_score'] >= 75;
    });
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Administration - Vert Chasseur</title>
    <link rel="icon" type="image/png" href="logo_new.png">
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {}
            }
        }
    </script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600&display=swap" rel="stylesheet">
    <style>
        * {
            font-family: 'Inter', sans-serif;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(4px);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }
        
        .modal.show {
            display: flex;
        }
        
        .modal-content {
            background: white;
            border-radius: 16px;
            padding: 32px;
            max-width: 600px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
            animation: slideUp 0.3s ease;
        }
        
        .dark .modal-content {
            background: #1c1917;
        }
        
        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .widget {
            background: white;
            border-radius: 16px;
            padding: 16px;
            border: 1px solid #e7e5e4;
            transition: all 0.3s ease;
        }
        
        .dark .widget {
            background: #292524;
            border-color: #44403c;
        }
        
        .widget:active {
            transform: scale(0.98);
        }
        
        .spam-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 6px;
            font-size: 11px;
            font-weight: 600;
        }
        
        .spam-low {
            background: #dcfce7;
            color: #166534;
        }
        
        .dark .spam-low {
            background: #166534;
            color: #86efac;
        }
        
        .spam-medium {
            background: #fed7aa;
            color: #92400e;
        }
        
        .dark .spam-medium {
            background: #92400e;
            color: #fdba74;
        }
        
        .spam-high {
            background: #fecaca;
            color: #991b1b;
        }
        
        .dark .spam-high {
            background: #991b1b;
            color: #fca5a5;
        }
        
        .spam-critical {
            background: #fca5a5;
            color: #7f1d1d;
        }
        
        .dark .spam-critical {
            background: #7f1d1d;
            color: #fecaca;
        }
    </style>
</head>
<body class="bg-stone-50 dark:bg-black min-h-screen">
    <script>
        if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
            document.documentElement.classList.add('dark');
        }
    </script>

    <div class="max-w-7xl mx-auto px-4 py-6 lg:py-8">
        <div class="flex flex-col lg:flex-row lg:items-center justify-between mb-8 gap-4">
            <div>
                <h1 class="text-3xl lg:text-4xl font-light text-stone-800 dark:text-white mb-2">Administration</h1>
                <p class="text-sm lg:text-base text-stone-600 dark:text-stone-400">Gestion des utilisateurs de Vert Chasseur</p>
            </div>
            <a href="index.php" class="inline-block px-5 py-2.5 bg-stone-800 dark:bg-white text-white dark:text-black rounded-lg hover:bg-stone-700 dark:hover:bg-stone-100 transition-all text-sm lg:text-base">
                ← Retour
            </a>
        </div>

        <div class="grid grid-cols-2 sm:grid-cols-2 lg:grid-cols-4 gap-3 mb-8">
            <div class="widget">
                <p class="text-xs uppercase tracking-wider text-stone-500 dark:text-stone-400 mb-2">Total</p>
                <p class="text-2xl lg:text-3xl font-light text-stone-900 dark:text-white"><?= $stats['total_users'] ?></p>
                <p class="text-xs text-stone-500 dark:text-stone-400 mt-2">utilisateurs</p>
            </div>
            
            <div class="widget">
                <p class="text-xs uppercase tracking-wider text-stone-500 dark:text-stone-400 mb-2">Premium</p>
                <p class="text-2xl lg:text-3xl font-light text-green-600 dark:text-green-400"><?= $stats['premium_users'] ?></p>
                <p class="text-xs text-stone-500 dark:text-stone-400 mt-2">membres actifs</p>
            </div>
            
            <div class="widget">
                <p class="text-xs uppercase tracking-wider text-stone-500 dark:text-stone-400 mb-2">Revenue</p>
                <p class="text-2xl lg:text-3xl font-light text-stone-900 dark:text-white"><?= number_format($monthly_revenue, 2, ',', '') ?>€</p>
                <p class="text-xs text-stone-500 dark:text-stone-400 mt-2">ce mois</p>
            </div>
            
            <div class="widget">
                <p class="text-xs uppercase tracking-wider text-stone-500 dark:text-stone-400 mb-2">7 jours</p>
                <p class="text-2xl lg:text-3xl font-light text-blue-600 dark:text-blue-400"><?= $new_users_week ?></p>
                <p class="text-xs text-stone-500 dark:text-stone-400 mt-2">nouvelles inscriptions</p>
            </div>
        </div>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-3 mb-8">
            <div class="widget lg:col-span-2">
                <p class="text-xs uppercase tracking-wider text-stone-500 dark:text-stone-400 mb-3">Période d'activité</p>
                <div class="space-y-2">
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-stone-600 dark:text-stone-300">Premier utilisateur</span>
                        <span class="text-sm font-medium text-stone-900 dark:text-white"><?= $stats['first_user_date'] ?></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-stone-600 dark:text-stone-300">Dernier utilisateur</span>
                        <span class="text-sm font-medium text-stone-900 dark:text-white"><?= $stats['last_user_date'] ?></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-stone-600 dark:text-stone-300">Familles uniques</span>
                        <span class="text-sm font-medium text-stone-900 dark:text-white"><?= $stats['unique_families'] ?></span>
                    </div>
                </div>
            </div>

            <div class="widget">
                <p class="text-xs uppercase tracking-wider text-stone-500 dark:text-stone-400 mb-3">Alertes</p>
                <div class="space-y-2">
                    <div class="flex items-center gap-2">
                        <span class="text-red-600 dark:text-red-400 text-lg">⚠</span>
                        <span class="text-sm text-stone-700 dark:text-stone-300"><?= count(array_filter($residents_with_spam, function($r) { return $r['spam_score'] >= 50; })) ?> suspects</span>
                    </div>
                </div>
            </div>
        </div>

        <?php if ($success_message): ?>
        <div class="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 text-green-800 dark:text-green-200 px-4 py-3 rounded-lg mb-6 text-sm">
            <?= htmlspecialchars($success_message) ?>
        </div>
        <?php endif; ?>

        <?php if ($error_message): ?>
        <div class="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 text-red-800 dark:text-red-200 px-4 py-3 rounded-lg mb-6 text-sm">
            <?= htmlspecialchars($error_message) ?>
        </div>
        <?php endif; ?>

        <div class="widget mb-6">
            <form method="GET" class="flex flex-col gap-3">
                <div class="flex flex-col sm:flex-row gap-3">
                    <input type="text" name="search" value="<?= htmlspecialchars($search) ?>" placeholder="Rechercher..." class="flex-1 px-4 py-2.5 bg-stone-100 dark:bg-stone-800 border border-stone-200 dark:border-stone-700 rounded-lg text-stone-800 dark:text-white text-sm">
                    
                    <select name="filter" class="px-4 py-2.5 bg-stone-100 dark:bg-stone-800 border border-stone-200 dark:border-stone-700 rounded-lg text-stone-800 dark:text-white text-sm">
                        <option value="all" <?= $filter == 'all' ? 'selected' : '' ?>>Tous</option>
                        <option value="premium" <?= $filter == 'premium' ? 'selected' : '' ?>>Premium</option>
                        <option value="standard" <?= $filter == 'standard' ? 'selected' : '' ?>>Standard</option>
                    </select>
                    
                    <select name="spam_filter" class="px-4 py-2.5 bg-stone-100 dark:bg-stone-800 border border-stone-200 dark:border-stone-700 rounded-lg text-stone-800 dark:text-white text-sm">
                        <option value="none" <?= $spam_filter == 'none' ? 'selected' : '' ?>>Tous</option>
                        <option value="suspicious" <?= $spam_filter == 'suspicious' ? 'selected' : '' ?>>Suspects (50+)</option>
                        <option value="high" <?= $spam_filter == 'high' ? 'selected' : '' ?>>Très suspects (75+)</option>
                    </select>
                    
                    <button type="submit" class="px-5 py-2.5 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors text-sm whitespace-nowrap">
                        Filtrer
                    </button>
                </div>
            </form>
        </div>

        <div id="bulkActions" class="widget mb-6 hidden">
            <div class="flex flex-col sm:flex-row items-center justify-between gap-4">
                <p class="text-sm text-stone-800 dark:text-stone-200">
                    <span id="selectedCount">0</span> sélectionné(s)
                </p>
                <div class="flex gap-2 w-full sm:w-auto">
                    <button onclick="bulkAction('bulk_promote')" class="flex-1 sm:flex-initial px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors text-sm">
                        Promouvoir
                    </button>
                    <button onclick="bulkAction('bulk_demote')" class="flex-1 sm:flex-initial px-4 py-2 bg-yellow-600 text-white rounded-lg hover:bg-yellow-700 transition-colors text-sm">
                        Rétrograder
                    </button>
                    <button onclick="bulkAction('bulk_delete')" class="flex-1 sm:flex-initial px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors text-sm">
                        Supprimer
                    </button>
                </div>
            </div>
        </div>

        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 mb-8">
            <?php foreach($residents_with_spam as $resident): 
                $spam_class = $resident['spam_score'] >= 75 ? 'spam-critical' : 
                             ($resident['spam_score'] >= 50 ? 'spam-high' : 
                             ($resident['spam_score'] >= 25 ? 'spam-medium' : 'spam-low'));
            ?>
            <div class="widget cursor-pointer hover:shadow-lg">
                <div class="flex items-start justify-between mb-4">
                    <div class="flex-1">
                        <p class="font-medium text-stone-900 dark:text-white text-sm"><?= htmlspecialchars($resident['first_name'] . ' ' . $resident['last_name']) ?></p>
                        <p class="text-xs text-stone-500 dark:text-stone-400 mt-1"><?= htmlspecialchars($resident['email']) ?></p>
                    </div>
                    <input type="checkbox" class="resident-checkbox" value="<?= $resident['id'] ?>" onchange="updateBulkActions()" class="w-4 h-4 rounded border-stone-300 text-green-600">
                </div>

                <div class="space-y-2 mb-4 pb-4 border-b border-stone-200 dark:border-stone-700">
                    <div class="flex justify-between items-center">
                        <span class="text-xs text-stone-500 dark:text-stone-400">ID</span>
                        <span class="text-xs font-medium text-stone-900 dark:text-white">#<?= $resident['id'] ?></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-xs text-stone-500 dark:text-stone-400">Statut</span>
                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= $resident['member_'] == 1 ? 'bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-200' : 'bg-stone-200 dark:bg-stone-700 text-stone-800 dark:text-stone-200' ?>">
                            <?= $resident['member_'] == 1 ? 'Premium' : 'Standard' ?>
                        </span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-xs text-stone-500 dark:text-stone-400">Inscrit</span>
                        <span class="text-xs font-medium text-stone-900 dark:text-white"><?= date('d/m/Y', strtotime($resident['created_at'])) ?></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-xs text-stone-500 dark:text-stone-400">Spam</span>
                        <span class="spam-badge <?= $spam_class ?>">
                            <?= $resident['spam_score'] ?>%
                        </span>
                    </div>
                </div>

                <div class="flex gap-2">
                    <button onclick="openEditModal(<?= htmlspecialchars(json_encode($resident)) ?>)" class="flex-1 px-3 py-2 text-sm bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                        Éditer
                    </button>
                    <?php if ($resident['id'] != $resident_id): ?>
                    <button onclick="confirmDelete(<?= $resident['id'] ?>)" class="flex-1 px-3 py-2 text-sm bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors">
                        Suppr.
                    </button>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <div id="editModal" class="modal">
        <div class="modal-content">
            <h2 class="text-2xl font-light text-stone-800 dark:text-white mb-6">Modifier l'utilisateur</h2>
            
            <form method="POST">
                <input type="hidden" name="action" value="update_resident">
                <input type="hidden" name="resident_id" id="edit_id">
                
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-stone-700 dark:text-stone-300 mb-2">Prénom</label>
                        <input type="text" name="first_name" id="edit_first_name" required class="w-full px-4 py-2.5 bg-stone-100 dark:bg-stone-800 border border-stone-200 dark:border-stone-700 rounded-lg text-stone-800 dark:text-white text-sm">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-stone-700 dark:text-stone-300 mb-2">Nom</label>
                        <input type="text" name="last_name" id="edit_last_name" required class="w-full px-4 py-2.5 bg-stone-100 dark:bg-stone-800 border border-stone-200 dark:border-stone-700 rounded-lg text-stone-800 dark:text-white text-sm">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-stone-700 dark:text-stone-300 mb-2">Email</label>
                        <input type="email" name="email" id="edit_email" required class="w-full px-4 py-2.5 bg-stone-100 dark:bg-stone-800 border border-stone-200 dark:border-stone-700 rounded-lg text-stone-800 dark:text-white text-sm">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-stone-700 dark:text-stone-300 mb-2">Adresse</label>
                        <textarea name="address" id="edit_address" rows="3" class="w-full px-4 py-2.5 bg-stone-100 dark:bg-stone-800 border border-stone-200 dark:border-stone-700 rounded-lg text-stone-800 dark:text-white text-sm"></textarea>
                    </div>
                    
                    <div class="flex items-center">
                        <input type="checkbox" name="member" id="edit_member" class="w-4 h-4 text-green-600 border-stone-300 rounded">
                        <label for="edit_member" class="ml-2 text-sm text-stone-700 dark:text-stone-300">Membre Premium</label>
                    </div>
                </div>
                
                <div class="flex gap-3 mt-6">
                    <button type="submit" class="flex-1 px-5 py-2.5 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors text-sm">
                        Enregistrer
                    </button>
                    <button type="button" onclick="closeEditModal()" class="flex-1 px-5 py-2.5 bg-stone-200 dark:bg-stone-700 text-stone-800 dark:text-white rounded-lg hover:bg-stone-300 dark:hover:bg-stone-600 transition-colors text-sm">
                        Annuler
                    </button>
                </div>
            </form>
        </div>
    </div>

    <form id="deleteForm" method="POST" style="display: none;">
        <input type="hidden" name="action" value="delete_resident">
        <input type="hidden" name="resident_id" id="delete_id">
    </form>

    <form id="bulkForm" method="POST" style="display: none;">
        <input type="hidden" name="action" id="bulk_action">
        <div id="selected_ids_container"></div>
    </form>

    <script>
        function openEditModal(resident) {
            document.getElementById('edit_id').value = resident.id;
            document.getElementById('edit_first_name').value = resident.first_name;
            document.getElementById('edit_last_name').value = resident.last_name;
            document.getElementById('edit_email').value = resident.email;
            document.getElementById('edit_address').value = resident.address;
            document.getElementById('edit_member').checked = resident.member_ == 1;
            document.getElementById('editModal').classList.add('show');
        }
        
        function closeEditModal() {
            document.getElementById('editModal').classList.remove('show');
        }
        
        function confirmDelete(id) {
            if (confirm('Es-tu sûr de vouloir supprimer cet utilisateur ?')) {
                document.getElementById('delete_id').value = id;
                document.getElementById('deleteForm').submit();
            }
        }
        
        function updateBulkActions() {
            const selected = document.querySelectorAll('.resident-checkbox:checked').length;
            const bulkActionsDiv = document.getElementById('bulkActions');
            
            if (selected > 0) {
                bulkActionsDiv.classList.remove('hidden');
                document.getElementById('selectedCount').textContent = selected;
            } else {
                bulkActionsDiv.classList.add('hidden');
            }
        }
        
        function bulkAction(action) {
            const selected = Array.from(document.querySelectorAll('.resident-checkbox:checked')).map(cb => cb.value);
            
            if (selected.length === 0) {
                alert('Aucun utilisateur sélectionné');
                return;
            }
            
            if (action === 'bulk_delete') {
                if (!confirm('Es-tu sûr de vouloir supprimer ' + selected.length + ' utilisateur(s) ?')) {
                    return;
                }
            }
            
            const bulkForm = document.getElementById('bulkForm');
            const container = document.getElementById('selected_ids_container');
            container.innerHTML = '';
            
            selected.forEach(id => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'selected_ids[]';
                input.value = id;
                container.appendChild(input);
            });
            
            document.getElementById('bulk_action').value = action;
            bulkForm.submit();
        }
        
        document.getElementById('editModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeEditModal();
            }
        });
    </script>
</body>
</html>