<?php
include 'db.php';
session_start();

$error = '';
$success = '';

if (isset($_SESSION['resident_id'])) {
    header('Location: resident-dashboard.php');
    exit;
}

if (isset($_GET['guest']) && $_GET['guest'] == '1') {
    $_SESSION['guest_mode'] = true;
    header('Location: resident-dashboard.php');
    exit;
}

if (isset($_GET['logout']) && $_GET['logout'] == '1') {
    $success = 'Vous avez été déconnecté avec succès.';
}

if (isset($_GET['guest_logout']) && $_GET['guest_logout'] == '1') {
    $success = 'Session invité terminée.';
}

if ($_POST) {
    $action = $_POST['action'];
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    
    if ($action === 'login') {
        $stmt = $conn->prepare("SELECT id, password, first_name FROM residents WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($user = $result->fetch_assoc()) {
            if (password_verify($password, $user['password'])) {
                $_SESSION['resident_id'] = $user['id'];
                $_SESSION['resident_name'] = $user['first_name'];
                
                $remember_token = bin2hex(random_bytes(32));
                $expires = time() + (365 * 24 * 60 * 60);
                
                $stmt = $conn->prepare("UPDATE residents SET remember_token = ? WHERE id = ?");
                $stmt->bind_param("si", $remember_token, $user['id']);
                $stmt->execute();
                
                setcookie('remember_token', $remember_token, $expires, '/', '', false, true);
                setcookie('resident_email', $email, $expires, '/', '', false, false);
                
                header('Location: resident-dashboard.php');
                exit;
            } else {
                $error = 'Email ou mot de passe incorrect';
            }
        } else {
            $error = 'Email ou mot de passe incorrect';
        }
    } 
    elseif ($action === 'register') {
        $first_name = trim($_POST['first_name']);
        $last_name = trim($_POST['last_name']);
        $address = trim($_POST['address']);
        
        if (empty($first_name) || empty($last_name) || empty($email) || empty($password) || empty($address)) {
            $error = 'Tous les champs sont requis';
        } elseif (strlen($password) < 6) {
            $error = 'Le mot de passe doit contenir au moins 6 caractères';
        } elseif (!preg_match('/Fort\s*Jaco|Vert\s*Chasseur/i', $address)) {
            $error = 'L\'adresse doit être située à Fort Jaco ou Vert Chasseur. Veuillez changer votre adresse.';
        } else {
            $stmt = $conn->prepare("SELECT id FROM residents WHERE email = ?");
            $stmt->bind_param("s", $email);
            $stmt->execute();
            
            if ($stmt->get_result()->num_rows > 0) {
                $error = 'Un compte existe déjà avec cet email';
            } else {
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $conn->prepare("INSERT INTO residents (first_name, last_name, email, password, address, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
                $stmt->bind_param("sssss", $first_name, $last_name, $email, $hashed_password, $address);
                
                if ($stmt->execute()) {
                    $new_resident_id = $conn->insert_id;
                    $_SESSION['resident_id'] = $new_resident_id;
                    $_SESSION['resident_name'] = $first_name;
                    
                    $remember_token = bin2hex(random_bytes(32));
                    $expires = time() + (365 * 24 * 60 * 60);
                    
                    $stmt = $conn->prepare("UPDATE residents SET remember_token = ? WHERE id = ?");
                    $stmt->bind_param("si", $remember_token, $new_resident_id);
                    $stmt->execute();
                    
                    setcookie('remember_token', $remember_token, $expires, '/', '', false, true);
                    setcookie('resident_email', $email, $expires, '/', '', false, false);
                    
                    header('Location: resident-dashboard.php');
                    exit;
                } else {
                    $error = 'Erreur lors de la création du compte';
                }
            }
        }
    }
}

if (isset($_COOKIE['remember_token'])) {
    $remember_token = $_COOKIE['remember_token'];
    $stmt = $conn->prepare("SELECT id, first_name, email FROM residents WHERE remember_token = ?");
    $stmt->bind_param("s", $remember_token);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($user = $result->fetch_assoc()) {
        $_SESSION['resident_id'] = $user['id'];
        $_SESSION['resident_name'] = $user['first_name'];
        header('Location: resident-dashboard.php');
        exit;
    } else {
        setcookie('remember_token', '', time() - 3600, '/');
        setcookie('resident_email', '', time() - 3600, '/');
    }
}

$saved_email = isset($_COOKIE['resident_email']) ? $_COOKIE['resident_email'] : '';
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Compte Résidant - Vert Chasseur</title>
    <link rel="icon" type="image/png" href="logo_new.png">
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {}
            }
        }
    </script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Archivo+Black&display=swap" rel="stylesheet">
    <style>
        @media (min-width: 768px) {
            body {
                background-image: url('font-homepage.png');
                background-size: cover;
                background-position: center;
                background-attachment: fixed;
                background-repeat: no-repeat;
            }
        }
        @media (max-width: 767px) {
            body {
                background-color: #ffffff !important;
                background-image: none !important;
            }
            .dark body {
                background-color: #000000 !important;
                background-image: none !important;
            }
        }
        body {
            font-family: 'Inter', sans-serif;
        }
        .txt-quarter {
              font-family: "Archivo Black", sans-serif;
              font-weight: 400;
              font-style: bold;
        }
        .address-results {
            max-height: 200px;
            overflow-y: auto;
        }
        .address-item {
            transition: all 0.2s ease;
        }
        .address-item:hover {
            background-color: #f5f5f4;
            transform: translateX(4px);
        }
        .dark .address-item:hover {
            background-color: #292524;
        }
    </style>
</head>
<body class="bg-white dark:bg-black transition-colors duration-300">
    <script>
        if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }
    </script>

<div class="min-h-screen flex items-center justify-center p-4">
    <div class="w-full max-w-md">
        <div class="mb-6">
            <h1 class="text-4xl font-light text-stone-800 dark:text-white mb-2">
                Bienvenue sur <span class="txt-quarter font-bold">Quarter</span>
            </h1>
            <p class="text-stone-600 dark:text-stone-400 mb-2">Soyez privilégié là où vous vivez</p>
            <p class="text-xs text-stone-500 dark:text-stone-500">Livraison privée, actualité locale et plus encore à Fort Jaco et Vert Chasseur</p>
        </div>

        <?php if ($error): ?>
        <div class="mb-4 p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-xl">
            <p class="text-red-600 dark:text-red-400 text-sm"><?= htmlspecialchars($error) ?></p>
        </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
        <div class="mb-4 p-3 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-xl">
            <p class="text-green-600 dark:text-green-400 text-sm"><?= htmlspecialchars($success) ?></p>
        </div>
        <?php endif; ?>

        <div class="mb-6">
            <div class="flex bg-stone-100 dark:bg-stone-800 rounded-xl p-1">
                <button id="login-tab" onclick="toggleForm()" class="flex-1 py-2.5 text-sm font-medium transition-all rounded-lg bg-white dark:bg-black text-stone-900 dark:text-white shadow-sm">
                    Connexion
                </button>
                <button id="register-tab" onclick="toggleForm()" class="flex-1 py-2.5 text-sm font-medium transition-all rounded-lg text-stone-600 dark:text-stone-400">
                    Créer un compte
                </button>
            </div>
        </div>

        <form id="login-form" method="POST" class="space-y-4">
            <input type="hidden" name="action" value="login">
            
            <input type="email" name="email" required placeholder="Email" value="<?= htmlspecialchars($saved_email) ?>"
                   class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-sm">
            
            <input type="password" name="password" required placeholder="Mot de passe" 
                   class="w-full px-4 py-3 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-sm">
            
            <button type="submit" 
                    class="w-full bg-stone-900 dark:bg-white text-white dark:text-black py-3 rounded-xl font-medium hover:opacity-90 transition-all text-sm">
                Se connecter
            </button>
        </form>

        <form id="register-form" method="POST" class="space-y-3 hidden">
            <input type="hidden" name="action" value="register">
            
            <div class="grid grid-cols-2 gap-3">
                <input type="text" name="first_name" required placeholder="Prénom" 
                       class="px-4 py-2.5 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-sm">
                <input type="text" name="last_name" required placeholder="Nom" 
                       class="px-4 py-2.5 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-sm">
            </div>
            
            <input type="email" name="email" required placeholder="Email" 
                   class="w-full px-4 py-2.5 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-sm">
            
            <div class="relative">
                <input type="text" id="address-input" name="address" required placeholder="Adresse à Fort Jaco ou Vert Chasseur" 
                       class="w-full px-4 py-2.5 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-sm">
                
                <div id="address-results" class="hidden absolute top-full left-0 right-0 mt-1 bg-white dark:bg-stone-900 border border-stone-200 dark:border-stone-700 rounded-xl shadow-xl z-50 address-results">
                </div>
            </div>
            
            <input type="password" name="password" required minlength="6" placeholder="Mot de passe (min. 6 caractères)" 
                   class="w-full px-4 py-2.5 rounded-xl border border-stone-200 dark:border-stone-700 bg-white dark:bg-stone-900 text-stone-900 dark:text-white placeholder-stone-500 dark:placeholder-stone-400 focus:ring-2 focus:ring-stone-900 dark:focus:ring-white focus:border-transparent transition-all text-sm">
            
            <button type="submit" 
                    class="w-full bg-stone-900 dark:bg-white text-white dark:text-black py-3 rounded-xl font-medium hover:opacity-90 transition-all text-sm">
                Créer mon compte
            </button>
        </form>

        <div class="mt-6 text-center">
            <a href="?guest=1" class="inline-flex items-center text-stone-600 dark:text-stone-400 hover:text-stone-800 dark:hover:text-stone-200 transition-colors text-xs mb-3">
                <svg class="w-3 h-3 mr-1.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
                Continuer sans compte
            </a>
        </div>
    </div>
</div>

<script>
    function toggleForm() {
        const loginForm = document.getElementById('login-form');
        const registerForm = document.getElementById('register-form');
        const loginTab = document.getElementById('login-tab');
        const registerTab = document.getElementById('register-tab');
        
        if (loginForm.classList.contains('hidden')) {
            loginForm.classList.remove('hidden');
            registerForm.classList.add('hidden');
            loginTab.classList.add('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
            loginTab.classList.remove('text-stone-600', 'dark:text-stone-400');
            registerTab.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
            registerTab.classList.add('text-stone-600', 'dark:text-stone-400');
        } else {
            loginForm.classList.add('hidden');
            registerForm.classList.remove('hidden');
            registerTab.classList.add('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
            registerTab.classList.remove('text-stone-600', 'dark:text-stone-400');
            loginTab.classList.remove('bg-white', 'dark:bg-black', 'text-stone-900', 'dark:text-white', 'shadow-sm');
            loginTab.classList.add('text-stone-600', 'dark:text-stone-400');
        }
    }

    let searchTimeout;

    document.addEventListener('DOMContentLoaded', function() {
        const addressInput = document.getElementById('address-input');
        const addressResults = document.getElementById('address-results');

        if (addressInput) {
            addressInput.addEventListener('input', function() {
                const query = this.value.trim();
                
                clearTimeout(searchTimeout);
                
                if (query.length < 3) {
                    addressResults.classList.add('hidden');
                    return;
                }

                searchTimeout = setTimeout(() => {
                    searchAddresses(query);
                }, 500);
            });

            document.addEventListener('click', function(e) {
                if (!addressInput.contains(e.target) && !addressResults.contains(e.target)) {
                    addressResults.classList.add('hidden');
                }
            });
        }
    });

    async function searchAddresses(query) {
        const addressResults = document.getElementById('address-results');
        
        try {
            addressResults.innerHTML = '<div class="p-3 text-center text-stone-500 dark:text-stone-400 text-xs">Recherche...</div>';
            addressResults.classList.remove('hidden');

            const responses = await Promise.all([
                fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(query + ', Fort Jaco, Belgium')}&countrycodes=be&limit=4&addressdetails=1`),
                fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(query + ', Vert Chasseur, Belgium')}&countrycodes=be&limit=4&addressdetails=1`)
            ]);
            
            if (!responses[0].ok || !responses[1].ok) {
                throw new Error('Erreur de recherche');
            }

            const [fortJacoData, vertChasseurData] = await Promise.all([
                responses[0].json(),
                responses[1].json()
            ]);
            
            const allResults = [...(fortJacoData || []), ...(vertChasseurData || [])];
            
            const uniqueResults = allResults.filter((item, index, self) => 
                index === self.findIndex(t => t.place_id === item.place_id)
            );

            const validResults = uniqueResults.filter(item => {
                const displayNameLower = item.display_name.toLowerCase();
                return displayNameLower.includes('fort jaco') || 
                       displayNameLower.includes('vert chasseur') ||
                       (item.address && (
                           (item.address.neighbourhood && (
                               item.address.neighbourhood.toLowerCase().includes('fort jaco') ||
                               item.address.neighbourhood.toLowerCase().includes('vert chasseur')
                           )) ||
                           (item.address.suburb && (
                               item.address.suburb.toLowerCase().includes('fort jaco') ||
                               item.address.suburb.toLowerCase().includes('vert chasseur')
                           ))
                       ));
            });
            
            if (validResults.length > 0) {
                const sortedResults = validResults.sort((a, b) => {
                    const aLower = a.display_name.toLowerCase();
                    const bLower = b.display_name.toLowerCase();
                    const queryLower = query.toLowerCase();
                    
                    const aStartsWithQuery = aLower.startsWith(queryLower);
                    const bStartsWithQuery = bLower.startsWith(queryLower);
                    
                    if (aStartsWithQuery && !bStartsWithQuery) return -1;
                    if (!aStartsWithQuery && bStartsWithQuery) return 1;
                    
                    return parseFloat(a.importance || 0) > parseFloat(b.importance || 0) ? -1 : 1;
                });

                addressResults.innerHTML = sortedResults.slice(0, 6).map(item => {
                    let displayName = item.display_name;
                    const parts = displayName.split(',');
                    
                    if (parts.length > 3) {
                        displayName = parts.slice(0, 2).join(',');
                    }
                    
                    const quarter = displayName.toLowerCase().includes('fort jaco') ? 'Fort Jaco' : 'Vert Chasseur';
                    
                    return `<div class="address-item p-3 cursor-pointer text-stone-900 dark:text-white text-xs border-b border-stone-100 dark:border-stone-800 last:border-b-0" onclick="selectAddress('${displayName.replace(/'/g, "\\'")}')">
                        <div class="font-medium leading-tight">${displayName}</div>
                        <div class="text-xs text-stone-500 dark:text-stone-400 mt-1 flex items-center justify-between">
                            <span>${item.address && item.address.postcode ? item.address.postcode : ''}</span>
                            <span class="text-green-600 dark:text-green-400 font-medium">${quarter}</span>
                        </div>
                    </div>`
                }).join('');
            } else {
                addressResults.innerHTML = `
                    <div class="p-3 text-center">
                        <div class="text-amber-600 dark:text-amber-400 text-xs font-medium mb-1">Aucune adresse trouvée</div>
                        <div class="text-xs text-stone-500 dark:text-stone-400">Seules les adresses à Fort Jaco ou Vert Chasseur</div>
                    </div>`;
            }
        } catch (error) {
            addressResults.innerHTML = '<div class="p-3 text-center text-red-500 text-xs">Erreur de recherche</div>';
        }
    }

    function selectAddress(address) {
        document.getElementById('address-input').value = address;
        document.getElementById('address-results').classList.add('hidden');
    }
</script>

</body>
</html>