<?php
$action = isset($_GET['action']) ? $_GET['action'] : 'view';
$album_id = isset($_GET['album']) ? intval($_GET['album']) : null;
$photo_id = isset($_GET['photo']) ? intval($_GET['photo']) : null;

if ($action === 'upload' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_FILES['photos']) && !empty($_FILES['photos']['name'][0])) {
        $album_id = isset($_POST['album_id']) ? intval($_POST['album_id']) : null;
        $titles = isset($_POST['titles']) ? $_POST['titles'] : [];
        $descriptions = isset($_POST['descriptions']) ? $_POST['descriptions'] : [];
        
        $upload_dir = 'families/' . $family['family_slug'] . '/';
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
        $max_size = 5 * 1024 * 1024; // 5MB
        
        $success_count = 0;
        $errors = [];
        
        for ($i = 0; $i < count($_FILES['photos']['name']); $i++) {
            if ($_FILES['photos']['error'][$i] === UPLOAD_ERR_OK) {
                $tmp_name = $_FILES['photos']['tmp_name'][$i];
                $name = $_FILES['photos']['name'][$i];
                $size = $_FILES['photos']['size'][$i];
                $type = $_FILES['photos']['type'][$i];
                
                if (!in_array($type, $allowed_types)) {
                    $errors[] = "File type not allowed for {$name}";
                    continue;
                }
                
                if ($size > $max_size) {
                    $errors[] = "File size too large for {$name}";
                    continue;
                }
                
                $timestamp = time();
                $ext = pathinfo($name, PATHINFO_EXTENSION);
                $new_filename = $timestamp . '_' . preg_replace('/[^a-zA-Z0-9]/', '', pathinfo($name, PATHINFO_FILENAME)) . '.' . $ext;
                $destination = $upload_dir . $new_filename;
                
                $title = isset($titles[$i]) ? $titles[$i] : '';
                $description = isset($descriptions[$i]) ? $descriptions[$i] : '';
                
                $title = $title ?: pathinfo($name, PATHINFO_FILENAME);
                
                try {
                    $image = new Imagick($tmp_name);
                    $image->stripImage();
                    $image->setImageCompressionQuality(85);
                    
                    if ($image->getImageWidth() > 2000 || $image->getImageHeight() > 2000) {
                        $image->resizeImage(2000, 2000, Imagick::FILTER_LANCZOS, 1, true);
                    }
                    
                    $api_url = 'https://cdn.imators.com/api/upload.php';
                    $data = [
                        'api_key' => 'your_api_key',
                        'file_data' => base64_encode($image->getImageBlob()),
                        'file_name' => $new_filename,
                        'destination' => $upload_dir
                    ];
                    
                    $options = [
                        'http' => [
                            'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                            'method' => 'POST',
                            'content' => http_build_query($data)
                        ]
                    ];
                    
                    $context = stream_context_create($options);
                    $result = file_get_contents($api_url, false, $context);
                    $response = json_decode($result, true);
                    
                    if ($response && isset($response['success']) && $response['success']) {
                        $stmt = $db->prepare("INSERT INTO photos (family_id, album_id, title, description, file_path, uploaded_by, uploaded_at) VALUES (?, ?, ?, ?, ?, ?, NOW())");
                        $stmt->execute([$family_id, $album_id, $title, $description, $destination, $user_id]);
                        $success_count++;
                    } else {
                        $errors[] = "CDN upload failed for {$name}: " . ($response['message'] ?? 'Unknown error');
                    }
                } catch (Exception $e) {
                    $errors[] = "Error processing {$name}: " . $e->getMessage();
                }
            } else {
                $errors[] = "Upload error for file #" . ($i + 1) . ": " . $_FILES['photos']['error'][$i];
            }
        }
        
        if ($success_count > 0) {
            $success_message = "{$success_count} photo" . ($success_count > 1 ? 's' : '') . " uploaded successfully";
        }
        
        if (count($errors) > 0) {
            $error_message = implode('<br>', $errors);
        }
        
        if ($album_id) {
            header("Location: ?page=photos&album={$album_id}");
        } else {
            header("Location: ?page=photos");
        }
        exit;
    }
}

if ($action === 'create_album' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $album_name = isset($_POST['album_name']) ? trim($_POST['album_name']) : '';
    $album_description = isset($_POST['album_description']) ? trim($_POST['album_description']) : '';
    
    if (!empty($album_name)) {
        $stmt = $db->prepare("INSERT INTO albums (family_id, name, description, created_by, created_at) VALUES (?, ?, ?, ?, NOW())");
        $stmt->execute([$family_id, $album_name, $album_description, $user_id]);
        $new_album_id = $db->lastInsertId();
        
        header("Location: ?page=photos&album={$new_album_id}");
        exit;
    }
}

if ($action === 'delete_photo' && $is_admin && $photo_id) {
    $stmt = $db->prepare("SELECT * FROM photos WHERE id = ? AND family_id = ?");
    $stmt->execute([$photo_id, $family_id]);
    $photo = $stmt->fetch();
    
    if ($photo) {
        $api_url = 'https://cdn.imators.com/api/delete.php';
        $data = [
            'api_key' => 'your_api_key',
            'file_path' => $photo['file_path']
        ];
        
        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data)
            ]
        ];
        
        $context = stream_context_create($options);
        file_get_contents($api_url, false, $context);
        
        $stmt = $db->prepare("DELETE FROM photos WHERE id = ?");
        $stmt->execute([$photo_id]);
        
        header("Location: ?page=photos" . ($album_id ? "&album={$album_id}" : ""));
        exit;
    }
}
?>

<div class="mb-6">
    <div class="flex items-center justify-between mb-4">
        <h1 class="text-2xl font-bold">Family Photos</h1>
        <div class="flex space-x-2">
            <button class="bg-white text-indigo-600 px-4 py-2 rounded-lg border border-indigo-600 hover:bg-indigo-50 transition" onclick="document.getElementById('create-album-modal').classList.remove('hidden')">
                New Album
            </button>
            <button class="bg-indigo-600 text-white px-4 py-2 rounded-lg hover:bg-indigo-700 transition" onclick="document.getElementById('upload-modal').classList.remove('hidden')">
                Upload Photos
            </button>
        </div>
    </div>

    <?php if ($album_id): ?>
        <?php
        $stmt = $db->prepare("SELECT * FROM albums WHERE id = ? AND family_id = ?");
        $stmt->execute([$album_id, $family_id]);
        $album = $stmt->fetch();
        
        if ($album):
        ?>
            <div class="mb-6">
                <div class="flex items-center justify-between">
                    <div>
                        <h2 class="text-xl font-bold"><?php echo htmlspecialchars($album['name']); ?></h2>
                        <?php if (!empty($album['description'])): ?>
                            <p class="text-gray-600"><?php echo htmlspecialchars($album['description']); ?></p>
                        <?php endif; ?>
                    </div>
                    <a href="?page=photos" class="text-indigo-600 hover:text-indigo-800">← Back to all photos</a>
                </div>
            </div>
        <?php endif; ?>
    <?php else: ?>
        <div class="mb-6">
            <h2 class="text-xl font-bold mb-4">Albums</h2>
            <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                <?php
                $stmt = $db->prepare("SELECT a.*, COUNT(p.id) AS photo_count, 
                                        (SELECT file_path FROM photos WHERE album_id = a.id ORDER BY uploaded_at DESC LIMIT 1) AS cover_photo 
                                     FROM albums a 
                                     LEFT JOIN photos p ON a.id = p.album_id AND p.family_id = a.family_id 
                                     WHERE a.family_id = ? 
                                     GROUP BY a.id
                                     ORDER BY a.created_at DESC");
                $stmt->execute([$family_id]);
                $albums = $stmt->fetchAll();
                
                foreach ($albums as $album):
                ?>
                    <a href="?page=photos&album=<?php echo $album['id']; ?>" class="bg-white shadow rounded-lg overflow-hidden hover:shadow-md transition">
                        <div class="aspect-square bg-gray-100">
                            <?php if ($album['cover_photo']): ?>
                                <img src="<?php echo htmlspecialchars($cdn_url . '/' . $album['cover_photo']); ?>" alt="<?php echo htmlspecialchars($album['name']); ?>" class="w-full h-full object-cover">
                            <?php else: ?>
                                <div class="w-full h-full flex items-center justify-center text-gray-400">
                                    <svg xmlns="http://www.w3.org/2000/svg" class="h-12 w-12" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                    </svg>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div class="p-4">
                            <h3 class="font-medium"><?php echo htmlspecialchars($album['name']); ?></h3>
                            <p class="text-sm text-gray-500"><?php echo $album['photo_count']; ?> photo<?php echo $album['photo_count'] != 1 ? 's' : ''; ?></p>
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>
        
        <h2 class="text-xl font-bold mb-4">All Photos</h2>
    <?php endif; ?>

    <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 mb-4">
        <?php
        $query = "SELECT * FROM photos WHERE family_id = ?";
        $params = [$family_id];
        
        if ($album_id) {
            $query .= " AND album_id = ?";
            $params[] = $album_id;
        }
        
        $query .= " ORDER BY uploaded_at DESC";
        
        $stmt = $db->prepare($query);
        $stmt->execute($params);
        $photos = $stmt->fetchAll();
        
        foreach ($photos as $photo):
        ?>
            <div class="bg-white shadow rounded-lg overflow-hidden group">
                <a href="#" onclick="openPhotoModal(<?php echo $photo['id']; ?>); return false;" class="block aspect-square">
                    <img src="<?php echo htmlspecialchars($cdn_url . '/' . $photo['file_path']); ?>" alt="<?php echo htmlspecialchars($photo['title']); ?>" class="w-full h-full object-cover group-hover:opacity-90 transition">
                </a>
                <div class="p-3">
                    <h3 class="font-medium truncate"><?php echo htmlspecialchars($photo['title']); ?></h3>
                    <div class="flex justify-between items-center mt-1">
                        <?php
                        $stmt = $db->prepare("SELECT username FROM utilisateurs WHERE id = ?");
                        $stmt->execute([$photo['uploaded_by']]);
                        $uploader = $stmt->fetch();
                        ?>
                        <span class="text-xs text-gray-500">By <?php echo htmlspecialchars($uploader['username']); ?></span>
                        <span class="text-xs text-gray-500"><?php echo date('M j, Y', strtotime($photo['uploaded_at'])); ?></span>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
        
        <?php if (count($photos) == 0): ?>
            <div class="col-span-full py-12 text-center text-gray-500">
                <svg xmlns="http://www.w3.org/2000/svg" class="h-12 w-12 mx-auto mb-4 text-gray-300" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                </svg>
                <p>No photos yet. Upload some to get started.</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Upload Modal -->
<div id="upload-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
    <div class="bg-white rounded-lg max-w-2xl w-full max-h-screen overflow-y-auto">
        <div class="p-6">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-xl font-bold">Upload Photos</h2>
                <button onclick="document.getElementById('upload-modal').classList.add('hidden')" class="text-gray-500 hover:text-gray-700">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                    </svg>
                </button>
            </div>
            
            <form action="?page=photos&action=upload" method="POST" enctype="multipart/form-data">
                <div class="mb-4">
                    <label for="album_id" class="block text-sm font-medium text-gray-700 mb-1">Album (optional)</label>
                    <select id="album_id" name="album_id" class="w-full rounded-lg border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        <option value="">No Album</option>
                        <?php
                        $stmt = $db->prepare("SELECT * FROM albums WHERE family_id = ? ORDER BY name");
                        $stmt->execute([$family_id]);
                        $albums = $stmt->fetchAll();
                        
                        foreach ($albums as $album) {
                            $selected = ($album['id'] == $album_id) ? 'selected' : '';
                            echo "<option value=\"{$album['id']}\" {$selected}>" . htmlspecialchars($album['name']) . "</option>";
                        }
                        ?>
                    </select>
                </div>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Photos</label>
                    <div class="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center hover:border-indigo-500 transition cursor-pointer" id="dropzone">
                        <input type="file" name="photos[]" id="file-input" class="hidden" multiple accept="image/*" onchange="handleFiles(this.files)">
                        <svg xmlns="http://www.w3.org/2000/svg" class="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12" />
                        </svg>
                        <p class="mt-1 text-sm text-gray-600">Drag and drop photos or click to select</p>
                        <p class="mt-1 text-xs text-gray-500">JPEG, PNG, GIF up to 5MB</p>
                    </div>
                </div>
                
                <div id="preview-container" class="mt-4 space-y-4 hidden">
                    <!-- Preview will be added here by JS -->
                </div>
                
                <div class="flex justify-end mt-6">
                    <button type="button" onclick="document.getElementById('upload-modal').classList.add('hidden')" class="bg-white px-4 py-2 border border-gray-300 rounded-lg text-gray-700 mr-2 hover:bg-gray-50 transition">Cancel</button>
                    <button type="submit" class="bg-indigo-600 text-white px-4 py-2 rounded-lg hover:bg-indigo-700 transition">Upload</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Create Album Modal -->
<div id="create-album-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
    <div class="bg-white rounded-lg max-w-md w-full">
        <div class="p-6">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-xl font-bold">Create New Album</h2>
                <button onclick="document.getElementById('create-album-modal').classList.add('hidden')" class="text-gray-500 hover:text-gray-700">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                    </svg>
                </button>
            </div>
            
            <form action="?page=photos&action=create_album" method="POST">
                <div class="mb-4">
                    <label for="album_name" class="block text-sm font-medium text-gray-700 mb-1">Album Name</label>
                    <input type="text" id="album_name" name="album_name" required class="w-full rounded-lg border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                </div>
                
                <div class="mb-4">
                    <label for="album_description" class="block text-sm font-medium text-gray-700 mb-1">Description (optional)</label>
                    <textarea id="album_description" name="album_description" rows="3" class="w-full rounded-lg border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500"></textarea>
                </div>
                
                <div class="flex justify-end mt-6">
                    <button type="button" onclick="document.getElementById('create-album-modal').classList.add('hidden')" class="bg-white px-4 py-2 border border-gray-300 rounded-lg text-gray-700 mr-2 hover:bg-gray-50 transition">Cancel</button>
                    <button type="submit" class="bg-indigo-600 text-white px-4 py-2 rounded-lg hover:bg-indigo-700 transition">Create Album</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Photo Modal -->
<div id="photo-modal" class="fixed inset-0 bg-black bg-opacity-90 flex items-center justify-center z-50 hidden">
    <div class="relative max-w-4xl w-full h-full flex flex-col items-center justify-center">
        <button onclick="closePhotoModal()" class="absolute top-4 right-4 text-white hover:text-gray-300">
            <svg xmlns="http://www.w3.org/2000/svg" class="h-8 w-8" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
            </svg>
        </button>
        
        <button id="prev-photo" class="absolute left-4 text-white hover:text-gray-300">
            <svg xmlns="http://www.w3.org/2000/svg" class="h-8 w-8" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7" />
            </svg>
        </button>
        
        <div id="photo-content" class="max-h-[80vh] max-w-full"></div>
        
        <button id="next-photo" class="absolute right-4 text-white hover:text-gray-300">
            <svg xmlns="http://www.w3.org/2000/svg" class="h-8 w-8" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7" />
            </svg>
        </button>
        
        <div id="photo-info" class="absolute bottom-0 left-0 right-0 bg-black bg-opacity-70 text-white p-4">
            <h3 id="photo-title" class="text-lg font-medium"></h3>
            <p id="photo-description" class="text-sm text-gray-300"></p>
            <div class="flex justify-between items-center mt-2">
                <div>
                    <span id="photo-uploader" class="text-xs text-gray-300"></span>
                    <span id="photo-date" class="text-xs text-gray-300 ml-2"></span>
                </div>
                <div>
                    <a id="download-link" href="#" class="text-indigo-300 hover:text-white text-sm mr-4" download>Download</a>
                    <?php if ($is_admin): ?>
                        <a id="delete-link" href="#" class="text-red-300 hover:text-white text-sm" onclick="return confirm('Are you sure you want to delete this photo? This cannot be undone.')">Delete</a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const dropzone = document.getElementById('dropzone');
    const fileInput = document.getElementById('file-input');
    
    dropzone.addEventListener('click', function() {
        fileInput.click();
    });
    
    dropzone.addEventListener('dragover', function(e) {
        e.preventDefault();
        dropzone.classList.add('border-indigo-500');
    });
    
    dropzone.addEventListener('dragleave', function() {
        dropzone.classList.remove('border-indigo-500');
    });
    
    dropzone.addEventListener('drop', function(e) {
        e.preventDefault();
        dropzone.classList.remove('border-indigo-500');
        
        if (e.dataTransfer.files.length > 0) {
            fileInput.files = e.dataTransfer.files;
            handleFiles(e.dataTransfer.files);
        }
    });
});

function handleFiles(files) {
    const previewContainer = document.getElementById('preview-container');
    previewContainer.innerHTML = '';
    previewContainer.classList.remove('hidden');
    
    for (let i = 0; i < files.length; i++) {
        const file = files[i];
        if (!file.type.startsWith('image/')) continue;
        
        const reader = new FileReader();
        reader.onload = function(e) {
            const preview = document.createElement('div');
            preview.className = 'flex items-start p-4 border border-gray-200 rounded-lg';
            preview.innerHTML = `
                <img src="${e.target.result}" alt="${file.name}" class="w-16 h-16 object-cover rounded mr-4">
                <div class="flex-1">
                    <input type="text" name="titles[]" placeholder="Title" value="${file.name.split('.')[0]}" class="w-full mb-2 border-gray-300 rounded-md shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    <textarea name="descriptions[]" placeholder="Description (optional)" rows="2" class="w-full border-gray-300 rounded-md shadow-sm focus:border-indigo-500 focus:ring-indigo-500"></textarea>
                </div>
            `;
            previewContainer.appendChild(preview);
        };
        reader.readAsDataURL(file);
    }
}

let currentPhotoId = null;
let photoIds = [];

function openPhotoModal(photoId) {
    currentPhotoId = photoId;
    loadPhotoData(photoId);
    document.getElementById('photo-modal').classList.remove('hidden');
    
    // Load photo IDs if not already loaded
    if (photoIds.length === 0) {
        const photoElements = document.querySelectorAll('[onclick^="openPhotoModal"]');
        photoIds = Array.from(photoElements).map(el => {
            const match = el.getAttribute('onclick').match(/openPhotoModal\((\d+)\)/);
            return match ? parseInt(match[1]) : null;
        }).filter(id => id !== null);
    }
    
    // Set up navigation buttons
    const currentIndex = photoIds.indexOf(photoId);
    const prevButton = document.getElementById('prev-photo');
    const nextButton = document.getElementById('next-photo');
    
    prevButton.style.display = currentIndex > 0 ? 'block' : 'none';
    nextButton.style.display = currentIndex < photoIds.length - 1 ? 'block' : 'none';
    
    prevButton.onclick = function() {
        if (currentIndex > 0) {
            openPhotoModal(photoIds[currentIndex - 1]);
        }
    };
    
    nextButton.onclick = function() {
        if (currentIndex < photoIds.length - 1) {
            openPhotoModal(photoIds[currentIndex + 1]);
        }
    };
}

function closePhotoModal() {
    document.getElementById('photo-modal').classList.add('hidden');
}

function loadPhotoData(photoId) {
    fetch(`?page=photos&action=get_photo&photo=${photoId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const photo = data.photo;
                document.getElementById('photo-content').innerHTML = `<img src="${photo.url}" alt="${photo.title}" class="max-h-[70vh] max-w-full">`;
                document.getElementById('photo-title').textContent = photo.title;
                document.getElementById('photo-description').textContent = photo.description || '';
                document.getElementById('photo-uploader').textContent = `Uploaded by ${photo.uploader}`;
                document.getElementById('photo-date').textContent = photo.date;
                document.getElementById('download-link').href = photo.url;
                document.getElementById('download-link').download = photo.title;
                
                if (document.getElementById('delete-link')) {
                    document.getElementById('delete-link').href = `?page=photos&action=delete_photo&photo=${photoId}${photo.album_id ? '&album=' + photo.album_id : ''}`;
                }
            }
        });
}

document.addEventListener('keydown', function(e) {
    if (document.getElementById('photo-modal').classList.contains('hidden')) return;
    
    if (e.key === 'Escape') {
        closePhotoModal();
    } else if (e.key === 'ArrowLeft') {
        document.getElementById('prev-photo').click();
    } else if (e.key === 'ArrowRight') {
        document.getElementById('next-photo').click();
    }
});
</script>