const axios = require('axios');
const config = require('../config.json');

const conversationBuffers = new Map();
const CONVERSATION_TIMEOUT = 300000;

class ClaudeHandler {
    static async makeClaudeRequest(prompt) {
        try {
            const response = await axios.post('https://api.anthropic.com/v1/messages', {
                model: 'claude-3-sonnet-20240229',
                max_tokens: 1000,
                messages: [
                    {
                        role: 'user',
                        content: prompt
                    }
                ]
            }, {
                headers: {
                    'Content-Type': 'application/json',
                    'x-api-key': config.claudeApiKey,
                    'anthropic-version': '2023-06-01'
                }
            });

            return response.data.content[0].text;
        } catch (error) {
            console.error('Erreur Claude API:', error.response?.data || error.message);
            return null;
        }
    }

    static async checkMessageSecurity(messageContent) {
        const prompt = `Analyse ce message Discord et détermine s'il contient du spam, des insultes, du contenu offensant ou inapproprié. Réponds uniquement par "OUI" si c'est inapproprié ou "NON" si c'est acceptable.

Message à analyser: "${messageContent}"

Réponds seulement OUI ou NON:`;

        const response = await this.makeClaudeRequest(prompt);
        if (!response) return false;
        
        return response.trim().toUpperCase() === 'OUI';
    }

    static async analyzeConversation(message) {
        const channelId = message.channel.id;
        
        if (!conversationBuffers.has(channelId)) {
            conversationBuffers.set(channelId, {
                messages: [],
                lastActivity: Date.now(),
                timeout: null
            });
        }

        const buffer = conversationBuffers.get(channelId);
        buffer.messages.push({
            author: message.author.username,
            content: message.content,
            timestamp: message.createdTimestamp
        });
        buffer.lastActivity = Date.now();

        if (buffer.timeout) {
            clearTimeout(buffer.timeout);
        }

        buffer.timeout = setTimeout(async () => {
            await this.processConversationEnd(message.channel, buffer.messages);
            conversationBuffers.delete(channelId);
        }, CONVERSATION_TIMEOUT);

        if (buffer.messages.length >= 10) {
            const shouldDelete = await this.shouldDeleteMessages(buffer.messages);
            if (shouldDelete) {
                try {
                    const messagesToDelete = await message.channel.messages.fetch({ limit: 5 });
                    await message.channel.bulkDelete(messagesToDelete);
                } catch (error) {
                    console.log('Impossible de supprimer les messages en masse');
                }
            }
        }
    }

    static async processConversationEnd(channel, messages) {
        if (messages.length < 3) return;

        const conversationText = messages.map(m => `${m.author}: ${m.content}`).join('\n');
        
        const prompt = `Analyse cette conversation Discord qui vient de se terminer. Détermine si elle contient du contenu offensant, des critiques malveillantes, du harcèlement ou tout autre comportement toxique qui justifierait la suppression des messages.

Conversation:
${conversationText}

Réponds uniquement par "SUPPRIMER" si les messages doivent être supprimés ou "GARDER" s'ils sont acceptables:`;

        const response = await this.makeClaudeRequest(prompt);
        if (!response) return;

        if (response.trim().toUpperCase() === 'SUPPRIMER') {
            try {
                const messagesToDelete = await channel.messages.fetch({ limit: messages.length });
                await channel.bulkDelete(messagesToDelete);
                
                await channel.send('⚠️ Des messages ont été supprimés par le système d\'analyse automatique pour violation des règles du serveur.');
            } catch (error) {
                console.log('Impossible de supprimer les messages de la conversation');
            }
        }
    }

    static async shouldDeleteMessages(messages) {
        const recentMessages = messages.slice(-5);
        const conversationText = recentMessages.map(m => `${m.author}: ${m.content}`).join('\n');
        
        const prompt = `Analyse ces messages Discord récents. Détermine s'ils contiennent du contenu qui nécessite une suppression immédiate (spam, insultes graves, harcèlement).

Messages:
${conversationText}

Réponds uniquement par "OUI" pour supprimer ou "NON" pour garder:`;

        const response = await this.makeClaudeRequest(prompt);
        if (!response) return false;
        
        return response.trim().toUpperCase() === 'OUI';
    }
}

module.exports = ClaudeHandler;