<?php
require_once 'db.php';

$articles = [];

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $stmt = $pdo->query("SELECT id, title, description, date_created FROM articles ORDER BY date_created DESC");
    $articles = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch(PDOException $e) {
    $error = 'Database connection failed: ' . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Blog - Izhak</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="icon" href="icon.png" type="image/png">
    <link href="https://api.fontshare.com/v2/css?f[]=cabinet-grotesk@400,500,700,800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        :root {
            --bg-light: #ffffff;
            --bg-dark: #121212;
            --text-light: #121212;
            --text-dark: #f5f5f5;
            --border-light: rgba(0, 0, 0, 0.1);
            --border-dark: rgba(255, 255, 255, 0.1);
        }

        @media (prefers-color-scheme: dark) {
            body {
                background-color: var(--bg-dark);
                color: var(--text-dark);
            }
            .header-border {
                border-top: 1px solid var(--text-dark);
                border-bottom: 1px solid var(--text-dark);
                background-color: rgba(18, 18, 18, 0.8);
            }
            .border-custom {
                border-color: var(--border-dark);
            }
        }

        @media (prefers-color-scheme: light) {
            body {
                background-color: var(--bg-light);
                color: var(--text-light);
            }
            .header-border {
                border-top: 1px solid var(--text-light);
                border-bottom: 1px solid var(--text-light);
                background-color: rgba(255, 255, 255, 0.8);
            }
            .border-custom {
                border-color: var(--border-light);
            }
        }

        body {
            font-family: 'Cabinet Grotesk', sans-serif;
            transition: background-color 0.3s ease, color 0.3s ease;
            margin: 0;
            padding: 0;
        }

        .header-border {
            backdrop-filter: blur(10px);
        }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .slide-up {
            opacity: 0;
            animation: slideUp 0.6s cubic-bezier(0.4, 0, 0.2, 1) forwards;
        }

        .delay-1 {animation-delay: 0.1s;}
        .delay-2 {animation-delay: 0.2s;}
        .delay-3 {animation-delay: 0.3s;}
        .delay-4 {animation-delay: 0.4s;}

        .hero-section {
            height: 60vh;
            display: flex;
            align-items: center;
            justify-content: center;
            text-align: center;
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 2rem;
        }

        .section-header {
            background: rgba(0,0,0,0.02);
            border-bottom: 1px solid var(--border-light);
            padding: 2rem 0;
        }

        @media (prefers-color-scheme: dark) {
            .section-header {
                background: rgba(255,255,255,0.02);
                border-color: var(--border-dark);
            }
        }

        .section-title {
            font-size: 0.8rem;
            text-transform: uppercase;
            letter-spacing: 0.15em;
            opacity: 0.4;
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 2rem;
        }

        .article-item {
            border-bottom: 1px solid var(--border-light);
            padding: 0;
        }

        @media (prefers-color-scheme: dark) {
            .article-item {
                border-color: var(--border-dark);
            }
        }

        .article-link {
            display: flex;
            align-items: center;
            gap: 2rem;
            padding: 3rem 2rem;
            text-decoration: none;
            color: inherit;
            transition: opacity 0.2s ease;
            max-width: 1200px;
            margin: 0 auto;
            cursor: pointer;
        }

        .article-link:hover {
            opacity: 0.7;
        }

        .article-number {
            font-size: 1.2rem;
            font-weight: 300;
            opacity: 0.5;
            min-width: 3rem;
        }

        .article-info {
            flex: 1;
        }

        .article-title {
            font-size: 1.8rem;
            font-weight: 300;
            margin-bottom: 0.5rem;
            letter-spacing: -0.01em;
        }

        .article-description {
            font-size: 0.95rem;
            opacity: 0.6;
            line-height: 1.5;
            margin-bottom: 0.5rem;
        }

        .article-date {
            font-size: 0.8rem;
            opacity: 0.4;
        }

        .article-meta {
            display: flex;
            align-items: center;
            gap: 1.5rem;
        }

        .article-status {
            font-size: 0.8rem;
            text-transform: uppercase;
            letter-spacing: 0.1em;
            opacity: 0.5;
        }

        .article-icon {
            font-size: 1.2rem;
            opacity: 0.3;
        }

        .cta-section {
            padding: 4rem 2rem;
            text-align: center;
            max-width: 1200px;
            margin: 0 auto;
        }

        .button-custom {
            background-color: var(--text-light);
            color: var(--bg-light);
            padding: 1rem 2rem;
            border-radius: 2rem;
            font-size: 0.9rem;
            font-weight: 300;
            letter-spacing: 0.05em;
            transition: opacity 0.2s ease;
            text-decoration: none;
            display: inline-block;
        }

        @media (prefers-color-scheme: dark) {
            .button-custom {
                background-color: var(--text-dark);
                color: var(--bg-dark);
            }
        }

        .button-custom:hover {
            opacity: 0.8;
        }

        .button-secondary {
            background-color: transparent;
            color: var(--text-light);
            border: 1px solid var(--border-light);
        }

        @media (prefers-color-scheme: dark) {
            .button-secondary {
                color: var(--text-dark);
                border-color: var(--border-dark);
            }
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            backdrop-filter: blur(10px);
        }

        .modal.active {
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .modal-content {
            background-color: var(--bg-light);
            color: var(--text-light);
            padding: 3rem;
            border-radius: 1rem;
            max-width: 400px;
            width: 90%;
            text-align: center;
            animation: slideUp 0.3s ease;
        }

        @media (prefers-color-scheme: dark) {
            .modal-content {
                background-color: var(--bg-dark);
                color: var(--text-dark);
            }
        }

        .modal-title {
            font-size: 1.5rem;
            font-weight: 300;
            margin-bottom: 1rem;
        }

        .modal-text {
            opacity: 0.7;
            margin-bottom: 2rem;
        }

        .modal-input {
            width: 100%;
            padding: 1rem;
            border: 1px solid var(--border-light);
            border-radius: 0.5rem;
            font-family: 'Cabinet Grotesk', sans-serif;
            font-size: 1rem;
            margin-bottom: 2rem;
            background-color: transparent;
            color: inherit;
        }

        @media (prefers-color-scheme: dark) {
            .modal-input {
                border-color: var(--border-dark);
            }
        }

        .modal-input:focus {
            outline: none;
            opacity: 1;
        }

        .modal-actions {
            display: flex;
            gap: 1rem;
            justify-content: center;
        }

        .modal-message {
            margin-top: 1rem;
            padding: 1rem;
            border-radius: 0.5rem;
            font-size: 0.9rem;
            display: none;
        }

        .modal-message.success {
            background-color: rgba(34, 197, 94, 0.1);
            border: 1px solid rgba(34, 197, 94, 0.3);
            color: #22c55e;
        }

        .modal-message.error {
            background-color: rgba(239, 68, 68, 0.1);
            border: 1px solid rgba(239, 68, 68, 0.3);
            color: #ef4444;
        }

        @media (max-width: 768px) {
            .hero-section {
                height: 50vh;
                padding: 0 1rem;
            }
            
            .article-link {
                padding: 2rem 1rem;
                flex-direction: column;
                align-items: flex-start;
                gap: 1rem;
            }
            
            .article-meta {
                align-self: flex-end;
            }
            
            .article-title {
                font-size: 1.4rem;
            }
            
            .section-title {
                padding: 0 1rem;
            }
            
            .cta-section {
                padding: 3rem 1rem;
            }
        }
    </style>
</head>
<body class="min-h-screen">
    <header class="header-border py-6 sticky top-0 z-50 slide-up">
        <div class="max-width-1200px mx-auto px-6">
            <div class="flex justify-between items-center">
                <h1 class="text-2xl font-light tracking-tight">Izhak</h1>
                <a href="index.html" class="text-sm font-light hover:opacity-70 transition-opacity">← Back</a>
            </div>
        </div>
    </header>

    <section class="hero-section slide-up delay-1">
        <div>
            <h2 class="text-4xl md:text-6xl font-light tracking-tight mb-4">Blog & Insights</h2>
            <p class="text-lg md:text-xl font-light opacity-70 max-w-2xl mx-auto">
                Thoughts, ideas and experiences from the world of technology and innovation
            </p>
        </div>
    </section>

    <main>
        <div class="section-header slide-up delay-2">
            <h3 class="section-title">Latest Articles</h3>
        </div>
        
        <?php if(isset($error)): ?>
            <div class="max-w-1200px mx-auto px-6 py-8 text-center">
                <p class="text-red-500">Error: <?php echo htmlspecialchars($error); ?></p>
            </div>
        <?php elseif(empty($articles)): ?>
            <div class="max-w-1200px mx-auto px-6 py-8 text-center">
                <p class="opacity-60">No articles found.</p>
            </div>
        <?php else: ?>
            <?php 
            function createSlug($title) {
                $slug = strtolower($title);
                $slug = str_replace(' ', '-', $slug);
                $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);
                $slug = preg_replace('/-+/', '-', $slug);
                $slug = trim($slug, '-');
                return $slug;
            }
            ?>
            <?php foreach($articles as $index => $article): ?>
                <div class="article-item slide-up delay-<?php echo ($index % 4) + 3; ?>">
                    <a href="view.php?title=<?php echo createSlug($article['title']); ?>" class="article-link">
                        <div class="article-number"><?php echo str_pad($index + 1, 2, '0', STR_PAD_LEFT); ?></div>
                        <div class="article-info">
                            <div class="article-title"><?php echo htmlspecialchars($article['title']); ?></div>
                            <div class="article-description"><?php echo htmlspecialchars($article['description']); ?></div>
                            <div class="article-date"><?php echo date('F d, Y', strtotime($article['date_created'])); ?></div>
                        </div>
                        <div class="article-meta">
                            <div class="article-status">Read more</div>
                            <div class="article-icon"><i class="fas fa-arrow-right"></i></div>
                        </div>
                    </a>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>

        <section class="cta-section">
            <h2 class="text-3xl md:text-4xl font-light mb-6">Want to stay updated?</h2>
            <p class="text-lg font-light opacity-70 mb-8 max-w-2xl mx-auto">
                Subscribe to get the latest insights and updates directly in your inbox
            </p>
            <button onclick="openModal()" class="button-custom">
                Get in touch
            </button>
        </section>

        <div id="modal" class="modal" onclick="closeModal(event)">
            <div class="modal-content" onclick="event.stopPropagation()">
                <h3 class="modal-title">Abonne-toi à la newsletter</h3>
                <p class="modal-text">Reçois les derniers articles directement dans ta boîte mail</p>
                
                <form id="subscribeForm" onsubmit="subscribe(event)">
                    <input type="email" id="emailInput" placeholder="ton@email.com" class="modal-input" required>
                    <div class="modal-actions">
                        <button type="submit" class="button-custom" id="subscribeBtn">
                            S'abonner
                        </button>
                        <button type="button" onclick="closeModal()" class="button-custom button-secondary">
                            Annuler
                        </button>
                    </div>
                </form>
                
                <div id="modalMessage" class="modal-message"></div>
            </div>
        </div>
    </main>

    <footer class="py-8 text-center text-sm font-light opacity-50">
        <p>Designed with a smile.</p>
    </footer>

    <script>
        function openModal() {
            document.getElementById('modal').classList.add('active');
        }

        function closeModal(event) {
            if(!event || event.target.id === 'modal') {
                document.getElementById('modal').classList.remove('active');
                document.getElementById('modalMessage').style.display = 'none';
                document.getElementById('subscribeForm').style.display = 'block';
                document.getElementById('emailInput').value = '';
            }
        }

        function subscribe(event) {
            event.preventDefault();
            
            const email = document.getElementById('emailInput').value;
            const button = document.getElementById('subscribeBtn');
            const message = document.getElementById('modalMessage');
            
            button.textContent = 'Inscription...';
            button.disabled = true;
            
            fetch('subscribe.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'email=' + encodeURIComponent(email)
            })
            .then(response => response.json())
            .then(data => {
                message.textContent = data.message;
                message.className = 'modal-message ' + (data.success ? 'success' : 'error');
                message.style.display = 'block';
                
                if(data.success) {
                    document.getElementById('subscribeForm').style.display = 'none';
                    setTimeout(() => {
                        closeModal();
                    }, 2000);
                }
                
                button.textContent = 'S\'abonner';
                button.disabled = false;
            })
            .catch(error => {
                message.textContent = 'Une erreur est survenue.';
                message.className = 'modal-message error';
                message.style.display = 'block';
                
                button.textContent = 'S\'abonner';
                button.disabled = false;
            });
        }
    </script>
</body>
</html>