<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vérification d'Identité</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/qrcode-generator/1.4.4/qrcode.min.js"></script>
    <style>
        .camera-frame {
            position: relative;
            width: 100%;
            aspect-ratio: 16/10;
            max-width: 450px;
            border: 2px solid #374151;
            border-radius: 12px;
            overflow: hidden;
            background: #111827;
        }
        video, canvas {
            width: 100%;
            height: 100%;
            object-fit: contain;
            background: #000;
        }
        .step {
            display: none;
            opacity: 0;
            transform: translateY(15px);
            transition: all 0.3s ease;
        }
        .step.active {
            display: block;
            opacity: 1;
            transform: translateY(0);
        }
        .desktop-only {
            display: none;
        }
        @media (min-width: 1024px) {
            .mobile-only {
                display: none;
            }
            .desktop-only {
                display: flex;
            }
        }
        .qr-container {
            width: 200px;
            height: 200px;
            background: white;
            border-radius: 12px;
            padding: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
    </style>
</head>
<body class="bg-black text-white min-h-screen">

    <div class="desktop-only flex-col items-center justify-center min-h-screen p-8">
        <div class="text-center space-y-6 max-w-md">
            <h1 class="text-3xl font-light">Vérification d'identité</h1>
            <p class="text-gray-400">Scanne ce QR code avec ton téléphone</p>
            <div class="qr-container mx-auto">
                <div id="qr-code">Chargement...</div>
            </div>
            <div class="bg-gray-900 rounded-xl p-4">
                <p class="text-xs text-gray-400 mb-2">URL mobile :</p>
                <p id="mobile-url" class="text-blue-400 text-xs break-all"></p>
            </div>
        </div>
    </div>

    <div class="mobile-only container mx-auto px-6 py-6 max-w-md">
        
        <div class="mb-6">
            <div class="flex items-center justify-between mb-4">
                <h1 class="text-xl font-light">Imators Identity</h1>
                <span class="text-xs text-gray-400" id="service-name">Service</span>
            </div>
            <div class="w-full bg-gray-800 rounded-full h-1">
                <div class="bg-white h-1 rounded-full transition-all duration-500" id="progress" style="width: 25%"></div>
            </div>
            <p class="text-xs text-gray-400 mt-2" id="step-indicator">Étape 1 sur 4</p>
        </div>

        <div class="step active" id="step-1">
            <div class="text-center space-y-6">
                <h2 class="text-2xl font-light mb-4">Vérifions que c'est bien toi</h2>
                <p class="text-gray-400 text-sm mb-6">Prépare ta carte d'identité pour la vérification</p>
                
                <div class="space-y-4 text-left">
                    <div class="flex items-start space-x-3">
                        <div class="w-8 h-8 bg-gray-800 rounded-full flex items-center justify-center flex-shrink-0">
                            <span class="text-xs">1</span>
                        </div>
                        <div>
                            <h3 class="font-medium text-sm">Carte propre et lisible</h3>
                            <p class="text-gray-400 text-xs">Assure-toi qu'elle soit bien visible</p>
                        </div>
                    </div>
                    <div class="flex items-start space-x-3">
                        <div class="w-8 h-8 bg-gray-800 rounded-full flex items-center justify-center flex-shrink-0">
                            <span class="text-xs">2</span>
                        </div>
                        <div>
                            <h3 class="font-medium text-sm">Bonne lumière</h3>
                            <p class="text-gray-400 text-xs">Évite les reflets et ombres</p>
                        </div>
                    </div>
                    <div class="flex items-start space-x-3">
                        <div class="w-8 h-8 bg-gray-800 rounded-full flex items-center justify-center flex-shrink-0">
                            <span class="text-xs">3</span>
                        </div>
                        <div>
                            <h3 class="font-medium text-sm">Photos recto et verso</h3>
                            <p class="text-gray-400 text-xs">Capture les deux faces</p>
                        </div>
                    </div>
                </div>
                
                <button class="w-full bg-white text-black py-3 rounded-xl font-medium hover:bg-gray-100 transition-colors mt-8" id="start-btn">
                    Commencer
                </button>
            </div>
        </div>

        <div class="step" id="step-2">
            <div class="text-center space-y-6">
                <h2 class="text-2xl font-light">Photo Recto</h2>
                <p class="text-gray-400 text-sm">Place ta carte dans le cadre</p>
                
                <div class="camera-frame mx-auto">
                    <video id="front-video" autoplay playsinline style="display: none;"></video>
                    <div id="front-placeholder" class="flex items-center justify-center h-full">
                        <button id="start-front-camera" class="bg-gray-800 hover:bg-gray-700 px-6 py-3 rounded-xl transition-colors">
                            Ouvrir caméra
                        </button>
                    </div>
                    <canvas id="front-preview" style="display: none;"></canvas>
                    <button id="capture-front" class="absolute bottom-3 left-1/2 transform -translate-x-1/2 bg-white text-black px-6 py-2 rounded-full font-medium hover:bg-gray-100 transition-colors" style="display: none;">
                        Capturer
                    </button>
                </div>
                
                <input type="file" id="front-file" accept="image/*" style="display: none;">
                <button id="front-gallery-btn" class="text-gray-400 text-sm underline hover:text-white transition-colors">
                    Choisir depuis la galerie
                </button>
                
                <div class="flex space-x-3">
                    <button class="flex-1 bg-gray-800 py-3 rounded-xl hover:bg-gray-700 transition-colors" id="back-1">
                        Retour
                    </button>
                    <button id="next-front" class="flex-1 bg-white text-black py-3 rounded-xl font-medium opacity-50 cursor-not-allowed" disabled>
                        Suivant
                    </button>
                </div>
            </div>
        </div>

        <div class="step" id="step-3">
            <div class="text-center space-y-6">
                <h2 class="text-2xl font-light">Photo Verso</h2>
                <p class="text-gray-400 text-sm">Retourne ta carte</p>
                
                <div class="camera-frame mx-auto">
                    <video id="back-video" autoplay playsinline style="display: none;"></video>
                    <div id="back-placeholder" class="flex items-center justify-center h-full">
                        <button id="start-back-camera" class="bg-gray-800 hover:bg-gray-700 px-6 py-3 rounded-xl transition-colors">
                            Ouvrir caméra
                        </button>
                    </div>
                    <canvas id="back-preview" style="display: none;"></canvas>
                    <button id="capture-back" class="absolute bottom-3 left-1/2 transform -translate-x-1/2 bg-white text-black px-6 py-2 rounded-full font-medium hover:bg-gray-100 transition-colors" style="display: none;">
                        Capturer
                    </button>
                </div>
                
                <input type="file" id="back-file" accept="image/*" style="display: none;">
                <button id="back-gallery-btn" class="text-gray-400 text-sm underline hover:text-white transition-colors">
                    Choisir depuis la galerie
                </button>
                
                <div class="flex space-x-3">
                    <button class="flex-1 bg-gray-800 py-3 rounded-xl hover:bg-gray-700 transition-colors" id="back-2">
                        Retour
                    </button>
                    <button id="next-back" class="flex-1 bg-white text-black py-3 rounded-xl font-medium opacity-50 cursor-not-allowed" disabled>
                        Suivant
                    </button>
                </div>
            </div>
        </div>

        <div class="step" id="step-4">
            <div class="space-y-6">
                <h2 class="text-2xl font-light text-center">Confirmation</h2>
                
                <div class="grid grid-cols-2 gap-3">
                    <div class="bg-gray-900 rounded-xl p-3">
                        <p class="text-xs text-gray-400 mb-2">Recto</p>
                        <canvas id="final-front-preview" class="w-full h-16 rounded bg-gray-800"></canvas>
                    </div>
                    <div class="bg-gray-900 rounded-xl p-3">
                        <p class="text-xs text-gray-400 mb-2">Verso</p>
                        <canvas id="final-back-preview" class="w-full h-16 rounded bg-gray-800"></canvas>
                    </div>
                </div>

                <div class="bg-gray-900 rounded-xl p-4">
                    <label class="text-sm font-medium mb-3 block">Pays de résidence</label>
                    <select id="country" required class="w-full bg-gray-800 border-0 rounded-lg px-3 py-3 text-white text-sm">
                        <option value="">Sélectionner</option>
                        <option value="FR">France</option>
                        <option value="BE">Belgique</option>
                        <option value="CH">Suisse</option>
                        <option value="CA">Canada</option>
                        <option value="US">États-Unis</option>
                        <option value="DE">Allemagne</option>
                        <option value="IT">Italie</option>
                        <option value="ES">Espagne</option>
                        <option value="PT">Portugal</option>
                        <option value="NL">Pays-Bas</option>
                        <option value="GB">Royaume-Uni</option>
                    </select>
                </div>

                <div class="space-y-3">
                    <label class="flex items-start space-x-3 cursor-pointer">
                        <input type="checkbox" id="no-hidden" required class="mt-1 h-4 w-4 rounded border-gray-600 bg-transparent">
                        <span class="text-sm">Aucune partie cachée</span>
                    </label>
                    <label class="flex items-start space-x-3 cursor-pointer">
                        <input type="checkbox" id="visible-clear" required class="mt-1 h-4 w-4 rounded border-gray-600 bg-transparent">
                        <span class="text-sm">Photos nettes et lisibles</span>
                    </label>
                </div>
                
                <div class="bg-red-500/10 border border-red-500/30 rounded-xl p-4">
                    <div class="flex items-start space-x-3">
                        <span class="text-red-400">ⓘ</span>
                        <div>
                            <h4 class="text-sm font-medium text-red-200 mb-1">Information importante</h4>
                            <p class="text-xs text-red-200">Tes données sont stockées 12h pour vérification puis supprimées définitivement selon l'E.S.P (Ephemeral Storage Protocol).</p>
                        </div>
                    </div>
                </div>
                
                <div class="flex space-x-3">
                    <button class="flex-1 bg-gray-800 py-3 rounded-xl hover:bg-gray-700 transition-colors" id="back-3">
                        Retour
                    </button>
                    <button id="submit-btn" class="flex-1 bg-white text-black py-3 rounded-xl font-medium hover:bg-gray-100 transition-colors">
                        Envoyer
                    </button>
                </div>
                
                <div class="text-center">
                    <p class="text-xs text-gray-500">
                        🔒 Données sécurisées et conformes RGPD
                    </p>
                </div>
            </div>
        </div>
    </div>

    <script>
        const urlParams = new URLSearchParams(window.location.search);
        const service = Object.keys(Object.fromEntries(urlParams))[0] || 'Service';
        
        if (document.getElementById('service-name')) {
            document.getElementById('service-name').textContent = service;
        }

        if (window.innerWidth >= 1024) {
            const currentUrl = window.location.href;
            document.getElementById('mobile-url').textContent = currentUrl;
            
            try {
                const qr = qrcode(0, 'M');
                qr.addData(currentUrl);
                qr.make();
                document.getElementById('qr-code').innerHTML = qr.createImgTag(3);
            } catch (error) {
                document.getElementById('qr-code').textContent = 'QR indisponible';
            }
        }

        let currentStep = 1;
        let frontCaptured = false, backCaptured = false;
        let frontBlob = null, backBlob = null;
        let frontDataUrl = null, backDataUrl = null;

        const progressWidths = ['25%', '50%', '75%', '100%'];

        function goToStep(step) {
            document.getElementById(`step-${currentStep}`).classList.remove('active');
            setTimeout(() => {
                currentStep = step;
                document.getElementById(`step-${currentStep}`).classList.add('active');
                document.getElementById('progress').style.width = progressWidths[step - 1];
                document.getElementById('step-indicator').textContent = `Étape ${step} sur 4`;
                
                if (step === 4) showFinalPreviews();
            }, 150);
        }

        function showFinalPreviews() {
            if (frontDataUrl) {
                const canvas = document.getElementById('final-front-preview');
                const ctx = canvas.getContext('2d');
                const img = new Image();
                img.onload = () => {
                    const ratio = Math.min(canvas.width / img.width, canvas.height / img.height);
                    const x = (canvas.width - img.width * ratio) / 2;
                    const y = (canvas.height - img.height * ratio) / 2;
                    ctx.clearRect(0, 0, canvas.width, canvas.height);
                    ctx.drawImage(img, x, y, img.width * ratio, img.height * ratio);
                };
                img.src = frontDataUrl;
            }
            if (backDataUrl) {
                const canvas = document.getElementById('final-back-preview');
                const ctx = canvas.getContext('2d');
                const img = new Image();
                img.onload = () => {
                    const ratio = Math.min(canvas.width / img.width, canvas.height / img.height);
                    const x = (canvas.width - img.width * ratio) / 2;
                    const y = (canvas.height - img.height * ratio) / 2;
                    ctx.clearRect(0, 0, canvas.width, canvas.height);
                    ctx.drawImage(img, x, y, img.width * ratio, img.height * ratio);
                };
                img.src = backDataUrl;
            }
        }

        function compressImage(file, maxWidth = 1200, quality = 0.85) {
            return new Promise((resolve) => {
                const canvas = document.createElement('canvas');
                const ctx = canvas.getContext('2d');
                const img = new Image();
                
                img.onload = () => {
                    const ratio = Math.min(maxWidth / img.width, maxWidth / img.height);
                    canvas.width = img.width * ratio;
                    canvas.height = img.height * ratio;
                    ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
                    canvas.toBlob(resolve, 'image/jpeg', quality);
                };
                
                const reader = new FileReader();
                reader.onload = (e) => img.src = e.target.result;
                reader.readAsDataURL(file);
            });
        }

        function blobToDataURL(blob) {
            return new Promise((resolve) => {
                const reader = new FileReader();
                reader.onload = (e) => resolve(e.target.result);
                reader.readAsDataURL(blob);
            });
        }

        async function startCamera(side) {
            try {
                const stream = await navigator.mediaDevices.getUserMedia({ 
                    video: { facingMode: 'environment', width: 1280, height: 720 }
                });
                
                const video = document.getElementById(`${side}-video`);
                const placeholder = document.getElementById(`${side}-placeholder`);
                const captureBtn = document.getElementById(`capture-${side}`);
                
                video.srcObject = stream;
                video.style.display = 'block';
                placeholder.style.display = 'none';
                captureBtn.style.display = 'block';
                
            } catch (error) {
                alert('Impossible d\'accéder à la caméra');
            }
        }

        async function captureImage(side) {
            const video = document.getElementById(`${side}-video`);
            const canvas = document.createElement('canvas');
            const preview = document.getElementById(`${side}-preview`);
            const captureBtn = document.getElementById(`capture-${side}`);
            
            canvas.width = video.videoWidth;
            canvas.height = video.videoHeight;
            canvas.getContext('2d').drawImage(video, 0, 0);
            
            canvas.toBlob(async (blob) => {
                const compressed = await compressImage(blob);
                const dataUrl = await blobToDataURL(compressed);
                
                const img = new Image();
                img.onload = () => {
                    preview.width = img.width;
                    preview.height = img.height;
                    const ctx = preview.getContext('2d');
                    ctx.drawImage(img, 0, 0);
                    preview.style.display = 'block';
                    video.style.display = 'none';
                    captureBtn.style.display = 'none';
                };
                img.src = dataUrl;
                
                if (side === 'front') {
                    frontCaptured = true;
                    frontBlob = compressed;
                    frontDataUrl = dataUrl;
                    const btn = document.getElementById('next-front');
                    btn.disabled = false;
                    btn.classList.remove('opacity-50', 'cursor-not-allowed');
                } else {
                    backCaptured = true;
                    backBlob = compressed;
                    backDataUrl = dataUrl;
                    const btn = document.getElementById('next-back');
                    btn.disabled = false;
                    btn.classList.remove('opacity-50', 'cursor-not-allowed');
                }
                
                video.srcObject.getTracks().forEach(track => track.stop());
            });
        }

        async function handleFileSelect(side, file) {
            const compressed = await compressImage(file);
            const dataUrl = await blobToDataURL(compressed);
            
            const preview = document.getElementById(`${side}-preview`);
            const img = new Image();
            
            img.onload = () => {
                preview.width = img.width;
                preview.height = img.height;
                preview.getContext('2d').drawImage(img, 0, 0);
                preview.style.display = 'block';
                document.getElementById(`${side}-placeholder`).style.display = 'none';
            };
            img.src = dataUrl;
            
            if (side === 'front') {
                frontBlob = compressed;
                frontDataUrl = dataUrl;
                frontCaptured = true;
                const btn = document.getElementById('next-front');
                btn.disabled = false;
                btn.classList.remove('opacity-50', 'cursor-not-allowed');
            } else {
                backBlob = compressed;
                backDataUrl = dataUrl;
                backCaptured = true;
                const btn = document.getElementById('next-back');
                btn.disabled = false;
                btn.classList.remove('opacity-50', 'cursor-not-allowed');
            }
        }

        if (window.innerWidth < 1024) {
            document.getElementById('start-btn').addEventListener('click', () => goToStep(2));
            document.getElementById('back-1').addEventListener('click', () => goToStep(1));
            document.getElementById('back-2').addEventListener('click', () => goToStep(2));
            document.getElementById('back-3').addEventListener('click', () => goToStep(3));
            document.getElementById('next-front').addEventListener('click', () => goToStep(3));
            document.getElementById('next-back').addEventListener('click', () => goToStep(4));

            document.getElementById('start-front-camera').addEventListener('click', () => startCamera('front'));
            document.getElementById('start-back-camera').addEventListener('click', () => startCamera('back'));
            document.getElementById('capture-front').addEventListener('click', () => captureImage('front'));
            document.getElementById('capture-back').addEventListener('click', () => captureImage('back'));

            document.getElementById('front-gallery-btn').addEventListener('click', () => {
                document.getElementById('front-file').click();
            });
            document.getElementById('back-gallery-btn').addEventListener('click', () => {
                document.getElementById('back-file').click();
            });

            document.getElementById('front-file').addEventListener('change', (e) => {
                if (e.target.files[0]) handleFileSelect('front', e.target.files[0]);
            });
            document.getElementById('back-file').addEventListener('change', (e) => {
                if (e.target.files[0]) handleFileSelect('back', e.target.files[0]);
            });

            document.getElementById('submit-btn').addEventListener('click', async () => {
                if (!frontCaptured || !backCaptured) {
                    alert('Photos manquantes');
                    return;
                }
                
                if (!document.getElementById('country').value) {
                    alert('Sélectionne ton pays');
                    return;
                }
                
                if (!document.getElementById('no-hidden').checked || !document.getElementById('visible-clear').checked) {
                    alert('Coche les déclarations');
                    return;
                }
                
                document.getElementById('submit-btn').textContent = 'Envoi...';
                document.getElementById('submit-btn').disabled = true;
                
                const formData = new FormData();
                formData.append('front_image', new File([frontBlob], 'front-id.jpg', { type: 'image/jpeg' }));
                formData.append('back_image', new File([backBlob], 'back-id.jpg', { type: 'image/jpeg' }));
                formData.append('country', document.getElementById('country').value);
                formData.append('service', service);
                formData.append('no_hidden', document.getElementById('no-hidden').checked);
                formData.append('visible_clear', document.getElementById('visible-clear').checked);
                
                try {
                    const response = await fetch('process_verification.php', {
                        method: 'POST',
                        body: formData
                    });
                    
                    const result = await response.json();
                    
                    if (result.success) {
                        alert('Vérification envoyée !');
                        window.location.href = 'success.php';
                    } else {
                        alert('Erreur: ' + result.message);
                        document.getElementById('submit-btn').textContent = 'Envoyer';
                        document.getElementById('submit-btn').disabled = false;
                    }
                } catch (error) {
                    alert('Erreur de connexion');
                    document.getElementById('submit-btn').textContent = 'Envoyer';
                    document.getElementById('submit-btn').disabled = false;
                }
            });
        }
    </script>
</body>
</html>