<?php
require_once '../search/db.php';
session_start();

// Vérification de la connexion et du rôle
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'writer') {
    header('Location: dashboard.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$message = '';
$error = '';

// Récupération des informations de l'utilisateur
$query = "SELECT first_name, last_name FROM users WHERE id = :id";
$stmt = $db->prepare($query);
$stmt->execute([':id' => $user_id]);
$user = $stmt->fetch();
$author_name = $user['first_name'] . ' ' . $user['last_name'];

// Traitement de la création/modification d'article
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = trim($_POST['title'] ?? '');
    $content = trim($_POST['content'] ?? '');
    $read_time = (int)($_POST['read_time'] ?? 5);
    $article_id = isset($_POST['article_id']) ? (int)$_POST['article_id'] : null;

    if (empty($title) || empty($content)) {
        $error = 'Title and content are required.';
    } else {
        try {
            if ($article_id) {
                // Mise à jour d'un article existant
                $query = "UPDATE data_article 
                         SET title = :title, content = :content, read_time = :read_time 
                         WHERE id = :id AND author = :author";
                $params = [
                    ':title' => $title,
                    ':content' => $content,
                    ':read_time' => $read_time,
                    ':id' => $article_id,
                    ':author' => $author_name
                ];
                $message = 'Article updated successfully.';
            } else {
                // Création d'un nouvel article
                $query = "INSERT INTO data_article (title, content, author, read_time) 
                         VALUES (:title, :content, :author, :read_time)";
                $params = [
                    ':title' => $title,
                    ':content' => $content,
                    ':author' => $author_name,
                    ':read_time' => $read_time
                ];
                $message = 'Article published successfully.';
            }

            $stmt = $db->prepare($query);
            $stmt->execute($params);

            if (!$article_id) {
                $article_id = $db->lastInsertId();
            }

            // Redirection vers l'article
            header("Location: ../search/article.php?id=$article_id");
            exit();
        } catch (PDOException $e) {
            $error = 'Error saving article.';
        }
    }
}

// Récupération de l'article si en mode édition
$edit_article = null;
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $query = "SELECT * FROM data_article WHERE id = :id AND author = :author";
    $stmt = $db->prepare($query);
    $stmt->execute([
        ':id' => $_GET['edit'],
        ':author' => $author_name
    ]);
    $edit_article = $stmt->fetch();
    
    if (!$edit_article) {
        header('Location: dashboard-write.php');
        exit();
    }
}

// Récupération des articles de l'utilisateur
$articles_query = "SELECT id, title, created_at FROM data_article 
                  WHERE author = :author 
                  ORDER BY created_at DESC";
$stmt = $db->prepare($articles_query);
$stmt->execute([':author' => $author_name]);
$articles = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en" class="h-full">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $edit_article ? 'Edit Article' : 'Write Article'; ?> - Peerkinton</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://api.fontshare.com/v2/css?f[]=satoshi@400,500,700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Satoshi', sans-serif;
            background: #fafafa;
        }

        .nav-link {
            transition: all 0.2s ease;
        }

        .nav-link:hover {
            background-color: rgba(0, 0, 0, 0.05);
        }

        .nav-link.active {
            background-color: #f3f4f6;
            color: #111827;
        }

        .editor-toolbar {
            position: sticky;
            top: 0;
            background: white;
            z-index: 10;
            border-bottom: 1px solid #e5e7eb;
        }

        .editor-content {
            min-height: 500px;
            max-height: none;
        }

        .editor-content[contenteditable=true]:empty:before {
            content: 'Start writing your article...';
            color: #9ca3af;
        }

        /* Style pour le contenu */
        .editor-content h1 { font-size: 2em; font-weight: 600; margin: 1em 0; }
        .editor-content h2 { font-size: 1.5em; font-weight: 600; margin: 0.83em 0; }
        .editor-content h3 { font-size: 1.17em; font-weight: 600; margin: 0.67em 0; }
        .editor-content ul { list-style-type: disc; margin-left: 2em; }
        .editor-content ol { list-style-type: decimal; margin-left: 2em; }
        .editor-content code { 
            background: #f3f4f6; 
            padding: 0.2em 0.4em; 
            border-radius: 0.25em;
            font-family: monospace;
        }
        .editor-content pre {
            background: #f3f4f6;
            padding: 1em;
            border-radius: 0.5em;
            margin: 1em 0;
            overflow-x: auto;
        }
        .editor-content blockquote {
            border-left: 4px solid #e5e7eb;
            padding-left: 1em;
            margin: 1em 0;
            color: #4b5563;
        }
    </style>
</head>
<body class="h-full">
    <?php include '../header.php'; ?>

    <div class="flex min-h-screen pt-20">
        <!-- Sidebar -->
        <aside class="w-64 border-r border-gray-200 px-4 py-6">
            <nav class="space-y-2">
                <a href="dashboard.php" class="nav-link flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"/>
                    </svg>
                    Dashboard
                </a>
                <a href="dashboard-files.php" class="nav-link flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 8h14M5 8a2 2 0 110-4h14a2 2 0 110 4M5 8v10a2 2 0 002 2h10a2 2 0 002-2V8m-9 4h4"/>
                    </svg>
                    Casier membre
                </a>
                <a href="dashboard-profile.php" class="nav-link flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                    </svg>
                    Profil
                </a>
                <a href="dashboard-write.php" class="nav-link active flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                    </svg>
                    Rédaction
                </a>
            </nav>

            <!-- Articles List -->
            <div class="mt-8">
                <h3 class="px-4 text-xs font-medium text-gray-500 uppercase tracking-wider">Your Articles</h3>
                <div class="mt-4 space-y-2">
                    <?php foreach ($articles as $article): ?>
                        <a href="?edit=<?php echo $article['id']; ?>" 
                           class="block px-4 py-2 text-sm text-gray-600 hover:bg-gray-50 rounded-lg">
                            <?php echo htmlspecialchars($article['title']); ?>
                            <span class="block text-xs text-gray-400">
                                <?php echo date('j M Y', strtotime($article['created_at'])); ?>
                            </span>
                        </a>
                    <?php endforeach; ?>
                </div>
            </div>
        </aside>

        <!-- Main Content -->
        <main class="flex-1 px-8 py-6">
            <?php if ($message): ?>
                <div class="bg-green-50 border border-green-200 text-green-800 rounded-lg p-4 mb-6">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div class="bg-red-50 border border-red-200 text-red-800 rounded-lg p-4 mb-6">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <form method="POST" id="articleForm">
                <?php if ($edit_article): ?>
                    <input type="hidden" name="article_id" value="<?php echo $edit_article['id']; ?>">
                <?php endif; ?>

                <!-- Title -->
                <div class="mb-6">
                    <input 
                        type="text" 
                        name="title" 
                        placeholder="Article title" 
                        value="<?php echo $edit_article ? htmlspecialchars($edit_article['title']) : ''; ?>"
                        class="w-full text-3xl font-medium text-gray-900 border-0 border-b border-gray-200 focus:outline-none focus:border-gray-300 p-0 pb-2"
                        required
                    >
                </div>

                <!-- Editor Toolbar -->
                <div class="editor-toolbar p-4 space-x-4 flex items-center mb-4 rounded-t-lg">
                    <button type="button" data-command="formatBlock" data-value="h1" class="p-2 hover:bg-gray-100 rounded">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path d="M4 6h16M4 12h16M4 18h7" stroke-width="2" stroke-linecap="round"/>
                        </svg>
                    </button>
                    <button type="button" data-command="formatBlock" data-value="h2" class="p-2 hover:bg-gray-100 rounded">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path d="M4 6h16M4 12h8" stroke-width="2" stroke-linecap="round"/>
                        </svg>
                    </button>
                    <span class="w-px h-6 bg-gray-200"></span>
                    <button type="button" data-command="bold" class="p-2 hover:bg-gray-100 rounded font-bold">B</button>
                    <button type="button" data-command="italic" class="p-2 hover:bg-gray-100 rounded italic">I</button>
                    <span class="w-px h-6 bg-gray-200"></span>
                    <button type="button" data-command="insertUnorderedList" class="p-2 hover:bg-gray-100 rounded">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path d="M8 6h13M8 12h13M8 18h13M3 6h.01M3 12h.01M3 18h.01" stroke-width="2" stroke-linecap="round"/>
                        </svg>
                    </button>
                    <button type="button" data-command="insertOrderedList" class="p-2 hover:bg-gray-100 rounded">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path d="M4 6h16M4 12h16M4 18h16" stroke-width="2" stroke-linecap="round"/>
                        </svg>
                    </button>
                    
                    <span class="w-px h-6 bg-gray-200"></span>
                    <button type="button" data-command="code" class="p-2 hover:bg-gray-100 rounded font-mono">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path d="M10 20l4-16m4 4l4 4-4 4M6 16l-4-4 4-4" stroke-width="2" stroke-linecap="round"/>
                        </svg>
                    </button>
                    <button type="button" data-command="blockquote" class="p-2 hover:bg-gray-100 rounded">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path d="M7 8h10M7 12h4m1 8l-4-4H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-3l-4 4z" stroke-width="2" stroke-linecap="round"/>
                        </svg>
                    </button>
                </div>

                <!-- Editor Content -->
                <div 
                    id="editor" 
                    class="editor-content p-6 bg-white rounded-b-lg border border-gray-200" 
                    contenteditable="true"
                ><?php echo $edit_article ? $edit_article['content'] : ''; ?></div>
                
                <textarea 
                    name="content" 
                    id="hiddenContent" 
                    style="display: none;"
                ><?php echo $edit_article ? htmlspecialchars($edit_article['content']) : ''; ?></textarea>

                <!-- Reading Time -->
                <div class="mt-6">
                    <label for="read_time" class="block text-sm font-medium text-gray-700 mb-2">
                        Estimated Reading Time (minutes)
                    </label>
                    <input 
                        type="number" 
                        id="read_time" 
                        name="read_time" 
                        value="<?php echo $edit_article ? $edit_article['read_time'] : '5'; ?>"
                        min="1"
                        class="w-32 px-4 py-2 rounded-lg border border-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-200"
                        required
                    >
                </div>

                <!-- Submit Buttons -->
                <div class="mt-8 flex justify-end space-x-4">
                    <a 
                        href="dashboard.php" 
                        class="px-6 py-2 text-gray-700 hover:text-gray-900"
                    >
                        Cancel
                    </a>
                    <button 
                        type="submit" 
                        class="px-6 py-2 bg-gray-900 text-white rounded-full hover:transform hover:-translate-y-0.5 transition-all duration-200"
                    >
                        <?php echo $edit_article ? 'Update Article' : 'Publish Article'; ?>
                    </button>
                </div>
            </form>
        </main>
    </div>

    <script>
        const editor = document.getElementById('editor');
        const hiddenContent = document.getElementById('hiddenContent');
        const form = document.getElementById('articleForm');

        // Initialize toolbar buttons
        document.querySelectorAll('.editor-toolbar button').forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                const command = this.dataset.command;
                const value = this.dataset.value || null;

                if (command === 'code') {
                    const selection = window.getSelection();
                    const range = selection.getRangeAt(0);
                    const text = range.toString();
                    
                    if (text) {
                        const code = document.createElement('code');
                        code.textContent = text;
                        range.deleteContents();
                        range.insertNode(code);
                    }
                } else if (command === 'blockquote') {
                    document.execCommand('formatBlock', false, 'blockquote');
                } else {
                    document.execCommand(command, false, value);
                }
                
                editor.focus();
            });
        });

        // Update hidden textarea before form submission
        form.addEventListener('submit', function(e) {
            hiddenContent.value = editor.innerHTML;
        });

        // Auto calculate reading time
        editor.addEventListener('input', function() {
            const words = this.innerText.trim().split(/\s+/).length;
            const readTime = Math.max(1, Math.ceil(words / 200)); // Moyenne de 200 mots par minute
            document.getElementById('read_time').value = readTime;
        });

        // Handle paste to remove formatting
        editor.addEventListener('paste', function(e) {
            e.preventDefault();
            const text = (e.originalEvent || e).clipboardData.getData('text/plain');
            document.execCommand('insertText', false, text);
        });

        // Initialize reading time on load if editing
        if (editor.innerText.trim()) {
            const words = editor.innerText.trim().split(/\s+/).length;
            const readTime = Math.max(1, Math.ceil(words / 200));
            document.getElementById('read_time').value = readTime;
        }
    </script>
</body>
</html>