<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);
$travel = is_traveling($_SESSION['user_id']);

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'start_travel') {
    if ($travel) {
        $error = "Tu es déjà en déplacement. Impossible d'en commencer un nouveau.";
    } else {
        $start_country = $_POST['start_country'] ?? '';
        $end_country = $_POST['end_country'] ?? '';
        $start_coords = $_POST['start_coords'] ?? '';
        $end_coords = $_POST['end_coords'] ?? '';
        $transport_type = $_POST['transport_type'] ?? 'car';
        
        if (empty($start_country) || empty($end_country) || empty($start_coords) || empty($end_coords)) {
            $error = "Informations de déplacement incomplètes.";
        } else {
            $result = start_travel(
                $_SESSION['user_id'],
                $start_country,
                $end_country,
                $start_coords,
                $end_coords,
                $transport_type
            );
            
            $travel_id = $result['travel_id'];
            $arrival_time = $result['arrival_time'];
            $travel_minutes = $result['travel_minutes'];
            
            header('Location: travel.php?id=' . $travel_id);
            exit;
        }
    }
}

$countries = db_get_all("SELECT id, name, code FROM countries ORDER BY name");
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Carte du monde | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.7.1/dist/leaflet.css" />
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        #map {
            height: 600px;
            width: 100%;
            z-index: 1;
        }
        .card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300 font-semibold">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300">Système</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300">Missions</a></li>
                    <?php if ($user['level'] >= 10): ?>
                        <li><a href="unions.php" class="hover:text-blue-300">Unions</a></li>
                    <?php endif; ?>
                    <li><a href="profile.php" class="hover:text-blue-300">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Carte du monde</h2>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($travel): ?>
            <div class="bg-yellow-100 border border-yellow-400 text-yellow-700 px-4 py-3 rounded mb-4">
                <p>Tu es actuellement en déplacement. Arrivée prévue le <?php echo format_datetime($travel['arrival_time']); ?></p>
                <a href="travel.php" class="text-blue-600 hover:underline">Voir les détails du voyage</a>
            </div>
        <?php endif; ?>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div class="lg:col-span-2">
                <div class="card mb-6">
                    <div id="map"></div>
                </div>
            </div>
            
            <div>
                <div class="card p-6 mb-6">
                    <h3 class="text-xl font-semibold mb-4">Ta position</h3>
                    <div id="current-position" class="mb-4">
                        <p class="text-gray-500 italic">Chargement de ta position...</p>
                    </div>
                    
                    <div>
                        <button id="update-position" class="w-full bg-blue-600 hover:bg-blue-700 text-white font-medium py-2 px-4 rounded transition-colors">
                            Actualiser ma position
                        </button>
                    </div>
                </div>
                
                <?php if (!$travel): ?>
                    <div class="card p-6">
                        <h3 class="text-xl font-semibold mb-4">Planifier un déplacement</h3>
                        <form id="travel-form" method="POST" action="map.php" class="space-y-4">
                            <input type="hidden" name="action" value="start_travel">
                            <input type="hidden" name="start_country" id="start_country" value="<?php echo $user['country_id']; ?>">
                            <input type="hidden" name="start_coords" id="start_coords" value="">
                            
                            <div>
                                <label for="end_country" class="block text-sm font-medium mb-1">Pays de destination</label>
                                <select id="end_country" name="end_country" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                                    <option value="">Sélectionne un pays</option>
                                    <?php foreach ($countries as $country): ?>
                                        <option value="<?php echo $country['id']; ?>">
                                            <?php echo htmlspecialchars($country['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div>
                                <label for="destination" class="block text-sm font-medium mb-1">Destination</label>
                                <input type="text" id="destination" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Recherche une ville ou un lieu" required>
                                <input type="hidden" name="end_coords" id="end_coords" value="">
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium mb-1">Mode de transport</label>
                                <div class="grid grid-cols-2 gap-2">
                                    <label class="flex items-center p-3 border border-gray-300 rounded-md cursor-pointer hover:bg-gray-50">
                                        <input type="radio" name="transport_type" value="car" class="mr-2" checked>
                                        <span>Voiture</span>
                                    </label>
                                    <label class="flex items-center p-3 border border-gray-300 rounded-md cursor-pointer hover:bg-gray-50">
                                        <input type="radio" name="transport_type" value="train" class="mr-2">
                                        <span>Train</span>
                                    </label>
                                    <label class="flex items-center p-3 border border-gray-300 rounded-md cursor-pointer hover:bg-gray-50">
                                        <input type="radio" name="transport_type" value="plane" class="mr-2">
                                        <span>Avion</span>
                                    </label>
                                    <label class="flex items-center p-3 border border-gray-300 rounded-md cursor-pointer hover:bg-gray-50">
                                        <input type="radio" name="transport_type" value="boat" class="mr-2">
                                        <span>Bateau</span>
                                    </label>
                                </div>
                            </div>
                            
                            <div id="travel-info" class="p-4 bg-gray-100 rounded-md hidden">
                                <p><span class="font-medium">Distance:</span> <span id="travel-distance">-</span></p>
                                <p><span class="font-medium">Durée estimée:</span> <span id="travel-duration">-</span></p>
                            </div>
                            
                            <div>
                                <button type="submit" id="start-travel" class="w-full bg-green-600 hover:bg-green-700 text-white font-medium py-2 px-4 rounded transition-colors" disabled>
                                    Commencer le déplacement
                                </button>
                            </div>
                        </form>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <script src="https://unpkg.com/leaflet@1.7.1/dist/leaflet.js"></script>
    <script>
        let map;
        let userMarker;
        let destinationMarker;
        let travelPath;
        let currentPosition = null;
        
        document.addEventListener('DOMContentLoaded', function() {
            initMap();
            
            document.getElementById('update-position').addEventListener('click', function() {
                updateUserPosition();
            });
            
            <?php if (!$travel): ?>
                const destinationInput = document.getElementById('destination');
                const endCountrySelect = document.getElementById('end_country');
                
                destinationInput.addEventListener('blur', function() {
                    if (this.value.trim() !== '') {
                        geocodeDestination(this.value);
                    }
                });
                
                endCountrySelect.addEventListener('change', function() {
                    if (destinationInput.value.trim() !== '') {
                        geocodeDestination(destinationInput.value);
                    }
                });
            <?php endif; ?>
        });
        
        function initMap() {
            map = L.map('map').setView([48.8566, 2.3522], 5);
            
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
            }).addTo(map);
            
            updateUserPosition();
        }
        
        function updateUserPosition() {
            if (navigator.geolocation) {
                navigator.geolocation.getCurrentPosition(
                    function(position) {
                        const lat = position.coords.latitude;
                        const lng = position.coords.longitude;
                        
                        displayUserPosition(lat, lng);
                        
                        fetch(`https://nominatim.openstreetmap.org/reverse?format=jsonv2&lat=${lat}&lon=${lng}`)
                            .then(response => response.json())
                            .then(data => {
                                const address = data.address;
                                let locationText = '';
                                
                                if (address.city || address.town || address.village) {
                                    locationText += address.city || address.town || address.village;
                                }
                                
                                if (address.country) {
                                    if (locationText) locationText += ', ';
                                    locationText += address.country;
                                }
                                
                                document.getElementById('current-position').innerHTML = `
                                    <p><span class="font-medium">Coordonnées:</span> ${lat.toFixed(6)}, ${lng.toFixed(6)}</p>
                                    <p><span class="font-medium">Lieu:</span> ${locationText}</p>
                                `;
                                
                                <?php if (!$travel): ?>
                                    document.getElementById('start_coords').value = `${lat},${lng}`;
                                <?php endif; ?>
                                
                                currentPosition = { lat, lng };
                                
                                saveUserPosition(lat, lng);
                            })
                            .catch(error => {
                                console.error('Erreur lors de la récupération des informations de localisation:', error);
                            });
                    },
                    function(error) {
                        console.error('Erreur de géolocalisation:', error);
                        document.getElementById('current-position').innerHTML = `
                            <p class="text-red-600">Impossible d'obtenir ta position. Vérifie que la géolocalisation est activée.</p>
                        `;
                    }
                );
            } else {
                document.getElementById('current-position').innerHTML = `
                    <p class="text-red-600">La géolocalisation n'est pas supportée par ton navigateur.</p>
                `;
            }
        }
        
        function displayUserPosition(lat, lng) {
            if (userMarker) {
                map.removeLayer(userMarker);
            }
            
            userMarker = L.marker([lat, lng], {
                icon: L.icon({
                    iconUrl: 'https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-2x-blue.png',
                    shadowUrl: 'https://cdnjs.cloudflare.com/ajax/libs/leaflet/0.7.7/images/marker-shadow.png',
                    iconSize: [25, 41],
                    iconAnchor: [12, 41],
                    popupAnchor: [1, -34],
                    shadowSize: [41, 41]
                })
            }).addTo(map);
            
            userMarker.bindPopup("<b>Ta position</b>").openPopup();
            map.setView([lat, lng], 10);
        }
        
        function saveUserPosition(lat, lng) {
            const token = getWebSocketToken();
            const userId = <?php echo $_SESSION['user_id']; ?>;
            
            if (window.websocket && window.websocket.readyState === WebSocket.OPEN) {
                window.websocket.send(JSON.stringify({
                    type: 'position_update',
                    data: {
                        latitude: lat,
                        longitude: lng
                    }
                }));
            } else {
                connectWebSocket(userId, token);
            }
        }
        
        function geocodeDestination(query) {
            const endCountry = document.getElementById('end_country').value;
            
            if (!endCountry) {
                alert('Sélectionne d\'abord un pays de destination');
                return;
            }
            
            fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(query)}&countrycodes=<?php echo strtolower($countries[0]['code']); ?>`)
                .then(response => response.json())
                .then(data => {
                    if (data.length > 0) {
                        const location = data[0];
                        const lat = parseFloat(location.lat);
                        const lng = parseFloat(location.lon);
                        
                        document.getElementById('end_coords').value = `${lat},${lng}`;
                        
                        if (destinationMarker) {
                            map.removeLayer(destinationMarker);
                        }
                        
                        destinationMarker = L.marker([lat, lng], {
                            icon: L.icon({
                                iconUrl: 'https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-2x-red.png',
                                shadowUrl: 'https://cdnjs.cloudflare.com/ajax/libs/leaflet/0.7.7/images/marker-shadow.png',
                                iconSize: [25, 41],
                                iconAnchor: [12, 41],
                                popupAnchor: [1, -34],
                                shadowSize: [41, 41]
                            })
                        }).addTo(map);
                        
                        destinationMarker.bindPopup("<b>Destination</b>").openPopup();
                        
                        if (currentPosition) {
                            drawPath(currentPosition.lat, currentPosition.lng, lat, lng);
                            calculateTravel(currentPosition.lat, currentPosition.lng, lat, lng);
                        }
                        
                        document.getElementById('start-travel').disabled = false;
                    } else {
                        alert('Destination non trouvée. Essaie avec une autre recherche.');
                    }
                })
                .catch(error => {
                    console.error('Erreur lors de la géolocalisation de la destination:', error);
                });
        }
        
        function drawPath(startLat, startLng, endLat, endLng) {
            if (travelPath) {
                map.removeLayer(travelPath);
            }
            
            travelPath = L.polyline([
                [startLat, startLng],
                [endLat, endLng]
            ], {
                color: 'blue',
                weight: 3,
                opacity: 0.7,
                dashArray: '10, 10'
            }).addTo(map);
            
            const bounds = L.latLngBounds([
                [startLat, startLng],
                [endLat, endLng]
            ]);
            
            map.fitBounds(bounds, { padding: [50, 50] });
        }
        
        function calculateTravel(startLat, startLng, endLat, endLng) {
            const transportType = document.querySelector('input[name="transport_type"]:checked').value;
            
            const R = 6371;
            const dLat = (endLat - startLat) * Math.PI / 180;
            const dLon = (endLng - startLng) * Math.PI / 180;
            const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                      Math.cos(startLat * Math.PI / 180) * Math.cos(endLat * Math.PI / 180) * 
                      Math.sin(dLon/2) * Math.sin(dLon/2);
            const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
            const distance = R * c;
            
            const speeds = {
                'car': 90,
                'train': 180,
                'plane': 800,
                'boat': 35
            };
            
            const speed = speeds[transportType] || speeds['car'];
            const hours = distance / speed;
            const minutes = Math.ceil(hours * 60);
            
            let durationText = '';
            if (hours >= 24) {
                const days = Math.floor(hours / 24);
                const remainingHours = Math.floor(hours % 24);
                durationText = `${days} jour${days > 1 ? 's' : ''} ${remainingHours} heure${remainingHours > 1 ? 's' : ''}`;
            } else if (hours >= 1) {
                const hoursPart = Math.floor(hours);
                const minutesPart = Math.floor((hours - hoursPart) * 60);
                durationText = `${hoursPart} heure${hoursPart > 1 ? 's' : ''} ${minutesPart} minute${minutesPart > 1 ? 's' : ''}`;
            } else {
                durationText = `${minutes} minute${minutes > 1 ? 's' : ''}`;
            }
            
            document.getElementById('travel-distance').textContent = `${distance.toFixed(1)} km`;
            document.getElementById('travel-duration').textContent = durationText;
            document.getElementById('travel-info').classList.remove('hidden');
        }
        
        function getWebSocketToken() {
            return Math.random().toString(36).substring(2, 15) + Math.random().toString(36).substring(2, 15);
        }
        
        function connectWebSocket(userId, token) {
            const wsProtocol = window.location.protocol === 'https:' ? 'wss:' : 'ws:';
            const wsUrl = `${wsProtocol}//${window.location.hostname}:3000?user_id=${userId}&token=${token}`;
            
            window.websocket = new WebSocket(wsUrl);
            
            window.websocket.onopen = function() {
                console.log('WebSocket connecté');
                
                if (currentPosition) {
                    window.websocket.send(JSON.stringify({
                        type: 'position_update',
                        data: {
                            latitude: currentPosition.lat,
                            longitude: currentPosition.lng
                        }
                    }));
                }
            };
            
            window.websocket.onmessage = function(event) {
                const message = JSON.parse(event.data);
                
                if (message.type === 'nearby_agent') {
                    const agent = message.data;
                    alert(`Agent détecté à proximité: ${agent.username} (Niveau ${agent.level})`);
                }
            };
            
            window.websocket.onerror = function(error) {
                console.error('Erreur WebSocket:', error);
            };
            
            window.websocket.onclose = function() {
                console.log('WebSocket déconnecté');
                setTimeout(function() {
                    connectWebSocket(userId, token);
                }, 5000);
            };
        }
    </script>
</body>
</html>