<?php
require_once 'db.php';

header('Content-Type: application/json');

// Fonction pour nettoyer les entrées
function sanitize_input($data) {
    return htmlspecialchars(strip_tags(trim($data)));
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Vérification de la présence des champs requis
        if (!isset($_POST['user-id']) || !isset($_POST['email']) || !isset($_POST['username'])) {
            throw new Exception('Missing required fields');
        }

        // Nettoyage et validation des entrées
        $userId = filter_var($_POST['user-id'], FILTER_VALIDATE_INT);
        $email = filter_var($_POST['email'], FILTER_VALIDATE_EMAIL);
        $username = sanitize_input($_POST['username']);
        $password = isset($_POST['password']) ? $_POST['password'] : '';
        $note = isset($_POST['note']) ? sanitize_input($_POST['note']) : '';

        // Validations supplémentaires
        if (!$userId || !$email) {
            throw new Exception('Invalid input data');
        }

        // Connexion à la base de données
        $db = getDB1Connection();
        
        // Vérification si l'utilisateur existe
        $checkUser = $db->prepare("SELECT id FROM utilisateurs WHERE id = ?");
        $checkUser->execute([$userId]);
        if (!$checkUser->fetch()) {
            throw new Exception('User not found');
        }

        // Préparation de la requête de base
        $updateFields = ['email = ?', 'username = ?', 'note = ?'];
        $params = [$email, $username, $note];

        // Si un nouveau mot de passe est fourni, on l'ajoute à la mise à jour
        if (!empty($password)) {
            // Validation du mot de passe (minimum 8 caractères)
            if (strlen($password) < 8) {
                throw new Exception('Password must be at least 8 characters long');
            }
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            $updateFields[] = 'password = ?';
            $params[] = $hashedPassword;
        }

        // Ajout de l'ID à la fin des paramètres
        $params[] = $userId;

        // Construction et exécution de la requête SQL
        $sql = "UPDATE utilisateurs SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $db->prepare($sql);
        $stmt->execute($params);

        // Log de l'opération (optionnel)
        error_log("User update - ID: $userId, Email: $email");

        // Retour du résultat
        if($stmt->rowCount() > 0) {
            echo json_encode([
                'success' => true,
                'message' => 'User updated successfully',
                'data' => [
                    'id' => $userId,
                    'email' => $email,
                    'username' => $username,
                    'note' => $note
                ]
            ]);
        } else {
            // Même si aucune modification n'a été faite, on considère que c'est un succès
            echo json_encode([
                'success' => true,
                'message' => 'No changes were necessary',
                'data' => [
                    'id' => $userId,
                    'email' => $email,
                    'username' => $username,
                    'note' => $note
                ]
            ]);
        }

    } catch(PDOException $e) {
        error_log("Database error: " . $e->getMessage());
        echo json_encode([
            'success' => false,
            'message' => 'Database error occurred',
            'debug' => $e->getMessage() // À retirer en production
        ]);
    } catch(Exception $e) {
        error_log("General error: " . $e->getMessage());
        echo json_encode([
            'success' => false,
            'message' => $e->getMessage()
        ]);
    }
} else {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request method'
    ]);
}