<?php
require_once 'config.php';
$data = json_decode(file_get_contents('stock.json'), true);
$token = generateAzureToken();
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Horaires des Trains</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/microsoft-cognitiveservices-speech-sdk/1.24.0/microsoft.cognitiveservices.speech.sdk.bundle.js"></script>
</head>
<body class="bg-gradient-to-b from-gray-900 to-gray-800 text-white min-h-screen">
    <div class="container mx-auto p-8">
        <div class="flex items-center justify-between mb-8">
            <div>
                <h1 class="text-5xl font-bold bg-gradient-to-r from-blue-500 to-purple-500 text-transparent bg-clip-text">Horaires des Trains</h1>
                <p class="mt-2 text-gray-400" id="last-update"></p>
            </div>
            <div class="text-right">
                <div id="clock" class="text-3xl font-mono"></div>
                <div id="date" class="text-gray-400"></div>
            </div>
        </div>
        
        <div class="flex gap-4 mb-8">
            <button onclick="filterTrains('all')" class="px-4 py-2 rounded-lg bg-blue-600 hover:bg-blue-700 transition-colors">Tous</button>
            <button onclick="filterTrains('on-time')" class="px-4 py-2 rounded-lg bg-green-600 hover:bg-green-700 transition-colors">À l'heure</button>
            <button onclick="filterTrains('delayed')" class="px-4 py-2 rounded-lg bg-red-600 hover:bg-red-700 transition-colors">Retardés</button>
        </div>

        <div id="trains" class="space-y-6"></div>
    </div>
    
    <div id="announcement" class="fixed bottom-0 left-0 w-full transform translate-y-full transition-transform duration-500">
        <div class="bg-gradient-to-r from-blue-600 to-blue-800 p-6">
            <div class="container mx-auto flex items-center gap-4">
                <div class="w-12 h-12 bg-white rounded-full flex items-center justify-center">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 text-blue-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5.882V19.24a1.76 1.76 0 01-3.417.592l-2.147-6.15M18 13a3 3 0 100-6M5.436 13.683A4.001 4.001 0 017 6h1.832c4.1 0 7.625-1.234 9.168-3v14c-1.543-1.766-5.067-3-9.168-3H7a3.988 3.988 0 01-1.564-.317z" />
                    </svg>
                </div>
                <p id="announcement-text" class="text-xl font-bold flex-grow"></p>
            </div>
        </div>
    </div>

    <script>
    const azureToken = '<?php echo $token; ?>';
    let currentFilter = 'all';
    let lastUpdate = new Date();

    class AzureSpeech {
        constructor(token, region = 'francecentral') {
            const speechConfig = SpeechSDK.SpeechConfig.fromAuthorizationToken(token, region);
            speechConfig.speechSynthesisVoiceName = "fr-FR-DeniseNeural";
            this.synthesizer = new SpeechSDK.SpeechSynthesizer(speechConfig);
        }

        speak(text) {
            this.synthesizer.speakTextAsync(
                text,
                result => {
                    if (result.errorDetails)
                        console.error(result.errorDetails);
                }
            );
        }
    }

    const azureSpeech = new AzureSpeech(azureToken);

    function updateDateTime() {
        const now = new Date();
        document.getElementById('clock').textContent = now.toLocaleTimeString('fr-FR');
        document.getElementById('date').textContent = now.toLocaleDateString('fr-FR', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
        document.getElementById('last-update').textContent = 
            `Dernière mise à jour: ${lastUpdate.toLocaleTimeString('fr-FR')}`;
    }

    function filterTrains(filter) {
        currentFilter = filter;
        updateTrains(true);
    }

    function updateTrains(skipAnnouncements = false) {
        fetch('stock.json?' + new Date().getTime())
        .then(response => response.json())
        .then(data => {
            const trainsDiv = document.getElementById('trains');
            trainsDiv.innerHTML = '';
            
            data.filter(train => {
                if (currentFilter === 'all') return true;
                return train.status === currentFilter;
            }).forEach(train => {
                const arrival = new Date(train.arrival);
                const now = new Date();
                const timeLeft = Math.max(0, arrival - now);
                const minutes = Math.floor(timeLeft / 60000);
                
                const statusColor = train.status === 'on-time' ? 'green' : 'red';
                const statusText = train.status === 'on-time' ? 'À l\'heure' : 'Retardé';
                
                const voie = train.platform || Math.floor(Math.random() * 20) + 1;
                
                const div = document.createElement('div');
                div.className = 'bg-gray-800 bg-opacity-50 backdrop-blur-lg p-8 rounded-xl shadow-lg transform transition-all duration-300 hover:scale-102 hover:bg-opacity-70';
                div.innerHTML = `
                    <div class="grid grid-cols-5 gap-8 items-center">
                        <div class="flex items-center gap-4">
                            <div class="w-12 h-12 rounded-full bg-blue-500 flex items-center justify-center text-xl font-bold">${train.line}</div>
                            <div>
                                <div class="text-xl">${train.station}</div>
                                <div class="text-sm text-gray-400">Voie ${voie}</div>
                            </div>
                        </div>
                        <div class="text-2xl font-mono">${arrival.toLocaleTimeString('fr-FR')}</div>
                        <div class="text-xl font-medium">${formatTimeLeft(minutes)}</div>
                        <div class="text-${statusColor}-500 text-xl font-medium flex items-center gap-2">
                            <span class="w-3 h-3 rounded-full bg-${statusColor}-500"></span>
                            ${statusText}
                        </div>
                        <div class="text-gray-400">
                            ${train.info ? train.info : 'Pas d\'information supplémentaire'}
                        </div>
                    </div>
                    <div class="mt-4 h-1 bg-gray-700 rounded-full overflow-hidden">
                        <div class="h-full bg-blue-500 transition-all duration-500" style="width: ${Math.min(100, (minutes / 60) * 100)}%"></div>
                    </div>
                `;
                
                if (!skipAnnouncements && minutes === 0) {
                    const announcement = `Votre attention s'il vous plaît. Le train de la ligne ${train.line} à destination de ${train.station} entre en gare voie ${voie}. ${train.info ? 'Information : ' + train.info : ''}`;
                    showAnnouncement(announcement);
                }
                
                trainsDiv.appendChild(div);
            });
            
            lastUpdate = new Date();
            updateDateTime();
        });
    }

    function formatTimeLeft(minutes) {
        if (minutes === 0) return 'À quai';
        if (minutes === 1) return '1 minute';
        return `${minutes} minutes`;
    }

    function showAnnouncement(text) {
        const announcement = document.getElementById('announcement');
        const announcementText = document.getElementById('announcement-text');
        announcementText.textContent = text;
        announcement.classList.remove('translate-y-full');
        azureSpeech.speak(text);
        setTimeout(() => {
            announcement.classList.add('translate-y-full');
        }, 10000);
    }

    updateDateTime();
    setInterval(updateDateTime, 1000);
    updateTrains();
    setInterval(() => updateTrains(), 5000);
    </script>
</body>
</html>