<?php
class FolderAPI {
    private $log_file = 'folder_api_log.txt';
    private $progress_file = 'creation_progress.json';
    private $total_steps = 4;
    private $current_step = 0;
    private $start_time = 0;
    
    public function __construct() {
        $this->log("FolderAPI initialized");
        $this->resetProgress();
    }
    
    private function log($message, $is_error = false) {
        $date = date('Y-m-d H:i:s');
        $log_entry = "[$date] " . ($is_error ? "ERROR: " : "") . $message . "\n";
        
        file_put_contents($this->log_file, $log_entry, FILE_APPEND);
        
        if ($is_error) {
            error_log("FolderAPI: $message");
        }
    }
    
    private function resetProgress() {
        $this->start_time = time();
        $this->current_step = 0;
        $progress = [
            'current_step' => 0,
            'total_steps' => $this->total_steps,
            'start_time' => $this->start_time,
            'current_time' => $this->start_time,
            'elapsed_time' => 0,
            'estimated_remaining' => 0,
            'percent_complete' => 0,
            'current_operation' => 'Initializing...',
            'steps_complete' => [],
            'status' => 'running'
        ];
        file_put_contents($this->progress_file, json_encode($progress));
    }
    
    private function updateProgress($step_name, $status = 'running', $message = '') {
        $this->current_step++;
        $current_time = time();
        $elapsed_time = $current_time - $this->start_time;
        
        $percent_complete = min(100, round(($this->current_step / $this->total_steps) * 100));
        
        $estimated_remaining = 0;
        if ($this->current_step > 0 && $this->current_step < $this->total_steps) {
            $avg_time_per_step = $elapsed_time / $this->current_step;
            $estimated_remaining = round($avg_time_per_step * ($this->total_steps - $this->current_step));
        }
        
        $steps_complete = json_decode(file_get_contents($this->progress_file), true)['steps_complete'] ?? [];
        $steps_complete[] = [
            'name' => $step_name,
            'time' => date('H:i:s'),
            'message' => $message
        ];
        
        $progress = [
            'current_step' => $this->current_step,
            'total_steps' => $this->total_steps,
            'start_time' => $this->start_time,
            'current_time' => $current_time,
            'elapsed_time' => $elapsed_time,
            'estimated_remaining' => $estimated_remaining,
            'percent_complete' => $percent_complete,
            'current_operation' => $step_name,
            'steps_complete' => $steps_complete,
            'status' => $status,
            'message' => $message
        ];
        
        file_put_contents($this->progress_file, json_encode($progress));
        return $progress;
    }
    
    public function getProgress() {
        if (file_exists($this->progress_file)) {
            return json_decode(file_get_contents($this->progress_file), true);
        }
        return null;
    }
    
    public function createDirectory($directory_name) {
        $this->updateProgress("Creating directory: $directory_name");
        $this->log("Creating directory: $directory_name");
        
        $full_path = dirname(__FILE__) . '/' . $directory_name;
        
        if (file_exists($full_path)) {
            $this->log("Directory already exists: $full_path");
            $this->updateProgress("Directory already exists", 'success');
            return [
                'status' => 'success',
                'message' => "Directory already exists"
            ];
        }
        
        $result = mkdir($full_path, 0755, true);
        
        if ($result) {
            $this->log("Directory created successfully: $full_path");
            $this->updateProgress("Directory created", 'success');
            return [
                'status' => 'success',
                'message' => "Directory created successfully"
            ];
        } else {
            $error_msg = "Failed to create directory: $full_path";
            $this->log($error_msg, true);
            $this->updateProgress("Failed to create directory", 'error', $error_msg);
            return [
                'status' => 'error',
                'message' => $error_msg
            ];
        }
    }
    
    public function copyFiles($source_dir, $destination_dir) {
        $this->updateProgress("Starting to copy files from $source_dir to $destination_dir");
        $this->log("Copying files from $source_dir to $destination_dir");
        
        $base_path = dirname(__FILE__);
        $source_path = $base_path . '/' . $source_dir;
        $dest_path = $base_path . '/' . $destination_dir;
        
        if (!file_exists($source_path)) {
            $error_msg = "Source directory does not exist: $source_path";
            $this->log($error_msg, true);
            $this->updateProgress("Failed to copy files", 'error', $error_msg);
            return [
                'status' => 'error',
                'message' => $error_msg
            ];
        }
        
        if (!file_exists($dest_path)) {
            if (!mkdir($dest_path, 0755, true)) {
                $error_msg = "Failed to create destination directory: $dest_path";
                $this->log($error_msg, true);
                $this->updateProgress("Failed to copy files", 'error', $error_msg);
                return [
                    'status' => 'error',
                    'message' => $error_msg
                ];
            }
        }
        
        $files = $this->scanDirectory($source_path);
        $total_files = count($files);
        $this->log("Found $total_files files/folders to copy");
        
        $this->updateProgress("Found $total_files files/folders to copy", 'running');
        
        $successful_copies = 0;
        $failed_copies = 0;
        
        foreach ($files as $index => $file) {
            $relative_path = substr($file, strlen($source_path));
            $destination_file = $dest_path . $relative_path;
            $directory = dirname($destination_file);
            
            $progress_percent = min(95, round(($index / $total_files) * 100));
            
            $op_progress = [
                'current_step' => $this->current_step,
                'total_steps' => $this->total_steps,
                'current_operation' => "Copying files (" . ($index + 1) . " of $total_files)",
                'current_file' => basename($file),
                'percent_complete' => $progress_percent,
                'total_files' => $total_files,
                'files_copied' => $index
            ];
            file_put_contents($this->progress_file, json_encode(array_merge(
                json_decode(file_get_contents($this->progress_file), true),
                $op_progress
            )));
            
            if (!file_exists($directory)) {
                if (!mkdir($directory, 0755, true)) {
                    $this->log("Failed to create directory: $directory", true);
                    $failed_copies++;
                    continue;
                }
            }
            
            if (is_dir($file)) {
                $successful_copies++;
                continue;
            }
            
            if (copy($file, $destination_file)) {
                $successful_copies++;
            } else {
                $this->log("Failed to copy file: $file to $destination_file", true);
                $failed_copies++;
            }
        }
        
        $this->log("Copied $successful_copies files/directories successfully, $failed_copies failed");
        
        $status = ($failed_copies > 0) ? 'partial' : 'success';
        $message = "Copied $successful_copies files/directories" . 
                   (($failed_copies > 0) ? ", $failed_copies failed" : " successfully");
        
        $this->updateProgress("Finished copying files", $status, $message);
        
        if ($failed_copies > 0) {
            return [
                'status' => 'partial',
                'message' => $message,
                'data' => [
                    'successful_copies' => $successful_copies,
                    'failed_copies' => $failed_copies
                ]
            ];
        }
        
        return [
            'status' => 'success',
            'message' => $message,
            'data' => [
                'successful_copies' => $successful_copies
            ]
        ];
    }
    
    private function scanDirectory($directory) {
        $files = [];
        
        if (!is_dir($directory)) {
            return $files;
        }
        
        $items = scandir($directory);
        
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }
            
            $path = $directory . '/' . $item;
            $files[] = $path;
            
            if (is_dir($path)) {
                $files = array_merge($files, $this->scanDirectory($path));
            }
        }
        
        return $files;
    }
    
    public function finalizeSetup() {
        $this->updateProgress("Finalizing family hub setup", 'success', "Setup completed successfully");
        return [
            'status' => 'success',
            'message' => 'Family hub setup completed successfully'
        ];
    }
    
    public function sendNotificationEmail($to, $family_name, $site_url) {
        $subject = "Your Family Hub is Ready - AndWeare";
        
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: AndWeare <noreply@andweare.com>" . "\r\n";
        
        $message = "
        <!DOCTYPE html>
        <html>
        <head>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    line-height: 1.6;
                    color: #333;
                    margin: 0;
                    padding: 0;
                }
                .container {
                    max-width: 600px;
                    margin: 0 auto;
                    padding: 20px;
                    background-color: #ffffff;
                }
                .header {
                    text-align: center;
                    padding: 20px 0;
                    border-bottom: 1px solid #eee;
                }
                .content {
                    padding: 20px 0;
                }
                .button {
                    display: inline-block;
                    padding: 10px 20px;
                    background-color: #000;
                    color: #fff;
                    text-decoration: none;
                    border-radius: 4px;
                    margin-top: 15px;
                }
                .footer {
                    padding: 20px 0;
                    font-size: 12px;
                    color: #888;
                    text-align: center;
                    border-top: 1px solid #eee;
                }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <img src='https://andweare.com/logo.png' alt='AndWeare Logo' height='50'>
                </div>
                <div class='content'>
                    <h2>Your Family Hub is Ready!</h2>
                    <p>Congratulations! Your family hub for <strong>$family_name</strong> has been successfully created.</p>
                    <p>You can access your family hub at the following URL:</p>
                    <p><a href='$site_url'>$site_url</a></p>
                    <p>From there, you can invite family members, share photos, events, and more.</p>
                    <p><a href='$site_url' class='button'>Go to my Family Hub</a></p>
                </div>
                <div class='footer'>
                    <p>&copy; " . date('Y') . " AndWeare. All rights reserved.</p>
                    <p>This is an automated message, please do not reply to this email.</p>
                </div>
            </div>
        </body>
        </html>
        ";
        
        $this->log("Sending notification email to: $to");
        $result = mail($to, $subject, $message, $headers);
        
        if ($result) {
            $this->log("Notification email sent successfully");
            return [
                'status' => 'success',
                'message' => 'Notification email sent successfully'
            ];
        } else {
            $this->log("Failed to send notification email", true);
            return [
                'status' => 'error',
                'message' => 'Failed to send notification email'
            ];
        }
    }
}
?>