document.addEventListener('DOMContentLoaded', () => {
    const app = {
        map: null,
        mapboxKey: '',
        markers: {},
        userMarker: null,
        tempMarker: null,
        selectedLightId: null,
        userWatchId: null,
        refreshTimerId: null,
        rafId: null,
        lastTimestamp: 0,
        currentRoute: null,
        isNavigating: false,
        routeLineLayer: null,
        trafficLayer: null,
        currentSpeedLimit: null,
        speedAlertPlayed: false,
        speedAudio: new Audio('speed-alert.mp3'),
        destinationWeather: null,
        settings: {
            darkMode: false,
            notifications: true,
            autoRefresh: true,
            predictions: true,
            autoMeasure: true,
            defaultZoom: 15,
            mapStyle: 'satellite',
            voiceAlerts: true,
            trafficView: true,
            weatherDisplay: true
        },
        navigation: {
            startPosition: null,
            endPosition: null,
            coordinates: [],
            lightsOnRoute: [],
            travelMode: 'driving',
            distance: null,
            duration: null
        },
        state: {
            selectingLocation: false,
            routePanelOpen: false,
            navigationActive: false,
            lightsPanelOpen: false,
            isAddingLight: false,
            isLoggedIn: false
        },
        
        init() {
            this.loadUserData();
            this.setupTheme();
            this.setupEventListeners();
            this.initializeApis()
                .then(() => {
                    this.initializeMap();
                    this.setupGeolocation();
                    this.loadTrafficLights();
                    this.hideLoadingScreen();
                })
                .catch(error => {
                    console.error('Failed to initialize app:', error);
                    this.showNotification('Failed to initialize app. Please refresh.', 'error');
                });
        },
        
        loadUserData() {
            try {
                const userData = this.getCookie('user_data');
                if (userData) {
                    this.state.isLoggedIn = true;
                    this.user = JSON.parse(userData);
                    this.updateUIForLoggedInUser(this.user);
                }
            } catch (e) {
                console.error('Error loading user data', e);
            }
        },
        
        getCookie(name) {
            const value = `; ${document.cookie}`;
            const parts = value.split(`; ${name}=`);
            if (parts.length === 2) return parts.pop().split(';').shift();
            return null;
        },
        
        setupTheme() {
            const isDark = localStorage.getItem('darkMode') === 'true';
            document.documentElement.classList.toggle('dark', isDark);
            if (isDark) {
                document.body.classList.add('dark-mode');
            }
        },
        
        async initializeApis() {
            try {
                const [mapboxResponse, appleResponse] = await Promise.all([
                    fetch('get-mapbox-key.php'),
                    fetch('get-apple-map-key.php')
                ]);
                
                const mapboxData = await mapboxResponse.json();
                const appleData = await appleResponse.json();
                
                if (mapboxData.success) this.mapboxKey = mapboxData.key;
                if (appleData.success) this.appleToken = appleData.token;
                
                return { mapbox: this.mapboxKey, apple: this.appleToken };
            } catch (error) {
                console.error('Failed to initialize APIs:', error);
                this.showNotification('Failed to initialize services', 'error');
                throw error;
            }
        },
        
        initializeMap() {
            if (!this.mapboxKey) {
                this.showNotification('MapBox key not available', 'error');
                return;
            }
            
            mapboxgl.accessToken = this.mapboxKey;
            this.map = new mapboxgl.Map({
                container: 'map',
                style: this.getMapStyle(),
                center: [0, 0],
                zoom: parseInt(this.settings.defaultZoom),
                pitch: 0,
                bearing: 0
            });
            
            this.map.addControl(new mapboxgl.NavigationControl(), 'bottom-right');
            
            this.map.on('load', () => {
                if (this.settings.trafficView) {
                    this.addTrafficLayer();
                }
                this.map.on('click', this.handleMapClick.bind(this));
            });
        },
        
        getMapStyle() {
            switch (this.settings.mapStyle) {
                case 'streets':
                    return 'mapbox://styles/mapbox/streets-v12';
                case 'navigation':
                    return 'mapbox://styles/mapbox/navigation-day-v1';
                case 'dark':
                    return 'mapbox://styles/mapbox/dark-v11';
                case 'satellite':
                default:
                    return 'mapbox://styles/mapbox/satellite-streets-v12';
            }
        },
        
        addTrafficLayer() {
            this.map.addLayer({
                id: 'traffic',
                type: 'line',
                source: {
                    type: 'vector',
                    url: 'mapbox://mapbox.mapbox-traffic-v1'
                },
                'source-layer': 'traffic',
                paint: {
                    'line-width': 3,
                    'line-color': [
                        'match',
                        ['get', 'congestion'],
                        'low', '#10b981',
                        'moderate', '#f59e0b',
                        'heavy', '#ef4444',
                        'severe', '#991b1b',
                        '#6b7280'
                    ],
                    'line-opacity': 0.7
                }
            });
            this.trafficLayer = 'traffic';
        },
        
        setupGeolocation() {
            if (navigator.geolocation) {
                this.userWatchId = navigator.geolocation.watchPosition(
                    this.updateUserLocation.bind(this),
                    this.handleGeolocationError.bind(this),
                    { enableHighAccuracy: true, maximumAge: 10000, timeout: 5000 }
                );
            }
        },
        
        updateUserLocation(position) {
            const lat = position.coords.latitude;
            const lng = position.coords.longitude;
            
            if (!this.userMarker) {
                const el = document.createElement('div');
                el.className = 'location-dot';
                
                this.userMarker = new mapboxgl.Marker(el)
                    .setLngLat([lng, lat])
                    .addTo(this.map);
                
                this.map.flyTo({
                    center: [lng, lat],
                    zoom: 16,
                    essential: true
                });
            } else {
                this.userMarker.setLngLat([lng, lat]);
            }
            
            if (this.state.navigationActive) {
                this.updateNavigation(position);
            }
        },
        
        handleGeolocationError(error) {
            console.error('Geolocation error:', error);
            const locateButton = document.getElementById('headerLocateBtn');
            if (locateButton) {
                locateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i>';
            }
            this.showNotification('Location access denied', 'error');
        },
        
        setupEventListeners() {
            document.getElementById('menuToggle')?.addEventListener('click', this.toggleSidebar.bind(this));
            document.getElementById('headerLocateBtn')?.addEventListener('click', this.locateUser.bind(this));
            document.getElementById('headerAddBtn')?.addEventListener('click', () => this.openModal('addLightModal'));
            document.getElementById('settingsBtn')?.addEventListener('click', () => this.openModal('settingsModal'));
            document.getElementById('refreshBtn')?.addEventListener('click', this.refreshLights.bind(this));
            document.getElementById('nearbyBtn')?.addEventListener('click', this.findNearbyLights.bind(this));
            document.getElementById('routeBtn')?.addEventListener('click', this.toggleRoutePanel.bind(this));
            document.getElementById('toggleTraffic')?.addEventListener('click', this.toggleTrafficLayer.bind(this));
            
            document.getElementById('startLocation')?.addEventListener('input', e => this.handleLocationSearch(e, 'start'));
            document.getElementById('endLocation')?.addEventListener('input', e => this.handleLocationSearch(e, 'end'));
            
            document.addEventListener('click', (e) => {
                if (!e.target.closest('#startLocation') && !e.target.closest('#startSuggestions')) {
                    document.getElementById('startSuggestions')?.classList.add('hidden');
                }
                if (!e.target.closest('#endLocation') && !e.target.closest('#endSuggestions')) {
                    document.getElementById('endSuggestions')?.classList.add('hidden');
                }
            });
            
            document.getElementById('closePopup')?.addEventListener('click', () => {
                document.getElementById('lightPopup')?.classList.add('hidden');
            });
            
            document.getElementById('routeTypeCar')?.addEventListener('click', () => this.setTravelMode('driving'));
            document.getElementById('routeTypeBike')?.addEventListener('click', () => this.setTravelMode('cycling'));
            document.getElementById('routeTypeWalk')?.addEventListener('click', () => this.setTravelMode('walking'));
            document.getElementById('calculateRoute')?.addEventListener('click', this.calculateRoute.bind(this));
            document.getElementById('startNavigation')?.addEventListener('click', this.startNavigation.bind(this));
            document.getElementById('endNavigation')?.addEventListener('click', this.endNavigation.bind(this));
            
            document.getElementById('closeRoutePanel')?.addEventListener('click', this.closeRoutePanel.bind(this));
            
            document.querySelectorAll('.close-modal').forEach(btn => {
                btn.addEventListener('click', () => {
                    const modal = btn.closest('.modal');
                    this.closeModal(modal.id);
                });
            });
            
            document.getElementById('darkModeToggle')?.addEventListener('change', this.toggleDarkMode.bind(this));
            document.getElementById('trafficViewToggle')?.addEventListener('change', this.toggleTrafficLayer.bind(this));
            
            document.getElementById('loginForm')?.addEventListener('submit', this.handleLogin.bind(this));
            document.getElementById('registerForm')?.addEventListener('submit', this.handleRegister.bind(this));
            document.getElementById('addLightForm')?.addEventListener('submit', this.handleAddLight.bind(this));
            
            document.getElementById('showLoginForm')?.addEventListener('click', () => {
                document.getElementById('loginForm')?.classList.remove('hidden');
                document.getElementById('registerForm')?.classList.add('hidden');
                document.getElementById('authTitle').textContent = 'Login';
            });
            
            document.getElementById('showRegisterForm')?.addEventListener('click', () => {
                document.getElementById('loginForm')?.classList.add('hidden');
                document.getElementById('registerForm')?.classList.remove('hidden');
                document.getElementById('authTitle').textContent = 'Register';
            });
        },
        
        toggleSidebar() {
            const sidebar = document.querySelector('.sidebar');
            sidebar.classList.toggle('active');
            
            const menuButton = document.getElementById('menuToggle');
            menuButton.innerHTML = sidebar.classList.contains('active') 
                ? '<i class="fas fa-times"></i>' 
                : '<i class="fas fa-bars"></i>';
        },
        
        locateUser() {
            if (this.userMarker) {
                this.map.flyTo({
                    center: this.userMarker.getLngLat(),
                    zoom: 16,
                    essential: true
                });
            } else {
                this.setupGeolocation();
            }
        },
        
        openModal(modalId) {
            const modal = document.getElementById(modalId);
            if (!modal) return;
            
            if (modalId === 'addLightModal') {
                this.state.isAddingLight = true;
                
                if (this.userMarker) {
                    const pos = this.userMarker.getLngLat();
                    document.querySelector('input[name="latitude"]').value = pos.lat.toFixed(6);
                    document.querySelector('input[name="longitude"]').value = pos.lng.toFixed(6);
                    
                    if (this.tempMarker) this.tempMarker.remove();
                    this.tempMarker = new mapboxgl.Marker({ draggable: true })
                        .setLngLat([pos.lng, pos.lat])
                        .addTo(this.map);
                    
                    this.tempMarker.on('dragend', this.updateMarkerPosition.bind(this));
                }
            }
            
            modal.style.display = 'flex';
            setTimeout(() => {
                modal.classList.add('active');
            }, 50);
        },
        
        closeModal(modalId) {
            const modal = document.getElementById(modalId);
            if (!modal) return;
            
            modal.classList.remove('active');
            setTimeout(() => {
                modal.style.display = 'none';
                
                if (modalId === 'addLightModal') {
                    this.state.isAddingLight = false;
                    if (this.tempMarker) {
                        this.tempMarker.remove();
                        this.tempMarker = null;
                    }
                }
            }, 300);
        },
        
        handleMapClick(e) {
            if (this.state.isAddingLight) {
                document.querySelector('input[name="latitude"]').value = e.lngLat.lat.toFixed(6);
                document.querySelector('input[name="longitude"]').value = e.lngLat.lng.toFixed(6);
                
                if (this.tempMarker) this.tempMarker.remove();
                this.tempMarker = new mapboxgl.Marker({ draggable: true })
                    .setLngLat([e.lngLat.lng, e.lngLat.lat])
                    .addTo(this.map);
                
                this.tempMarker.on('dragend', this.updateMarkerPosition.bind(this));
            }
        },
        
        updateMarkerPosition() {
            const pos = this.tempMarker.getLngLat();
            document.querySelector('input[name="latitude"]').value = pos.lat.toFixed(6);
            document.querySelector('input[name="longitude"]').value = pos.lng.toFixed(6);
        },
        
        toggleRoutePanel() {
            const routePanel = document.getElementById('routePanel');
            if (this.state.routePanelOpen) {
                routePanel.classList.remove('active');
                this.state.routePanelOpen = false;
            } else {
                if (this.userMarker) {
                    const position = this.userMarker.getLngLat();
                    this.reverseGeocode(position.lng, position.lat)
                        .then(placeName => {
                            document.getElementById('startLocation').value = placeName || `${position.lat.toFixed(6)}, ${position.lng.toFixed(6)}`;
                            this.navigation.startPosition = {
                                center: [position.lng, position.lat],
                                place_name: placeName || `${position.lat.toFixed(6)}, ${position.lng.toFixed(6)}`
                            };
                        });
                }
                
                routePanel.classList.add('active');
                this.state.routePanelOpen = true;
            }
        },
        
        closeRoutePanel() {
            const routePanel = document.getElementById('routePanel');
            routePanel.classList.remove('active');
            this.state.routePanelOpen = false;
        },
        
        async handleLocationSearch(event, type) {
            const query = event.target.value;
            const suggestionsDiv = document.getElementById(`${type}Suggestions`);
            
            if (query.length < 3) {
                suggestionsDiv.classList.add('hidden');
                return;
            }
            
            try {
                const response = await fetch(`https://api.mapbox.com/geocoding/v5/mapbox.places/${encodeURIComponent(query)}.json?access_token=${this.mapboxKey}`);
                const data = await response.json();
                
                suggestionsDiv.innerHTML = '';
                suggestionsDiv.classList.remove('hidden');
                
                if (!data.features || data.features.length === 0) {
                    const div = document.createElement('div');
                    div.className = 'px-4 py-3 text-gray-500';
                    div.textContent = 'No results found';
                    suggestionsDiv.appendChild(div);
                    return;
                }
                
                data.features.forEach(result => {
                    const div = document.createElement('div');
                    div.className = 'suggestion-item px-4 py-3 text-gray-900 dark:text-white cursor-pointer';
                    div.textContent = result.place_name;
                    div.onclick = () => this.selectLocation(result, type);
                    suggestionsDiv.appendChild(div);
                });
            } catch (error) {
                console.error('Location search error:', error);
            }
        },
        
        selectLocation(result, type) {
            const input = document.getElementById(`${type}Location`);
            const suggestionsDiv = document.getElementById(`${type}Suggestions`);
            
            input.value = result.place_name;
            suggestionsDiv.classList.add('hidden');
            
            this.navigation[`${type}Position`] = {
                center: result.center,
                place_name: result.place_name
            };
        },
        
        async reverseGeocode(lng, lat) {
            try {
                const response = await fetch(`https://api.mapbox.com/geocoding/v5/mapbox.places/${lng},${lat}.json?access_token=${this.mapboxKey}`);
                const data = await response.json();
                return data.features[0]?.place_name || null;
            } catch (error) {
                console.error('Reverse geocoding error:', error);
                return null;
            }
        },
        
        setTravelMode(mode) {
            this.navigation.travelMode = mode;
            
            document.querySelectorAll('.travel-type-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            
            const modes = {
                'driving': document.getElementById('drivingMode'),
                'cycling': document.getElementById('cyclingMode'),
                'walking': document.getElementById('walkingMode')
            };
            
            if (modes[mode]) {
                modes[mode].classList.add('active');
            }
        },
        
        async calculateRoute() {
            if (!this.navigation.startPosition || !this.navigation.endPosition) {
                this.showNotification('Please enter both start and end locations', 'error');
                return;
            }
            
            const routeButton = document.getElementById('calculateRoute');
            routeButton.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Calculating...';
            routeButton.disabled = true;
            
            try {
                const response = await fetch(
                    `https://api.mapbox.com/directions/v5/mapbox/${this.navigation.travelMode}/` +
                    `${this.navigation.startPosition.center[0]},${this.navigation.startPosition.center[1]};` +
                    `${this.navigation.endPosition.center[0]},${this.navigation.endPosition.center[1]}` +
                    `?geometries=geojson&overview=full&steps=true&annotations=maxspeed,congestion&access_token=${this.mapboxKey}`
                );
                
                const data = await response.json();
                
                if (!data.routes || data.routes.length === 0) {
                    throw new Error('No route found');
                }
                
                const route = data.routes[0];
                this.navigation.coordinates = route.geometry.coordinates;
                this.navigation.distance = route.distance;
                this.navigation.duration = route.duration;
                
                this.drawRoute(route.geometry);
                this.displayRouteDetails(route);
                this.fitMapToRoute(route.geometry.coordinates);
                
                document.getElementById('routeInfo').classList.remove('hidden');
                
            } catch (error) {
                console.error('Route calculation error:', error);
                this.showNotification('Error calculating route', 'error');
            } finally {
                routeButton.innerHTML = '<i class="fas fa-route mr-2"></i>Find Route';
                routeButton.disabled = false;
            }
        },
        
        drawRoute(geometry) {
            if (this.map.getSource('route')) {
                this.map.removeLayer('route-line');
                this.map.removeSource('route');
            }
            
            this.map.addSource('route', {
                type: 'geojson',
                data: {
                    type: 'Feature',
                    properties: {},
                    geometry: geometry
                }
            });
            
            this.map.addLayer({
                id: 'route-line',
                type: 'line',
                source: 'route',
                layout: {
                    'line-join': 'round',
                    'line-cap': 'round'
                },
                paint: {
                    'line-color': '#0ea5e9',
                    'line-width': 6,
                    'line-opacity': 0.8
                }
            });
        },
        
        fitMapToRoute(coordinates) {
            if (!coordinates || coordinates.length === 0) return;
            
            const bounds = coordinates.reduce((bounds, coord) => {
                return bounds.extend(coord);
            }, new mapboxgl.LngLatBounds(coordinates[0], coordinates[0]));
            
            this.map.fitBounds(bounds, {
                padding: 50,
                maxZoom: 15,
                duration: 1000
            });
        },
        
        displayRouteDetails(route) {
            const duration = Math.round(route.duration / 60);
            const distance = (route.distance / 1000).toFixed(1);
            const eta = new Date(Date.now() + route.duration * 1000);
            
            document.getElementById('routeDuration').textContent = `${duration} min`;
            document.getElementById('routeDistance').textContent = `${distance} km`;
            document.getElementById('routeETA').textContent = eta.toLocaleTimeString([], {hour: '2-digit', minute: '2-digit'});
            
            const startNavBtn = document.getElementById('startNavigation');
            if (startNavBtn) {
                startNavBtn.classList.remove('hidden');
            }
        },
        
        startNavigation() {
            if (!this.navigation.coordinates || this.navigation.coordinates.length === 0) {
                this.showNotification('Please calculate a route first', 'error');
                return;
            }
            
            this.state.navigationActive = true;
            document.getElementById('routePanel').classList.remove('active');
            document.getElementById('navigationPanel').classList.remove('hidden');
            
            this.map.easeTo({
                center: this.map.getCenter(),
                pitch: 60,
                bearing: 0,
                essential: true
            });
            
            if (this.userWatchId) {
                navigator.geolocation.clearWatch(this.userWatchId);
            }
            
            this.userWatchId = navigator.geolocation.watchPosition(
                this.updateNavigation.bind(this),
                this.handleGeolocationError.bind(this),
                { enableHighAccuracy: true, maximumAge: 2000, timeout: 5000 }
            );
            
            this.showNotification('Navigation started. Follow the blue route.', 'success');
        },
        
        updateNavigation(position) {
            const { latitude, longitude, speed, heading } = position.coords;
            const speedKmh = speed ? Math.round(speed * 3.6) : 0;
            
            document.getElementById('currentSpeed').textContent = `${speedKmh} km/h`;
            
            this.map.easeTo({
                center: [longitude, latitude],
                bearing: heading || this.map.getBearing(),
                duration: 1000,
                essential: true
            });
            
            if (this.userMarker) {
                this.userMarker.setLngLat([longitude, latitude]);
            }
        },
        
        endNavigation() {
            this.state.navigationActive = false;
            document.getElementById('navigationPanel').classList.add('hidden');
            
            this.map.easeTo({
                pitch: 0,
                bearing: 0,
                duration: 1000,
                essential: true
            });
            
            if (this.map.getSource('route')) {
                this.map.removeLayer('route-line');
                this.map.removeSource('route');
            }
            
            if (this.userWatchId) {
                navigator.geolocation.clearWatch(this.userWatchId);
            }
            
            this.setupGeolocation();
        },
        
        async loadTrafficLights() {
            try {
                const response = await fetch('db.php?action=getTrafficLights');
                const lights = await response.json();
                
                const lightsList = document.getElementById('lightsList');
                const loadingLights = document.getElementById('loadingLights');
                const noLightsMessage = document.getElementById('noLightsMessage');
                
                if (loadingLights) loadingLights.style.display = 'none';
                
                if (!Array.isArray(lights) || lights.length === 0) {
                    if (noLightsMessage) noLightsMessage.classList.remove('hidden');
                    return;
                }
                
                if (noLightsMessage) noLightsMessage.classList.add('hidden');
                
                lights.forEach(light => {
                    this.addLightToMap(light);
                    this.addLightToSidebar(light, lightsList);
                });
                
                this.startStatusUpdates();
            } catch (error) {
                console.error('Error loading traffic lights:', error);
                this.showNotification('Failed to load traffic lights', 'error');
            }
        },
        
        addLightToMap(light) {
            const status = this.getLightStatus(light);
            
            const el = document.createElement('div');
            el.className = `w-8 h-8 bg-traffic-${status.color} rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white`;
            el.innerHTML = '<i class="fas fa-traffic-light"></i>';
            
            const marker = new mapboxgl.Marker(el)
                .setLngLat([parseFloat(light.longitude), parseFloat(light.latitude)])
                .addTo(this.map);
            
            marker.getElement().addEventListener('click', () => this.showLightDetail(light));
            
            this.markers[light.id] = {
                marker: marker,
                data: light
            };
        },
        
        addLightToSidebar(light, container) {
            if (!container) return;
            
            const status = this.getLightStatus(light);
            
            const card = document.createElement('div');
            card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg rounded-xl';
            card.setAttribute('data-id', light.id);
            card.setAttribute('data-name', light.name);
            
            card.innerHTML = `
                <div class="flex justify-between items-center">
                    <h3 class="font-medium">${light.name}</h3>
                    <span id="status-${light.id}" class="px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white">${status.label}</span>
                </div>
                <div class="flex items-center text-sm text-gray-500 mt-2">
                    <i class="fas fa-location-dot mr-2"></i>
                    <span class="capitalize">${light.direction}</span>
                    <div class="ml-auto flex items-center" id="timer-${light.id}">
                        <i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s
                    </div>
                </div>
                <div class="mt-2 text-xs">
                    <div class="w-full bg-gray-100 rounded-full h-1.5">
                        <div class="bg-traffic-${status.color} h-1.5 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                    </div>
                    <div class="flex justify-between mt-1 text-gray-400">
                        <span>0s</span>
                        <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                    </div>
                </div>
            `;
            
            card.addEventListener('click', () => {
                document.querySelectorAll('.light-card').forEach(c => c.classList.remove('active'));
                card.classList.add('active');
                
                this.showLightDetail(light);
                
                const sidebar = document.querySelector('.sidebar');
                if (window.innerWidth <= 768 && sidebar) {
                    sidebar.classList.remove('active');
                    document.getElementById('menuToggle').innerHTML = '<i class="fas fa-bars"></i>';
                }
            });
            
            container.appendChild(card);
        },
        
        getLightStatus(light) {
            const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
            const currentTime = Math.floor(Date.now() / 1000);
            const timeInCycle = currentTime % totalCycle;
            
            if (timeInCycle < light.red_duration) {
                return {
                    isRed: true,
                    color: 'red',
                    label: 'RED',
                    timeLeft: light.red_duration - timeInCycle
                };
            } else {
                return {
                    isRed: false,
                    color: 'green',
                    label: 'GREEN',
                    timeLeft: totalCycle - timeInCycle
                };
            }
        },
        
        showLightDetail(light) {
            this.selectedLightId = light.id;
            
            this.map.flyTo({
                center: [parseFloat(light.longitude), parseFloat(light.latitude)],
                zoom: 17,
                essential: true
            });
            
            const lightPopup = document.getElementById('lightPopup');
            if (!lightPopup) return;
            
            const popupTitle = document.getElementById('popupTitle');
            const popupContent = document.getElementById('popupContent');
            const popupStatus = document.getElementById('popupStatus');
            
            if (popupTitle) popupTitle.textContent = light.name;
            
            if (popupContent) {
                popupContent.innerHTML = `
                    <div class="grid grid-cols-2 gap-3">
                        <div class="text-gray-500">Direction:</div>
                        <div class="font-medium capitalize">${light.direction}</div>
                        
                        <div class="text-gray-500">Red duration:</div>
                        <div class="font-medium">${light.red_duration} seconds</div>
                        
                        <div class="text-gray-500">Green duration:</div>
                        <div class="font-medium">${light.green_duration} seconds</div>
                        
                        <div class="text-gray-500">Total cycle:</div>
                        <div class="font-medium">${parseInt(light.red_duration) + parseInt(light.green_duration)} seconds</div>
                        
                        <div class="text-gray-500">Location:</div>
                        <div class="font-medium truncate">${light.latitude.substring(0, 8)}, ${light.longitude.substring(0, 8)}</div>
                    </div>
                `;
            }
            
            const status = this.getLightStatus(light);
            
            if (popupStatus) {
                popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
                popupStatus.innerHTML = `
                    <div class="text-xl font-semibold mb-1">${status.label}</div>
                    <div class="flex justify-center items-center gap-2">
                        <i class="fas fa-clock"></i>
                        <span>Changes in ${status.timeLeft} seconds</span>
                    </div>
                    <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                        <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                    </div>
                `;
            }
            
            lightPopup.classList.remove('hidden');
        },
        
        startStatusUpdates() {
            this.stopStatusUpdates();
            
            const updateStatuses = (timestamp) => {
                if (!this.lastTimestamp || timestamp - this.lastTimestamp > 1000) {
                    this.lastTimestamp = timestamp;
                    this.updateAllLightStatus();
                }
                this.rafId = requestAnimationFrame(updateStatuses);
            };
            
            this.rafId = requestAnimationFrame(updateStatuses);
        },
        
        stopStatusUpdates() {
            if (this.rafId) {
                cancelAnimationFrame(this.rafId);
                this.rafId = null;
            }
        },
        
        updateAllLightStatus() {
            for (const id in this.markers) {
                this.updateLightStatus(this.markers[id].data);
            }
        },
        
        updateLightStatus(light) {
            const status = this.getLightStatus(light);
            
            // Update marker
            if (this.markers[light.id]) {
                const el = this.markers[light.id].marker.getElement().firstChild;
                el.className = el.className.replace(/bg-traffic-\w+/, `bg-traffic-${status.color}`);
            }
            
            // Update sidebar entry
            const statusElement = document.getElementById(`status-${light.id}`);
            const timerElement = document.getElementById(`timer-${light.id}`);
            
            if (statusElement) {
                statusElement.className = `px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white`;
                statusElement.textContent = status.label;
            }
            
            if (timerElement) {
                timerElement.innerHTML = `<i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s`;
            }
            
            // Update progress bar
            const card = document.querySelector(`.light-card[data-id="${light.id}"]`);
            if (card) {
                const progressBar = card.querySelector('.bg-traffic-red, .bg-traffic-green');
                if (progressBar) {
                    progressBar.className = `bg-traffic-${status.color} h-1.5 rounded-full`;
                    progressBar.style.width = `${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%`;
                }
            }
            
            // Update popup if visible
            const lightPopup = document.getElementById('lightPopup');
            if (this.selectedLightId === light.id && lightPopup && !lightPopup.classList.contains('hidden')) {
                const popupStatus = document.getElementById('popupStatus');
                if (popupStatus) {
                    popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
                    popupStatus.innerHTML = `
                        <div class="text-xl font-semibold mb-1">${status.label}</div>
                        <div class="flex justify-center items-center gap-2">
                            <i class="fas fa-clock"></i>
                            <span>Changes in ${status.timeLeft} seconds</span>
                        </div>
                        <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                            <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                        </div>
                    `;
                }
            }
        },
        
        refreshLights() {
            const refreshBtn = document.getElementById('refreshBtn');
            if (refreshBtn) {
                refreshBtn.innerHTML = '<i class="fas fa-sync-alt fa-spin"></i>';
            }
            
            // Clear existing markers
            for (const id in this.markers) {
                this.markers[id].marker.remove();
            }
            this.markers = {};
            
            // Clear sidebar
            const lightsList = document.getElementById('lightsList');
            if (lightsList) {
                lightsList.innerHTML = '';
            }
            
            this.loadTrafficLights();
            
            setTimeout(() => {
                if (refreshBtn) {
                    refreshBtn.innerHTML = '<i class="fas fa-sync-alt"></i>';
                }
                this.showNotification('Traffic lights refreshed', 'success');
            }, 800);
        },
        
        findNearbyLights() {
            if (!this.userMarker) {
                this.setupGeolocation();
                return;
            }
            
            this.map.flyTo({
                center: this.userMarker.getLngLat(),
                zoom: 16,
                essential: true
            });
            
            const userPos = this.userMarker.getLngLat();
            const nearbyLights = [];
            
            for (const id in this.markers) {
                const markerPos = this.markers[id].marker.getLngLat();
                const distance = this.getDistance(
                    [userPos.lng, userPos.lat],
                    [markerPos.lng, markerPos.lat]
                );
                
                if (distance < 1000) {
                    nearbyLights.push({
                        id: id,
                        distance: distance
                    });
                }
            }
            
            nearbyLights.sort((a, b) => a.distance - b.distance);
            
            document.querySelectorAll('.light-card').forEach(card => {
                card.classList.remove('active');
            });
            
            if (nearbyLights.length > 0) {
                this.showNotification(`${nearbyLights.length} traffic lights found within 1km`, 'info');
                
                nearbyLights.slice(0, 5).forEach(light => {
                    const card = document.querySelector(`.light-card[data-id="${light.id}"]`);
                    if (card) {
                        card.classList.add('active');
                        card.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                    }
                });
            } else {
                this.showNotification('No traffic lights nearby', 'info');
            }
        },
        
        getDistance(point1, point2) {
            const lat1 = point1[1] * Math.PI / 180;
            const lon1 = point1[0] * Math.PI / 180;
            const lat2 = point2[1] * Math.PI / 180;
            const lon2 = point2[0] * Math.PI / 180;
            
            const dlon = lon2 - lon1;
            const dlat = lat2 - lat1;
            const a = Math.sin(dlat/2)**2 + Math.cos(lat1) * Math.cos(lat2) * Math.sin(dlon/2)**2;
            const c = 2 * Math.asin(Math.sqrt(a));
            const r = 6371; // Radius of earth in kilometers
            
            return c * r * 1000; // Convert to meters
        },
        
        async handleAddLight(e) {
            e.preventDefault();
            
            const form = e.target;
            const submitBtn = form.querySelector('[type="submit"]');
            const originalText = submitBtn.textContent;
            
            submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Adding...';
            submitBtn.disabled = true;
            
            const formData = new FormData(form);
            const lightData = {
                name: formData.get('name'),
                latitude: formData.get('latitude'),
                longitude: formData.get('longitude'),
                direction: formData.get('direction'),
                red_duration: formData.get('red_duration'),
                green_duration: formData.get('green_duration')
            };
            
            try {
                const response = await fetch('db.php?action=addTrafficLight', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(lightData)
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.addLightToMap(data.light);
                    this.addLightToSidebar(data.light, document.getElementById('lightsList'));
                    
                    this.showNotification(data.message || 'Traffic light added successfully', 'success');
                    this.closeModal('addLightModal');
                    
                    if (this.tempMarker) {
                        this.tempMarker.remove();
                        this.tempMarker = null;
                    }
                    
                    document.getElementById('noLightsMessage')?.classList.add('hidden');
                    
                    this.map.flyTo({
                        center: [parseFloat(data.light.longitude), parseFloat(data.light.latitude)],
                        zoom: 16,
                        essential: true
                    });
                    
                    form.reset();
                } else {
                    this.showNotification(data.message || 'Error adding traffic light', 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                this.showNotification('Connection error', 'error');
            } finally {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }
        },
        
        async handleLogin(e) {
            e.preventDefault();
            
            const form = e.target;
            const formData = new FormData(form);
            const submitBtn = form.querySelector('[type="submit"]');
            const originalText = submitBtn.textContent;
            
            submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Logging in...';
            submitBtn.disabled = true;
            
            try {
                const response = await fetch('db.php?action=login', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        email: formData.get('email'),
                        password: formData.get('password'),
                        remember: formData.get('remember') === 'on'
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.showNotification('Login successful', 'success');
                    this.closeModal('loginModal');
                    this.state.isLoggedIn = true;
                    this.updateUIForLoggedInUser(data.user);
                    setTimeout(() => window.location.reload(), 1000);
                } else {
                    this.showNotification(data.message || 'Login failed', 'error');
                }
            } catch (error) {
                console.error('Login error:', error);
                this.showNotification('Connection error', 'error');
            } finally {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }
        },
        
        async handleRegister(e) {
            e.preventDefault();
            
            const form = e.target;
            const formData = new FormData(form);
            const submitBtn = form.querySelector('[type="submit"]');
            const originalText = submitBtn.textContent;
            
            if (formData.get('password') !== formData.get('password_confirm')) {
                this.showNotification('Passwords do not match', 'error');
                return;
            }
            
            submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Registering...';
            submitBtn.disabled = true;
            
            try {
                const response = await fetch('db.php?action=register', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        username: formData.get('username'),
                        email: formData.get('email'),
                        password: formData.get('password'),
                        remember: true
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.showNotification('Registration successful', 'success');
                    this.closeModal('loginModal');
                    this.state.isLoggedIn = true;
                    this.updateUIForLoggedInUser(data.user);
                    setTimeout(() => window.location.reload(), 1000);
                } else {
                    this.showNotification(data.message || 'Registration failed', 'error');
                }
            } catch (error) {
                console.error('Registration error:', error);
                this.showNotification('Connection error', 'error');
            } finally {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }
        },
        
        updateUIForLoggedInUser(user) {
            const loginSection = document.querySelector('.sidebar > .p-4.border-t');
            if (loginSection) {
                loginSection.innerHTML = `
                    <div class="flex items-center mb-3">
                        <div class="w-10 h-10 rounded-full bg-primary-500 text-white flex items-center justify-center mr-3">
                            <i class="fas fa-user"></i>
                        </div>
                        <div>
                            <p class="font-medium text-sm">${user.username}</p>
                            <p class="text-xs text-gray-500">Logged in</p>
                        </div>
                        <button id="logoutBtn" class="ml-auto text-gray-500 hover:text-gray-700">
                            <i class="fas fa-sign-out-alt"></i>
                        </button>
                    </div>
                    <button id="settingsBtn" class="w-full bg-gray-100 hover:bg-gray-200 text-gray-800 font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center justify-center gap-2">
                        <i class="fas fa-cog"></i> Settings
                    </button>
                `;
                
                document.getElementById('logoutBtn')?.addEventListener('click', this.handleLogout.bind(this));
                document.getElementById('settingsBtn')?.addEventListener('click', () => this.openModal('settingsModal'));
            }
        },
        
        async handleLogout() {
            try {
                const response = await fetch('db.php?action=logout');
                const data = await response.json();
                
                if (data.success) {
                    document.cookie = 'user_data=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';
                    this.showNotification('Logout successful', 'success');
                    this.state.isLoggedIn = false;
                    setTimeout(() => window.location.reload(), 1000);
                } else {
                    this.showNotification(data.message || 'Logout failed', 'error');
                }
            } catch (error) {
                console.error('Logout error:', error);
                this.showNotification('Connection error', 'error');
            }
        },
        
        toggleDarkMode() {
            const darkModeToggle = document.getElementById('darkModeToggle');
            if (!darkModeToggle) return;
            
            this.settings.darkMode = darkModeToggle.checked;
            
            if (this.settings.darkMode) {
                document.documentElement.classList.add('dark');
                document.body.classList.add('dark-mode');
            } else {
                document.documentElement.classList.remove('dark');
                document.body.classList.remove('dark-mode');
            }
            
            localStorage.setItem('darkMode', this.settings.darkMode);
        },
        
        toggleTrafficLayer() {
            this.settings.trafficView = !this.settings.trafficView;
            
            if (this.settings.trafficView) {
                if (!this.map.getLayer('traffic')) {
                    this.addTrafficLayer();
                }
            } else {
                if (this.map.getLayer('traffic')) {
                    this.map.removeLayer('traffic');
                    this.map.removeSource('traffic');
                }
            }
            
            // Update toggle in settings if necessary
            const trafficViewToggle = document.getElementById('trafficViewToggle');
            if (trafficViewToggle) {
                trafficViewToggle.checked = this.settings.trafficView;
            }
            
            localStorage.setItem('trafficSettings', JSON.stringify(this.settings));
        },
        
        showNotification(message, type = 'info') {
            const colors = {
                success: 'bg-green-500',
                error: 'bg-red-500',
                info: 'bg-primary-500'
            };
            
            const icons = {
                success: 'check-circle',
                error: 'exclamation-circle',
                info: 'info-circle'
            };
            
            const notificationArea = document.getElementById('notificationArea');
            if (!notificationArea) return;
            
            const notification = document.createElement('div');
            notification.className = `p-4 rounded-xl shadow-lg ${colors[type]} text-white mb-3 flex items-center notification`;
            notification.innerHTML = `<i class="fas fa-${icons[type]} mr-2"></i>${message}`;
            
            notificationArea.appendChild(notification);
            
            setTimeout(() => {
                notification.classList.add('show');
            }, 100);
            
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        },
        
        hideLoadingScreen() {
            const loadingScreen = document.getElementById('loadingScreen');
            if (loadingScreen) {
                loadingScreen.style.opacity = '0';
                setTimeout(() => {
                    loadingScreen.style.display = 'none';
                }, 300);
            }
        }
    };
    
    // Initialize app
    app.init();
});