document.addEventListener('DOMContentLoaded', () => {
    let map;
    let userMarker;
    let tempMarker;
    let directionsControl;
    let mapboxKey;
    let mapkitToken;
    let userLocation = null;
    let isMenuOpen = false;
    let selectedLightId = null;
    let pendingLights = [];
    let markers = {};
    let rafId = null;
    let lastTimestamp = 0;
    let isAddingLight = false;
    let isNavigating = false;
    let navigationRoute = null;
    let measureMode = null;
    let measureTimer = 0;
    let measureInterval = null;
    let currentTransportMode = 'driving';
    
    // Settings
    const settings = {
        darkMode: false,
        notifications: true,
        autoRefresh: true,
        autoMeasure: true,
        mapStyle: 'navigation-day-v1',
        defaultTransport: 'driving'
    };
    
    // User
    const user = {
        isLoggedIn: false,
        firstName: '',
        lastName: '',
        email: '',
        preferredVehicle: 'car'
    };
    
    // DOM Elements
    const dom = {
        map: document.getElementById('map'),
        fabMenu: document.getElementById('fabMenuBtn'),
        menuItems: document.querySelectorAll('.menu-item'),
        routePanel: document.querySelector('.route-panel'),
        searchContainer: document.querySelector('.search-container'),
        infoPanel: document.getElementById('infoPanel'),
        modals: {
            route: document.getElementById('routeSearchModal'),
            add: document.getElementById('addLightModal'),
            measure: document.getElementById('measureModal'),
            settings: document.getElementById('settingsModal'),
            account: document.getElementById('accountModal'),
            onboarding: document.getElementById('onboardingModal')
        }
    };
    
    // Initialize app
    init();
    
    function init() {
        loadSettings();
        loadUserData();
        initMap();
        bindEvents();
        checkFirstVisit();
    }
    
    function loadSettings() {
        const savedSettings = localStorage.getItem('trafficLightSettings');
        if (savedSettings) {
            try {
                const parsed = JSON.parse(savedSettings);
                Object.assign(settings, parsed);
            } catch (e) {
                console.error('Error parsing settings', e);
            }
        }
        
        // Apply settings to UI
        document.getElementById('darkModeToggle').checked = settings.darkMode;
        document.getElementById('notificationsToggle').checked = settings.notifications;
        document.getElementById('autoRefreshToggle').checked = settings.autoRefresh;
        document.getElementById('autoMeasureToggle').checked = settings.autoMeasure;
        document.getElementById('mapStyle').value = settings.mapStyle;
        document.getElementById('defaultTransport').value = settings.defaultTransport;
        
        if (settings.darkMode) {
            document.documentElement.classList.add('dark');
            document.body.classList.add('dark-mode');
        }
    }
    
    function saveSettings() {
        settings.darkMode = document.getElementById('darkModeToggle').checked;
        settings.notifications = document.getElementById('notificationsToggle').checked;
        settings.autoRefresh = document.getElementById('autoRefreshToggle').checked;
        settings.autoMeasure = document.getElementById('autoMeasureToggle').checked;
        settings.mapStyle = document.getElementById('mapStyle').value;
        settings.defaultTransport = document.getElementById('defaultTransport').value;
        
        localStorage.setItem('trafficLightSettings', JSON.stringify(settings));
    }
    
    function loadUserData() {
        const userId = localStorage.getItem('userId') || sessionStorage.getItem('userId');
        if (userId) {
            fetch(`api/user-info.php?id=${userId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        user.isLoggedIn = true;
                        user.firstName = data.user.first_name;
                        user.lastName = data.user.last_name;
                        user.email = data.user.email;
                        user.preferredVehicle = data.user.preferred_vehicle;
                        updateUserDisplay();
                    }
                })
                .catch(error => {
                    console.error('Error loading user data', error);
                });
        } else {
            updateUserDisplay();
        }
    }
    
    function updateUserDisplay() {
        const nameElement = document.getElementById('accountName');
        const emailElement = document.getElementById('accountEmail');
        const loginButtons = document.getElementById('loginButtons');
        const logoutButton = document.getElementById('logoutButton');
        
        if (user.isLoggedIn) {
            nameElement.textContent = `${user.firstName} ${user.lastName}`;
            emailElement.textContent = user.email;
            loginButtons.classList.add('hidden');
            logoutButton.classList.remove('hidden');
        } else {
            nameElement.textContent = 'Guest User';
            emailElement.textContent = 'Not signed in';
            loginButtons.classList.remove('hidden');
            logoutButton.classList.add('hidden');
        }
    }
    
    function initMap() {
        // Load MapBox key
        fetch('get-mapbox-key.php')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    mapboxKey = data.key;
                    mapboxgl.accessToken = mapboxKey;
                    
                    // Initialize map
                    map = new mapboxgl.Map({
                        container: 'map',
                        style: `mapbox://styles/mapbox/${settings.mapStyle}`,
                        center: [-0.09, 51.505],
                        zoom: 15
                    });
                    
                    map.on('load', () => {
                        // Add navigation control
                        map.addControl(new mapboxgl.NavigationControl({
                            showCompass: false
                        }), 'bottom-right');
                        
                        // Add geolocate control
                        const geolocate = new mapboxgl.GeolocateControl({
                            positionOptions: {
                                enableHighAccuracy: true
                            },
                            trackUserLocation: true,
                            showUserHeading: true
                        });
                        
                        map.addControl(geolocate, 'bottom-right');
                        
                        // Add directions control
                        directionsControl = new MapboxDirections({
                            accessToken: mapboxKey,
                            unit: 'metric',
                            profile: 'mapbox/driving',
                            controls: {
                                instructions: false,
                                profileSwitcher: false,
                                inputs: false
                            }
                        });
                        
                        map.addControl(directionsControl, 'top-left');
                        
                        // Hide directions UI
                        document.querySelector('.mapboxgl-ctrl-directions').style.display = 'none';
                        
                        // Add search
                        setupSearchBoxes();
                        
                        // Load traffic lights
                        loadTrafficLights();
                        
                        // Listen for map clicks
                        map.on('click', handleMapClick);
                        
                        // Get user location
                        geolocate.trigger();
                        
                        // Start status updates
                        if (settings.autoRefresh) {
                            startStatusUpdates();
                        }
                    });
                    
                    // Listen for user location
                    map.on('geolocate', (e) => {
                        userLocation = [e.coords.longitude, e.coords.latitude];
                    });
                } else {
                    showNotification('Error', 'Failed to load MapBox key', 'error');
                }
            })
            .catch(error => {
                console.error('Error loading MapBox key', error);
                showNotification('Error', 'Failed to load MapBox key', 'error');
            });
            
        // Load MapKit token
        fetch('get-apple-map-key.php')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    mapkitToken = data.token;
                }
            })
            .catch(error => {
                console.error('Error loading MapKit token', error);
            });
    }
    
    function setupSearchBoxes() {
        // Main search box
        const searchBox = new MapboxGeocoder({
            accessToken: mapboxgl.accessToken,
            mapboxgl: mapboxgl,
            marker: false,
            placeholder: 'Search locations or traffic lights...'
        });
        
        document.getElementById('searchBox').appendChild(searchBox.onAdd(map));
        
        // Route modal search boxes
        const startLocationSearch = new MapboxGeocoder({
            accessToken: mapboxgl.accessToken,
            mapboxgl: mapboxgl,
            marker: false,
            placeholder: 'Enter starting point'
        });
        
        const endLocationSearch = new MapboxGeocoder({
            accessToken: mapboxgl.accessToken,
            mapboxgl: mapboxgl,
            marker: false,
            placeholder: 'Enter destination'
        });
        
        document.getElementById('startLocationSearch').appendChild(startLocationSearch.onAdd(map));
        document.getElementById('endLocationSearch').appendChild(endLocationSearch.onAdd(map));
        
        // Store search results
        startLocationSearch.on('result', (e) => {
            directionsControl.setOrigin(e.result.center);
        });
        
        endLocationSearch.on('result', (e) => {
            directionsControl.setDestination(e.result.center);
        });
    }
    
    function bindEvents() {
        // FAB Menu
        dom.fabMenu.addEventListener('click', toggleMenu);
        
        // Menu items
        dom.menuItems.forEach(item => {
            const button = item.querySelector('button');
            button.addEventListener('click', () => {
                const id = item.id;
                handleMenuItemClick(id);
            });
        });
        
        // Modal close buttons
        document.querySelectorAll('.close-modal').forEach(button => {
            button.addEventListener('click', () => {
                const modal = button.closest('.modal');
                closeModal(modal);
            });
        });
        
        // Modal overlays
        document.querySelectorAll('.modal-overlay').forEach(overlay => {
            overlay.addEventListener('click', () => {
                const modal = overlay.closest('.modal');
                closeModal(modal);
            });
        });
        
        // Info panel
        document.getElementById('closeInfoPanel').addEventListener('click', () => {
            dom.infoPanel.classList.remove('active');
        });
        
        document.getElementById('infoPanelNavigate').addEventListener('click', () => {
            if (selectedLightId && markers[selectedLightId]) {
                const light = markers[selectedLightId].data;
                dom.infoPanel.classList.remove('active');
                calculateRoute([userLocation[0], userLocation[1]], [light.longitude, light.latitude]);
            }
        });
        
        document.getElementById('infoPanelMeasure').addEventListener('click', () => {
            if (selectedLightId) {
                dom.infoPanel.classList.remove('active');
                openMeasureModal(selectedLightId);
            }
        });
        
        // Route panel
        document.getElementById('cancelRouteBtn').addEventListener('click', cancelRoute);
        document.getElementById('startNavigationBtn').addEventListener('click', startNavigation);
        
        // Transport mode buttons - route panel
        document.querySelectorAll('.transport-mode-btn').forEach(button => {
            button.addEventListener('click', () => {
                document.querySelectorAll('.transport-mode-btn').forEach(btn => {
                    btn.classList.remove('active', 'bg-white', 'text-primary-600');
                    btn.classList.add('text-gray-600');
                });
                
                button.classList.add('active', 'bg-white', 'text-primary-600');
                button.classList.remove('text-gray-600');
                
                let mode = 'driving';
                if (button.id === 'bikingModeBtn') mode = 'cycling';
                if (button.id === 'walkingModeBtn') mode = 'walking';
                
                currentTransportMode = mode;
                directionsControl.setProfile(`mapbox/${mode}`);
            });
        });
        
        // Transport mode buttons - route search modal
        document.querySelectorAll('.transport-mode-choice').forEach(button => {
            button.addEventListener('click', () => {
                document.querySelectorAll('.transport-mode-choice').forEach(btn => {
                    btn.classList.remove('active', 'bg-primary-50', 'border-primary-200', 'text-primary-600');
                    btn.classList.add('bg-gray-50', 'border-gray-200', 'text-gray-600');
                });
                
                button.classList.add('active', 'bg-primary-50', 'border-primary-200', 'text-primary-600');
                button.classList.remove('bg-gray-50', 'border-gray-200', 'text-gray-600');
                
                currentTransportMode = button.getAttribute('data-mode');
            });
        });
        
        // Calculate route button
        document.getElementById('calculateRouteBtn').addEventListener('click', () => {
            if (directionsControl.getOrigin() && directionsControl.getDestination()) {
                calculateRoute(
                    directionsControl.getOrigin().geometry.coordinates,
                    directionsControl.getDestination().geometry.coordinates
                );
                closeModal(dom.modals.route);
            } else {
                showNotification('Error', 'Please select start and destination points', 'error');
            }
        });
        
        // Settings form
        document.getElementById('darkModeToggle').addEventListener('change', () => {
            toggleDarkMode();
            saveSettings();
        });
        
        document.getElementById('notificationsToggle').addEventListener('change', saveSettings);
        document.getElementById('autoRefreshToggle').addEventListener('change', () => {
            settings.autoRefresh = document.getElementById('autoRefreshToggle').checked;
            if (settings.autoRefresh) {
                startStatusUpdates();
            } else {
                stopStatusUpdates();
            }
            saveSettings();
        });
        
        document.getElementById('autoMeasureToggle').addEventListener('change', saveSettings);
        
        document.getElementById('mapStyle').addEventListener('change', () => {
            settings.mapStyle = document.getElementById('mapStyle').value;
            if (map) {
                map.setStyle(`mapbox://styles/mapbox/${settings.mapStyle}`);
            }
            saveSettings();
        });
        
        document.getElementById('defaultTransport').addEventListener('change', saveSettings);
        
        // Reset app button
        document.getElementById('resetAppBtn').addEventListener('click', resetApp);
        
        // Add light form
        document.getElementById('addLightForm').addEventListener('submit', handleAddLight);
        
        // Measure buttons
        document.getElementById('measureRedBtn').addEventListener('click', () => startMeasure('red'));
        document.getElementById('measureGreenBtn').addEventListener('click', () => startMeasure('green'));
        document.getElementById('startTimer').addEventListener('click', startTimer);
        document.getElementById('stopTimer').addEventListener('click', stopTimer);
        document.getElementById('saveTimer').addEventListener('click', saveTimer);
        
        // Account buttons
        document.getElementById('loginBtn').addEventListener('click', () => {
            showLoginForm();
            openModal(dom.modals.account);
        });
        
        document.getElementById('signupBtn').addEventListener('click', () => {
            showSignupForm();
            openModal(dom.modals.account);
        });
        
        document.getElementById('logoutBtn').addEventListener('click', logoutUser);
        
        document.getElementById('switchToSignupBtn').addEventListener('click', (e) => {
            e.preventDefault();
            showSignupForm();
        });
        
        document.getElementById('switchToLoginBtn').addEventListener('click', (e) => {
            e.preventDefault();
            showLoginForm();
        });
        
        // Account forms
        document.getElementById('loginForm').addEventListener('submit', (e) => {
            e.preventDefault();
            loginUser();
        });
        
        document.getElementById('signupForm').addEventListener('submit', (e) => {
            e.preventDefault();
            signupUser();
        });
        
        document.getElementById('onboardingForm').addEventListener('submit', (e) => {
            e.preventDefault();
            createAccount(true);
        });
        
        // Notification close
        document.getElementById('closeNotification').addEventListener('click', () => {
            document.getElementById('notificationBanner').style.opacity = '0';
        });
        
        // Certification banner
        document.getElementById('dismissCertBanner').addEventListener('click', () => {
            document.getElementById('certificationBanner').classList.add('hidden');
        });
    }
    
    function toggleMenu() {
        isMenuOpen = !isMenuOpen;
        
        if (isMenuOpen) {
            dom.fabMenu.innerHTML = '<i class="fas fa-times"></i>';
            
            dom.menuItems.forEach(item => {
                const position = parseInt(item.getAttribute('data-position'));
                setTimeout(() => {
                    item.classList.add('active');
                    item.style.transform = `translateY(-${position * 70}px)`;
                }, position * 50);
            });
        } else {
            dom.fabMenu.innerHTML = '<i class="fas fa-plus"></i>';
            
            dom.menuItems.forEach(item => {
                item.classList.remove('active');
                item.style.transform = '';
            });
        }
    }
    
    function handleMenuItemClick(id) {
        toggleMenu();
        
        switch (id) {
            case 'accountMenuItem':
                if (user.isLoggedIn) {
                    showAccountForm();
                } else {
                    showLoginForm();
                }
                openModal(dom.modals.account);
                break;
                
            case 'addLightMenuItem':
                isAddingLight = true;
                if (userLocation) {
                    document.getElementById('latitude').value = userLocation[1].toFixed(6);
                    document.getElementById('longitude').value = userLocation[0].toFixed(6);
                    
                    if (tempMarker) tempMarker.remove();
                    
                    tempMarker = new mapboxgl.Marker({
                        draggable: true,
                        color: '#10b981'
                    })
                    .setLngLat([userLocation[0], userLocation[1]])
                    .addTo(map);
                    
                    tempMarker.on('dragend', updateMarkerPosition);
                }
                openModal(dom.modals.add);
                break;
                
            case 'searchMenuItem':
                openModal(dom.modals.route);
                break;
                
            case 'locateMenuItem':
                if (userLocation) {
                    map.flyTo({
                        center: userLocation,
                        zoom: 16
                    });
                } else {
                    showNotification('Location', 'Getting your location...', 'info');
                    // Trigger geolocate control
                    document.querySelector('.mapboxgl-ctrl-geolocate').click();
                }
                break;
                
            case 'settingsMenuItem':
                openModal(dom.modals.settings);
                break;
        }
    }
    
    function openModal(modal) {
        modal.style.display = 'flex';
        setTimeout(() => {
            const content = modal.querySelector('.modal-content');
            content.classList.add('active');
        }, 10);
    }
    
    function closeModal(modal) {
        const content = modal.querySelector('.modal-content');
        content.classList.remove('active');
        
        setTimeout(() => {
            modal.style.display = 'none';
            
            if (modal === dom.modals.add && isAddingLight) {
                isAddingLight = false;
                if (tempMarker) {
                    tempMarker.remove();
                    tempMarker = null;
                }
            }
        }, 300);
    }
    
    function handleMapClick(e) {
        if (isAddingLight) {
            document.getElementById('latitude').value = e.lngLat.lat.toFixed(6);
            document.getElementById('longitude').value = e.lngLat.lng.toFixed(6);
            
            if (tempMarker) tempMarker.remove();
            
            tempMarker = new mapboxgl.Marker({
                draggable: true,
                color: '#10b981'
            })
            .setLngLat([e.lngLat.lng, e.lngLat.lat])
            .addTo(map);
            
            tempMarker.on('dragend', updateMarkerPosition);
        }
    }
    
    function updateMarkerPosition() {
        const lngLat = tempMarker.getLngLat();
        document.getElementById('latitude').value = lngLat.lat.toFixed(6);
        document.getElementById('longitude').value = lngLat.lng.toFixed(6);
    }
    
    function handleAddLight(e) {
        e.preventDefault();
        
        const submitBtn = e.target.querySelector('[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Adding...';
        submitBtn.disabled = true;
        
        const lightData = {
            name: document.getElementById('lightName').value,
            latitude: document.getElementById('latitude').value,
            longitude: document.getElementById('longitude').value,
            direction: document.getElementById('direction').value,
            red_duration: document.getElementById('redDuration').value,
            green_duration: document.getElementById('greenDuration').value,
            user_id: user.isLoggedIn ? (localStorage.getItem('userId') || sessionStorage.getItem('userId')) : null
        };
        
        fetch('api/add-traffic-light.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(lightData)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                pendingLights.push(data.light.id);
                document.getElementById('certificationBanner').classList.remove('hidden');
                
                showNotification('Success', 'Traffic light added successfully! Pending certification.', 'success');
                closeModal(dom.modals.add);
                
                e.target.reset();
                
                if (tempMarker) {
                    tempMarker.remove();
                    tempMarker = null;
                }
                
                isAddingLight = false;
                addLightToMap(data.light);
            } else {
                showNotification('Error', data.message, 'error');
            }
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        })
        .catch(error => {
            console.error('Error adding light:', error);
            showNotification('Error', 'Connection error', 'error');
            
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        });
    }
    
    function loadTrafficLights() {
        fetch('api/get-traffic-lights.php')
            .then(response => response.json())
            .then(data => {
                if (Array.isArray(data) && data.length > 0) {
                    data.forEach(light => {
                        addLightToMap(light);
                    });
                }
            })
            .catch(error => {
                console.error('Error loading traffic lights:', error);
                showNotification('Error', 'Failed to load traffic lights', 'error');
            });
    }
    
    function addLightToMap(light) {
        const status = getLightStatus(light);
        
        const el = document.createElement('div');
        el.className = `w-8 h-8 bg-traffic-${status.color} rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white`;
        el.innerHTML = '<i class="fas fa-traffic-light"></i>';
        
        const marker = new mapboxgl.Marker(el)
            .setLngLat([parseFloat(light.longitude), parseFloat(light.latitude)])
            .addTo(map);
        
        marker.getElement().addEventListener('click', () => {
            showLightDetail(light);
        });
        
        markers[light.id] = {
            marker: marker,
            element: el,
            data: light
        };
    }
    
    function getLightStatus(light) {
        const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
        const currentTime = Math.floor(Date.now() / 1000);
        const timeInCycle = currentTime % totalCycle;
        
        if (timeInCycle < light.red_duration) {
            return {
                isRed: true,
                color: 'red',
                label: 'RED',
                timeLeft: light.red_duration - timeInCycle,
                percentage: Math.round(((light.red_duration - timeInCycle) / totalCycle) * 100)
            };
        } else {
            return {
                isRed: false,
                color: 'green',
                label: 'GREEN',
                timeLeft: totalCycle - timeInCycle,
                percentage: Math.round(((totalCycle - timeInCycle) / totalCycle) * 100)
            };
        }
    }
    
    function updateLightStatus(light) {
        const status = getLightStatus(light);
        
        if (markers[light.id]) {
            const el = markers[light.id].element;
            el.className = `w-8 h-8 bg-traffic-${status.color} rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white`;
        }
        
        if (selectedLightId === light.id && dom.infoPanel.classList.contains('active')) {
            document.getElementById('lightStatusText').textContent = status.label;
            document.getElementById('lightStatusTime').textContent = `${status.timeLeft}s until change`;
            document.getElementById('lightStatusBar').className = `bg-traffic-${status.color} h-2 rounded-full`;
            document.getElementById('lightStatusBar').style.width = `${status.percentage}%`;
        }
    }
    
    function updateAllLightStatus() {
        for (const id in markers) {
            updateLightStatus(markers[id].data);
        }
    }
    
    function startStatusUpdates() {
        stopStatusUpdates();
        
        function updateStatuses(timestamp) {
            if (!lastTimestamp || timestamp - lastTimestamp > 1000) {
                lastTimestamp = timestamp;
                updateAllLightStatus();
            }
            rafId = requestAnimationFrame(updateStatuses);
        }
        
        rafId = requestAnimationFrame(updateStatuses);
    }
    
    function stopStatusUpdates() {
        if (rafId) {
            cancelAnimationFrame(rafId);
            rafId = null;
        }
    }
    
    function showLightDetail(light) {
        selectedLightId = light.id;
        
        map.flyTo({
            center: [parseFloat(light.longitude), parseFloat(light.latitude)],
            zoom: 17
        });
        
        document.getElementById('infoPanelTitle').textContent = light.name;
        
        const status = getLightStatus(light);
        
        document.getElementById('lightStatusText').textContent = status.label;
        document.getElementById('lightStatusTime').textContent = `${status.timeLeft}s until change`;
        document.getElementById('lightStatusBar').className = `bg-traffic-${status.color} h-2 rounded-full`;
        document.getElementById('lightStatusBar').style.width = `${status.percentage}%`;
        
        dom.infoPanel.classList.add('active');
    }
    
    function calculateRoute(origin, destination) {
        if (!origin || !destination) {
            showNotification('Error', 'Start and destination are required', 'error');
            return;
        }
        
        directionsControl.setOrigin(origin);
        directionsControl.setDestination(destination);
        directionsControl.setProfile(`mapbox/${currentTransportMode}`);
        
        document.querySelector('.mapboxgl-ctrl-directions').style.display = 'block';
        
        dom.routePanel.classList.add('active');
        dom.searchContainer.classList.add('route-active');
        
        directionsControl.on('route', (e) => {
            if (e.route && e.route[0]) {
                const route = e.route[0];
                const duration = Math.round(route.duration / 60);
                
                document.getElementById('estimatedDuration').textContent = `${duration} min`;
                
                const now = new Date();
                const arrivalTime = new Date(now.getTime() + route.duration * 1000);
                document.getElementById('arrivalTime').textContent = `Arrival at ${arrivalTime.getHours().toString().padStart(2, '0')}:${arrivalTime.getMinutes().toString().padStart(2, '0')}`;
                
                navigationRoute = route;
            }
        });
    }
    
    function cancelRoute() {
        dom.routePanel.classList.remove('active');
        dom.searchContainer.classList.remove('route-active');
        
        document.querySelector('.mapboxgl-ctrl-directions').style.display = 'none';
        directionsControl.removeRoutes();
        
        isNavigating = false;
        document.getElementById('startNavigationBtn').innerHTML = '<i class="fas fa-play"></i>';
    }
    
    function startNavigation() {
        if (isNavigating) {
            isNavigating = false;
            document.getElementById('startNavigationBtn').innerHTML = '<i class="fas fa-play"></i>';
            return;
        }
        
        if (!navigationRoute) {
            showNotification('Error', 'No route calculated', 'error');
            return;
        }
        
        isNavigating = true;
        document.getElementById('startNavigationBtn').innerHTML = '<i class="fas fa-stop"></i>';
        
        showNotification('Navigation', 'Navigation started', 'success');
    }
    
    function openMeasureModal(lightId) {
        const light = markers[lightId].data;
        
        document.getElementById('measureTitle').textContent = light.name;
        document.getElementById('measureTitle').setAttribute('data-id', lightId);
        
        const status = getLightStatus(light);
        document.getElementById('measureStatus').innerHTML = `
            <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                ${status.label} (${status.timeLeft}s)
            </span>
        `;
        
        document.getElementById('timerContainer').classList.add('hidden');
        measureMode = null;
        resetTimer();
        
        openModal(dom.modals.measure);
    }
    
    function startMeasure(mode) {
        measureMode = mode;
        document.getElementById('timerContainer').classList.remove('hidden');
        
        if (mode === 'red') {
            document.getElementById('timerInstructions').textContent = 'Press "Start" when the light turns red, then "Stop" when it turns green.';
            document.getElementById('startTimer').className = 'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors';
           document.getElementById('stopTimer').className = 'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors opacity-50';
       } else {
           document.getElementById('timerInstructions').textContent = 'Press "Start" when the light turns green, then "Stop" when it turns red.';
           document.getElementById('startTimer').className = 'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors';
           document.getElementById('stopTimer').className = 'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors opacity-50';
       }
       
       resetTimer();
   }
   
   function startTimer() {
       if (measureInterval) clearInterval(measureInterval);
       
       measureTimer = 0;
       document.getElementById('timerDisplay').textContent = '00:00';
       document.getElementById('measureResult').textContent = 'Measuring...';
       
       document.getElementById('startTimer').disabled = true;
       document.getElementById('startTimer').classList.add('opacity-50');
       document.getElementById('stopTimer').disabled = false;
       document.getElementById('stopTimer').classList.remove('opacity-50');
       document.getElementById('saveTimer').disabled = true;
       document.getElementById('saveTimer').classList.add('opacity-50');
       
       const startTime = Date.now();
       
       measureInterval = setInterval(() => {
           const elapsed = Math.floor((Date.now() - startTime) / 1000);
           measureTimer = elapsed;
           
           const minutes = Math.floor(elapsed / 60);
           const seconds = elapsed % 60;
           document.getElementById('timerDisplay').textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
       }, 1000);
   }
   
   function stopTimer() {
       if (measureInterval) {
           clearInterval(measureInterval);
           measureInterval = null;
           
           document.getElementById('measureResult').textContent = `Measured duration: ${measureTimer} seconds. Click Save to confirm.`;
           
           document.getElementById('stopTimer').disabled = true;
           document.getElementById('stopTimer').classList.add('opacity-50');
           document.getElementById('saveTimer').disabled = false;
           document.getElementById('saveTimer').classList.remove('opacity-50');
       }
   }
   
   function resetTimer() {
       if (measureInterval) {
           clearInterval(measureInterval);
           measureInterval = null;
       }
       
       measureTimer = 0;
       document.getElementById('timerDisplay').textContent = '00:00';
       document.getElementById('measureResult').textContent = '';
       
       document.getElementById('startTimer').disabled = false;
       document.getElementById('startTimer').classList.remove('opacity-50');
       document.getElementById('stopTimer').disabled = true;
       document.getElementById('stopTimer').classList.add('opacity-50');
       document.getElementById('saveTimer').disabled = true;
       document.getElementById('saveTimer').classList.add('opacity-50');
   }
   
   function saveTimer() {
       const lightId = document.getElementById('measureTitle').getAttribute('data-id');
       
       if (!lightId || measureTimer <= 0 || !measureMode) {
           showNotification('Error', 'Invalid measurement data', 'error');
           return;
       }
       
       document.getElementById('measureResult').textContent = 'Saving...';
       
       fetch('api/update-traffic-light-timing.php', {
           method: 'POST',
           headers: { 'Content-Type': 'application/json' },
           body: JSON.stringify({
               id: lightId,
               duration_type: measureMode,
               duration: measureTimer,
               user_id: user.isLoggedIn ? (localStorage.getItem('userId') || sessionStorage.getItem('userId')) : null
           })
       })
       .then(response => response.json())
       .then(data => {
           if (data.success) {
               document.getElementById('measureResult').innerHTML = '<i class="fas fa-check text-green-500 mr-1"></i> Timing updated!';
               
               if (markers[lightId]) {
                   markers[lightId].data = data.light;
                   updateLightStatus(data.light);
               }
               
               document.getElementById('saveTimer').disabled = true;
               document.getElementById('saveTimer').classList.add('opacity-50');
               
               setTimeout(() => {
                   document.getElementById('timerContainer').classList.add('hidden');
                   measureMode = null;
               }, 1500);
           } else {
               document.getElementById('measureResult').innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Error: ' + data.message;
           }
       })
       .catch(error => {
           console.error('Error saving timer:', error);
           document.getElementById('measureResult').innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Connection error';
       });
   }
   
   function toggleDarkMode() {
       settings.darkMode = document.getElementById('darkModeToggle').checked;
       
       if (settings.darkMode) {
           document.documentElement.classList.add('dark');
           document.body.classList.add('dark-mode');
           
           if (map && settings.mapStyle.includes('day')) {
               settings.mapStyle = settings.mapStyle.replace('day', 'night');
               document.getElementById('mapStyle').value = settings.mapStyle;
               map.setStyle(`mapbox://styles/mapbox/${settings.mapStyle}`);
           }
       } else {
           document.documentElement.classList.remove('dark');
           document.body.classList.remove('dark-mode');
           
           if (map && settings.mapStyle.includes('night')) {
               settings.mapStyle = settings.mapStyle.replace('night', 'day');
               document.getElementById('mapStyle').value = settings.mapStyle;
               map.setStyle(`mapbox://styles/mapbox/${settings.mapStyle}`);
           }
       }
   }
   
   function resetApp() {
       if (confirm('Are you sure you want to reset all app data? This will clear all your settings and logout.')) {
           localStorage.removeItem('trafficLightSettings');
           localStorage.removeItem('trafficLightOnboardingCompleted');
           localStorage.removeItem('userId');
           sessionStorage.removeItem('userId');
           
           showNotification('Reset', 'App data reset. Refreshing...', 'info');
           
           setTimeout(() => {
               window.location.reload();
           }, 1500);
       }
   }
   
   function showNotification(title, message, type) {
       if (!settings.notifications && type !== 'error') return;
       
       const banner = document.getElementById('notificationBanner');
       const titleElement = document.getElementById('notificationTitle');
       const messageElement = document.getElementById('notificationMessage');
       const iconElement = document.getElementById('notificationIcon');
       
       titleElement.textContent = title;
       messageElement.textContent = message;
       
       let bgColor, iconClass;
       
       switch (type) {
           case 'success':
               bgColor = 'bg-traffic-green';
               iconClass = 'fas fa-check';
               break;
           case 'error':
               bgColor = 'bg-traffic-red';
               iconClass = 'fas fa-exclamation-circle';
               break;
           default:
               bgColor = 'bg-primary-500';
               iconClass = 'fas fa-info-circle';
       }
       
       iconElement.className = `w-8 h-8 ${bgColor} text-white rounded-full flex items-center justify-center mr-3`;
       iconElement.innerHTML = `<i class="${iconClass}"></i>`;
       
       banner.style.display = 'block';
       setTimeout(() => {
           banner.style.opacity = '1';
       }, 10);
       
       setTimeout(() => {
           banner.style.opacity = '0';
           setTimeout(() => {
               banner.style.display = 'none';
           }, 300);
       }, 4000);
   }
   
   function showAccountForm() {
       document.getElementById('accountForm').classList.remove('hidden');
       document.getElementById('loginForm').classList.add('hidden');
       document.getElementById('signupForm').classList.add('hidden');
       
       if (user.isLoggedIn) {
           document.getElementById('firstName').value = user.firstName;
           document.getElementById('lastName').value = user.lastName;
           document.getElementById('email').value = user.email;
           document.getElementById('preferredVehicle').value = user.preferredVehicle;
       }
   }
   
   function showLoginForm() {
       document.getElementById('accountForm').classList.add('hidden');
       document.getElementById('loginForm').classList.remove('hidden');
       document.getElementById('signupForm').classList.add('hidden');
   }
   
   function showSignupForm() {
       document.getElementById('accountForm').classList.add('hidden');
       document.getElementById('loginForm').classList.add('hidden');
       document.getElementById('signupForm').classList.remove('hidden');
   }
   
   function loginUser() {
       const email = document.getElementById('loginEmail').value;
       const password = document.getElementById('loginPassword').value;
       const rememberMe = document.getElementById('rememberMe').checked;
       
       fetch('api/login.php', {
           method: 'POST',
           headers: { 'Content-Type': 'application/json' },
           body: JSON.stringify({ email, password })
       })
       .then(response => response.json())
       .then(data => {
           if (data.success) {
               user.isLoggedIn = true;
               user.firstName = data.user.first_name;
               user.lastName = data.user.last_name;
               user.email = data.user.email;
               user.preferredVehicle = data.user.preferred_vehicle;
               
               if (rememberMe) {
                   localStorage.setItem('userId', data.user.id);
               } else {
                   sessionStorage.setItem('userId', data.user.id);
               }
               
               updateUserDisplay();
               closeModal(dom.modals.account);
               showNotification('Success', 'Logged in successfully', 'success');
           } else {
               showNotification('Error', data.message, 'error');
           }
       })
       .catch(error => {
           console.error('Login error:', error);
           showNotification('Error', 'Connection error', 'error');
       });
   }
   
   function signupUser() {
       createAccount(false);
   }
   
   function createAccount(isOnboarding = false) {
       const prefix = isOnboarding ? 'onboarding' : 'signup';
       
       const firstName = document.getElementById(`${prefix}FirstName`).value;
       const lastName = document.getElementById(`${prefix}LastName`).value;
       const email = document.getElementById(`${prefix}Email`).value;
       const password = document.getElementById(`${prefix}Password`).value;
       const preferredVehicle = document.getElementById(`${prefix}Vehicle`).value;
       
       fetch('api/signup.php', {
           method: 'POST',
           headers: { 'Content-Type': 'application/json' },
           body: JSON.stringify({ firstName, lastName, email, password, preferredVehicle })
       })
       .then(response => response.json())
       .then(data => {
           if (data.success) {
               user.isLoggedIn = true;
               user.firstName = firstName;
               user.lastName = lastName;
               user.email = email;
               user.preferredVehicle = preferredVehicle;
               
               localStorage.setItem('userId', data.user.id);
               
               if (isOnboarding) {
                   localStorage.setItem('trafficLightOnboardingCompleted', 'true');
                   closeModal(dom.modals.onboarding);
               } else {
                   closeModal(dom.modals.account);
               }
               
               updateUserDisplay();
               showNotification('Success', 'Account created successfully', 'success');
               
               settings.defaultTransport = preferredVehicle === 'car' ? 'driving' : 
                                          preferredVehicle === 'bicycle' ? 'cycling' : 'walking';
               saveSettings();
           } else {
               showNotification('Error', data.message, 'error');
           }
       })
       .catch(error => {
           console.error('Signup error:', error);
           showNotification('Error', 'Connection error', 'error');
       });
   }
   
   function logoutUser() {
       user.isLoggedIn = false;
       user.firstName = '';
       user.lastName = '';
       user.email = '';
       user.preferredVehicle = 'car';
       
       localStorage.removeItem('userId');
       sessionStorage.removeItem('userId');
       
       updateUserDisplay();
       closeModal(dom.modals.account);
       
       showNotification('Logout', 'You have been logged out', 'info');
   }
   
   function checkFirstVisit() {
       if (!localStorage.getItem('trafficLightOnboardingCompleted') && !user.isLoggedIn) {
           openModal(dom.modals.onboarding);
       }
   }
});