<?php 
require_once 'config.php';

$query = $pdo->query("
    SELECT 
        c.*,
        u.username as author_name,
        COUNT(DISTINCT CASE WHEN i.type = 'view' THEN i.id END) as view_count,
        COUNT(DISTINCT CASE WHEN i.type = 'like' THEN i.id END) as like_count,
        COUNT(DISTINCT CASE WHEN i.type = 'dislike' THEN i.id END) as dislike_count,
        COUNT(DISTINCT CASE WHEN i.type = 'comment' THEN i.id END) as comment_count
    FROM content c
    LEFT JOIN users u ON c.author_id = u.id
    LEFT JOIN interactions i ON c.id = i.content_id
    GROUP BY c.id
    ORDER BY c.created_at DESC
");

$videos = $query->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ForMore</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-black to-blue-900 min-h-screen text-white p-8">
    <nav class="flex justify-between items-center mb-8">
        <h1 class="text-3xl font-bold">ForMore</h1>
        <div>
            <?php if(isset($_SESSION['user_id'])): ?>
                <span class="mr-4">Welcome, <?= htmlspecialchars($_SESSION['username']) ?></span>
                <a href="profile.php" class="text-blue-400 hover:text-blue-300">Profile</a>
                <a href="logout.php" class="ml-4 text-red-400 hover:text-red-300">Logout</a>
            <?php else: ?>
                <a href="login.php" class="text-blue-400 hover:text-blue-300">Login</a>
                <a href="signup.php" class="ml-4 text-green-400 hover:text-green-300">Sign Up</a>
            <?php endif; ?>
        </div>
    </nav>

    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
        <?php foreach($videos as $video): ?>
            <div class="bg-gray-800 rounded-lg overflow-hidden cursor-pointer hover:scale-105 transition-transform" onclick="openModal(<?= $video['id'] ?>)">
                <div class="relative h-48 bg-cover bg-center" style="background-image: url('<?= htmlspecialchars($video['picture_cover']) ?>');">
                    <div class="absolute bottom-2 right-2 bg-black bg-opacity-70 px-2 py-1 rounded">
                        <?= htmlspecialchars($video['time']) ?>
                    </div>
                </div>
                <div class="p-4">
                    <h2 class="text-xl font-bold mb-2"><?= htmlspecialchars($video['name']) ?></h2>
                    <p class="text-gray-400 mb-2"><?= substr(htmlspecialchars($video['description']), 0, 100) ?>...</p>
                    <div class="flex justify-between items-center mt-4">
                        <span class="text-sm text-gray-400"><?= htmlspecialchars($video['author_name']) ?></span>
                        <div class="flex space-x-2 text-sm text-gray-400">
                            <span><?= $video['view_count'] ?> views</span>
                        </div>
                    </div>
                    <div class="flex flex-wrap gap-2 mt-2">
                        <?php foreach(explode(',', $video['tags']) as $tag): ?>
                            <span class="bg-blue-900 px-2 py-1 rounded-full text-xs"><?= htmlspecialchars(trim($tag)) ?></span>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>

    <div id="videoModal" class="hidden fixed inset-0 bg-black bg-opacity-90 flex items-center justify-center p-4">
        <div class="bg-gray-800 rounded-lg w-full max-w-4xl">
            <div id="modalContent" class="p-4">
                <div class="flex justify-end">
                    <button onclick="closeModal()" class="text-gray-400 hover:text-white">&times; Close</button>
                </div>
                <div id="videoContainer" class="aspect-w-16 aspect-h-9 mb-4">
                    <div class="flex items-center justify-center h-full">
                        <div class="loading">Loading...</div>
                    </div>
                </div>
                <div id="videoInfo"></div>
            </div>
        </div>
    </div>

    <script>
        function openModal(videoId) {
            document.getElementById('videoModal').classList.remove('hidden');
            fetch('get_video.php?id=' + videoId)
                .then(response => response.text())
                .then(html => {
                    document.getElementById('videoContainer').innerHTML = html;
                });
        }

        function closeModal() {
            document.getElementById('videoModal').classList.add('hidden');
            document.getElementById('videoContainer').innerHTML = '<div class="flex items-center justify-center h-full"><div class="loading">Loading...</div></div>';
        }
    </script>
</body>
</html>