const WebSocket = require('ws');
const http = require('http');
const url = require('url');
const mysql = require('mysql2/promise');

const PORT = process.env.PORT || 3000;

const dbConfig = {
    host: 'localhost',
    user: 'root',
    password: '',
    database: 'cergies'
};

const server = http.createServer();
const wss = new WebSocket.Server({ server });

const clients = new Map();

async function getDbConnection() {
    return await mysql.createConnection(dbConfig);
}

const messageHandlers = {
    'chat': async (data, connection, sender) => {
        const { receiver_id, content, type } = data;
        
        if (!receiver_id || !content) {
            return { error: 'Données manquantes' };
        }
        
        try {
            const query = `
                INSERT INTO communications 
                (sender_id, receiver_id, content, type, read, created_at) 
                VALUES (?, ?, ?, ?, 0, NOW())
            `;
            
            const [result] = await connection.execute(query, [
                sender.user_id, 
                receiver_id, 
                content, 
                type || 'text'
            ]);
            
            const receiverClient = Array.from(clients.values()).find(
                client => client.user_id === parseInt(receiver_id)
            );
            
            if (receiverClient && receiverClient.ws.readyState === WebSocket.OPEN) {
                receiverClient.ws.send(JSON.stringify({
                    type: 'chat',
                    data: {
                        message_id: result.insertId,
                        sender_id: sender.user_id,
                        sender_username: sender.username,
                        content,
                        message_type: type || 'text',
                        created_at: new Date().toISOString()
                    }
                }));
            }
            
            return { 
                success: true, 
                message_id: result.insertId
            };
        } catch (error) {
            console.error('Erreur de base de données:', error);
            return { error: 'Erreur lors de l\'envoi du message' };
        }
    },
    
    'position_update': async (data, connection, sender) => {
        const { latitude, longitude } = data;
        
        if (!latitude || !longitude) {
            return { error: 'Coordonnées manquantes' };
        }
        
        try {
            const query = `
                UPDATE users 
                SET last_position_lat = ?, last_position_lng = ?, last_active = NOW() 
                WHERE id = ?
            `;
            
            await connection.execute(query, [latitude, longitude, sender.user_id]);
            
            const nearbyQuery = `
                SELECT id, username, level
                FROM users
                WHERE 
                    id != ? AND
                    last_position_lat IS NOT NULL AND 
                    last_position_lng IS NOT NULL AND
                    (6371 * acos(
                        cos(radians(?)) * cos(radians(last_position_lat)) * 
                        cos(radians(last_position_lng) - radians(?)) + 
                        sin(radians(?)) * sin(radians(last_position_lat))
                    )) < 10
            `;
            
            const [nearbyUsers] = await connection.execute(nearbyQuery, [
                sender.user_id, 
                latitude, 
                longitude, 
                latitude
            ]);
            
            nearbyUsers.forEach(user => {
                const userClient = Array.from(clients.values()).find(
                    client => client.user_id === user.id
                );
                
                if (userClient && userClient.ws.readyState === WebSocket.OPEN) {
                    userClient.ws.send(JSON.stringify({
                        type: 'nearby_agent',
                        data: {
                            user_id: sender.user_id,
                            username: sender.username,
                            level: sender.level
                        }
                    }));
                }
            });
            
            return { 
                success: true,
                nearby_users: nearbyUsers
            };
        } catch (error) {
            console.error('Erreur de base de données:', error);
            return { error: 'Erreur lors de la mise à jour de position' };
        }
    },
    
    'alliance_request': async (data, connection, sender) => {
        const { target_user_id } = data;
        
        if (!target_user_id) {
            return { error: 'ID utilisateur cible manquant' };
        }
        
        try {
            const checkQuery = `
                SELECT * FROM alliances 
                WHERE 
                    (user1_id = ? AND user2_id = ?) OR 
                    (user1_id = ? AND user2_id = ?)
            `;
            
            const [existing] = await connection.execute(checkQuery, [
                sender.user_id, 
                target_user_id,
                target_user_id,
                sender.user_id
            ]);
            
            if (existing.length > 0) {
                return { error: 'Une relation existe déjà avec cet utilisateur' };
            }
            
            const query = `
                INSERT INTO alliances 
                (user1_id, user2_id, status, created_at) 
                VALUES (?, ?, 'pending', NOW())
            `;
            
            const [result] = await connection.execute(query, [
                sender.user_id, 
                target_user_id
            ]);
            
            const targetClient = Array.from(clients.values()).find(
                client => client.user_id === parseInt(target_user_id)
            );
            
            if (targetClient && targetClient.ws.readyState === WebSocket.OPEN) {
                targetClient.ws.send(JSON.stringify({
                    type: 'alliance_request',
                    data: {
                        alliance_id: result.insertId,
                        user_id: sender.user_id,
                        username: sender.username
                    }
                }));
            }
            
            return { 
                success: true, 
                alliance_id: result.insertId
            };
        } catch (error) {
            console.error('Erreur de base de données:', error);
            return { error: 'Erreur lors de la demande d\'alliance' };
        }
    },
    
    'alliance_response': async (data, connection, sender) => {
        const { alliance_id, accept } = data;
        
        if (!alliance_id) {
            return { error: 'ID alliance manquant' };
        }
        
        try {
            const status = accept ? 'active' : 'rejected';
            
            const query = `
                UPDATE alliances 
                SET status = ? 
                WHERE id = ? AND user2_id = ?
            `;
            
            const [result] = await connection.execute(query, [
                status, 
                alliance_id, 
                sender.user_id
            ]);
            
            if (result.affectedRows === 0) {
                return { error: 'Alliance non trouvée ou vous n\'êtes pas autorisé à la modifier' };
            }
            
            const allianceQuery = `
                SELECT user1_id FROM alliances WHERE id = ?
            `;
            
            const [allianceData] = await connection.execute(allianceQuery, [alliance_id]);
            
            if (allianceData.length > 0) {
                const requesterId = allianceData[0].user1_id;
                const requesterClient = Array.from(clients.values()).find(
                    client => client.user_id === requesterId
                );
                
                if (requesterClient && requesterClient.ws.readyState === WebSocket.OPEN) {
                    requesterClient.ws.send(JSON.stringify({
                        type: 'alliance_response',
                        data: {
                            alliance_id,
                            user_id: sender.user_id,
                            username: sender.username,
                            accepted: accept
                        }
                    }));
                }
            }
            
            return { 
                success: true
            };
        } catch (error) {
            console.error('Erreur de base de données:', error);
            return { error: 'Erreur lors de la réponse à la demande d\'alliance' };
        }
    },
    
    'voice_call': async (data, connection, sender) => {
        const { target_user_id, signal_data } = data;
        
        if (!target_user_id || !signal_data) {
            return { error: 'Données manquantes' };
        }
        
        const targetClient = Array.from(clients.values()).find(
            client => client.user_id === parseInt(target_user_id)
        );
        
        if (targetClient && targetClient.ws.readyState === WebSocket.OPEN) {
            targetClient.ws.send(JSON.stringify({
                type: 'voice_call',
                data: {
                    caller_id: sender.user_id,
                    caller_username: sender.username,
                    signal_data
                }
            }));
            return { success: true };
        } else {
            return { error: 'Utilisateur non connecté' };
        }
    },
    
    'voice_answer': async (data, connection, sender) => {
        const { caller_id, signal_data, accept } = data;
        
        if (!caller_id || !signal_data) {
            return { error: 'Données manquantes' };
        }
        
        const callerClient = Array.from(clients.values()).find(
            client => client.user_id === parseInt(caller_id)
        );
        
        if (callerClient && callerClient.ws.readyState === WebSocket.OPEN) {
            callerClient.ws.send(JSON.stringify({
                type: 'voice_answer',
                data: {
                    user_id: sender.user_id,
                    username: sender.username,
                    signal_data,
                    accept
                }
            }));
            return { success: true };
        } else {
            return { error: 'Appelant non connecté' };
        }
    }
};

wss.on('connection', async (ws, req) => {
    const parameters = url.parse(req.url, true).query;
    const userId = parameters.user_id;
    const token = parameters.token;
    
    if (!userId || !token) {
        ws.send(JSON.stringify({ error: 'Authentification requise' }));
        ws.close();
        return;
    }
    
    let connection;
    try {
        connection = await getDbConnection();
        
        const [users] = await connection.execute(
            'SELECT id, username, level FROM users WHERE id = ?',
            [userId]
        );
        
        if (users.length === 0) {
            ws.send(JSON.stringify({ error: 'Utilisateur non trouvé' }));
            ws.close();
            return;
        }
        
        const user = users[0];
        clients.set(ws, {
            ws,
            user_id: user.id,
            username: user.username,
            level: user.level
        });
        
        ws.send(JSON.stringify({ type: 'connected', user_id: user.id }));
        
        ws.on('message', async (message) => {
            try {
                const data = JSON.parse(message);
                
                if (!data.type || !messageHandlers[data.type]) {
                    ws.send(JSON.stringify({ error: 'Type de message non reconnu' }));
                    return;
                }
                
                const conn = await getDbConnection();
                const sender = clients.get(ws);
                const response = await messageHandlers[data.type](data.data, conn, sender);
                
                ws.send(JSON.stringify({
                    type: data.type + '_response',
                    data: response
                }));
                
                await conn.end();
            } catch (error) {
                console.error('Erreur de traitement du message:', error);
                ws.send(JSON.stringify({ error: 'Erreur de traitement du message' }));
            }
        });
        
        ws.on('close', () => {
            clients.delete(ws);
        });
        
    } catch (error) {
        console.error('Erreur de connexion à la base de données:', error);
        ws.send(JSON.stringify({ error: 'Erreur de connexion au serveur' }));
        ws.close();
    } finally {
        if (connection) {
            await connection.end();
        }
    }
});

server.listen(PORT, () => {
    console.log(`Serveur WebSocket démarré sur le port ${PORT}`);
});