document.addEventListener('DOMContentLoaded', function() {
    // Fonctions pour la galerie photo
    initPhotoGallery();
    
    // Fonctions pour le calendrier
    initCalendar();
    
    // Gestion des modals et popovers
    initModals();
    
    // Gestion des formulaires
    initForms();
    
    // Initialisation des menus déroulants
    initDropdowns();
});

/**
 * Initialise les fonctionnalités de la galerie photo
 */
function initPhotoGallery() {
    // Sélection multiple de photos
    const selectAllCheckbox = document.getElementById('select-all-photos');
    if (selectAllCheckbox) {
        selectAllCheckbox.addEventListener('change', function() {
            const checkboxes = document.querySelectorAll('.photo-checkbox');
            checkboxes.forEach(checkbox => {
                checkbox.checked = selectAllCheckbox.checked;
            });
            updateBulkActions();
        });
        
        document.querySelectorAll('.photo-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', updateBulkActions);
        });
    }
    
    // Gestion du glisser-déposer pour l'upload
    const dropzone = document.getElementById('dropzone');
    if (dropzone) {
        dropzone.addEventListener('dragover', function(e) {
            e.preventDefault();
            this.classList.add('border-indigo-500');
        });
        
        dropzone.addEventListener('dragleave', function() {
            this.classList.remove('border-indigo-500');
        });
        
        dropzone.addEventListener('drop', function(e) {
            e.preventDefault();
            this.classList.remove('border-indigo-500');
            
            if (e.dataTransfer.files.length > 0) {
                const fileInput = document.getElementById('file-input');
                if (fileInput) {
                    fileInput.files = e.dataTransfer.files;
                    handleFiles(e.dataTransfer.files);
                }
            }
        });
    }
    
    // Initialisation de la visionneuse de photos
    initPhotoViewer();
}

/**
 * Gère la sélection de fichiers pour l'upload
 */
function handleFiles(files) {
    const previewContainer = document.getElementById('preview-container');
    if (!previewContainer) return;
    
    previewContainer.innerHTML = '';
    previewContainer.classList.remove('hidden');
    
    for (let i = 0; i < files.length; i++) {
        const file = files[i];
        if (!file.type.startsWith('image/')) continue;
        
        const reader = new FileReader();
        reader.onload = function(e) {
            const preview = document.createElement('div');
            preview.className = 'flex items-start p-4 border border-gray-200 rounded-lg mb-4';
            preview.innerHTML = `
                <img src="${e.target.result}" alt="${file.name}" class="w-16 h-16 object-cover rounded mr-4">
                <div class="flex-1">
                    <input type="text" name="titles[]" placeholder="Title" value="${file.name.split('.')[0]}" class="w-full mb-2 border-gray-300 rounded-md shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    <textarea name="descriptions[]" placeholder="Description (optional)" rows="2" class="w-full border-gray-300 rounded-md shadow-sm focus:border-indigo-500 focus:ring-indigo-500"></textarea>
                </div>
            `;
            previewContainer.appendChild(preview);
        };
        reader.readAsDataURL(file);
    }
}

/**
 * Met à jour l'affichage des actions en masse pour les photos
 */
function updateBulkActions() {
    const checkedPhotos = document.querySelectorAll('.photo-checkbox:checked');
    const bulkActions = document.getElementById('bulk-actions');
    
    if (bulkActions) {
        if (checkedPhotos.length > 0) {
            bulkActions.classList.remove('hidden');
            const count = document.getElementById('selected-count');
            if (count) {
                count.textContent = checkedPhotos.length;
            }
        } else {
            bulkActions.classList.add('hidden');
        }
    }
}

/**
 * Initialise la visionneuse de photos
 */
function initPhotoViewer() {
    // Collection des images pour la navigation
    let photoIds = [];
    let currentPhotoIndex = -1;
    
    // Ouvre la visionneuse pour une photo spécifique
    window.openPhotoModal = function(photoId) {
        // Construire la liste des photos si ce n'est pas déjà fait
        if (photoIds.length === 0) {
            const photoElements = document.querySelectorAll('[data-photo-id]');
            photoIds = Array.from(photoElements).map(el => parseInt(el.dataset.photoId));
        }
        
        currentPhotoIndex = photoIds.indexOf(photoId);
        if (currentPhotoIndex === -1) return;
        
        // Charger les données de la photo
        loadPhotoData(photoId);
        
        // Afficher la modal
        document.getElementById('photo-modal').classList.remove('hidden');
        
        // Mettre à jour les boutons de navigation
        updatePhotoNavigation();
    };
    
    // Ferme la visionneuse de photos
    window.closePhotoModal = function() {
        document.getElementById('photo-modal').classList.add('hidden');
    };
    
    // Navigation vers la photo précédente
    window.prevPhoto = function() {
        if (currentPhotoIndex > 0) {
            currentPhotoIndex--;
            loadPhotoData(photoIds[currentPhotoIndex]);
            updatePhotoNavigation();
        }
    };
    
    // Navigation vers la photo suivante
    window.nextPhoto = function() {
        if (currentPhotoIndex < photoIds.length - 1) {
            currentPhotoIndex++;
            loadPhotoData(photoIds[currentPhotoIndex]);
            updatePhotoNavigation();
        }
    };
    
    // Met à jour les boutons de navigation
    function updatePhotoNavigation() {
        const prevButton = document.getElementById('prev-photo');
        const nextButton = document.getElementById('next-photo');
        
        if (prevButton) {
            prevButton.style.display = currentPhotoIndex > 0 ? 'block' : 'none';
        }
        
        if (nextButton) {
            nextButton.style.display = currentPhotoIndex < photoIds.length - 1 ? 'block' : 'none';
        }
    }
    
    // Charge les données d'une photo spécifique
    function loadPhotoData(photoId) {
        fetch(`?page=photos&action=get_photo&photo=${photoId}`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const photo = data.photo;
                    
                    // Mettre à jour le contenu de la modal
                    document.getElementById('photo-content').innerHTML = `<img src="${photo.url}" alt="${photo.title}" class="max-h-[70vh] max-w-full">`;
                    document.getElementById('photo-title').textContent = photo.title;
                    document.getElementById('photo-description').textContent = photo.description || '';
                    document.getElementById('photo-uploader').textContent = `Uploaded by ${photo.uploader}`;
                    document.getElementById('photo-date').textContent = photo.date;
                    
                    // Mettre à jour les liens d'action
                    const downloadLink = document.getElementById('download-link');
                    if (downloadLink) {
                        downloadLink.href = photo.url;
                        downloadLink.download = photo.title;
                    }
                    
                    const deleteLink = document.getElementById('delete-link');
                    if (deleteLink) {
                        deleteLink.href = `?page=photos&action=delete_photo&photo=${photoId}${photo.album_id ? '&album=' + photo.album_id : ''}`;
                    }
                }
            })
            .catch(error => {
                console.error('Error loading photo data:', error);
            });
    }
    
    // Gestion des touches clavier pour la navigation
    document.addEventListener('keydown', function(e) {
        const photoModal = document.getElementById('photo-modal');
        if (photoModal && !photoModal.classList.contains('hidden')) {
            if (e.key === 'Escape') {
                closePhotoModal();
            } else if (e.key === 'ArrowLeft') {
                prevPhoto();
            } else if (e.key === 'ArrowRight') {
                nextPhoto();
            }
        }
    });
}

/**
 * Initialise les fonctionnalités du calendrier
 */
function initCalendar() {
    // Gestion de la case à cocher "All day" dans le formulaire d'événement
    const allDayCheckbox = document.getElementById('all_day');
    const timeInput = document.getElementById('time');
    
    if (allDayCheckbox && timeInput) {
        allDayCheckbox.addEventListener('change', function() {
            timeInput.disabled = allDayCheckbox.checked;
            timeInput.classList.toggle('opacity-50', allDayCheckbox.checked);
        });
    }
}

/**
 * Initialise la gestion des modals
 */
function initModals() {
    // Fonction pour ouvrir une modal
    window.openModal = function(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.remove('hidden');
        }
    };
    
    // Fonction pour fermer une modal
    window.closeModal = function(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.add('hidden');
        }
    };
    
    // Fermer les modals quand on clique en dehors
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('modal-backdrop')) {
            const modal = e.target.closest('.modal-container');
            if (modal) {
                modal.classList.add('hidden');
            }
        }
    });
}

/**
 * Initialise la gestion des formulaires
 */
function initForms() {
    // Validation côté client des formulaires
    document.querySelectorAll('form[data-validate]').forEach(form => {
        form.addEventListener('submit', function(e) {
            let isValid = true;
            
            // Validation des champs requis
            form.querySelectorAll('[required]').forEach(field => {
                if (!field.value.trim()) {
                    isValid = false;
                    field.classList.add('border-red-500');
                    
                    // Afficher un message d'erreur
                    let errorMessage = field.dataset.errorMessage || 'This field is required';
                    let errorElement = document.createElement('p');
                    errorElement.className = 'text-red-500 text-xs mt-1';
                    errorElement.textContent = errorMessage;
                    
                    // Supprimer les anciens messages d'erreur
                    const existingError = field.parentNode.querySelector('.text-red-500');
                    if (existingError) {
                        existingError.remove();
                    }
                    
                    field.parentNode.appendChild(errorElement);
                } else {
                    field.classList.remove('border-red-500');
                    const existingError = field.parentNode.querySelector('.text-red-500');
                    if (existingError) {
                        existingError.remove();
                    }
                }
            });
            
            // Validation des emails
            form.querySelectorAll('[type="email"]').forEach(field => {
                if (field.value.trim() && !validateEmail(field.value)) {
                    isValid = false;
                    field.classList.add('border-red-500');
                    
                    // Afficher un message d'erreur
                    let errorMessage = 'Please enter a valid email address';
                    let errorElement = document.createElement('p');
                    errorElement.className = 'text-red-500 text-xs mt-1';
                    errorElement.textContent = errorMessage;
                    
                    // Supprimer les anciens messages d'erreur
                    const existingError = field.parentNode.querySelector('.text-red-500');
                    if (existingError) {
                        existingError.remove();
                    }
                    
                    field.parentNode.appendChild(errorElement);
                }
            });
            
            if (!isValid) {
                e.preventDefault();
            }
        });
    });
    
    // Fonction de validation d'email
    function validateEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }
}

/**
 * Initialise les menus déroulants
 */
function initDropdowns() {
    document.querySelectorAll('[data-dropdown-toggle]').forEach(button => {
        const targetId = button.dataset.dropdownToggle;
        const target = document.getElementById(targetId);
        
        if (target) {
            button.addEventListener('click', function(e) {
                e.stopPropagation();
                target.classList.toggle('hidden');
            });
        }
    });
    
    // Fermer les menus déroulants quand on clique ailleurs
    document.addEventListener('click', function() {
        document.querySelectorAll('[data-dropdown]').forEach(dropdown => {
            if (!dropdown.classList.contains('hidden')) {
                dropdown.classList.add('hidden');
            }
        });
    });
}