<?php
session_start();

$config = [
    'alltophotels_api_key' => 'YOUR_ALLTOPHOTELS_API_KEY',
    'db_host' => 'localhost',
    'db_name' => 'hizhat_bookings',
    'db_user' => 'root',
    'db_pass' => ''
];

$pdo = new PDO(
    "mysql:host={$config['db_host']};dbname={$config['db_name']};charset=utf8mb4",
    $config['db_user'],
    $config['db_pass'],
    [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $booking_type = $_POST['booking_type'] ?? '';
    
    switch ($booking_type) {
        case 'flight':
            $results = searchFlights($_POST);
            break;
        case 'hotel':
            $results = searchHotels($_POST);
            break;
        case 'train':
            $results = searchTrains($_POST);
            break;
        case 'combined':
            $results = searchCombined($_POST);
            break;
        default:
            header('Location: error.php?msg=invalid_booking_type');
            exit;
    }
    
    $_SESSION['search_results'] = $results;
    $_SESSION['search_params'] = $_POST;
    
    header('Location: results.php');
    exit;
}

function searchFlights($params) {
    $origin = $params['origin'] ?? '';
    $destination = $params['destination'] ?? '';
    $departure_date = $params['departure_date'] ?? '';
    $return_date = $params['return_date'] ?? '';
    
    if (empty($origin) || empty($destination) || empty($departure_date)) {
        return ['error' => 'Paramètres manquants'];
    }
    
    $departure_date_formatted = date('d/m/Y', strtotime($departure_date));
    $return_date_formatted = !empty($return_date) ? date('d/m/Y', strtotime($return_date)) : '';
    
    $api_url = "https://api.skyscanner.net/v3/flights/live/search/create";
    $request_data = [
        "query": [
            "market": "FR",
            "locale": "fr-FR",
            "currency": "EUR",
            "queryLegs": [
                [
                    "originPlaceId": ["entityId" => $origin],
                    "destinationPlaceId": ["entityId" => $destination],
                    "date": ["year" => (int)date('Y', strtotime($departure_date)), 
                             "month" => (int)date('m', strtotime($departure_date)), 
                             "day" => (int)date('d', strtotime($departure_date))]
                ]
            ],
            "adults": (int)($params['passengers'] ?? 1),
            "cabinClass": strtoupper($params['class'] ?? 'ECONOMY')
        ]
    ];
    
    if (!empty($return_date)) {
        $request_data["query"]["queryLegs"][] = [
            "originPlaceId": ["entityId" => $destination],
            "destinationPlaceId": ["entityId" => $origin],
            "date": ["year" => (int)date('Y', strtotime($return_date)), 
                     "month" => (int)date('m', strtotime($return_date)), 
                     "day" => (int)date('d', strtotime($return_date))]
        ];
    }
    
    $ch = curl_init($api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($request_data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'x-api-key: VOTRE_API_KEY_SKYSCANNER',
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code != 200) {
        saveSearchToDb($params, 'flight', 'error');
        return simulateFlightResults($params);
    }
    
    $data = json_decode($response, true);
    saveSearchToDb($params, 'flight', 'success');
    
    return processFlightResults($data);
}

function processFlightResults($data) {
    $results = [];
    
    foreach ($data['content']['results']['itineraries'] as $itinerary_id => $itinerary) {
        $leg_ids = $itinerary['legIds'];
        $price_info = $data['content']['results']['prices'][$itinerary['priceIds'][0]];
        
        $flight_info = [
            'id' => $itinerary_id,
            'price' => $price_info['amount'],
            'currency' => $price_info['currency'],
            'legs' => []
        ];
        
        foreach ($leg_ids as $leg_id) {
            $leg = $data['content']['results']['legs'][$leg_id];
            
            $flight_info['legs'][] = [
                'departure' => [
                    'airport' => $leg['origin']['name'],
                    'time' => $leg['departureDateTime']
                ],
                'arrival' => [
                    'airport' => $leg['destination']['name'],
                    'time' => $leg['arrivalDateTime']
                ],
                'duration' => $leg['durationInMinutes'],
                'stops' => count($leg['stopCount']),
                'carriers' => $leg['marketingCarrierIds']
            ];
        }
        
        $results[] = $flight_info;
    }
    
    return ['data' => $results];
}

function searchHotels($params) {
    $destination = $params['destination'] ?? '';
    $checkin_date = $params['checkin_date'] ?? '';
    $checkout_date = $params['checkout_date'] ?? '';
    $rooms = (int)($params['rooms'] ?? 1);
    $guests = (int)($params['guests'] ?? 1);
    
    if (empty($destination) || empty($checkin_date) || empty($checkout_date)) {
        return ['error' => 'Paramètres manquants'];
    }
    
    $api_params = [
        'destination' => $destination,
        'checkin' => $checkin_date,
        'checkout' => $checkout_date,
        'rooms' => $rooms,
        'guests' => $guests,
        'apikey' => $GLOBALS['config']['alltophotels_api_key'],
    ];
    
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => 'https://api.alltophotels.io/hotels/search',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($api_params)
    ]);
    
    $response = curl_exec($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);
    
    if ($http_code != 200) {
        saveSearchToDb($params, 'hotel', 'error');
        return simulateHotelResults($params);
    }
    
    $data = json_decode($response, true);
    saveSearchToDb($params, 'hotel', 'success');
    
    return $data;
}

function searchTrains($params) {
    $origin = $params['origin'] ?? '';
    $destination = $params['destination'] ?? '';
    $departure_date = $params['departure_date'] ?? '';
    $departure_time = $params['departure_time'] ?? '';
    
    if (empty($origin) || empty($destination) || empty($departure_date)) {
        return ['error' => 'Paramètres manquants'];
    }
    
    $api_url = "https://api.trainline.eu/api/v5_1/search";
    
    $request_data = [
        'search' => [
            'departure_date' => $departure_date,
            'origin_station_id' => getTrainStationId($origin),
            'destination_station_id' => getTrainStationId($destination),
            'passengers' => [
                ['id' => '1', 'age' => 30, 'cards' => []]
            ]
        ]
    ];
    
    $ch = curl_init($api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($request_data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Token token="YOUR_TRAINLINE_API_KEY"',
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code != 200) {
        saveSearchToDb($params, 'train', 'error');
        return simulateTrainResults($params);
    }
    
    $data = json_decode($response, true);
    saveSearchToDb($params, 'train', 'success');
    
    return processTrainResults($data);
}

function getTrainStationId($station_name) {
    $cache_file = 'cache/stations.json';
    
    if (file_exists($cache_file)) {
        $stations = json_decode(file_get_contents($cache_file), true);
        foreach ($stations as $station) {
            if (strtolower($station['name']) === strtolower($station_name)) {
                return $station['id'];
            }
        }
    }
    
    return substr(md5($station_name), 0, 8);
}

function processTrainResults($data) {
    $results = [];
    
    foreach ($data['trips'] as $trip) {
        $segments = [];
        foreach ($trip['segments'] as $segment_id) {
            $segment = $data['segments'][$segment_id];
            $segments[] = [
                'departure' => [
                    'station' => $data['stations'][$segment['departure_station_id']]['name'],
                    'time' => $segment['departure_date']
                ],
                'arrival' => [
                    'station' => $data['stations'][$segment['arrival_station_id']]['name'],
                    'time' => $segment['arrival_date']
                ],
                'train_number' => $segment['train_number'],
                'carrier' => $data['carriers'][$segment['carrier_id']]['name']
            ];
        }
        
        $results[] = [
            'id' => $trip['id'],
            'price' => $data['prices'][$trip['price_id']]['amount'],
            'currency' => $data['prices'][$trip['price_id']]['currency'],
            'segments' => $segments
        ];
    }
    
    return ['data' => $results];
}

function searchCombined($params) {
    $destination = $params['destination'] ?? '';
    $start_date = $params['start_date'] ?? '';
    $end_date = $params['end_date'] ?? '';
    $package_type = $params['package_type'] ?? 'flight_hotel';
    
    $flight_params = [
        'destination' => $destination,
        'departure_date' => $start_date,
        'return_date' => $end_date,
        'passengers' => $params['travelers'] ?? 1
    ];
    
    $hotel_params = [
        'destination' => $destination,
        'checkin_date' => $start_date,
        'checkout_date' => $end_date,
        'guests' => $params['travelers'] ?? 1
    ];
    
    $results = [
        'package_type' => $package_type,
        'destination' => $destination,
        'dates' => [
            'start' => $start_date,
            'end' => $end_date
        ]
    ];
    
    switch ($package_type) {
        case 'flight_hotel':
            $results['flights'] = searchFlights($flight_params)['data'] ?? [];
            $results['hotels'] = searchHotels($hotel_params)['data'] ?? [];
            break;
            
        case 'train_hotel':
            $train_params = [
                'destination' => $destination,
                'departure_date' => $start_date,
                'return_date' => $end_date,
                'passengers' => $params['travelers'] ?? 1
            ];
            $results['trains'] = searchTrains($train_params)['data'] ?? [];
            $results['hotels'] = searchHotels($hotel_params)['data'] ?? [];
            break;
            
        case 'flight_hotel_car':
            $results['flights'] = searchFlights($flight_params)['data'] ?? [];
            $results['hotels'] = searchHotels($hotel_params)['data'] ?? [];
            $results['cars'] = simulateCarRental($params);
            break;
    }
    
    saveSearchToDb($params, 'combined', 'success');
    return ['data' => $results];
}

function simulateFlightResults($params) {
    $airlines = ['Air France', 'Lufthansa', 'British Airways', 'KLM', 'Ryanair', 'EasyJet'];
    $results = [];
    
    $num_flights = rand(5, 10);
    
    for ($i = 0; $i < $num_flights; $i++) {
        $departure_hour = rand(6, 22);
        $flight_duration = rand(90, 360);
        $arrival_hour = ($departure_hour + floor($flight_duration / 60)) % 24;
        
        $results[] = [
            'id' => 'f' . rand(10000, 99999),
            'airline' => $airlines[array_rand($airlines)],
            'flight_number' => 'FL' . rand(1000, 9999),
            'departure' => [
                'airport' => $params['origin'],
                'time' => sprintf('%02d:%02d', $departure_hour, rand(0, 59))
            ],
            'arrival' => [
                'airport' => $params['destination'],
                'time' => sprintf('%02d:%02d', $arrival_hour, rand(0, 59))
            ],
            'duration' => floor($flight_duration / 60) . 'h ' . ($flight_duration % 60) . 'm',
            'price' => rand(99, 599) . '.00',
            'class' => $params['class'],
            'stops' => rand(0, 2)
        ];
    }
    
    usort($results, function($a, $b) {
        return (float)$a['price'] <=> (float)$b['price'];
    });
    
    return ['data' => $results];
}

function simulateHotelResults($params) {
    $hotel_names = ['Grand Hotel', 'Seaside Resort', 'City Center Hotel', 'Mountain View', 'Ocean Breeze', 'Urban Stay'];
    $results = [];
    
    $num_hotels = rand(5, 15);
    
    for ($i = 0; $i < $num_hotels; $i++) {
        $stars = rand(2, 5);
        $base_price = 50 + ($stars * 30);
        
        $checkin = new DateTime($params['checkin_date']);
        $checkout = new DateTime($params['checkout_date']);
        $nights = $checkout->diff($checkin)->days;
        
        $results[] = [
            'id' => 'h' . rand(10000, 99999),
            'name' => $hotel_names[array_rand($hotel_names)] . ' ' . $params['destination'],
            'address' => rand(1, 150) . ' ' . ['Main St', 'Beach Rd', 'Central Ave', 'Park Blvd'][array_rand(['Main St', 'Beach Rd', 'Central Ave', 'Park Blvd'])],
            'stars' => $stars,
            'rating' => round(rand(60, 95) / 10, 1),
            'price_per_night' => $base_price + rand(0, 100),
            'total_price' => ($base_price + rand(0, 100)) * $nights,
            'thumbnail' => 'https://placehold.co/600x400',
            'amenities' => array_slice(['WiFi', 'Pool', 'Spa', 'Restaurant', 'Gym', 'Bar', 'Parking'], 0, rand(3, 7)),
            'available_rooms' => rand(1, 10)
        ];
    }
    
    usort($results, function($a, $b) {
        return $a['price_per_night'] <=> $b['price_per_night'];
    });
    
    return $results;
}

function simulateTrainResults($params) {
    $train_companies = ['Eurostar', 'SNCF', 'Deutsche Bahn', 'Trenitalia', 'Renfe'];
    $results = [];
    
    $num_trains = rand(3, 8);
    
    $departure_times = [
        'morning' => ['06:30', '07:15', '08:45', '09:30', '10:45', '11:20'],
        'afternoon' => ['12:10', '13:45', '14:30', '15:15', '16:40', '17:20'],
        'evening' => ['18:15', '19:30', '20:45', '21:10'],
        'night' => ['22:30', '23:15', '00:20', '05:45']
    ];
    
    $departure_period = $params['departure_time'] ?? array_rand($departure_times);
    $times = $departure_times[$departure_period];
    
    for ($i = 0; $i < $num_trains; $i++) {
        $departure_time = $times[array_rand($times)];
        $duration_minutes = rand(45, 300);
        
        $departure_time_obj = DateTime::createFromFormat('H:i', $departure_time);
        $arrival_time_obj = clone $departure_time_obj;
        $arrival_time_obj->add(new DateInterval('PT' . $duration_minutes . 'M'));
        
        $results[] = [
            'id' => 't' . rand(10000, 99999),
            'company' => $train_companies[array_rand($train_companies)],
            'train_number' => 'TN' . rand(1000, 9999),
            'departure' => [
                'station' => $params['origin'],
                'time' => $departure_time
            ],
            'arrival' => [
                'station' => $params['destination'],
                'time' => $arrival_time_obj->format('H:i')
            ],
            'duration' => floor($duration_minutes / 60) . 'h ' . ($duration_minutes % 60) . 'm',
            'price' => rand(29, 199) . '.00',
            'class' => $params['class'],
            'stops' => rand(0, 3)
        ];
    }
    
    usort($results, function($a, $b) {
        return (float)$a['price'] <=> (float)$b['price'];
    });
    
    return ['data' => $results];
}

function simulateCarRental($params) {
    $car_types = ['Economy', 'Compact', 'Midsize', 'SUV', 'Luxury'];
    $car_brands = ['Renault', 'Peugeot', 'Volkswagen', 'Toyota', 'Ford', 'BMW', 'Mercedes'];
    $rental_companies = ['Avis', 'Europcar', 'Hertz', 'Sixt', 'Enterprise'];
    
    $results = [];
    
    $num_cars = rand(3, 8);
    
    for ($i = 0; $i < $num_cars; $i++) {
        $car_type = $car_types[array_rand($car_types)];
        $base_price = ['Economy' => 30, 'Compact' => 40, 'Midsize' => 55, 'SUV' => 70, 'Luxury' => 100][$car_type];
        
        $start_date = new DateTime($params['start_date']);
        $end_date = new DateTime($params['end_date']);
        $days = $end_date->diff($start_date)->days;
        
        $results[] = [
            'id' => 'c' . rand(10000, 99999),
            'company' => $rental_companies[array_rand($rental_companies)],
            'car' => $car_brands[array_rand($car_brands)] . ' ' . ['Clio', 'Golf', 'Corolla', 'Focus', '3 Series', 'C-Class'][array_rand(['Clio', 'Golf', 'Corolla', 'Focus', '3 Series', 'C-Class'])],
            'type' => $car_type,
            'price_per_day' => $base_price + rand(0, 15),
            'total_price' => ($base_price + rand(0, 15)) * $days,
            'pickup' => $params['destination'],
            'dropoff' => $params['destination'],
            'available' => rand(1, 5),
            'features' => array_slice(['A/C', 'Automatic', 'GPS', 'Bluetooth', '5 Doors', 'Diesel'], 0, rand(3, 6))
        ];
    }
    
    usort($results, function($a, $b) {
        return $a['price_per_day'] <=> $b['price_per_day'];
    });
    
    return $results;
}

function saveSearchToDb($params, $type, $status) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        INSERT INTO searches (
            search_type, 
            search_params, 
            ip_address, 
            user_agent, 
            status, 
            created_at
        ) VALUES (?, ?, ?, ?, ?, NOW())
    ");
    
    $stmt->execute([
        $type,
        json_encode($params),
        $_SERVER['REMOTE_ADDR'] ?? '',
        $_SERVER['HTTP_USER_AGENT'] ?? '',
        $status
    ]);
    
    return $pdo->lastInsertId();
}
?>