<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);

// Vérifier le niveau minimal requis
if ($user['level'] < 10) {
    header('Location: index.php?error=' . urlencode('Tu dois être au moins niveau 10 pour accéder aux unions.'));
    exit;
}

$error = null;
$success = null;

// Vérifier si l'utilisateur est déjà dans une union
$existing_membership = db_get_one(
    "SELECT um.* FROM union_members um WHERE um.user_id = ?",
    [$_SESSION['user_id']]
);

if ($existing_membership) {
    header('Location: unions.php?error=' . urlencode('Tu es déjà membre d\'une union.'));
    exit;
}

// Récupérer l'ID de l'union
$union_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($union_id <= 0) {
    header('Location: unions.php?error=' . urlencode('Union invalide.'));
    exit;
}

// Récupérer les informations de l'union
$union = db_get_one(
    "SELECT u.*, c.name as country_name
     FROM unions u
     JOIN countries c ON u.country_id = c.id
     WHERE u.id = ?",
    [$union_id]
);

if (!$union) {
    header('Location: unions.php?error=' . urlencode('Union non trouvée.'));
    exit;
}

// Vérifier que l'union est dans le même pays que l'utilisateur
if ($union['country_id'] != $user['country_id']) {
    header('Location: unions.php?error=' . urlencode('Tu ne peux rejoindre que des unions de ton pays.'));
    exit;
}

// Traitement de la demande d'adhésion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'join_union') {
    // Vérifier si une invitation existe
    $invitation = db_get_one(
        "SELECT * FROM union_invitations 
         WHERE union_id = ? AND user_id = ? AND status = 'pending'",
        [$union_id, $_SESSION['user_id']]
    );
    
    // Permettre de rejoindre même sans invitation (mais le leader devra approuver)
    $member_data = [
        'union_id' => $union_id,
        'user_id' => $_SESSION['user_id'],
        'role' => 'member',
        'joined_at' => date('Y-m-d H:i:s')
    ];
    
    if ($invitation) {
        // Si invitation, accepter directement
        db_update(
            'union_invitations',
            ['status' => 'accepted'],
            'id = ?',
            [$invitation['id']]
        );
        
        db_insert('union_members', $member_data);
        
        $success = "Tu as rejoint l'union " . $union['name'] . "!";
    } else {
        // Sinon, créer une demande d'adhésion
        $request_data = [
            'union_id' => $union_id,
            'user_id' => $_SESSION['user_id'],
            'status' => 'pending',
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        db_insert('union_join_requests', $request_data);
        
        $success = "Demande d'adhésion envoyée à " . $union['name'] . ". Attends l'approbation du leader.";
    }
    
    header('Location: unions.php' . ($success ? '?success=' . urlencode($success) : ''));
    exit;
}

// Récupérer les membres de l'union
$union_members = db_get_all(
    "SELECT um.role, u.username, u.level
     FROM union_members um
     JOIN users u ON um.user_id = u.id
     WHERE um.union_id = ?
     ORDER BY 
        CASE um.role 
            WHEN 'leader' THEN 1 
            WHEN 'officer' THEN 2 
            ELSE 3 
        END,
        u.username
     LIMIT 10",
    [$union_id]
);

// Vérifier si une invitation existe
$invitation = db_get_one(
    "SELECT * FROM union_invitations 
     WHERE union_id = ? AND user_id = ? AND status = 'pending'",
    [$union_id, $_SESSION['user_id']]
);

// Vérifier si une demande d'adhésion existe déjà
$join_request = db_get_one(
    "SELECT * FROM union_join_requests 
     WHERE union_id = ? AND user_id = ? AND status = 'pending'",
    [$union_id, $_SESSION['user_id']]
);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rejoindre une union | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        .card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300">Système</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300">Missions</a></li>
                    <li><a href="unions.php" class="hover:text-blue-300 font-semibold">Unions</a></li>
                    <li><a href="profile.php" class="hover:text-blue-300">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Rejoindre une union</h2>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div class="lg:col-span-2">
                <div class="card p-6 mb-6">
                    <div class="flex justify-between items-start">
                        <div>
                            <h3 class="text-xl font-semibold"><?php echo htmlspecialchars($union['name']); ?></h3>
                            <p class="text-sm text-gray-500">
                                Niveau <?php echo $union['level']; ?> • 
                                <?php echo htmlspecialchars($union['country_name']); ?> •
                                Créée le <?php echo format_datetime($union['created_at']); ?>
                            </p>
                        </div>
                    </div>
                    
                    <?php if ($union['description']): ?>
                        <div class="mt-4 p-4 bg-gray-50 rounded-lg">
                            <p><?php echo nl2br(htmlspecialchars($union['description'])); ?></p>
                        </div>
                    <?php endif; ?>
                    
                    <div class="mt-6">
                        <?php if ($invitation): ?>
                            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                                <p>Tu as reçu une invitation à rejoindre cette union!</p>
                            </div>
                        <?php elseif ($join_request): ?>
                            <div class="bg-yellow-100 border border-yellow-400 text-yellow-700 px-4 py-3 rounded mb-4">
                                <p>Ta demande d'adhésion est en attente d'approbation.</p>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!$join_request): ?>
                            <form method="POST" action="join_union.php?id=<?php echo $union_id; ?>">
                                <input type="hidden" name="action" value="join_union">
                                
                                <?php if ($invitation): ?>
                                    <button type="submit" class="bg-green-600 hover:bg-green-700 text-white py-2 px-4 rounded transition-colors">
                                        Accepter l'invitation
                                    </button>
                                <?php else: ?>
                                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded transition-colors">
                                        Demander à rejoindre
                                    </button>
                                <?php endif; ?>
                                
                                <a href="unions.php" class="inline-block ml-2 text-gray-600 hover:text-gray-800">
                                    Retour aux unions
                                </a>
                            </form>
                        <?php else: ?>
                            <a href="unions.php" class="inline-block text-blue-600 hover:text-blue-800">
                                Retour aux unions
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="card p-6">
                    <h3 class="text-lg font-semibold mb-4">Membres</h3>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <?php foreach ($union_members as $member): ?>
                            <div class="border rounded-lg p-3">
                                <div class="flex justify-between items-center">
                                    <div>
                                        <h4 class="font-medium"><?php echo htmlspecialchars($member['username']); ?></h4>
                                        <p class="text-sm text-gray-500">Niveau <?php echo $member['level']; ?></p>
                                    </div>
                                    <div>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                            <?php 
                                                echo $member['role'] === 'leader' ? 'bg-purple-100 text-purple-800' : 
                                                    ($member['role'] === 'officer' ? 'bg-yellow-100 text-yellow-800' : 
                                                    'bg-gray-100 text-gray-800');
                                            ?>">
                                            <?php echo ucfirst($member['role']); ?>
                                        </span>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            
            <div>
                <div class="card p-6">
                    <h3 class="text-lg font-semibold mb-4">Avantages de l'union</h3>
                    
                    <div class="space-y-3">
                        <div>
                            <p class="font-medium">Communications sécurisées</p>
                            <p class="text-sm text-gray-600">Échange d'informations cryptées entre les membres</p>
                        </div>
                        <div>
                            <p class="font-medium">Missions conjointes</p>
                            <p class="text-sm text-gray-600">Accès à des missions spéciales réservées aux unions</p>
                        </div>
                        <div>
                            <p class="font-medium">Bonus de défense</p>
                            <p class="text-sm text-gray-600">Protection accrue du système national</p>
                        </div>
                        <div>
                            <p class="font-medium">Base d'opérations</p>
                            <p class="text-sm text-gray-600">Lieu sécurisé pour planifier des stratégies</p>
                        </div>
                        <div>
                            <p class="font-medium">Partage de ressources</p>
                            <p class="text-sm text-gray-600">Possibilité de partager des objets et des devises</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>
</body>
</html>