document.addEventListener('DOMContentLoaded', () => {
    const app = {
        map: null,
        mapboxKey: '',
        mapboxDirections: null,
        markers: {},
        userMarker: null,
        tempMarker: null,
        selectedLightId: null,
        userWatchId: null,
        refreshTimerId: null,
        rafId: null,
        lastTimestamp: 0,
        currentRoute: null,
        isNavigating: false,
        routeLineLayer: null,
        trafficLayer: null,
        currentSpeedLimit: null,
        speedAlertPlayed: false,
        speedAudio: new Audio('speed-alert.mp3'),
        destinationWeather: null,
        settings: {
            darkMode: false,
            notifications: true,
            autoRefresh: true,
            predictions: true,
            autoMeasure: true,
            defaultZoom: 15,
            mapStyle: 'satellite',
            voiceAlerts: true,
            trafficView: true,
            weatherDisplay: true
        },
        measure: {
            timer: 0,
            startTime: 0,
            rafId: null,
            mode: null,
            lightId: null
        },
        navigation: {
            startPosition: null,
            endPosition: null,
            coordinates: [],
            lightsOnRoute: [],
            travelMode: 'driving',
            distance: null,
            duration: null,
            startTime: null,
            estimatedArrival: null,
            currentStep: 0,
            lastStop: null,
            stopDuration: 0,
            currentSpeed: 0,
            lastPosition: null,
            steps: [],
            speedLimits: [],
            currentSpeedLimit: null,
            nextTurn: null,
            trafficDelay: 0
        },
        state: {
            selectingLocation: false,
            currentSlide: 1,
            routePanelOpen: false,
            navigationActive: false,
            lightsPanelOpen: false,
            accountPanelOpen: false
        },

        init() {
            Promise.all([
                this.fetchMapboxKey(),
                this.registerServiceWorker()
            ])
            .then(() => {
                this.loadSettings();
                this.initMap();
                this.bindEvents();
                this.setupDraggableModals();
                this.checkFirstVisit();
                this.checkLoggedInUser();
                
                setTimeout(() => {
                    const loadingScreen = document.getElementById('loadingScreen');
                    if (loadingScreen) {
                        loadingScreen.style.opacity = '0';
                        setTimeout(() => {
                            loadingScreen.style.display = 'none';
                        }, 500);
                    }
                }, 800);
            })
            .catch(error => {
                console.error('Failed to initialize app:', error);
                this.showNotification('Failed to initialize app. Please refresh.', 'error');
                const loadingScreen = document.getElementById('loadingScreen');
                if (loadingScreen) {
                    loadingScreen.innerHTML = '<div class="text-center"><p class="text-xl text-red-500 mb-2">Failed to load app</p><p>Please refresh the page</p></div>';
                }
            });
        },
        
        fetchMapboxKey() {
            return fetch('get-mapbox-key.php')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        this.mapboxKey = data.key;
                    } else {
                        throw new Error('Failed to get Mapbox API key');
                    }
                });
        },

        async registerServiceWorker() {
            if ('serviceWorker' in navigator) {
                try {
                    await navigator.serviceWorker.register('service-worker.js');
                    console.log('Service Worker registered');
                } catch (error) {
                    console.error('Service Worker registration failed:', error);
                }
            }
        },

        loadSettings() {
            try {
                const saved = localStorage.getItem('trafficLightSettings');
                if (saved) {
                    const parsed = JSON.parse(saved);
                    this.settings = { ...this.settings, ...parsed };
                } else {
                    this.settings.mapStyle = 'satellite';
                }
                this.applySettings();
            } catch (e) {
                console.error('Error loading settings');
                this.settings.mapStyle = 'satellite';
                this.applySettings();
            }
        },

        applySettings() {
            if (document.getElementById('darkModeToggle')) document.getElementById('darkModeToggle').checked = this.settings.darkMode;
            if (document.getElementById('notificationsToggle')) document.getElementById('notificationsToggle').checked = this.settings.notifications;
            if (document.getElementById('autoRefreshToggle')) document.getElementById('autoRefreshToggle').checked = this.settings.autoRefresh;
            if (document.getElementById('predictionToggle')) document.getElementById('predictionToggle').checked = this.settings.predictions;
            if (document.getElementById('autoMeasureToggle')) document.getElementById('autoMeasureToggle').checked = this.settings.autoMeasure;
            if (document.getElementById('defaultZoom')) document.getElementById('defaultZoom').value = this.settings.defaultZoom;
            if (document.getElementById('voiceAlertsToggle')) document.getElementById('voiceAlertsToggle').checked = this.settings.voiceAlerts;
            if (document.getElementById('trafficViewToggle')) document.getElementById('trafficViewToggle').checked = this.settings.trafficView;
            if (document.getElementById('weatherDisplayToggle')) document.getElementById('weatherDisplayToggle').checked = this.settings.weatherDisplay;
            
            if (this.settings.darkMode) {
                document.documentElement.classList.add('dark');
                document.body.classList.add('dark-mode');
            }
        },

        saveSettings() {
            if (document.getElementById('darkModeToggle')) this.settings.darkMode = document.getElementById('darkModeToggle').checked;
            if (document.getElementById('notificationsToggle')) this.settings.notifications = document.getElementById('notificationsToggle').checked;
            if (document.getElementById('autoRefreshToggle')) this.settings.autoRefresh = document.getElementById('autoRefreshToggle').checked;
            if (document.getElementById('predictionToggle')) this.settings.predictions = document.getElementById('predictionToggle').checked;
            if (document.getElementById('autoMeasureToggle')) this.settings.autoMeasure = document.getElementById('autoMeasureToggle').checked;
            if (document.getElementById('defaultZoom')) this.settings.defaultZoom = document.getElementById('defaultZoom').value;
            if (document.getElementById('voiceAlertsToggle')) this.settings.voiceAlerts = document.getElementById('voiceAlertsToggle').checked;
            if (document.getElementById('trafficViewToggle')) this.settings.trafficView = document.getElementById('trafficViewToggle').checked;
            if (document.getElementById('weatherDisplayToggle')) this.settings.weatherDisplay = document.getElementById('weatherDisplayToggle').checked;
            
            try {
                localStorage.setItem('trafficLightSettings', JSON.stringify(this.settings));
            } catch (e) {
                console.error('Error saving settings');
            }
        },

        checkFirstVisit() {
            if (!localStorage.getItem('trafficLightAppVisited')) {
                const welcomeModal = document.getElementById('welcomeModal');
                if (welcomeModal) {
                    welcomeModal.style.display = 'flex';
                }
                localStorage.setItem('trafficLightAppVisited', 'true');
            } else {
                const welcomeModal = document.getElementById('welcomeModal');
                if (welcomeModal) {
                    welcomeModal.style.display = 'none';
                }
            }
            
            const modals = document.querySelectorAll('.modal');
            modals.forEach(modal => {
                if (modal.id !== 'welcomeModal') modal.style.display = 'none';
            });
        },
        
        checkLoggedInUser() {
            const userData = this.getCookie('user_data');
            
            if (userData) {
                try {
                    const user = JSON.parse(userData);
                    this.updateUIForLoggedInUser(user);
                } catch (e) {
                    console.error('Error parsing user data from cookie');
                }
            }
        },
        
        getCookie(name) {
            const value = `; ${document.cookie}`;
            const parts = value.split(`; ${name}=`);
            if (parts.length === 2) return parts.pop().split(';').shift();
            return null;
        },

        initMap() {
            if (!this.mapboxKey) {
                console.error('MapBox API key not available');
                return;
            }
            
            mapboxgl.accessToken = this.mapboxKey;
            
            this.map = new mapboxgl.Map({
                container: 'map',
                style: this.getMapStyle(),
                center: [0, 0],
                zoom: parseInt(this.settings.defaultZoom),
                pitch: 60,
                bearing: 0
            });
            
            this.map.addControl(new mapboxgl.NavigationControl({
                showCompass: true,
                showZoom: true,
                visualizePitch: true
            }), 'bottom-right');
            
            this.map.on('load', () => {
                if (this.settings.trafficView) {
                    this.addTrafficLayer();
                }
                this.map.on('click', this.handleMapClick.bind(this));
                this.loadTrafficLights();
                this.startGeolocation();
                
                this.mapboxDirections = new MapboxDirections({
                    accessToken: mapboxgl.accessToken,
                    unit: 'metric',
                    profile: 'mapbox/driving',
                    alternatives: false,
                    controls: false,
                    interactive: false
                });
            });
        },

        getMapStyle() {
            switch (this.settings.mapStyle) {
                case 'streets':
                    return 'mapbox://styles/mapbox/streets-v12';
                case 'navigation':
                    return 'mapbox://styles/mapbox/navigation-day-v1';
                case 'dark':
                    return 'mapbox://styles/mapbox/dark-v11';
                case 'satellite':
                default:
                    return 'mapbox://styles/mapbox/satellite-streets-v12';
            }
        },

        addTrafficLayer() {
            this.map.addLayer({
                id: 'traffic',
                type: 'line',
                source: {
                    type: 'vector',
                    url: 'mapbox://mapbox.mapbox-traffic-v1'
                },
                'source-layer': 'traffic',
                paint: {
                    'line-width': 3,
                    'line-color': [
                        'match',
                        ['get', 'congestion'],
                        'low', '#10b981',
                        'moderate', '#f59e0b',
                        'heavy', '#ef4444',
                        'severe', '#991b1b',
                        '#6b7280'
                    ],
                    'line-opacity': 0.7
                }
            });
            this.trafficLayer = 'traffic';
        },

        setupDraggableModals() {
            const draggableHandles = document.querySelectorAll('.draggable-handle');
            draggableHandles.forEach(handle => {
                const modal = handle.closest('.modal-body');
                if (!modal) return;
                
                let startY, currentY, initialY;
                let isDragging = false;
                
                handle.addEventListener('touchstart', e => {
                    startY = e.touches[0].clientY;
                    initialY = 0;
                    isDragging = true;
                    modal.style.transition = 'none';
                }, { passive: true });
                
                handle.addEventListener('touchmove', e => {
                    if (!isDragging) return;
                    currentY = e.touches[0].clientY;
                    const diffY = currentY - startY;
                    if (diffY < 0) return;
                    
                    modal.style.transform = `translateY(${diffY}px)`;
                }, { passive: true });
                
                handle.addEventListener('touchend', e => {
                    modal.style.transition = 'transform 0.3s ease';
                    if (!isDragging) return;
                    
                    const diffY = currentY - startY;
                    if (diffY > 100) {
                        modal.style.transform = 'translateY(100%)';
                        const modalContainer = modal.closest('.modal');
                        setTimeout(() => {
                            if (modalContainer) {
                                modalContainer.style.display = 'none';
                                modal.style.transform = 'translateY(0)';
                            }
                        }, 300);
                    } else {
                        modal.style.transform = 'translateY(0)';
                    }
                    
                    isDragging = false;
                }, { passive: true });
            });
            
            const overlays = document.querySelectorAll('.modal-overlay');
            overlays.forEach(overlay => {
                overlay.addEventListener('click', () => {
                    const modal = overlay.closest('.modal');
                    if (modal) this.closeModal(modal);
                });
            });
        },

        bindEvents() {
            const dom = {
                menuToggle: document.getElementById('menuToggle'),
                headerLocate: document.getElementById('headerLocateBtn'),
                headerAdd: document.getElementById('headerAddBtn'),
                route: document.getElementById('routeBtn'),
                refresh: document.getElementById('refreshBtn'),
                settings: document.getElementById('settingsBtn'),
                nearby: document.getElementById('nearbyBtn'),
                routes: document.getElementById('routesBtn'),
                toggleTraffic: document.getElementById('toggleTraffic'),
                closePopup: document.getElementById('closePopup'),
                popupNavigate: document.getElementById('popupNavigate'),
                popupMeasure: document.getElementById('popupMeasure'),
                closeRoutePanel: document.getElementById('closeRoutePanel'),
                calculateRoute: document.getElementById('calculateRoute'),
                startNavigation: document.getElementById('startNavigation'),
                routeTypeCar: document.getElementById('routeTypeCar'),
                routeTypeBike: document.getElementById('routeTypeBike'),
                routeTypeWalk: document.getElementById('routeTypeWalk'),
                endNavigation: document.getElementById('endNavigation'),
                resetApp: document.getElementById('resetAppBtn'),
                measureRed: document.getElementById('measureRedBtn'),
                measureGreen: document.getElementById('measureGreenBtn'),
                startTimer: document.getElementById('startTimer'),
                stopTimer: document.getElementById('stopTimer'),
                saveTimer: document.getElementById('saveTimer'),
                skipWelcome: document.getElementById('skipWelcome'),
                nextSlide: document.getElementById('nextSlide'),
                searchInput: document.getElementById('searchInput'),
                startLocation: document.getElementById('startLocation'),
                endLocation: document.getElementById('endLocation'),
                loginForm: document.getElementById('loginForm'),
                registerForm: document.getElementById('registerForm'),
                showLogin: document.getElementById('showLoginForm'),
                showRegister: document.getElementById('showRegisterForm'),
                authTitle: document.getElementById('authTitle'),
                login: document.getElementById('loginBtn'),
                closeButtons: document.querySelectorAll('.close-modal'),
                darkModeToggle: document.getElementById('darkModeToggle'),
                autoRefreshToggle: document.getElementById('autoRefreshToggle'),
                notificationsToggle: document.getElementById('notificationsToggle'),
                predictionToggle: document.getElementById('predictionToggle'),
                autoMeasureToggle: document.getElementById('autoMeasureToggle'),
                defaultZoom: document.getElementById('defaultZoom'),
                voiceAlertsToggle: document.getElementById('voiceAlertsToggle'),
                trafficViewToggle: document.getElementById('trafficViewToggle'),
                weatherDisplayToggle: document.getElementById('weatherDisplayToggle')
            };
            
            if (dom.menuToggle) dom.menuToggle.addEventListener('click', this.toggleSidebar.bind(this));
            if (dom.headerLocate) dom.headerLocate.addEventListener('click', this.locateUser.bind(this));
            if (dom.headerAdd) dom.headerAdd.addEventListener('click', () => this.openModal('addModal'));
            if (dom.settings) dom.settings.addEventListener('click', () => this.openModal('settingsModal'));
            if (dom.refresh) dom.refresh.addEventListener('click', this.refreshLights.bind(this));
            if (dom.nearby) dom.nearby.addEventListener('click', this.findNearbyLights.bind(this));
            if (dom.route) dom.route.addEventListener('click', this.toggleRoutePanel.bind(this));
            if (dom.routes) dom.routes.addEventListener('click', this.showSavedRoutes.bind(this));
            if (dom.toggleTraffic) dom.toggleTraffic.addEventListener('click', this.toggleTrafficLayer.bind(this));
            
            if (dom.searchInput) {
                dom.searchInput.addEventListener('input', this.handleSearch.bind(this));
            }
            
            if (dom.closePopup) dom.closePopup.addEventListener('click', () => {
                const lightPopup = document.getElementById('lightPopup');
                if (lightPopup) lightPopup.classList.add('hidden');
            });
            
            if (dom.popupNavigate) dom.popupNavigate.addEventListener('click', this.navigateToSelectedLight.bind(this));
            if (dom.popupMeasure) dom.popupMeasure.addEventListener('click', () => {
                const lightPopup = document.getElementById('lightPopup');
                if (lightPopup) lightPopup.classList.add('hidden');
                this.openMeasureModal(this.selectedLightId);
            });
            
            if (dom.closeButtons) {
                dom.closeButtons.forEach(btn => {
                    btn.addEventListener('click', () => {
                        const modal = btn.closest('.modal');
                        if (modal) this.closeModal(modal);
                    });
                });
            }
            
            if (dom.closeRoutePanel) {
                dom.closeRoutePanel.addEventListener('click', () => {
                    const routePlannerPanel = document.getElementById('routePlannerPanel');
                    if (routePlannerPanel) {
                        routePlannerPanel.classList.add('hidden');
                        this.state.routePanelOpen = false;
                    }
                });
            }
            
            if (dom.darkModeToggle) {
                dom.darkModeToggle.addEventListener('change', this.toggleDarkMode.bind(this));
            }
            if (dom.autoRefreshToggle) {
                dom.autoRefreshToggle.addEventListener('change', this.toggleAutoRefresh.bind(this));
            }
            if (dom.notificationsToggle) {
                dom.notificationsToggle.addEventListener('change', this.saveSettings.bind(this));
            }
            if (dom.predictionToggle) {
                dom.predictionToggle.addEventListener('change', () => {
                    this.saveSettings();
                    this.updateAllLightStatus();
                });
            }
            if (dom.autoMeasureToggle) {
                dom.autoMeasureToggle.addEventListener('change', this.saveSettings.bind(this));
            }
            if (dom.defaultZoom) {
                dom.defaultZoom.addEventListener('change', this.saveSettings.bind(this));
            }
            if (dom.voiceAlertsToggle) {
                dom.voiceAlertsToggle.addEventListener('change', this.saveSettings.bind(this));
            }
            if (dom.trafficViewToggle) {
                dom.trafficViewToggle.addEventListener('change', this.toggleTrafficLayer.bind(this));
            }
            if (dom.weatherDisplayToggle) {
                dom.weatherDisplayToggle.addEventListener('change', this.saveSettings.bind(this));
            }
            
            if (dom.routeTypeCar) dom.routeTypeCar.addEventListener('click', () => this.setTravelMode('driving'));
            if (dom.routeTypeBike) dom.routeTypeBike.addEventListener('click', () => this.setTravelMode('cycling'));
            if (dom.routeTypeWalk) dom.routeTypeWalk.addEventListener('click', () => this.setTravelMode('walking'));
            if (dom.calculateRoute) dom.calculateRoute.addEventListener('click', this.calculateRoute.bind(this));
            if (dom.startNavigation) dom.startNavigation.addEventListener('click', this.startRouteNavigation.bind(this));
            if (dom.endNavigation) dom.endNavigation.addEventListener('click', this.endNavigation.bind(this));
            
            if (dom.resetApp) dom.resetApp.addEventListener('click', this.resetAppData.bind(this));
            if (dom.measureRed) dom.measureRed.addEventListener('click', () => this.startMeasure('red'));
            if (dom.measureGreen) dom.measureGreen.addEventListener('click', () => this.startMeasure('green'));
            if (dom.startTimer) dom.startTimer.addEventListener('click', this.startMeasureTimer.bind(this));
            if (dom.stopTimer) dom.stopTimer.addEventListener('click', this.stopMeasureTimer.bind(this));
            if (dom.saveTimer) dom.saveTimer.addEventListener('click', this.saveMeasureTimer.bind(this));
            
            if (dom.showLogin) dom.showLogin.addEventListener('click', this.showLoginForm.bind(this));
            if (dom.showRegister) dom.showRegister.addEventListener('click', this.showRegisterForm.bind(this));
            if (dom.login) dom.login.addEventListener('click', () => this.openModal('loginModal'));
            if (dom.loginForm) dom.loginForm.addEventListener('submit', this.handleLogin.bind(this));
            if (dom.registerForm) dom.registerForm.addEventListener('submit', this.handleRegister.bind(this));
            
            if (dom.skipWelcome) dom.skipWelcome.addEventListener('click', () => this.closeModal('welcomeModal'));
            if (dom.nextSlide) dom.nextSlide.addEventListener('click', this.nextOnboardingSlide.bind(this));
            
            const welcomeDots = document.querySelectorAll('.onboarding-dot');
            if (welcomeDots) {
                welcomeDots.forEach(dot => {
                    dot.addEventListener('click', () => {
                        const slide = parseInt(dot.getAttribute('data-dot'));
                        this.goToSlide(slide);
                    });
                });
            }
            
            const addLightForm = document.getElementById('addLightForm');
            if (addLightForm) addLightForm.addEventListener('submit', this.handleAddLight.bind(this));
            
            if (dom.startLocation) this.setupAddressAutocomplete(dom.startLocation);
            if (dom.endLocation) this.setupAddressAutocomplete(dom.endLocation);
            
            window.addEventListener('online', this.handleOnline.bind(this));
            window.addEventListener('offline', this.handleOffline.bind(this));
            document.addEventListener('visibilitychange', this.handleVisibilityChange.bind(this));
        },

        toggleSidebar() {
            const sidebar = document.querySelector('.sidebar');
            if (!sidebar) return;
            
            sidebar.classList.toggle('active');
            
            const menuToggle = document.getElementById('menuToggle');
            if (menuToggle) {
                menuToggle.innerHTML = sidebar.classList.contains('active') 
                    ? '<i class="fas fa-times"></i>' 
                    : '<i class="fas fa-bars"></i>';
            }
        },

        locateUser() {
            if (this.userMarker) {
                this.map.flyTo({
                    center: this.userMarker.getLngLat(),
                    zoom: 16,
                    essential: true
                });
            } else {
                this.startGeolocation();
            }
        },

        refreshLights() {
            const refreshBtn = document.getElementById('refreshBtn');
            if (refreshBtn) {
                refreshBtn.innerHTML = '<i class="fas fa-sync-alt fa-spin"></i>';
            }
            
            this.loadTrafficLights(true);
            
            setTimeout(() => {
                if (refreshBtn) {
                    refreshBtn.innerHTML = '<i class="fas fa-sync-alt"></i>';
                }
                if (this.settings.notifications) {
                    this.showNotification('Traffic lights refreshed', 'success');
                }
            }, 800);
        },

        findNearbyLights() {
            if (this.userMarker) {
                this.map.flyTo({
                    center: this.userMarker.getLngLat(),
                    zoom: 16,
                    essential: true
                });
                this.highlightNearbyLights();
            } else {
                this.startGeolocation();
            }
        },
        
        toggleRoutePanel() {
            const routePlannerPanel = document.getElementById('routePlannerPanel');
            if (!routePlannerPanel) return;
            
            if (this.state.routePanelOpen) {
                routePlannerPanel.classList.add('hidden');
                this.state.routePanelOpen = false;
            } else {
                if (this.userMarker) {
                    const position = this.userMarker.getLngLat();
                    this.reverseGeocode(position.lng, position.lat)
                        .then(placeName => {
                            const startLocation = document.getElementById('startLocation');
                            if (startLocation) {
                                startLocation.value = placeName || `${position.lat.toFixed(6)}, ${position.lng.toFixed(6)}`;
                                this.navigation.startPosition = {
                                    center: [position.lng, position.lat],
                                    place_name: placeName || `${position.lat.toFixed(6)}, ${position.lng.toFixed(6)}`
                                };
                            }
                        });
                }
                
                routePlannerPanel.classList.remove('hidden');
                this.state.routePanelOpen = true;
            }
        },
        
        showSavedRoutes() {
            fetch('db.php?action=getUserRoutes')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        this.showRoutesListInSidebar(data.routes);
                    } else {
                        if (this.settings.notifications) {
                            this.showNotification(data.message, 'error');
                        }
                    }
                })
                .catch(error => {
                    console.error('Error fetching routes:', error);
                    if (this.settings.notifications) {
                        this.showNotification('Error loading saved routes', 'error');
                    }
                });
        },
        
        showRoutesListInSidebar(routes) {
            const lightsList = document.getElementById('lightsList');
            const noLightsMessage = document.getElementById('noLightsMessage');
            
            if (!lightsList || !noLightsMessage) return;
            
            lightsList.innerHTML = '';
            
            if (routes.length === 0) {
                noLightsMessage.querySelector('p').textContent = 'No saved routes';
                noLightsMessage.classList.remove('hidden');
                return;
            }
            
            noLightsMessage.classList.add('hidden');
            
            routes.forEach(route => {
                const card = document.createElement('div');
                card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg rounded-xl';
                
                const startName = route.route_data.start_point.place_name || 'Start';
                const endName = route.route_data.end_point.place_name || 'Destination';
                const distance = route.route_data.distance ? `${(route.route_data.distance / 1000).toFixed(1)} km` : '';
                const duration = route.route_data.duration ? `${Math.round(route.route_data.duration / 60)} min` : '';
                
                card.innerHTML = `
                    <div class="flex justify-between items-center">
                        <h3 class="font-medium">${route.name}</h3>
                        <span class="px-2.5 py-1 text-xs font-medium rounded-full bg-primary-100 text-primary-800">
                            <i class="fas fa-route mr-1"></i>${distance}
                        </span>
                    </div>
                    <div class="flex items-center text-sm text-gray-500 mt-2">
                        <i class="fas fa-map-marker-alt text-primary-500 mr-2"></i>
                        <span class="truncate">${startName} → ${endName}</span>
                    </div>
                    <div class="mt-2 flex justify-between text-xs">
                        <span class="text-gray-500">${duration}</span>
                        <span class="text-gray-500">${this.formatDate(route.created_at)}</span>
                    </div>
                `;
                
                card.addEventListener('click', () => {
                    this.loadSavedRoute(route);
                });
                
                lightsList.appendChild(card);
            });
        },
        
        formatDate(dateString) {
            const date = new Date(dateString);
            return date.toLocaleDateString();
        },
        
        loadSavedRoute(route) {
            const startLocation = document.getElementById('startLocation');
            const endLocation = document.getElementById('endLocation');
            const routePlannerPanel = document.getElementById('routePlannerPanel');
            
            if (!startLocation || !endLocation || !routePlannerPanel) return;
            
            startLocation.value = route.route_data.start_point.place_name || '';
            endLocation.value = route.route_data.end_point.place_name || '';
            
            this.navigation.startPosition = route.route_data.start_point;
            this.navigation.endPosition = route.route_data.end_point;
            this.navigation.travelMode = route.route_data.travel_mode || 'driving';
            
            this.setTravelMode(this.navigation.travelMode, false);
            
            routePlannerPanel.classList.remove('hidden');
            this.state.routePanelOpen = true;
            
            if (window.innerWidth <= 768) {
                const sidebar = document.querySelector('.sidebar');
                if (sidebar) {
                    sidebar.classList.remove('active');
                    const menuToggle = document.getElementById('menuToggle');
                    if (menuToggle) {
                        menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
                    }
                }
            }
            
            this.displayRouteDetails(route.route_data);
            
            this.drawRoute(route.route_data.coordinates);
        },

        openModal(modal) {
            const modalElement = document.getElementById(modal);
            if (!modalElement) return;
            
            if (modal === 'addModal') {
                this.state.selectingLocation = true;
                if (this.userMarker) {
                    const pos = this.userMarker.getLngLat();
                    const latitudeInput = document.getElementById('latitude');
                    const longitudeInput = document.getElementById('longitude');
                    
                    if (latitudeInput && longitudeInput) {
                        latitudeInput.value = pos.lat.toFixed(6);
                        longitudeInput.value = pos.lng.toFixed(6);
                    }
                    
                    if (this.tempMarker) this.tempMarker.remove();
                    this.tempMarker = new mapboxgl.Marker({ draggable: true })
                        .setLngLat([pos.lng, pos.lat])
                        .addTo(this.map);
                    
                    this.tempMarker.on('dragend', this.updateMarkerPosition.bind(this));
                }
            }
            
            modalElement.style.display = 'flex';
            const modalBody = modalElement.querySelector('.modal-body');
            if (modalBody) {
                modalBody.style.transform = 'translateY(0)';
            }
        },

        closeModal(modal) {
            const modalElement = typeof modal === 'string' ? document.getElementById(modal) : modal;
            if (!modalElement) return;
            
            const modalBody = modalElement.querySelector('.modal-body');
            if (modalBody) {
                modalBody.style.transition = 'transform 0.3s ease';
                modalBody.style.transform = 'translateY(100%)';
            }
            
            setTimeout(() => {
                modalElement.style.display = 'none';
                if (modalBody) {
                    modalBody.style.transform = 'translateY(0)';
                }
                
                if (modalElement.id === 'addModal') {
                    this.state.selectingLocation = false;
                    if (this.tempMarker) {
                        this.tempMarker.remove();
                        this.tempMarker = null;
                    }
                }
            }, 300);
        },

        handleMapClick(e) {
            if (this.state.selectingLocation) {
                const latitudeInput = document.getElementById('latitude');
                const longitudeInput = document.getElementById('longitude');
                
                if (latitudeInput && longitudeInput) {
                    latitudeInput.value = e.lngLat.lat.toFixed(6);
                    longitudeInput.value = e.lngLat.lng.toFixed(6);
                }
                
                if (this.tempMarker) this.tempMarker.remove();
                this.tempMarker = new mapboxgl.Marker({ draggable: true })
                    .setLngLat([e.lngLat.lng, e.lngLat.lat])
                    .addTo(this.map);
                
                this.tempMarker.on('dragend', this.updateMarkerPosition.bind(this));
            }
        },

        updateMarkerPosition() {
            if (!this.tempMarker) return;
            
            const pos = this.tempMarker.getLngLat();
            const latitudeInput = document.getElementById('latitude');
            const longitudeInput = document.getElementById('longitude');
            
            if (latitudeInput && longitudeInput) {
                latitudeInput.value = pos.lat.toFixed(6);
                longitudeInput.value = pos.lng.toFixed(6);
            }
        },

        handleSearch() {
            const searchInput = document.getElementById('searchInput');
            if (!searchInput) return;
            
            const searchTerm = searchInput.value.toLowerCase();
            const lights = document.querySelectorAll('.light-card');
            
            let hasResults = false;
            
            lights.forEach(light => {
                const name = light.getAttribute('data-name')?.toLowerCase() || '';
                const visible = name.includes(searchTerm);
                light.style.display = visible ? 'block' : 'none';
                if (visible) hasResults = true;
            });
            
            const noLightsMessage = document.getElementById('noLightsMessage');
            if (noLightsMessage) {
                if (lights.length > 0 && !hasResults) {
                    noLightsMessage.classList.remove('hidden');
                    const messageParagraph = noLightsMessage.querySelector('p');
                    if (messageParagraph) {
                        messageParagraph.textContent = 'No results found';
                    }
                } else {
                    noLightsMessage.classList.add('hidden');
                }
            }
        },

        toggleDarkMode() {
            const darkModeToggle = document.getElementById('darkModeToggle');
            if (!darkModeToggle) return;
            
            this.settings.darkMode = darkModeToggle.checked;
            
            if (this.settings.darkMode) {
                document.documentElement.classList.add('dark');
                document.body.classList.add('dark-mode');
            } else {
                document.documentElement.classList.remove('dark');
                document.body.classList.remove('dark-mode');
            }
            
            this.saveSettings();
        },

        toggleAutoRefresh() {
            const autoRefreshToggle = document.getElementById('autoRefreshToggle');
            if (!autoRefreshToggle) return;
            
            this.settings.autoRefresh = autoRefreshToggle.checked;
            
            if (this.settings.autoRefresh) {
                this.startStatusUpdates();
            } else {
                this.stopStatusUpdates();
            }
            
            this.saveSettings();
        },

        resetAppData() {
            if (confirm('Are you sure you want to reset all app data? This will clear all your settings and cache.')) {
                localStorage.clear();
                this.showNotification('App data reset. Refreshing...', 'info');
                setTimeout(() => {
                    window.location.reload();
                }, 1500);
            }
        },

        navigateToSelectedLight() {
            if (!this.selectedLightId || !this.markers[this.selectedLightId]) return;
            
            const light = this.markers[this.selectedLightId].data;
            
            const startLocation = document.getElementById('startLocation');
            const endLocation = document.getElementById('endLocation');
            const lightPopup = document.getElementById('lightPopup');
            const routePlannerPanel = document.getElementById('routePlannerPanel');
            
            if (!startLocation || !endLocation || !lightPopup || !routePlannerPanel) return;
            
            startLocation.value = '';
            endLocation.value = light.name;
            this.navigation.endPosition = {
                center: [parseFloat(light.longitude), parseFloat(light.latitude)],
                place_name: light.name
            };
            
            if (this.userMarker) {
                const userPosition = this.userMarker.getLngLat();
                this.reverseGeocode(userPosition.lng, userPosition.lat)
                    .then(placeName => {
                        startLocation.value = placeName || `${userPosition.lat.toFixed(6)}, ${userPosition.lng.toFixed(6)}`;
                        this.navigation.startPosition = {
                            center: [userPosition.lng, userPosition.lat],
                            place_name: placeName || `${userPosition.lat.toFixed(6)}, ${userPosition.lng.toFixed(6)}`
                        };
                        
                        lightPopup.classList.add('hidden');
                        routePlannerPanel.classList.remove('hidden');
                        this.state.routePanelOpen = true;
                    });
            } else {
                lightPopup.classList.add('hidden');
                routePlannerPanel.classList.remove('hidden');
                this.state.routePanelOpen = true;
                
                this.startGeolocation();
            }
        },

        reverseGeocode(lng, lat) {
            return fetch(`https://api.mapbox.com/geocoding/v5/mapbox.places/${lng},${lat}.json?access_token=${this.mapboxKey}`)
                .then(response => response.json())
                .then(data => {
                    if (data.features && data.features.length > 0) {
                        return data.features[0].place_name;
                    }
                    return null;
                })
                .catch(error => {
                    console.error('Reverse geocoding error:', error);
                    return null;
                });
        },
        
        setTravelMode(mode, calculate = true) {
            this.navigation.travelMode = mode;
            
            const carBtn = document.getElementById('routeTypeCar');
            const bikeBtn = document.getElementById('routeTypeBike');
            const walkBtn = document.getElementById('routeTypeWalk');
            
            if (carBtn) carBtn.classList.remove('active');
            if (bikeBtn) bikeBtn.classList.remove('active');
            if (walkBtn) walkBtn.classList.remove('active');
            
            let activeBtn;
            switch (mode) {
                case 'cycling':
                    activeBtn = bikeBtn;
                    break;
                case 'walking':
                    activeBtn = walkBtn;
                    break;
                default:
                    activeBtn = carBtn;
            }
            
            if (activeBtn) activeBtn.classList.add('active');
            
            if (calculate && this.navigation.startPosition && this.navigation.endPosition) {
                this.calculateRoute();
            }
        },

        