<?php
session_start();
require_once 'db.php';
require_once 'folder-api.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: /login');
    exit;
}

if(isset($_SESSION['andweare_member']) && $_SESSION['andweare_member'] == 1 && !empty($_SESSION['family_site_url'])) {
    header('Location: ' . $_SESSION['family_site_url']);
    exit;
}

$error = '';
$success = '';
$is_loading = false;
$current_step = 1;

if (isset($_POST['step'])) {
    $current_step = intval($_POST['step']);
}

if (isset($_GET['check_progress']) && isset($_SESSION['create_id'])) {
    header('Content-Type: application/json');
    $progress_file = 'creation_progress_' . $_SESSION['create_id'] . '.json';
    
    if (file_exists($progress_file)) {
        $progress = json_decode(file_get_contents($progress_file), true);
        echo json_encode($progress);
    } else {
        echo json_encode([
            'status' => 'error',
            'message' => 'Progress file not found'
        ]);
    }
    exit;
}

if (isset($_GET['process_creation']) && isset($_SESSION['create_data'])) {
    $data = $_SESSION['create_data'];
    $create_id = $_SESSION['create_id'];
    $progress_file = 'creation_progress_' . $create_id . '.json';
    
    $db = new Database();
    $conn = $db->connect();
    $user_id = $_SESSION['user_id'];
    
    try {
        $family_name = $data['family_name'];
        $language = $data['language'];
        $family_slug = strtolower(preg_replace('/[^a-z0-9]+/i', '-', $family_name));
        
        $progress = [
            'status' => 'running',
            'percent_complete' => 0,
            'current_operation' => 'Checking family name availability',
            'started_at' => time(),
            'updated_at' => time(),
            'steps' => [
                ['name' => 'Validation', 'status' => 'running'],
                ['name' => 'Creating Directory', 'status' => 'pending'],
                ['name' => 'Copying Files', 'status' => 'pending'],
                ['name' => 'Database Setup', 'status' => 'pending'],
                ['name' => 'Finalizing', 'status' => 'pending']
            ],
            'current_file' => '',
            'total_files' => 0,
            'files_copied' => 0,
            'errors' => []
        ];
        file_put_contents($progress_file, json_encode($progress));
        
        // Vérifier si le nom est disponible
        $stmt = $conn->prepare("SELECT COUNT(*) FROM family_sites WHERE family_slug = ?");
        $stmt->execute([$family_slug]);
        $exists = $stmt->fetchColumn();
        
        if ($exists) {
            $progress['status'] = 'error';
            $progress['errors'][] = 'This family name is already taken. Please choose another one.';
            file_put_contents($progress_file, json_encode($progress));
            exit;
        }
        
        // Validation terminée
        $progress['percent_complete'] = 10;
        $progress['steps'][0]['status'] = 'complete';
        $progress['steps'][1]['status'] = 'running';
        $progress['current_operation'] = 'Creating directory';
        $progress['updated_at'] = time();
        file_put_contents($progress_file, json_encode($progress));
        
        // Création du dossier
        $folder_api = new FolderAPI();
        $result = $folder_api->createDirectory('/public_html/' . $family_slug);
        
        if ($result['status'] != 'success') {
            $progress['status'] = 'error';
            $progress['errors'][] = 'Failed to create directory: ' . ($result['message'] ?? 'Unknown error');
            file_put_contents($progress_file, json_encode($progress));
            exit;
        }
        
        // Site URL
        $site_url = 'https://andweare.com/' . $family_slug;
        
        // Dossier créé
        $progress['percent_complete'] = 30;
        $progress['steps'][1]['status'] = 'complete';
        $progress['steps'][2]['status'] = 'running';
        $progress['current_operation'] = 'Copying application files';
        $progress['updated_at'] = time();
        file_put_contents($progress_file, json_encode($progress));
        
        // Copie des fichiers
        $app_source = '/app-' . $language;
        $site_destination = '/public_html/' . $family_slug;
        
        $copy_result = $folder_api->copyFiles($app_source, $site_destination);
        
        if ($copy_result['status'] != 'success' && $copy_result['status'] != 'partial') {
            $progress['status'] = 'error';
            $progress['errors'][] = 'Failed to copy application files: ' . ($copy_result['message'] ?? 'Unknown error');
            file_put_contents($progress_file, json_encode($progress));
            exit;
        }
        
        // Fichiers copiés
        $progress['percent_complete'] = 70;
        $progress['steps'][2]['status'] = 'complete';
        $progress['steps'][3]['status'] = 'running';
        $progress['current_operation'] = 'Setting up database';
        $progress['updated_at'] = time();
        file_put_contents($progress_file, json_encode($progress));
        
        // Mise à jour de la base de données
        $stmt = $conn->prepare("UPDATE utilisateurs SET andweare_member = 1, family_site_url = ?, family_role = 'admin' WHERE id = ?");
        $stmt->execute([$site_url, $user_id]);
        
        $stmt = $conn->prepare("INSERT INTO family_sites (family_name, family_slug, site_url, created_by, language) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([
            $family_name,
            $family_slug,
            $site_url,
            $user_id,
            $language
        ]);
        
        $family_site_id = $conn->lastInsertId();
        $stmt = $conn->prepare("INSERT INTO family_members (family_site_id, user_id, role) VALUES (?, ?, 'admin')");
        $stmt->execute([$family_site_id, $user_id]);
        
        // Base de données mise à jour
        $progress['percent_complete'] = 90;
        $progress['steps'][3]['status'] = 'complete';
        $progress['steps'][4]['status'] = 'running';
        $progress['current_operation'] = 'Finalizing your family hub';
        $progress['updated_at'] = time();
        file_put_contents($progress_file, json_encode($progress));
        
        // Finalisation
        $_SESSION['andweare_member'] = 1;
        $_SESSION['family_site_url'] = $site_url;
        $_SESSION['family_role'] = 'admin';
        
        // Processus terminé
        $progress['percent_complete'] = 100;
        $progress['steps'][4]['status'] = 'complete';
        $progress['status'] = 'complete';
        $progress['current_operation'] = 'Family hub created successfully!';
        $progress['redirect_url'] = $site_url;
        $progress['updated_at'] = time();
        $progress['completion_time'] = time() - $progress['started_at'];
        file_put_contents($progress_file, json_encode($progress));
        
    } catch(PDOException $e) {
        $progress['status'] = 'error';
        $progress['errors'][] = 'Database error: ' . $e->getMessage();
        file_put_contents($progress_file, json_encode($progress));
    } catch(Exception $e) {
        $progress['status'] = 'error';
        $progress['errors'][] = 'Error: ' . $e->getMessage();
        file_put_contents($progress_file, json_encode($progress));
    }
    
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_family'])) {
    $family_name = htmlspecialchars(trim($_POST['family_name'] ?? ''));
    $language = $_POST['language'] ?? 'en';
    
    if(empty($family_name)) {
        $error = "Please enter a family name";
    } else {
        $is_loading = true;
        $_SESSION['create_id'] = uniqid('create_');
        $_SESSION['create_data'] = [
            'family_name' => $family_name,
            'language' => $language
        ];
    }
}

if (isset($_POST['next_step'])) {
    $current_step++;
} elseif (isset($_POST['prev_step'])) {
    $current_step--;
}

$current_step = max(1, min(2, $current_step));

function sanitize_slug($input) {
    return strtolower(preg_replace('/[^a-z0-9]+/i', '-', $input));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Create Your Family Hub - AndWeare</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+KR:wght@300;400;500;700&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        body {
            font-family: 'Noto Sans KR', sans-serif;
        }
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(255, 255, 255, 0.95);
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            z-index: 50;
        }
        .progress-step {
            transition: all 0.3s ease;
        }
        .progress-step.active {
            transform: translateX(8px);
        }
        .progress-step.complete .step-icon {
            color: #10B981;
        }
        .progress-step.running .step-icon {
            color: #3B82F6;
            animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
        }
        .progress-step.error .step-icon {
            color: #EF4444;
        }
        @keyframes pulse {
            0%, 100% {
                opacity: 1;
            }
            50% {
                opacity: 0.5;
            }
        }
        .faq-item.active .faq-content {
            display: block;
        }
        .faq-item.active .faq-icon {
            transform: rotate(180deg);
        }
    </style>
</head>
<body class="bg-white min-h-screen">
    <?php if ($is_loading): ?>
    <div id="loading-overlay" class="loading-overlay">
        <div class="max-w-xl w-full bg-white shadow-xl rounded-lg p-8">
            <div class="text-center mb-8">
                <h2 class="text-xl font-bold mb-2" id="loading-title">Creating your family hub...</h2>
                <p class="text-gray-600 mb-6" id="loading-subtitle">Please wait while we set everything up. This may take a few minutes.</p>
                
                <div class="relative pt-1 mb-6">
                    <div class="flex items-center justify-between mb-2">
                        <div>
                            <span class="text-xs font-semibold inline-block py-1 px-2 uppercase rounded-full bg-blue-200 text-blue-600" id="progress-text">
                                0%
                            </span>
                        </div>
                        <div class="text-right">
                            <span class="text-xs font-semibold inline-block text-blue-600" id="time-estimate">
                                Estimating time...
                            </span>
                        </div>
                    </div>
                    <div class="overflow-hidden h-2 mb-4 text-xs flex rounded bg-blue-200">
                        <div id="progress-bar" class="shadow-none flex flex-col text-center whitespace-nowrap text-white justify-center bg-blue-500 transition-all duration-500" style="width: 0%"></div>
                    </div>
                </div>
                
                <div class="text-left">
                    <h3 class="font-bold text-lg mb-4 border-b pb-2">Progress Details</h3>
                    <div class="space-y-6" id="steps-container">
                        <!-- Étapes générées par JS -->
                    </div>
                </div>
            </div>
            
            <div id="error-container" class="mt-8 hidden">
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-4">
                    <div class="flex">
                        <svg class="h-6 w-6 text-red-500 mr-3" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                        <div>
                            <p class="font-bold">Error</p>
                            <p id="error-message">Something went wrong during the creation process.</p>
                        </div>
                    </div>
                </div>
                <div class="flex justify-end">
                    <a href="create-family.php" class="inline-block px-6 py-2.5 bg-gray-600 text-white font-medium text-xs leading-tight uppercase rounded shadow-md hover:bg-gray-700 hover:shadow-lg focus:bg-gray-700 focus:shadow-lg focus:outline-none focus:ring-0 active:bg-gray-800 active:shadow-lg transition duration-150 ease-in-out">
                        Go Back
                    </a>
                </div>
            </div>
            
            <div id="success-container" class="mt-8 hidden">
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-4">
                    <div class="flex">
                        <svg class="h-6 w-6 text-green-500 mr-3" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                        </svg>
                        <div>
                            <p class="font-bold">Success!</p>
                            <p>Your family hub has been created successfully.</p>
                        </div>
                    </div>
                </div>
                <div class="flex justify-end">
                    <button id="goto-hub-btn" class="inline-block px-6 py-2.5 bg-blue-600 text-white font-medium text-xs leading-tight uppercase rounded shadow-md hover:bg-blue-700 hover:shadow-lg focus:bg-blue-700 focus:shadow-lg focus:outline-none focus:ring-0 active:bg-blue-800 active:shadow-lg transition duration-150 ease-in-out">
                        Go to My Family Hub
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    const createId = "<?php echo $_SESSION['create_id']; ?>";
    const steps = [
        { id: 0, name: 'Validation', description: 'Checking family name availability', icon: 'check-circle' },
        { id: 1, name: 'Creating Directory', description: 'Setting up your family space', icon: 'folder' },
        { id: 2, name: 'Copying Files', description: 'Copying application files', icon: 'document' },
        { id: 3, name: 'Database Setup', description: 'Setting up your family database', icon: 'database' },
        { id: 4, name: 'Finalizing', description: 'Completing the setup process', icon: 'sparkles' }
    ];
    
    let startTime = Date.now();
    let intervalId;
    let redirectUrl = '';
    
    function getStepIcon(type) {
        switch(type) {
            case 'check-circle':
                return '<svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 step-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" /></svg>';
            case 'folder':
                return '<svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 step-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z" /></svg>';
            case 'document':
                return '<svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 step-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" /></svg>';
            case 'database':
                return '<svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 step-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 7v10c0 2.21 3.582 4 8 4s8-1.79 8-4V7M4 7c0 2.21 3.582 4 8 4s8-1.79 8-4M4 7c0-2.21 3.582-4 8-4s8 1.79 8 4m0 5c0 2.21-3.582 4-8 4s-8-1.79-8-4" /></svg>';
            case 'sparkles':
                return '<svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 step-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16l2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143L13 3z" /></svg>';
            default:
                return '<svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 step-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" /></svg>';
        }
    }
    
    function initializeSteps() {
        const container = document.getElementById('steps-container');
        steps.forEach(step => {
            const stepElement = document.createElement('div');
            stepElement.id = `step-${step.id}`;
            stepElement.className = 'progress-step flex items-start';
            stepElement.dataset.status = 'pending';
            
            stepElement.innerHTML = `
                <div class="flex-shrink-0 mt-1 mr-3 text-gray-400">
                    ${getStepIcon(step.icon)}
                </div>
                <div class="flex-1">
                    <h4 class="font-medium">${step.name}</h4>
                    <p class="text-sm text-gray-500">${step.description}</p>
                    <div class="step-details mt-1 text-xs text-gray-400"></div>
                </div>
            `;
            
            container.appendChild(stepElement);
        });
    }
    
    function updateStepStatus(stepId, status, details = '') {
        const stepElement = document.getElementById(`step-${stepId}`);
        if (!stepElement) return;
        
        // Supprimer toutes les classes de statut
        stepElement.classList.remove('pending', 'running', 'complete', 'error', 'active');
        
        // Ajouter les nouvelles classes
        stepElement.classList.add(status);
        stepElement.dataset.status = status;
        
        if (status === 'running') {
            stepElement.classList.add('active');
        }
        
        // Mettre à jour les détails si fournis
        if (details) {
            const detailsElement = stepElement.querySelector('.step-details');
            if (detailsElement) {
                detailsElement.textContent = details;
            }
        }
    }
    
    function formatTimeRemaining(seconds) {
        if (seconds < 60) {
            return `${Math.round(seconds)} seconds remaining`;
        } else if (seconds < 3600) {
            const minutes = Math.floor(seconds / 60);
            const remainingSeconds = Math.round(seconds % 60);
            return `${minutes} min ${remainingSeconds} sec remaining`;
        } else {
            const hours = Math.floor(seconds / 3600);
            const minutes = Math.floor((seconds % 3600) / 60);
            return `${hours} hr ${minutes} min remaining`;
        }
    }
    
    function checkProgress() {
        fetch(`create-family.php?check_progress=1&id=${createId}&t=${Date.now()}`)
            .then(response => response.json())
            .then(data => {
                // Mettre à jour la barre de progression
                const percent = data.percent_complete || 0;
                document.getElementById('progress-bar').style.width = `${percent}%`;
                document.getElementById('progress-text').textContent = `${Math.round(percent)}%`;
                
                // Mettre à jour le titre de l'opération en cours
                document.getElementById('loading-title').textContent = data.current_operation || 'Creating your family hub...';
                
                // Mettre à jour l'estimation du temps restant
                if (percent > 0) {
                    const elapsedSeconds = (Date.now() - startTime) / 1000;
                    const estimatedTotalSeconds = (elapsedSeconds / percent) * 100;
                    const remainingSeconds = estimatedTotalSeconds - elapsedSeconds;
                    
                    document.getElementById('time-estimate').textContent = formatTimeRemaining(remainingSeconds);
                }
                
                // Mettre à jour le statut des étapes
                if (data.steps) {
                    data.steps.forEach((step, index) => {
                        updateStepStatus(index, step.status, step.details || '');
                    });
                }
                
                // Afficher les détails de progression pour la copie de fichiers
                if (data.current_file && data.total_files > 0) {
                    const copyingStep = document.getElementById('step-2');
                    if (copyingStep) {
                        const details = copyingStep.querySelector('.step-details');
                        if (details) {
                            details.textContent = `Copying file ${data.files_copied} of ${data.total_files}: ${data.current_file}`;
                        }
                    }
                }
                
                // Gérer les erreurs
                if (data.status === 'error') {
                    clearInterval(intervalId);
                    document.getElementById('error-container').classList.remove('hidden');
                    document.getElementById('error-message').textContent = data.errors && data.errors.length > 0 
                        ? data.errors[0] 
                        : 'An error occurred during the creation process.';
                }
                
                // Gérer le succès
                if (data.status === 'complete' && percent === 100) {
                    clearInterval(intervalId);
                    document.getElementById('success-container').classList.remove('hidden');
                    
                    // Configurer la redirection
                    if (data.redirect_url) {
                        redirectUrl = data.redirect_url;
                        document.getElementById('goto-hub-btn').addEventListener('click', function() {
                            window.location.href = redirectUrl;
                        });
                    }
                }
            })
            .catch(error => {
                console.error('Error checking progress:', error);
            });
    }
    
    // Initialiser et démarrer le processus
    document.addEventListener('DOMContentLoaded', function() {
        initializeSteps();
        updateStepStatus(0, 'running');
        
        // Démarrer le processus de création en arrière-plan
        fetch('create-family.php?process_creation=1', {
            method: 'GET'
        }).catch(err => console.error('Error starting creation process:', err));
        
        // Vérifier la progression toutes les secondes
        intervalId = setInterval(checkProgress, 1000);
    });
    </script>
    <?php else: ?>
    <div class="max-w-4xl mx-auto px-4 py-8">
        <div class="flex justify-between items-center mb-8">
            <a href="/" class="block">
                <img src="logo.png" alt="AndWeare Logo" class="h-10">
            </a>
            <a href="/login/logout.php" class="text-gray-600 hover:text-black transition-colors">
                Sign Out
            </a>
        </div>
        
        <div class="text-center mb-12">
            <h1 class="text-3xl font-light mb-2">Create Your Family Hub</h1>
            <p class="text-gray-600">Welcome <?php echo htmlspecialchars($_SESSION['username']); ?>, let's set up your family's online space</p>
        </div>
        
        <?php if($error): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-6">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>
        
        <div class="bg-gray-50 rounded-lg shadow-sm p-6 mb-8">
            <div class="flex justify-between mb-8">
                <div class="flex items-center <?php echo $current_step >= 1 ? 'text-blue-600' : 'text-gray-400'; ?>">
                    <div class="w-8 h-8 rounded-full flex items-center justify-center mr-2 <?php echo $current_step >= 1 ? 'bg-blue-600 text-white' : 'bg-gray-200'; ?>">1</div>
                    <span>Family Details</span>
                </div>
                <div class="flex items-center <?php echo $current_step >= 2 ? 'text-blue-600' : 'text-gray-400'; ?>">
                    <div class="w-8 h-8 rounded-full flex items-center justify-center mr-2 <?php echo $current_step >= 2 ? 'bg-blue-600 text-white' : 'bg-gray-200'; ?>">2</div>
                    <span>Language</span>
                </div>
            </div>
            
            <form method="POST" id="familyForm">
                <input type="hidden" name="step" value="<?php echo $current_step; ?>">
                
                <!-- Step 1: Family Details -->
                <div id="step1" <?php if ($current_step != 1): ?>style="display: none;"<?php endif; ?>>
                    <h2 class="text-xl font-medium mb-4">What's your family's name?</h2>
                    <p class="text-gray-600 mb-4">This will be the name of your family hub.</p>
                    
                    <div class="mb-6">
                        <label for="family_name" class="block text-sm font-medium text-gray-700 mb-1">Family Name</label>
                        <input 
                            type="text" 
                            id="family_name" 
                            name="family_name" 
                            class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            placeholder="e.g. The Smiths, Johnson Family"
                            required
                            value="<?php echo isset($_POST['family_name']) ? htmlspecialchars($_POST['family_name']) : ''; ?>"
                            onkeyup="updatePreview()"
                        >
                        <p class="mt-2 text-sm text-gray-500">
                            This will be used to create your family's URL: <span id="url-preview" class="font-medium"></span>
                        </p>
                    </div>
                    
                    <div class="flex justify-end">
                        <button 
                            type="submit" 
                            name="next_step" 
                            class="px-4 py-2 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors"
                            id="nextButton"
                            disabled
                        >
                            Next
                        </button>
                    </div>
                </div>
                
                <!-- Step 2: Language -->
                <div id="step2" <?php if ($current_step != 2): ?>style="display: none;"<?php endif; ?>>
                    <h2 class="text-xl font-medium mb-4">Choose a language</h2>
                    <p class="text-gray-600 mb-4">Select the primary language for your family hub.</p>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all language-option <?php echo (!isset($_POST['language']) || $_POST['language'] == 'en') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            onclick="selectLanguage('en')"
                        >
                            <div class="flex items-center">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (!isset($_POST['language']) || $_POST['language'] == 'en') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (!isset($_POST['language']) || $_POST['language'] == 'en') ? '' : 'style="display: none;"'; ?> id="en-indicator"></div>
                                </div>
                                <span class="font-medium">English</span>
                            </div>
                        </div>
                        
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all language-option <?php echo (isset($_POST['language']) && $_POST['language'] == 'fr') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            onclick="selectLanguage('fr')"
                        >
                            <div class="flex items-center">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (isset($_POST['language']) && $_POST['language'] == 'fr') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (isset($_POST['language']) && $_POST['language'] == 'fr') ? '' : 'style="display: none;"'; ?> id="fr-indicator"></div>
                                </div>
                                <span class="font-medium">Français</span>
                            </div>
                        </div>
                        
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all language-option <?php echo (isset($_POST['language']) && $_POST['language'] == 'de') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            onclick="selectLanguage('de')"
                        >
                            <div class="flex items-center">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (isset($_POST['language']) && $_POST['language'] == 'de') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (isset($_POST['language']) && $_POST['language'] == 'de') ? '' : 'style="display: none;"'; ?> id="de-indicator"></div>
                                </div>
                                <span class="font-medium">Deutsch</span>
                            </div>
                        </div>
                        
                        <div 
                            class="border rounded-lg p-4 cursor-pointer transition-all language-option <?php echo (isset($_POST['language']) && $_POST['language'] == 'nl') ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'; ?>"
                            onclick="selectLanguage('nl')"
                        >
                            <div class="flex items-center">
                                <div class="w-5 h-5 rounded-full border border-gray-300 mr-2 flex items-center justify-center <?php echo (isset($_POST['language']) && $_POST['language'] == 'nl') ? 'border-blue-500' : ''; ?>">
                                    <div class="w-3 h-3 rounded-full bg-blue-500" <?php echo (isset($_POST['language']) && $_POST['language'] == 'nl') ? '' : 'style="display: none;"'; ?> id="nl-indicator"></div>
                                </div>
                                <span class="font-medium">Nederlands</span>
                            </div>
                        </div>
                    </div>
                    
                    <input type="hidden" name="language" id="language" value="<?php echo isset($_POST['language']) ? htmlspecialchars($_POST['language']) : 'en'; ?>">
                    
                    <div class="bg-white p-4 rounded-lg border border-gray-200 mb-6">
                        <h3 class="font-medium mb-2">Your family hub summary</h3>
                        <ul class="space-y-2 text-sm">
                            <li class="flex justify-between">
                                <span class="text-gray-600">Family name:</span>
                                <span class="font-medium" id="summary-name"></span>
                            </li>
                            <li class="flex justify-between">
                                <span class="text-gray-600">URL:</span>
                                <span class="font-medium" id="summary-url"></span>
                            </li>
                            <li class="flex justify-between">
                                <span class="text-gray-600">Language:</span>
                                <span class="font-medium" id="summary-language"></span>
                            </li>
                        </ul>
                    </div>
                    
                    <div class="flex justify-between">
                        <button 
                            type="submit" 
                            name="prev_step" 
                            class="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
                        >
                            Back
                        </button>
                        <button 
                            type="submit" 
                            name="create_family" 
                            class="px-4 py-2 bg-black text-white rounded-lg hover:bg-gray-800 transition-colors"
                        >
                            Create Family Hub
                        </button>
                    </div>
                </div>
            </form>
        </div>
        
        <div class="mt-8">
            <h3 class="text-lg font-medium mb-4">Frequently Asked Questions</h3>
            
            <div class="space-y-4">
                <div class="border border-gray-200 rounded-lg overflow-hidden faq-item">
                    <button 
                        class="flex justify-between items-center w-full p-4 text-left bg-white hover:bg-gray-50 transition-colors"
                        onclick="toggleFAQ(this)"
                    >
                        <span class="font-medium">Can I change my family hub URL later?</span>
                        <svg class="h-5 w-5 text-gray-500 faq-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd" />
                        </svg>
                    </button>
                    <div class="p-4 border-t border-gray-200 bg-white faq-content" style="display: none;">
                        <p class="text-gray-600">
                            Once you've created your family hub URL, it cannot be changed. This ensures the stability of your family's online home. Choose a name that will stand the test of time!
                        </p>
                    </div>
                </div>
                
                <div class="border border-gray-200 rounded-lg overflow-hidden faq-item">
                    <button 
                        class="flex justify-between items-center w-full p-4 text-left bg-white hover:bg-gray-50 transition-colors"
                        onclick="toggleFAQ(this)"
                    >
                        <span class="font-medium">How do I invite family members?</span>
                        <svg class="h-5 w-5 text-gray-500 faq-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd" />
                        </svg>
                    </button>
                    <div class="p-4 border-t border-gray-200 bg-white faq-content" style="display: none;">
                        <p class="text-gray-600">
                            After creating your family hub, you can invite family members by sending them an email invitation directly from your dashboard. They'll receive a link to join your family hub, and can create their own account if they don't already have one.
                        </p>
                    </div>
                </div>
                
                <div class="border border-gray-200 rounded-lg overflow-hidden faq-item">
                    <button 
                        class="flex justify-between items-center w-full p-4 text-left bg-white hover:bg-gray-50 transition-colors"
                        onclick="toggleFAQ(this)"
                    >
                        <span class="font-medium">Can I change the language later?</span>
                        <svg class="h-5 w-5 text-gray-500 faq-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd" />
                        </svg>
                    </button>
                    <div class="p-4 border-t border-gray-200 bg-white faq-content" style="display: none;">
                        <p class="text-gray-600">
                            Yes, you can change the language of your family hub after creation. You can do this in the settings section of your family hub dashboard.
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        let familyName = "<?php echo isset($_POST['family_name']) ? htmlspecialchars($_POST['family_name']) : ''; ?>";
        let familySlug = "<?php echo isset($_POST['family_name']) ? sanitize_slug($_POST['family_name']) : ''; ?>";
        let language = "<?php echo isset($_POST['language']) ? htmlspecialchars($_POST['language']) : 'en'; ?>";
        
        function updatePreview() {
            let nameInput = document.getElementById('family_name');
            let nextButton = document.getElementById('nextButton');
            familyName = nameInput.value.trim();
            
            if (familyName) {
                nextButton.disabled = false;
                familySlug = familyName.toLowerCase().replace(/[^a-z0-9]+/g, '-');
                if (familySlug.length > 30) {
                    familySlug = familySlug.substring(0, 30);
                }
            } else {
                nextButton.disabled = true;
            }
            
            document.getElementById('url-preview').textContent = 'andweare.com/' + familySlug;
            updateSummary();
        }
        
        function selectLanguage(lang) {
            language = lang;
            document.getElementById('language').value = lang;
            
            // Réinitialiser tous les indicateurs de langue
            document.querySelectorAll('.language-option [id$="-indicator"]').forEach(indicator => {
                indicator.style.display = 'none';
            });
            
            // Afficher l'indicateur pour la langue sélectionnée
            document.getElementById(`${lang}-indicator`).style.display = 'block';
            
            // Mettre à jour les classes CSS
            document.querySelectorAll('.language-option').forEach(option => {
                const indicator = option.querySelector(`#${lang}-indicator`);
                if (indicator) {
                    option.classList.add('border-blue-500', 'bg-blue-50');
                    option.classList.remove('border-gray-200', 'hover:border-gray-300');
                    
                    const border = option.querySelector('.w-5');
                    if (border) {
                        border.classList.add('border-blue-500');
                        border.classList.remove('border-gray-300');
                    }
                } else {
                    option.classList.remove('border-blue-500', 'bg-blue-50');
                    option.classList.add('border-gray-200', 'hover:border-gray-300');
                    
                    const border = option.querySelector('.w-5');
                    if (border) {
                        border.classList.remove('border-blue-500');
                        border.classList.add('border-gray-300');
                    }
                }
            });
            
            updateSummary();
        }
        
        function updateSummary() {
            const summaryName = document.getElementById('summary-name');
            const summaryUrl = document.getElementById('summary-url');
            const summaryLanguage = document.getElementById('summary-language');
            
            if (summaryName) summaryName.textContent = familyName;
            
            if (summaryUrl) {
                summaryUrl.textContent = 'andweare.com/' + familySlug;
            }
            
            if (summaryLanguage) {
                let langDisplay = '';
                switch(language) {
                    case 'en': langDisplay = 'English'; break;
                    case 'fr': langDisplay = 'Français'; break;
                    case 'de': langDisplay = 'Deutsch'; break;
                    case 'nl': langDisplay = 'Nederlands'; break;
                    default: langDisplay = 'English';
                }
                summaryLanguage.textContent = langDisplay;
            }
        }
        
        function toggleFAQ(button) {
            const faqItem = button.closest('.faq-item');
            const content = faqItem.querySelector('.faq-content');
            const icon = faqItem.querySelector('.faq-icon');
            
            document.querySelectorAll('.faq-content').forEach(item => {
                if (item !== content) {
                    item.style.display = 'none';
                    item.closest('.faq-item').classList.remove('active');
                    item.closest('.faq-item').querySelector('.faq-icon').style.transform = '';
                }
            });
            
            if (content.style.display === 'none' || content.style.display === '') {
                content.style.display = 'block';
                faqItem.classList.add('active');
                icon.style.transform = 'rotate(180deg)';
            } else {
                content.style.display = 'none';
                faqItem.classList.remove('active');
                icon.style.transform = '';
            }
        }
        
        document.addEventListener('DOMContentLoaded', function() {
            updatePreview();
            updateSummary();
        });
    </script>
    <?php endif; ?>
</body>
</html>