<?php
require_once '../includes/db.php';
require_once '../includes/functions.php';

require_login();

$user = get_user_info($_SESSION['user_id']);
$country = get_user_country($_SESSION['user_id']);
$level_info = get_user_level_info($user['level']);
$inventory = get_user_inventory($_SESSION['user_id']);

$error = null;
$success = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'update_profile') {
        $bio = trim($_POST['bio'] ?? '');
        
        $data = [
            'bio' => $bio
        ];
        
        $updated = db_update('users', $data, 'id = ?', [$_SESSION['user_id']]);
        
        if ($updated) {
            $success = "Profil mis à jour avec succès.";
            $user['bio'] = $bio;
        } else {
            $error = "Erreur lors de la mise à jour du profil.";
        }
    } elseif (isset($_POST['action']) && $_POST['action'] === 'change_password') {
        $current_password = $_POST['current_password'] ?? '';
        $new_password = $_POST['new_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        
        if (empty($current_password)) {
            $error = "Le mot de passe actuel est requis.";
        } elseif (empty($new_password)) {
            $error = "Le nouveau mot de passe est requis.";
        } elseif (strlen($new_password) < 8) {
            $error = "Le nouveau mot de passe doit contenir au moins 8 caractères.";
        } elseif ($new_password !== $confirm_password) {
            $error = "Les nouveaux mots de passe ne correspondent pas.";
        } else {
            $stored_hash = db_get_value("SELECT password_hash FROM users WHERE id = ?", [$_SESSION['user_id']]);
            
            if (password_verify($current_password, $stored_hash)) {
                $new_hash = password_hash($new_password, PASSWORD_DEFAULT);
                
                $updated = db_update(
                    'users',
                    ['password_hash' => $new_hash],
                    'id = ?',
                    [$_SESSION['user_id']]
                );
                
                if ($updated) {
                    $success = "Mot de passe modifié avec succès.";
                } else {
                    $error = "Erreur lors de la modification du mot de passe.";
                }
            } else {
                $error = "Le mot de passe actuel est incorrect.";
            }
        }
    }
}

$missions_completed = db_get_value(
    "SELECT COUNT(*) FROM user_missions WHERE user_id = ? AND status = 'completed'",
    [$_SESSION['user_id']]
);

$alliances_count = db_get_value(
    "SELECT COUNT(*) FROM alliances 
     WHERE (user1_id = ? OR user2_id = ?) AND status = 'active'",
    [$_SESSION['user_id'], $_SESSION['user_id']]
);

$systems_attacked = db_get_value(
    "SELECT COUNT(DISTINCT country_id) FROM system_combinations WHERE added_by = ?",
    [$_SESSION['user_id']]
);

$travel_stats = db_get_one(
    "SELECT 
        COUNT(*) as total_travels,
        SUM(CASE WHEN transport_type = 'car' THEN 1 ELSE 0 END) as car_travels,
        SUM(CASE WHEN transport_type = 'train' THEN 1 ELSE 0 END) as train_travels,
        SUM(CASE WHEN transport_type = 'plane' THEN 1 ELSE 0 END) as plane_travels,
        SUM(CASE WHEN transport_type = 'boat' THEN 1 ELSE 0 END) as boat_travels
     FROM travels 
     WHERE user_id = ? AND status = 'completed'",
    [$_SESSION['user_id']]
);

$skills = db_get_all(
    "SELECT * FROM skills WHERE user_id = ? ORDER BY level DESC, skill_type",
    [$_SESSION['user_id']]
);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profil | Cergies</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: white;
        }
        .card {
            background-color: white;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
        }
        .progress-bar {
            height: 8px;
            border-radius: 4px;
            background-color: #E5E7EB;
        }
        .progress-value {
            height: 8px;
            border-radius: 4px;
            background-color: #3B82F6;
        }
    </style>
</head>
<body class="bg-white text-gray-800">
    <header class="bg-gray-900 text-white">
        <div class="container mx-auto px-4 py-4 flex justify-between items-center">
            <div class="flex items-center">
                <h1 class="text-2xl font-bold">CERGIES</h1>
            </div>
            <nav>
                <ul class="flex space-x-6">
                    <li><a href="index.php" class="hover:text-blue-300">Accueil</a></li>
                    <li><a href="map.php" class="hover:text-blue-300">Carte</a></li>
                    <li><a href="communication.php" class="hover:text-blue-300">Communications</a></li>
                    <li><a href="system.php" class="hover:text-blue-300">Système</a></li>
                    <li><a href="inventory.php" class="hover:text-blue-300">Inventaire</a></li>
                    <li><a href="missions.php" class="hover:text-blue-300">Missions</a></li>
                    <?php if ($user['level'] >= 10): ?>
                        <li><a href="unions.php" class="hover:text-blue-300">Unions</a></li>
                    <?php endif; ?>
                    <li><a href="profile.php" class="hover:text-blue-300 font-semibold">Profil</a></li>
                    <li><a href="logout.php" class="hover:text-red-300">Déconnexion</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container mx-auto px-4 py-8">
        <h2 class="text-2xl font-bold mb-6">Profil de l'agent</h2>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <div class="lg:col-span-2">
                <div class="card p-6 mb-6">
                    <div class="flex flex-col md:flex-row md:items-center">
                        <div class="flex-shrink-0 mb-4 md:mb-0 md:mr-6">
                            <div class="w-24 h-24 bg-gray-200 rounded-full flex items-center justify-center text-gray-500 text-2xl font-semibold">
                                <?php echo strtoupper(substr($user['username'], 0, 2)); ?>
                            </div>
                        </div>
                        <div>
                            <h3 class="text-xl font-semibold mb-1"><?php echo htmlspecialchars($user['username']); ?></h3>
                            <p class="text-gray-500 mb-2">Niveau <?php echo $user['level']; ?> - <?php echo $level_info['title']; ?></p>
                            <div class="flex items-center space-x-4">
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                    <?php echo htmlspecialchars($country['name']); ?>
                                </span>
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                                    <?php echo htmlspecialchars($user['city']); ?>
                                </span>
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                    <?php
                                        $languages = [
                                            'fr' => 'Français',
                                            'en' => 'Anglais',
                                            'es' => 'Espagnol',
                                            'de' => 'Allemand',
                                            'it' => 'Italien'
                                        ];
                                        echo $languages[$user['language']] ?? $user['language'];
                                    ?>
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card p-6 mb-6">
                    <h3 class="text-lg font-semibold mb-4">Progression</h3>
                    <div class="mb-4">
                        <div class="flex justify-between mb-1">
                            <span class="text-sm font-medium">Expérience: <?php echo $user['experience']; ?> / <?php echo $level_info['max_exp']; ?></span>
                            <span class="text-sm font-medium"><?php echo round(($user['experience'] - $level_info['min_exp']) / ($level_info['max_exp'] - $level_info['min_exp']) * 100); ?>%</span>
                        </div>
                        <div class="progress-bar">
                            <div class="progress-value" style="width: <?php echo ($user['experience'] - $level_info['min_exp']) / ($level_info['max_exp'] - $level_info['min_exp']) * 100; ?>%"></div>
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
                        <div class="bg-gray-50 p-3 rounded-lg text-center">
                            <p class="text-xl font-semibold"><?php echo $missions_completed; ?></p>
                            <p class="text-sm text-gray-500">Missions accomplies</p>
                        </div>
                        <div class="bg-gray-50 p-3 rounded-lg text-center">
                            <p class="text-xl font-semibold"><?php echo $alliances_count; ?></p>
                            <p class="text-sm text-gray-500">Alliances</p>
                        </div>
                        <div class="bg-gray-50 p-3 rounded-lg text-center">
                            <p class="text-xl font-semibold"><?php echo $systems_attacked; ?></p>
                            <p class="text-sm text-gray-500">Systèmes infiltrés</p>
                        </div>
                        <div class="bg-gray-50 p-3 rounded-lg text-center">
                            <p class="text-xl font-semibold"><?php echo $travel_stats['total_travels'] ?? 0; ?></p>
                            <p class="text-sm text-gray-500">Déplacements</p>
                        </div>
                    </div>
                </div>
                
                <div class="card p-6 mb-6">
                    <h3 class="text-lg font-semibold mb-4">Compétences</h3>
                    
                    <?php if (empty($skills)): ?>
                        <p class="text-gray-500 italic">Tu n'as pas encore développé de compétences. Complète des missions et interagis avec le monde pour les améliorer.</p>
                    <?php else: ?>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <?php foreach ($skills as $skill): ?>
                                <div class="border rounded-lg p-3">
                                    <div class="flex justify-between items-center mb-2">
                                        <h4 class="font-medium">
                                            <?php
                                                $skill_names = [
                                                    'hacking' => 'Piratage',
                                                    'combat' => 'Combat',
                                                    'stealth' => 'Discrétion',
                                                    'charisma' => 'Charisme',
                                                    'intelligence' => 'Intelligence',
                                                    'survival' => 'Survie'
                                                ];
                                                echo $skill_names[$skill['skill_type']] ?? $skill['skill_type'];
                                            ?>
                                        </h4>
                                        <span class="text-sm">Niveau <?php echo $skill['level']; ?></span>
                                    </div>
                                    <div class="progress-bar">
                                        <div class="progress-value" style="width: <?php echo min(100, $skill['experience'] / (100 * $skill['level']) * 100); ?>%"></div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="card p-6">
                    <h3 class="text-lg font-semibold mb-4">Statistiques de voyage</h3>
                    
                    <?php if ($travel_stats['total_travels'] > 0): ?>
                        <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
                            <div class="bg-gray-50 p-3 rounded-lg text-center">
                                <p class="text-xl font-semibold"><?php echo $travel_stats['car_travels']; ?></p>
                                <p class="text-sm text-gray-500">Voyages en voiture</p>
                            </div>
                            <div class="bg-gray-50 p-3 rounded-lg text-center">
                                <p class="text-xl font-semibold"><?php echo $travel_stats['train_travels']; ?></p>
                                <p class="text-sm text-gray-500">Voyages en train</p>
                            </div>
                            <div class="bg-gray-50 p-3 rounded-lg text-center">
                                <p class="text-xl font-semibold"><?php echo $travel_stats['plane_travels']; ?></p>
                                <p class="text-sm text-gray-500">Voyages en avion</p>
                            </div>
                            <div class="bg-gray-50 p-3 rounded-lg text-center">
                                <p class="text-xl font-semibold"><?php echo $travel_stats['boat_travels']; ?></p>
                                <p class="text-sm text-gray-500">Voyages en bateau</p>
                            </div>
                        </div>
                    <?php else: ?>
                        <p class="text-gray-500 italic">Tu n'as pas encore effectué de voyages.</p>
                    <?php endif; ?>
                </div>
            </div>
            
            <div>
                <div class="card p-6 mb-6">
                    <h3 class="text-lg font-semibold mb-4">Modifier le profil</h3>
                    
                    <form method="POST" action="profile.php" class="space-y-4">
                        <input type="hidden" name="action" value="update_profile">
                        
                        <div>
                            <label for="bio" class="block text-sm font-medium mb-1">Histoire de ton personnage</label>
                            <textarea id="bio" name="bio" rows="5" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"><?php echo htmlspecialchars($user['bio']); ?></textarea>
                        </div>
                        
                        <div>
                            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded transition-colors">
                                Enregistrer les modifications
                            </button>
                        </div>
                    </form>
                </div>
                
                <div class="card p-6">
                    <h3 class="text-lg font-semibold mb-4">Changer le mot de passe</h3>
                    
                    <form method="POST" action="profile.php" class="space-y-4">
                        <input type="hidden" name="action" value="change_password">
                        
                        <div>
                            <label for="current_password" class="block text-sm font-medium mb-1">Mot de passe actuel</label>
                            <input type="password" id="current_password" name="current_password" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                        </div>
                        
                        <div>
                            <label for="new_password" class="block text-sm font-medium mb-1">Nouveau mot de passe</label>
                            <input type="password" id="new_password" name="new_password" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                        </div>
                        
                        <div>
                            <label for="confirm_password" class="block text-sm font-medium mb-1">Confirmer le nouveau mot de passe</label>
                            <input type="password" id="confirm_password" name="confirm_password" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                        </div>
                        
                        <div>
                            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded transition-colors">
                                Changer le mot de passe
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </main>
</body>
</html>