<?php
require_once 'db.php';
session_start();


$message = '';
$messageType = '';

// Traitement du formulaire
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Validation des données de base
        if (empty($_POST['title'])) {
            throw new Exception('Le titre est obligatoire');
        }

        // Vérification de la vidéo
        if (!isset($_FILES['video']) || $_FILES['video']['error'] !== 0) {
            throw new Exception('La vidéo est obligatoire');
        }

        $videoFile = $_FILES['video'];
        $allowedVideoTypes = ['video/mp4', 'video/webm', 'video/quicktime'];
        
        if (!in_array($videoFile['type'], $allowedVideoTypes)) {
            throw new Exception('Format vidéo non supporté. Utilisez MP4, WEBM ou MOV');
        }

        if ($videoFile['size'] > 100 * 1024 * 1024) {
            throw new Exception('La vidéo est trop volumineuse (max 100MB)');
        }

        // Création des dossiers si nécessaire
        $uploadDirs = ['uploads', 'uploads/videos', 'uploads/thumbnails'];
        foreach ($uploadDirs as $dir) {
            if (!file_exists($dir)) {
                mkdir($dir, 0777, true);
            }
        }

        // Upload de la vidéo
        $videoExtension = pathinfo($videoFile['name'], PATHINFO_EXTENSION);
        $videoNewName = uniqid('video_') . '_' . time() . '.' . $videoExtension;
        $videoPath = 'uploads/videos/' . $videoNewName;
        
        if (!move_uploaded_file($videoFile['tmp_name'], $videoPath)) {
            throw new Exception('Erreur lors de l\'upload de la vidéo');
        }

        // Gestion de la miniature optionnelle
        $thumbnailPath = null;
        if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] === 0) {
            $thumbFile = $_FILES['thumbnail'];
            $allowedImageTypes = ['image/jpeg', 'image/png', 'image/gif'];
            
            if (!in_array($thumbFile['type'], $allowedImageTypes)) {
                throw new Exception('Format d\'image non supporté');
            }

            if ($thumbFile['size'] > 5 * 1024 * 1024) {
                throw new Exception('Image trop volumineuse (max 5MB)');
            }

            $thumbExtension = pathinfo($thumbFile['name'], PATHINFO_EXTENSION);
            $thumbNewName = uniqid('thumb_') . '_' . time() . '.' . $thumbExtension;
            $thumbnailPath = 'uploads/thumbnails/' . $thumbNewName;
            
            if (!move_uploaded_file($thumbFile['tmp_name'], $thumbnailPath)) {
                throw new Exception('Erreur lors de l\'upload de la miniature');
            }
        }

        // Insertion dans la base de données
        $query = "INSERT INTO videos (user_id, video_url, thumbnail_url, title, description) VALUES (?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($query);
        $stmt->bind_param("issss", 
            $_SESSION['user_id'],
            $videoPath,
            $thumbnailPath,
            $_POST['title'],
            $_POST['description']
        );

        if (!$stmt->execute()) {
            throw new Exception('Erreur lors de l\'enregistrement en base de données');
        }

        // Succès
        $message = "Vidéo uploadée avec succès !";
        $messageType = "success";
        
        // Redirection après 2 secondes
        header("refresh:2;url=profile.php?id=" . $_SESSION['user_id']);

    } catch (Exception $e) {
        $message = $e->getMessage();
        $messageType = "error";
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload de vidéo - Hothings</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body class="bg-gray-50">
    <div class="min-h-screen">
        <!-- Navigation -->
        <nav class="bg-white border-b">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div class="flex justify-between h-16">
                    <div class="flex">
                        <a href="/" class="flex items-center">
                            <span class="text-2xl font-bold text-purple-600">Hothings</span>
                        </a>
                    </div>
                </div>
            </div>
        </nav>

        <!-- Contenu principal -->
        <main class="max-w-3xl mx-auto py-12 px-4 sm:px-6 lg:px-8">
            <!-- En-tête -->
            <div class="md:flex md:items-center md:justify-between mb-8">
                <div class="flex-1 min-w-0">
                    <h1 class="text-3xl font-bold leading-tight text-gray-900">Upload de vidéo</h1>
                </div>
            </div>

            <!-- Message de statut -->
            <?php if ($message): ?>
            <div class="mb-6 p-4 rounded-lg <?php echo $messageType === 'error' ? 'bg-red-100 text-red-700' : 'bg-green-100 text-green-700'; ?>">
                <?php echo $message; ?>
            </div>
            <?php endif; ?>

            <!-- Formulaire -->
            <div class="bg-white shadow-xl rounded-lg overflow-hidden">
                <form id="uploadForm" method="POST" enctype="multipart/form-data" class="p-8">
                    <!-- Zone de vidéo -->
                    <div class="mb-8">
                        <div class="space-y-2">
                            <label class="block text-sm font-medium text-gray-700">
                                Votre vidéo
                            </label>
                            <div id="dropZone" class="relative border-2 border-dashed border-gray-300 rounded-lg p-8 text-center hover:border-purple-500 transition-colors cursor-pointer group">
                                <input type="file" 
                                       id="video" 
                                       name="video" 
                                       accept="video/mp4,video/webm,video/quicktime" 
                                       class="absolute inset-0 w-full h-full opacity-0 cursor-pointer" 
                                       required>
                                <div id="dropContent" class="space-y-4">
                                    <i class="fas fa-cloud-upload-alt text-4xl text-gray-400 group-hover:text-purple-500"></i>
                                    <div class="space-y-1">
                                        <p class="text-base text-gray-600">
                                            Glissez votre vidéo ici ou cliquez pour parcourir
                                        </p>
                                        <p class="text-sm text-gray-500">
                                            MP4, WEBM ou MOV jusqu'à 100MB
                                        </p>
                                    </div>
                                </div>
                                <!-- Zone de prévisualisation vidéo -->
                                <div id="videoPreview" class="hidden mt-4">
                                    <video controls class="mx-auto max-h-64 rounded-lg"></video>
                                    <p class="mt-2 text-sm text-gray-600 video-name"></p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Zone de miniature -->
                    <div class="mb-8">
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Miniature de la vidéo
                        </label>
                        <div id="thumbnailDropZone" class="relative border-2 border-dashed border-gray-300 rounded-lg p-6 text-center hover:border-purple-500 transition-colors cursor-pointer group">
                            <input type="file" 
                                   id="thumbnail" 
                                   name="thumbnail" 
                                   accept="image/*" 
                                   class="absolute inset-0 w-full h-full opacity-0 cursor-pointer">
                            <div id="thumbnailDropContent" class="space-y-2">
                                <i class="fas fa-image text-3xl text-gray-400 group-hover:text-purple-500"></i>
                                <p class="text-sm text-gray-500">
                                    PNG, JPG, GIF jusqu'à 5MB
                                </p>
                            </div>
                            <!-- Zone de prévisualisation miniature -->
                            <div id="thumbnailPreview" class="hidden mt-4">
                                <img src="" alt="Prévisualisation" class="mx-auto max-h-48 rounded-lg">
                                <p class="mt-2 text-sm text-gray-600 thumbnail-name"></p>
                            </div>
                        </div>
                    </div>

                    <!-- Informations de la vidéo -->
                    <div class="space-y-6">
                        <div>
                            <label for="title" class="block text-sm font-medium text-gray-700 mb-2">
                                Titre de la vidéo
                            </label>
                            <input type="text" 
                                   id="title" 
                                   name="title" 
                                   required
                                   class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500">
                        </div>

                        <div>
                            <label for="description" class="block text-sm font-medium text-gray-700 mb-2">
                                Description
                            </label>
                            <textarea id="description" 
                                      name="description" 
                                      rows="4"
                                      class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-purple-500"></textarea>
                        </div>
                    </div>

                    <!-- Barre de progression -->
                    <div id="uploadProgress" class="hidden mt-8 space-y-2">
                        <div class="w-full bg-gray-200 rounded-full h-2">
                            <div class="bg-purple-600 h-2 rounded-full transition-all duration-300" style="width: 0%"></div>
                        </div>
                        <div class="flex justify-between text-sm text-gray-600">
                            <span class="progress-text">0%</span>
                            <span class="upload-status"></span>
                        </div>
                    </div>

                    <!-- Message d'erreur -->
                    <div id="errorMessage" class="hidden mt-4 p-4 bg-red-100 border border-red-400 text-red-700 rounded-lg">
                    </div>

                    <!-- Bouton d'envoi -->
                    <div class="mt-8">
                        <button type="submit" 
                                class="w-full bg-purple-600 text-white px-6 py-3 rounded-lg hover:bg-purple-700 focus:outline-none focus:ring-2 focus:ring-purple-500 focus:ring-offset-2 transition-colors">
                            Publier la vidéo
                        </button>
                    </div>
                </form>
            </div>
        </main>
    </div>
    <script>
document.addEventListener('DOMContentLoaded', function() {
    // Éléments du DOM
    const form = document.getElementById('uploadForm');
    const videoInput = document.getElementById('video');
    const thumbnailInput = document.getElementById('thumbnail');
    const videoPreview = document.getElementById('videoPreview');
    const thumbnailPreview = document.getElementById('thumbnailPreview');
    const videoDropZone = document.getElementById('dropZone');
    const thumbnailDropZone = document.getElementById('thumbnailDropZone');
    const uploadProgress = document.getElementById('uploadProgress');
    const progressBar = uploadProgress.querySelector('.bg-purple-600');
    const progressText = uploadProgress.querySelector('.progress-text');
    const uploadStatus = uploadProgress.querySelector('.upload-status');
    const errorMessage = document.getElementById('errorMessage');

    // Configuration du drag & drop
    function setupDragAndDrop(dropZone, input, isVideo = true) {
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, preventDefault, false);
        });

        function preventDefault(e) {
            e.preventDefault();
            e.stopPropagation();
        }

        // Ajout de la classe de surbrillance
        ['dragenter', 'dragover'].forEach(eventName => {
            dropZone.addEventListener(eventName, () => {
                dropZone.classList.add('border-purple-500', 'bg-purple-50');
            });
        });

        // Retrait de la classe de surbrillance
        ['dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, () => {
                dropZone.classList.remove('border-purple-500', 'bg-purple-50');
            });
        });

        // Gestion du drop
        dropZone.addEventListener('drop', function(e) {
            const dt = e.dataTransfer;
            input.files = dt.files;
            handleFile(input.files[0], isVideo);
        });
    }

    // Configuration des zones de drop
    setupDragAndDrop(videoDropZone, videoInput, true);
    setupDragAndDrop(thumbnailDropZone, thumbnailInput, false);

    // Gestion des changements de fichiers
    videoInput.addEventListener('change', (e) => handleFile(e.target.files[0], true));
    thumbnailInput.addEventListener('change', (e) => handleFile(e.target.files[0], false));

    // Fonction de gestion des fichiers
    function handleFile(file, isVideo) {
        if (!file) return;

        // Validation de la taille et du type
        if (isVideo) {
            if (file.size > 100 * 1024 * 1024) {
                showError('La vidéo est trop volumineuse (maximum 100MB)');
                videoInput.value = '';
                return;
            }
            if (!file.type.startsWith('video/')) {
                showError('Veuillez sélectionner un fichier vidéo valide');
                videoInput.value = '';
                return;
            }
            
            // Prévisualisation de la vidéo
            const video = videoPreview.querySelector('video');
            const videoName = videoPreview.querySelector('.video-name');
            video.src = URL.createObjectURL(file);
            videoName.textContent = file.name;
            videoPreview.classList.remove('hidden');
            document.getElementById('dropContent').classList.add('hidden');
            
            // Ajout d'un événement pour libérer la mémoire
            video.onload = () => URL.revokeObjectURL(video.src);
        } else {
            if (file.size > 5 * 1024 * 1024) {
                showError('L\'image est trop volumineuse (maximum 5MB)');
                thumbnailInput.value = '';
                return;
            }
            if (!file.type.startsWith('image/')) {
                showError('Veuillez sélectionner une image valide');
                thumbnailInput.value = '';
                return;
            }
            
            // Prévisualisation de l'image
            const img = thumbnailPreview.querySelector('img');
            const thumbName = thumbnailPreview.querySelector('.thumbnail-name');
            img.src = URL.createObjectURL(file);
            thumbName.textContent = file.name;
            thumbnailPreview.classList.remove('hidden');
            document.getElementById('thumbnailDropContent').classList.add('hidden');
            
            // Ajout d'un événement pour libérer la mémoire
            img.onload = () => URL.revokeObjectURL(img.src);
        }
    }

    // Fonction d'affichage des erreurs
    function showError(message) {
        errorMessage.textContent = message;
        errorMessage.classList.remove('hidden');
        setTimeout(() => {
            errorMessage.classList.add('hidden');
        }, 5000);
    }

    // Gestion de l'envoi du formulaire
    form.addEventListener('submit', function(e) {
        e.preventDefault();

        // Validation des champs requis
        if (!videoInput.files[0]) {
            showError('Veuillez sélectionner une vidéo');
            return;
        }
        if (!form.title.value.trim()) {
            showError('Le titre est obligatoire');
            return;
        }

        // Préparation de l'envoi
        const formData = new FormData(this);
        uploadProgress.classList.remove('hidden');
        progressBar.style.width = '0%';
        progressText.textContent = '0%';
        uploadStatus.textContent = 'Préparation...';

        // Configuration de la requête AJAX
        const xhr = new XMLHttpRequest();
        
        // Gestion de la progression
        xhr.upload.onprogress = function(e) {
            if (e.lengthComputable) {
                const percentComplete = Math.round((e.loaded / e.total) * 100);
                progressBar.style.width = `${percentComplete}%`;
                progressText.textContent = `${percentComplete}%`;
                uploadStatus.textContent = 'Upload en cours...';
            }
        };

        // Gestion de la fin du téléchargement
        xhr.onload = function() {
            if (xhr.status === 200) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.success) {
                        uploadStatus.textContent = 'Upload terminé !';
                        // Redirection après succès
                        window.location.href = response.redirect || `profile.php?id=${response.user_id}`;
                    } else {
                        showError(response.message || 'Une erreur est survenue');
                        uploadProgress.classList.add('hidden');
                    }
                } catch (error) {
                    // Si la réponse n'est pas du JSON, recharger la page
                    window.location.reload();
                }
            } else {
                showError(`Erreur serveur: ${xhr.status}`);
                uploadProgress.classList.add('hidden');
            }
        };

        // Gestion des erreurs réseau
        xhr.onerror = function() {
            showError('Erreur réseau lors de l\'upload');
            uploadProgress.classList.add('hidden');
        };

        // Envoi de la requête
        xhr.open('POST', form.action || window.location.href, true);
        xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');
        xhr.send(formData);
    });

    // Nettoyage des ressources lors du déchargement de la page
    window.addEventListener('beforeunload', function() {
        const video = videoPreview.querySelector('video');
        const img = thumbnailPreview.querySelector('img');
        
        if (video.src) URL.revokeObjectURL(video.src);
        if (img.src) URL.revokeObjectURL(img.src);
    });
});

</script>
</body>
</html>