<!doctype html>
<html lang="fr">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>AWPO Terminal</title>
<script src="https://cdn.tailwindcss.com"></script>
<style>
  body { background: #012a66; font-family: "Courier New", monospace; }
  .xp-shadow { box-shadow: 5px 5px 0 rgba(0,0,0,0.55); border: 2px solid #002a5a; }
  .titlebar { -webkit-user-select: none; user-select: none; }
  .term-scroll { max-height: 320px; overflow:auto; white-space: pre-wrap; }
</style>
</head>
<body class="min-h-screen flex items-center justify-center p-6">

  <div class="w-[720px] xp-shadow bg-gradient-to-b from-[#0b3b8a] to-[#06306a] rounded">
    <div class="titlebar flex items-center gap-3 px-3 py-2 bg-gradient-to-r from-[#002a5a] to-[#003f8f] text-white">
      <div class="flex items-center gap-2">
        <div class="w-8 h-8 bg-white border border-gray-300 flex items-center justify-center text-sm font-bold text-[#003f8f]">AW</div>
        <div class="font-bold">AWPO Terminal</div>
      </div>

      <div class="ml-auto flex gap-2">
        <button id="btnShutdown" class="px-3 py-1 bg-[#f8f9fb] text-[#003f8f] border border-[#002a5a] rounded text-sm">Shutdown</button>
        <button id="btnRestart" class="px-3 py-1 bg-[#f8f9fb] text-[#003f8f] border border-[#002a5a] rounded text-sm">Restart</button>
        <button id="btnLock" class="px-3 py-1 bg-[#f8f9fb] text-[#003f8f] border border-[#002a5a] rounded text-sm">Lock</button>
        <button id="btnClear" class="px-3 py-1 bg-[#f8f9fb] text-[#003f8f] border border-[#002a5a] rounded text-sm">Clear</button>
      </div>
    </div>

    <div class="p-4 bg-[#0b4aa6]">
      <div id="terminal" class="term-scroll bg-[#03254d] text-[#cfe9ff] p-4 rounded min-h-[160px] max-h-[320px]"></div>

      <div class="mt-3 flex gap-2">
        <input id="cmdInput" autofocus type="text" placeholder="Tape une commande (help pour la liste)" 
               class="flex-1 px-3 py-2 bg-[#042f57] text-[#dff4ff] border border-[#053a6d] rounded outline-none" />
        <button id="runBtn" class="px-4 py-2 bg-[#e6f3ff] text-[#003663] border border-[#001f3a] rounded">Run</button>
      </div>

      <div class="mt-2 text-xs text-[#cfe9ff] opacity-80">
        Commandes supportées: <code>help</code>, <code>shutdown</code>, <code>restart</code>, <code>lock</code>, <code>time</code>, <code>echo &lt;text&gt;</code>, <code>cls</code>.
      </div>
    </div>

    <div class="bg-[#06306a] px-3 py-2 text-sm text-[#cfe9ff] rounded-b">AWPO Terminal — local</div>
  </div>

  <!-- Lock overlay -->
  <div id="lockOverlay" class="fixed inset-0 bg-[#00183c]/80 hidden items-center justify-center z-50">
    <div class="bg-white text-[#002a5a] p-6 rounded shadow-xl w-[360px] text-center">
      <div class="font-bold mb-2">Système verrouillé</div>
      <div class="mb-4 text-sm">La session est verrouillée. Clique sur Unlock pour revenir.</div>
      <button id="unlockBtn" class="px-4 py-2 bg-[#e6f3ff] text-[#003663] border border-[#001f3a] rounded">Unlock</button>
    </div>
  </div>

<script>
/* Terminal logique minimale */
const terminal = document.getElementById('terminal');
const input = document.getElementById('cmdInput');
const runBtn = document.getElementById('runBtn');
const btnShutdown = document.getElementById('btnShutdown');
const btnRestart = document.getElementById('btnRestart');
const btnLock = document.getElementById('btnLock');
const btnClear = document.getElementById('btnClear');
const lockOverlay = document.getElementById('lockOverlay');
const unlockBtn = document.getElementById('unlockBtn');

function appendLine(text, cls = '') {
  const el = document.createElement('div');
  el.className = cls;
  el.textContent = text;
  terminal.appendChild(el);
  terminal.scrollTop = terminal.scrollHeight;
}

function prompt() {
  appendLine('> ', 'text-[#9fe0ff]');
}

function listHelp() {
  appendLine('help        : Affiche les commandes');
  appendLine('shutdown    : Ferme l\'onglet (tentative)');
  appendLine('restart     : Recharge la page');
  appendLine('lock        : Verrouille la session');
  appendLine('time        : Affiche l\'heure');
  appendLine('echo <txt>  : Écrit le texte');
  appendLine('cls         : Vide le terminal');
}

function tryClose() {
  // Tentative normale puis fallback (peut être bloqué par le navigateur)
  try {
    window.close();
    // fallback: navigate to about:blank then try close
    setTimeout(()=> {
      try {
        window.open('','_self');
        window.close();
      } catch(e) {}
    }, 200);
  } catch(e){}
}

function runCommand(raw) {
  const cmd = raw.trim();
  if(!cmd) return;
  appendLine('> ' + cmd);
  const [c, ...args] = cmd.split(' ');
  switch(c.toLowerCase()) {
    case 'help':
      listHelp(); break;
    case 'shutdown':
      appendLine('Shutting down... closing tab.');
      // slight delay to let user see message
      setTimeout(tryClose, 400);
      break;
    case 'restart':
      appendLine('Restarting...');
      setTimeout(()=> location.reload(), 250);
      break;
    case 'lock':
      appendLine('Locking session...');
      lockOverlay.classList.remove('hidden');
      lockOverlay.classList.add('flex');
      break;
    case 'time':
      appendLine(new Date().toLocaleString());
      break;
    case 'echo':
      appendLine(args.join(' '));
      break;
    case 'cls':
    case 'clear':
      terminal.innerHTML = '';
      break;
    default:
      appendLine('Commande inconnue. tape "help" pour la liste.');
  }
}

/* Événements */
runBtn.addEventListener('click', ()=>{
  runCommand(input.value);
  input.value = '';
  input.focus();
});
input.addEventListener('keydown', (e)=>{
  if(e.key === 'Enter'){ runBtn.click(); e.preventDefault(); }
});
btnShutdown.addEventListener('click', ()=>{
  appendLine('Shutdown demandé (bouton).');
  setTimeout(tryClose, 300);
});
btnRestart.addEventListener('click', ()=>{
  appendLine('Restart demandé (bouton).');
  setTimeout(()=> location.reload(), 250);
});
btnLock.addEventListener('click', ()=>{
  appendLine('Lock demandé (bouton).');
  lockOverlay.classList.remove('hidden'); lockOverlay.classList.add('flex');
});
btnClear.addEventListener('click', ()=>{ terminal.innerHTML = ''; });
unlockBtn.addEventListener('click', ()=>{ lockOverlay.classList.add('hidden'); lockOverlay.classList.remove('flex'); appendLine('Session déverrouillée.'); });

/* Démarrage - message initial */
appendLine('AWPO Terminal prêt. Tape "help" pour commencer.');
appendLine('---');
</script>

</body>
</html>
