let liveUpdateInterval;
let currentTab = 'notifications';

document.addEventListener('DOMContentLoaded', function() {
    initTabs();
    startLiveUpdates();
    setupForms();
});

function initTabs() {
    const tabs = document.querySelectorAll('.nav-tab');
    const contents = document.querySelectorAll('.tab-content');
    
    tabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const target = this.dataset.tab;
            
            tabs.forEach(t => t.classList.remove('active'));
            contents.forEach(c => c.classList.remove('active'));
            
            this.classList.add('active');
            document.getElementById(target).classList.add('active');
            
            currentTab = target;
        });
    });
}

function startLiveUpdates() {
    updateLiveData();
    liveUpdateInterval = setInterval(updateLiveData, 1000);
}

async function updateLiveData() {
    try {
        const response = await fetch('../api/live-data.php');
        const data = await response.json();
        
        if (data.success) {
            updateCounters(data.counters);
            updateSystemStatus(data.system);
            updateMessages(data.messages);
            updateShopStatus(data.shop_updates);
        }
    } catch (error) {
        console.error('Erreur mise à jour live:', error);
    }
}

function updateCounters(counters) {
    const notifCounter = document.getElementById('notif-counter');
    const emailCounter = document.getElementById('email-counter');
    
    if (notifCounter) {
        notifCounter.textContent = counters.notifications_remaining;
        notifCounter.className = counters.notifications_remaining > 0 ? 'counter-badge text-green-500' : 'counter-badge text-red-500';
    }
    
    if (emailCounter) {
        emailCounter.textContent = counters.emails_remaining;
        emailCounter.className = counters.emails_remaining > 0 ? 'counter-badge text-green-500' : 'counter-badge text-red-500';
    }
    
    toggleFormAccess('notification-form', counters.notifications_remaining > 0 && counters.in_time_window);
    toggleFormAccess('email-form', counters.emails_remaining > 0 && counters.in_time_window && !counters.email_cooldown);
}

function updateSystemStatus(system) {
    const iosStatus = document.getElementById('ios-status');
    const androidStatus = document.getElementById('android-status');
    
    if (iosStatus) {
        iosStatus.className = system.ios_enabled ? 'status-online w-3 h-3 rounded-full' : 'status-offline w-3 h-3 rounded-full';
    }
    
    if (androidStatus) {
        androidStatus.className = system.android_enabled ? 'status-online w-3 h-3 rounded-full' : 'status-offline w-3 h-3 rounded-full';
    }
}

function updateMessages(messages) {
    const messagesBadge = document.getElementById('messages-badge');
    if (messagesBadge) {
        if (messages.unread > 0) {
            messagesBadge.textContent = messages.unread;
            messagesBadge.classList.remove('hidden');
        } else {
            messagesBadge.classList.add('hidden');
        }
    }
}

function updateShopStatus(shopUpdates) {
    const pendingBadge = document.getElementById('pending-badge');
    if (pendingBadge) {
        if (shopUpdates.pending > 0) {
            pendingBadge.textContent = shopUpdates.pending;
            pendingBadge.classList.remove('hidden');
        } else {
            pendingBadge.classList.add('hidden');
        }
    }
}

function toggleFormAccess(formId, enabled) {
    const form = document.getElementById(formId);
    if (form) {
        if (enabled) {
            form.classList.remove('form-disabled');
        } else {
            form.classList.add('form-disabled');
        }
    }
}

function setupForms() {
    const forms = document.querySelectorAll('form[data-ajax]');
    forms.forEach(form => {
        form.addEventListener('submit', handleAjaxSubmit);
    });
}

async function handleAjaxSubmit(e) {
    e.preventDefault();
    const form = e.target;
    const formData = new FormData(form);
    const submitBtn = form.querySelector('button[type="submit"]');
    
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<div class="loading-spinner w-4 h-4 border-2 border-current border-t-transparent rounded-full mx-auto"></div>';
    
    try {
        const response = await fetch(form.action, {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            showNotification(result.message, 'success');
            form.reset();
        } else {
            showNotification(result.message, 'error');
        }
    } catch (error) {
        showNotification('Erreur de connexion', 'error');
    } finally {
        submitBtn.disabled = false;
        submitBtn.innerHTML = submitBtn.dataset.originalText || 'Envoyer';
    }
}

function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `notification-alert fixed top-4 right-4 p-4 rounded-lg text-white z-50 ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.remove();
    }, 3000);
}

async function loadShopInfo() {
    try {
        const response = await fetch('../api/get-shop-info.php');
        const data = await response.json();
        
        if (data.success && data.shop) {
            const shop = data.shop;
            
            document.getElementById('current-name').textContent = shop.name || 'Non défini';
            document.getElementById('current-description').textContent = shop.description || 'Non définie';
            document.getElementById('current-address').textContent = shop.address || 'Non définie';
            document.getElementById('current-phone').textContent = shop.phone || 'Non défini';
            
            document.getElementById('shop-loading').classList.add('hidden');
            document.getElementById('shop-form').classList.remove('hidden');
        } else {
            document.getElementById('shop-loading').innerHTML = '<p class="text-red-500">Erreur lors du chargement des informations</p>';
        }
    } catch (error) {
        document.getElementById('shop-loading').innerHTML = '<p class="text-red-500">Erreur de connexion</p>';
    }
}