admin.php
<?php
session_start();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $data = json_decode(file_get_contents('stock.json'), true);
    
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'add') {
            $stops = [];
            foreach ($_POST['stop_station'] as $key => $station) {
                if (!empty($station) && !empty($_POST['stop_time'][$key])) {
                    $stops[] = [
                        'station' => $station,
                        'time' => $_POST['stop_time'][$key]
                    ];
                }
            }

            $newTrain = [
                'id' => uniqid(),
                'line' => $_POST['line'],
                'type' => $_POST['type'],
                'destination' => $_POST['destination'],
                'arrival' => $_POST['arrival'],
                'platform' => $_POST['platform'],
                'status' => 'on-time',
                'info' => $_POST['info'],
                'stops' => $stops,
                'composition' => [
                    'first' => isset($_POST['first']),
                    'second' => isset($_POST['second']),
                    'bar' => isset($_POST['bar']),
                    'carriages' => $_POST['carriages']
                ]
            ];
            
            $data['trains'][] = $newTrain;

        } elseif ($_POST['action'] === 'update') {
            foreach ($data['trains'] as &$train) {
                if ($train['id'] === $_POST['id']) {
                    $train['status'] = $_POST['status'];
                    $train['platform'] = $_POST['platform'];
                    $train['info'] = $_POST['info'];
                    break;
                }
            }
        } elseif ($_POST['action'] === 'delete') {
            $data['trains'] = array_filter($data['trains'], function($train) {
                return $train['id'] !== $_POST['id'];
            });
            $data['trains'] = array_values($data['trains']);
        }
    }
    
    file_put_contents('stock.json', json_encode($data, JSON_PRETTY_PRINT));
    header('Location: admin.php');
    exit;
}

$data = json_decode(file_get_contents('stock.json'), true);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Administration | <?= htmlspecialchars($data['settings']['station']['name']) ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-black text-white">
    <div class="container mx-auto p-4">
        <!-- Header -->
        <header class="flex justify-between items-center mb-8">
            <h1 class="text-2xl font-light">Administration</h1>
            <div class="flex gap-4">
                <div id="clock" class="text-xl font-mono"></div>
                <a href="index.php" class="text-blue-400 hover:text-blue-300">← Affichage</a>
            </div>
        </header>

        <!-- Formulaire d'ajout -->
        <div x-data="{ open: false }" class="mb-8">
            <button @click="open = !open" 
                class="bg-blue-500 px-4 py-2 rounded hover:bg-blue-600 transition-colors mb-4">
                + Nouveau train
            </button>

            <form x-show="open" method="POST" class="bg-gray-900 p-6 rounded-lg space-y-6" 
                x-transition:enter="transition ease-out duration-300"
                x-transition:enter-start="opacity-0 transform -translate-y-4"
                x-transition:enter-end="opacity-100 transform translate-y-0">
                <input type="hidden" name="action" value="add">

                <!-- Infos principales -->
                <div class="grid grid-cols-4 gap-4">
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">Type</label>
                        <select name="type" class="w-full bg-gray-800 p-2 rounded border border-gray-700 focus:border-blue-500 focus:outline-none">
                            <option>TGV</option>
                            <option>TER</option>
                            <option>Intercités</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">Numéro</label>
                        <input type="text" name="line" maxlength="5" class="w-full bg-gray-800 p-2 rounded border border-gray-700 focus:border-blue-500 focus:outline-none" required>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">Destination</label>
                        <input type="text" name="destination" class="w-full bg-gray-800 p-2 rounded border border-gray-700 focus:border-blue-500 focus:outline-none" required>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">Voie</label>
                        <input type="number" name="platform" min="1" max="99" class="w-full bg-gray-800 p-2 rounded border border-gray-700 focus:border-blue-500 focus:outline-none" required>
                    </div>
                </div>

                <!-- Horaire et composition -->
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">Heure d'arrivée</label>
                        <input type="datetime-local" name="arrival" class="w-full bg-gray-800 p-2 rounded border border-gray-700 focus:border-blue-500 focus:outline-none" required>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">Composition</label>
                        <div class="flex gap-4 items-center h-[38px]">
                            <label class="inline-flex items-center">
                                <input type="checkbox" name="first" class="form-checkbox bg-gray-800 border-gray-700">
                                <span class="ml-2">1ère</span>
                            </label>
                            <label class="inline-flex items-center">
                                <input type="checkbox" name="second" class="form-checkbox bg-gray-800 border-gray-700">
                                <span class="ml-2">2nde</span>
                            </label>
                            <label class="inline-flex items-center">
                                <input type="checkbox" name="bar" class="form-checkbox bg-gray-800 border-gray-700">
                                <span class="ml-2">Bar</span>
                            </label>
                            <input type="text" name="carriages" placeholder="Nb voitures" class="w-32 bg-gray-800 p-2 rounded border border-gray-700 focus:border-blue-500 focus:outline-none">
                        </div>
                    </div>
                </div>

                <!-- Arrêts -->
                <div>
                    <label class="block text-sm text-gray-400 mb-2">Arrêts intermédiaires</label>
                    <div class="space-y-2" id="stopsContainer">
                        <template x-for="i in 5" :key="i">
                            <div class="grid grid-cols-2 gap-4">
                                <input type="text" :name="'stop_station[]'" placeholder="Gare" 
                                    class="bg-gray-800 p-2 rounded border border-gray-700 focus:border-blue-500 focus:outline-none">
                                <input type="time" :name="'stop_time[]'" 
                                    class="bg-gray-800 p-2 rounded border border-gray-700 focus:border-blue-500 focus:outline-none">
                            </div>
                        </template>
                    </div>
                </div>

                <!-- Information -->
                <div>
                    <label class="block text-sm text-gray-400 mb-1">Information</label>
                    <input type="text" name="info" class="w-full bg-gray-800 p-2 rounded border border-gray-700 focus:border-blue-500 focus:outline-none">
                </div>

                <button type="submit" class="bg-green-600 px-6 py-2 rounded hover:bg-green-700 transition-colors">
                    Ajouter
                </button>
            </form>
        </div>

        <!-- Liste des trains -->
        <div class="space-y-4">
            <?php foreach ($data['trains'] as $train): 
                $arrival = new DateTime($train['arrival']);
                $now = new DateTime();
                $timeLeft = $arrival->getTimestamp() - $now->getTimestamp();
                if ($timeLeft < -1200) continue; // Ne pas afficher les trains partis depuis plus de 20min
            ?>
            <div class="bg-gray-900 p-4 rounded-lg <?= $timeLeft < -600 ? 'opacity-50' : '' ?>">
                <div class="flex items-center justify-between">
                    <div class="flex items-start gap-6">
                        <div>
                            <div class="px-3 py-1 bg-blue-900 rounded text-sm">
                                <?= htmlspecialchars($train['type']) ?> <?= htmlspecialchars($train['line']) ?>
                            </div>
                            <div class="mt-1 text-sm text-gray-500">
                                <?= $arrival->format('H:i') ?>
                            </div>
                        </div>
                        <div>
                            <div class="font-light"><?= htmlspecialchars($train['destination']) ?></div>
                            <div class="text-sm text-gray-500 mt-1">
                                <?= count($train['stops']) ?> arrêts • 
                                <?= $train['composition']['carriages'] ?> •
                                <?= implode(', ', array_filter([
                                    $train['composition']['first'] ? '1ère' : '',
                                    $train['composition']['second'] ? '2nde' : '',
                                    $train['composition']['bar'] ? 'Bar' : ''
                                ])) ?>
                            </div>
                            <?php if (!empty($train['stops'])): ?>
                            <div class="text-sm text-gray-500 mt-1">
                                <?= implode(' • ', array_map(function($stop) {
                                    return $stop['station'] . ' (' . $stop['time'] . ')';
                                }, $train['stops'])) ?>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div class="flex items-start gap-4">
                        <form method="POST" class="flex gap-2">
                            <input type="hidden" name="action" value="update">
                            <input type="hidden" name="id" value="<?= $train['id'] ?>">
                            
                            <select name="status" class="bg-gray-800 px-3 py-1 rounded text-sm border border-gray-700">
                                <option value="on-time" <?= $train['status'] === 'on-time' ? 'selected' : '' ?>>À l'heure</option>
                                <option value="delayed" <?= $train['status'] === 'delayed' ? 'selected' : '' ?>>Retardé</option>
                                <option value="cancelled" <?= $train['status'] === 'cancelled' ? 'selected' : '' ?>>Supprimé</option>
                            </select>
                            
                            <input type="number" name="platform" value="<?= htmlspecialchars($train['platform']) ?>" 
                                min="1" max="99" class="w-16 bg-gray-800 px-3 py-1 rounded text-sm border border-gray-700">
                            
                            <input type="text" name="info" value="<?= htmlspecialchars($train['info'] ?? '') ?>" 
                                placeholder="Information" class="w-48 bg-gray-800 px-3 py-1 rounded text-sm border border-gray-700">
                            
                            <button type="submit" class="bg-blue-600 px-3 py-1 rounded text-sm hover:bg-blue-700">
                                Maj
                            </button>
                        </form>
                        
                        <form method="POST" class="flex" onsubmit="return confirm('Supprimer ce train ?')">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="id" value="<?= $train['id'] ?>">
                            <button type="submit" class="bg-red-600 px-3 py-1 rounded text-sm hover:bg-red-700">×</button>
                        </form>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>

    <script>
    // Mise à jour de l'heure
    function updateClock() {
        const now = new Date();
        document.getElementById('clock').textContent = now.toLocaleTimeString('fr-FR', {
            hour: '2-digit',
            minute: '2-digit',
            second: '2-digit'
        });
    }

    // Initialisation de l'heure actuelle dans les champs datetime-local
    document.querySelectorAll('input[type="datetime-local"]').forEach(input => {
        if (!input.value) {
            const now = new Date();
            now.setMinutes(now.getMinutes() - now.getTimezoneOffset());
            input.value = now.toISOString().slice(0,16);
        }
    });

    updateClock();
    setInterval(updateClock, 1000);
    </script>
</body>
</html>