<?php 
require_once 'config.php';

$query = $pdo->query("
   SELECT 
       c.*,
       u.username as author_name,
       COUNT(DISTINCT CASE WHEN i.type = 'view' THEN i.id END) as view_count,
       COUNT(DISTINCT CASE WHEN i.type = 'like' THEN i.id END) as like_count,
       COUNT(DISTINCT CASE WHEN i.type = 'dislike' THEN i.id END) as dislike_count,
       COUNT(DISTINCT CASE WHEN i.type = 'comment' THEN i.id END) as comment_count
   FROM content c
   LEFT JOIN users u ON c.author_id = u.id
   LEFT JOIN interactions i ON c.id = i.content_id
   GROUP BY c.id
   ORDER BY c.created_at DESC
");

$videos = $query->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
   <meta charset="UTF-8">
   <meta name="viewport" content="width=device-width, initial-scale=1.0">
   <title>ForMore</title>
   <script src="https://cdn.tailwindcss.com"></script>
   <link rel="preconnect" href="https://fonts.googleapis.com">
   <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
   <link href="https://fonts.googleapis.com/css2?family=Anton&family=Titillium+Web:ital,wght@0,200;0,300;0,400;0,600;0,700;0,900;1,200;1,300;1,400;1,600;1,700&display=swap" rel="stylesheet">
</head>
<style>
   .formore {
       font-family: 'Anton', sans-serif;
       letter-spacing: 2px;
       text-transform: uppercase;
       background: linear-gradient(to right, #3b82f6, #1d4ed8);
       -webkit-background-clip: text;
       -webkit-text-fill-color: transparent;
       text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.1);
   }
   .card-animation {
       transform: translateY(20px);
       opacity: 0;
       animation: cardAppear 0.5s ease forwards;
   }
   @keyframes cardAppear {
       to {
           transform: translateY(0);
           opacity: 1;
       }
   }
   .modal-animation {
       animation: modalAppear 0.3s ease-out;
   }
   @keyframes modalAppear {
       from {
           opacity: 0;
           transform: scale(0.9);
       }
       to {
           opacity: 1;
           transform: scale(1);
       }
   }
   body {
       font-family: 'Titillium Web', sans-serif;
   }
   .video-card {
       transition: all 0.3s ease;
   }
   .video-card:hover {
       transform: translateY(-5px);
       box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
   }
</style>
<body class="bg-gradient-to-br from-black to-blue-900 min-h-screen text-white p-8">
   <nav class="flex justify-between items-center mb-12 px-4">
       <h1 class="text-4xl formore">ForMore</h1>
       <div class="flex items-center gap-6">
           <?php if(isset($_SESSION['user_id'])): ?>
               <span class="text-gray-300">Welcome, <?= htmlspecialchars($_SESSION['username']) ?></span>
               <a href="profile.php" class="text-blue-400 hover:text-blue-300 transition-colors">Profile</a>
               <a href="upload.php" class="bg-blue-600 hover:bg-blue-700 px-4 py-2 rounded-lg transition-colors">Upload</a>
               <a href="logout.php" class="text-red-400 hover:text-red-300 transition-colors">Logout</a>
           <?php else: ?>
               <a href="login.php" class="text-blue-400 hover:text-blue-300 transition-colors">Login</a>
               <a href="signup.php" class="bg-blue-600 hover:bg-blue-700 px-4 py-2 rounded-lg transition-colors">Sign Up</a>
           <?php endif; ?>
       </div>
   </nav>

   <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-8 px-4">
       <?php foreach($videos as $index => $video): ?>
           <div class="video-card bg-gray-800 rounded-xl overflow-hidden cursor-pointer card-animation" style="animation-delay: <?= $index * 0.1 ?>s" onclick="openModal(<?= $video['id'] ?>)">
               <div class="relative h-48 bg-cover bg-center" style="background-image: url('<?= htmlspecialchars($video['picture_cover']) ?>');">
                   <div class="absolute bottom-2 right-2 bg-black bg-opacity-70 px-3 py-1 rounded-lg text-sm">
                       <?= htmlspecialchars($video['time']) ?>
                   </div>
               </div>
               <div class="p-4">
                   <h2 class="text-xl font-bold mb-2 line-clamp-1"><?= htmlspecialchars($video['name']) ?></h2>
                   <p class="text-gray-400 mb-3 text-sm line-clamp-2"><?= htmlspecialchars($video['description']) ?></p>
                   <div class="flex justify-between items-center">
                       <span class="text-sm text-blue-400"><?= htmlspecialchars($video['author_name']) ?></span>
                       <div class="flex items-center gap-3 text-sm text-gray-400">
                           <span><?= number_format($video['view_count']) ?> views</span>
                       </div>
                   </div>
                   <div class="flex flex-wrap gap-2 mt-3">
                       <?php foreach(explode(',', $video['tags']) as $tag): ?>
                           <span class="bg-blue-900 px-3 py-1 rounded-full text-xs"><?= htmlspecialchars(trim($tag)) ?></span>
                       <?php endforeach; ?>
                   </div>
               </div>
           </div>
       <?php endforeach; ?>
   </div>

   <div id="videoModal" class="hidden fixed inset-0 bg-black bg-opacity-95 backdrop-blur-sm flex items-center justify-center p-4 z-50">
       <div class="bg-gray-800 rounded-xl w-full max-w-4xl modal-animation">
           <div id="modalContent" class="p-6">
               <div class="flex justify-end mb-4">
                   <button onclick="closeModal()" class="text-gray-400 hover:text-white transition-colors text-xl">&times; Close</button>
               </div>
               <div id="videoContainer" class="aspect-w-16 aspect-h-9 mb-4 rounded-lg overflow-hidden">
                   <div class="flex items-center justify-center h-full bg-gray-900">
                       <div class="animate-pulse">Loading...</div>
                   </div>
               </div>
               <div id="videoInfo"></div>
           </div>
       </div>
   </div>

   <script>
       function openModal(videoId) {
           document.getElementById('videoModal').classList.remove('hidden');
           document.body.style.overflow = 'hidden';
           fetch('get_video.php?id=' + videoId)
               .then(response => response.text())
               .then(html => {
                   document.getElementById('videoContainer').innerHTML = html;
               });
       }

       function closeModal() {
           document.getElementById('videoModal').classList.add('hidden');
           document.body.style.overflow = 'auto';
           document.getElementById('videoContainer').innerHTML = '<div class="flex items-center justify-center h-full bg-gray-900"><div class="animate-pulse">Loading...</div></div>';
       }
   </script>
</body>
</html>