document.addEventListener('DOMContentLoaded', () => {
    const app = {
        map: null,
        directionsService: null,
        directionsRenderer: null,
        markers: {},
        userMarker: null,
        tempMarker: null,
        selectedLightId: null,
        userWatchId: null,
        refreshTimerId: null,
        rafId: null,
        lastTimestamp: 0,
        currentRoute: null,
        isNavigating: false,
        geocoder: null,
        settings: {
            darkMode: false,
            notifications: true,
            autoRefresh: true,
            predictions: true,
            autoMeasure: true,
            defaultZoom: 15,
            mapStyle: 'satellite'
        },
        measure: {
            timer: 0,
            startTime: 0,
            rafId: null,
            mode: null,
            lightId: null
        },
        navigation: {
            startPosition: null,
            endPosition: null,
            route: null,
            coordinates: [],
            lightsOnRoute: [],
            travelMode: 'DRIVING',
            distance: null,
            duration: null,
            startTime: null,
            estimatedArrival: null,
            currentStep: 0,
            lastStop: null,
            stopDuration: 0,
            currentSpeed: 0,
            lastPosition: null
        },
        state: {
            selectingLocation: false,
            currentSlide: 1,
            routePanelOpen: false,
            accountPanelOpen: false
        }
    };

    const dom = {
        map: document.getElementById('map'),
        sidebar: document.querySelector('.sidebar'),
        lightsList: document.getElementById('lightsList'),
        loadingLights: document.getElementById('loadingLights'),
        noLightsMessage: document.getElementById('noLightsMessage'),
        lightPopup: document.getElementById('lightPopup'),
        popupTitle: document.getElementById('popupTitle'),
        popupContent: document.getElementById('popupContent'),
        popupStatus: document.getElementById('popupStatus'),
        closePopup: document.getElementById('closePopup'),
        popupNavigate: document.getElementById('popupNavigate'),
        popupMeasure: document.getElementById('popupMeasure'),
        routePlannerPanel: document.getElementById('routePlannerPanel'),
        directionsPanel: document.getElementById('directionsPanel'),
        directionsSteps: document.getElementById('directionsSteps'),
        routeDetails: document.getElementById('routeDetails'),
        routeTime: document.getElementById('routeTime'),
        routeDistance: document.getElementById('routeDistance'),
        routeLights: document.getElementById('routeLights'),
        arrivalTime: document.getElementById('arrivalTime'),
        closeRoutePanel: document.getElementById('closeRoutePanel'),
        calculateRoute: document.getElementById('calculateRoute'),
        startNavigation: document.getElementById('startNavigation'),
        navigationPanel: document.getElementById('navigationPanel'),
        accountPanel: document.getElementById('accountPanel'),
        modals: {
            add: document.getElementById('addModal'),
            measure: document.getElementById('measureModal'),
            settings: document.getElementById('settingsModal'),
            login: document.getElementById('loginModal'),
            welcome: document.getElementById('welcomeModal')
        },
        overlays: document.querySelectorAll('.modal-overlay'),
        modalBodies: document.querySelectorAll('.modal-body'),
        draggableHandles: document.querySelectorAll('.draggable-handle'),
        closeButtons: document.querySelectorAll('.close-modal'),
        loadingScreen: document.getElementById('loadingScreen'),
        buttons: {
            menuToggle: document.getElementById('menuToggle'),
            headerLocate: document.getElementById('headerLocateBtn'),
            headerAdd: document.getElementById('headerAddBtn'),
            account: document.getElementById('accountBtn'),
            route: document.getElementById('routeBtn'),
            refresh: document.getElementById('refreshBtn'),
            settings: document.getElementById('settingsBtn'),
            nearby: document.getElementById('nearbyBtn'),
            routes: document.getElementById('routesBtn'),
            skipWelcome: document.getElementById('skipWelcome'),
            nextSlide: document.getElementById('nextSlide'),
            resetApp: document.getElementById('resetAppBtn'),
            measureRed: document.getElementById('measureRedBtn'),
            measureGreen: document.getElementById('measureGreenBtn'),
            startTimer: document.getElementById('startTimer'),
            stopTimer: document.getElementById('stopTimer'),
            saveTimer: document.getElementById('saveTimer'),
            routeTypeCar: document.getElementById('routeTypeCar'),
            routeTypeBike: document.getElementById('routeTypeBike'),
            routeTypeWalk: document.getElementById('routeTypeWalk'),
            endNavigation: document.getElementById('endNavigation'),
            accountLogout: document.getElementById('accountLogoutBtn'),
            accountLogin: document.getElementById('accountLoginBtn'),
            accountSettings: document.getElementById('accountSettings'),
            accountRoutes: document.getElementById('accountRoutes'),
            accountLights: document.getElementById('accountLights'),
            changePassword: document.getElementById('changePasswordBtn'),
            manageAccount: document.getElementById('manageAccountBtn')
        },
        form: {
            addLight: document.getElementById('addLightForm'),
            searchInput: document.getElementById('searchInput'),
            startLocation: document.getElementById('startLocation'),
            endLocation: document.getElementById('endLocation'),
            login: document.getElementById('loginForm'),
            register: document.getElementById('registerForm')
        },
        measure: {
            title: document.getElementById('measureTitle'),
            status: document.getElementById('measureStatus'),
            container: document.getElementById('timerContainer'),
            instructions: document.getElementById('timerInstructions'),
            display: document.getElementById('timerDisplay'),
            result: document.getElementById('measureResult')
        },
        welcome: {
            slides: document.querySelectorAll('.onboarding-slide'),
            dots: document.querySelectorAll('.onboarding-dot')
        },
        settings: {
            darkMode: document.getElementById('darkModeToggle'),
            notifications: document.getElementById('notificationsToggle'),
            autoRefresh: document.getElementById('autoRefreshToggle'),
            predictions: document.getElementById('predictionToggle'),
            autoMeasure: document.getElementById('autoMeasureToggle'),
            defaultZoom: document.getElementById('defaultZoom'),
            rememberLogin: document.getElementById('rememberLoginToggle')
        },
        auth: {
            showLogin: document.getElementById('showLoginForm'),
            showRegister: document.getElementById('showRegisterForm'),
            authTitle: document.getElementById('authTitle'),
            login: document.getElementById('loginBtn')
        },
        navigation: {
            nextDirection: document.getElementById('nextDirection'),
            nextStreet: document.getElementById('nextStreet'),
            nextDistance: document.getElementById('nextDistance'),
            etaText: document.getElementById('etaText'),
            currentSpeed: document.getElementById('currentSpeed'),
            totalDistance: document.getElementById('totalDistance'),
            totalTime: document.getElementById('totalTime')
        }
    };

    init();

    function init() {
        loadSettings();
        initMap();
        bindEvents();
        setupDraggableModals();
        checkFirstVisit();
        checkLoggedInUser();
        
        setTimeout(() => {
            dom.loadingScreen.style.opacity = '0';
            setTimeout(() => {
                dom.loadingScreen.style.display = 'none';
            }, 500);
        }, 800);
    }

    function loadSettings() {
        try {
            const saved = localStorage.getItem('trafficLightSettings');
            if (saved) {
                const parsed = JSON.parse(saved);
                app.settings = { ...app.settings, ...parsed };
            } else {
                app.settings.mapStyle = 'satellite';
            }
            applySettings();
        } catch (e) {
            console.error('Error loading settings');
            app.settings.mapStyle = 'satellite';
            applySettings();
        }
    }

    function applySettings() {
        if (dom.settings.darkMode) dom.settings.darkMode.checked = app.settings.darkMode;
        if (dom.settings.notifications) dom.settings.notifications.checked = app.settings.notifications;
        if (dom.settings.autoRefresh) dom.settings.autoRefresh.checked = app.settings.autoRefresh;
        if (dom.settings.predictions) dom.settings.predictions.checked = app.settings.predictions;
        if (dom.settings.autoMeasure) dom.settings.autoMeasure.checked = app.settings.autoMeasure;
        if (dom.settings.defaultZoom) dom.settings.defaultZoom.value = app.settings.defaultZoom;
        
        if (app.settings.darkMode) {
            document.documentElement.classList.add('dark');
            document.body.classList.add('dark-mode');
        }
    }

    function saveSettings() {
        if (dom.settings.darkMode) app.settings.darkMode = dom.settings.darkMode.checked;
        if (dom.settings.notifications) app.settings.notifications = dom.settings.notifications.checked;
        if (dom.settings.autoRefresh) app.settings.autoRefresh = dom.settings.autoRefresh.checked;
        if (dom.settings.predictions) app.settings.predictions = dom.settings.predictions.checked;
        if (dom.settings.autoMeasure) app.settings.autoMeasure = dom.settings.autoMeasure.checked;
        if (dom.settings.defaultZoom) app.settings.defaultZoom = dom.settings.defaultZoom.value;
        
        if (dom.settings.rememberLogin && dom.settings.rememberLogin.checked) {
            app.settings.rememberLogin = dom.settings.rememberLogin.checked;
        }
        
        try {
            localStorage.setItem('trafficLightSettings', JSON.stringify(app.settings));
        } catch (e) {
            console.error('Error saving settings');
        }
    }

    function checkFirstVisit() {
        if (!localStorage.getItem('trafficLightAppVisited')) {
            dom.modals.welcome.style.display = 'flex';
            localStorage.setItem('trafficLightAppVisited', 'true');
        } else {
            dom.modals.welcome.style.display = 'none';
        }
        
        Object.values(dom.modals).forEach(modal => {
            if (modal !== dom.modals.welcome) modal.style.display = 'none';
        });
    }
    
    function checkLoggedInUser() {
        const userId = getCookie('user_id');
        const username = getCookie('username');
        
        if (userId && username) {
            fetch('db.php?action=autoLogin', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    user_id: userId,
                    username: username
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    updateUIForLoggedInUser(data.user);
                }
            })
            .catch(error => console.error('Auto login error:', error));
        }
    }
    
    function getCookie(name) {
        const value = `; ${document.cookie}`;
        const parts = value.split(`; ${name}=`);
        if (parts.length === 2) return parts.pop().split(';').shift();
        return null;
    }

    function initMap() {
        const mapOptions = {
            center: { lat: 48.8566, lng: 2.3522 },
            zoom: parseInt(app.settings.defaultZoom),
            mapTypeId: google.maps.MapTypeId.HYBRID,
            disableDefaultUI: true,
            zoomControl: true,
            fullscreenControl: true
        };
        
        app.map = new google.maps.Map(dom.map, mapOptions);
        app.directionsService = new google.maps.DirectionsService();
        app.directionsRenderer = new google.maps.DirectionsRenderer({
            map: app.map,
            suppressMarkers: true,
            polylineOptions: {
                strokeColor: '#0ea5e9',
                strokeWeight: 6,
                strokeOpacity: 0.8
            }
        });
        
        app.directionsRenderer.setPanel(dom.directionsSteps);
        
        setupAutocomplete(dom.form.startLocation);
        setupAutocomplete(dom.form.endLocation);
        
        app.map.addListener('click', handleMapClick);
        
        loadTrafficLights();
        startGeolocation();
    }
    
    function setupAutocomplete(input) {
        if (!input) return;
        
        const autocomplete = new google.maps.places.Autocomplete(input, {
            types: ['geocode'],
            fields: ['place_id', 'geometry', 'name', 'formatted_address']
        });
        
        autocomplete.addListener('place_changed', () => {
            const place = autocomplete.getPlace();
            
            if (!place.geometry || !place.geometry.location) {
                showNotification('No location found for this address', 'error');
                return;
            }
            
            if (input === dom.form.startLocation) {
                app.navigation.startPosition = {
                    lat: place.geometry.location.lat(),
                    lng: place.geometry.location.lng(),
                    formatted_address: place.formatted_address || place.name
                };
            } else if (input === dom.form.endLocation) {
                app.navigation.endPosition = {
                    lat: place.geometry.location.lat(),
                    lng: place.geometry.location.lng(),
                    formatted_address: place.formatted_address || place.name
                };
            }
        });
    }
    
    function setMapStyle() {
        switch (app.settings.mapStyle) {
            case 'roadmap':
                app.map.setMapTypeId(google.maps.MapTypeId.ROADMAP);
                break;
            case 'terrain':
                app.map.setMapTypeId(google.maps.MapTypeId.TERRAIN);
                break;
            case 'satellite':
            default:
                app.map.setMapTypeId(google.maps.MapTypeId.HYBRID);
                break;
        }
    }

    function setupDraggableModals() {
        dom.draggableHandles.forEach(handle => {
            const modal = handle.closest('.modal-body');
            let startY, currentY, initialY;
            let isDragging = false;
            
            handle.addEventListener('touchstart', e => {
                startY = e.touches[0].clientY;
                initialY = 0;
                isDragging = true;
                modal.style.transition = 'none';
            }, { passive: true });
            
            handle.addEventListener('touchmove', e => {
                if (!isDragging) return;
                currentY = e.touches[0].clientY;
                const diffY = currentY - startY;
                if (diffY < 0) return;
                
                modal.style.transform = `translateY(${diffY}px)`;
            }, { passive: true });
            
            handle.addEventListener('touchend', e => {
                modal.style.transition = 'transform 0.3s ease';
                if (!isDragging) return;
                
                const diffY = currentY - startY;
                if (diffY > 100) {
                    modal.style.transform = 'translateY(100%)';
                    const modalContainer = modal.closest('.modal');
                    setTimeout(() => {
                        modalContainer.style.display = 'none';
                        modal.style.transform = 'translateY(0)';
                    }, 300);
                } else {
                    modal.style.transform = 'translateY(0)';
                }
                
                isDragging = false;
            }, { passive: true });
        });
        
        dom.overlays.forEach(overlay => {
            overlay.addEventListener('click', () => {
                const modal = overlay.closest('.modal');
                closeModal(modal);
            });
        });
    }

    function bindEvents() {
        dom.buttons.menuToggle.addEventListener('click', toggleSidebar);
        dom.buttons.headerLocate.addEventListener('click', locateUser);
        dom.buttons.headerAdd.addEventListener('click', () => openModal(dom.modals.add));
        dom.buttons.account.addEventListener('click', toggleAccountPanel);
        dom.buttons.settings.addEventListener('click', () => openModal(dom.modals.settings));
        dom.buttons.refresh.addEventListener('click', refreshLights);
        dom.buttons.nearby.addEventListener('click', findNearbyLights);
        dom.buttons.route.addEventListener('click', toggleRoutePanel);
        dom.buttons.routes.addEventListener('click', showSavedRoutes);
        dom.form.searchInput.addEventListener('input', handleSearch);
        
        dom.closePopup.addEventListener('click', () => dom.lightPopup.classList.add('hidden'));
        dom.popupNavigate.addEventListener('click', navigateToSelectedLight);
        dom.popupMeasure.addEventListener('click', () => {
            dom.lightPopup.classList.add('hidden');
            openMeasureModal(app.selectedLightId);
        });
        
        dom.closeButtons.forEach(btn => {
            btn.addEventListener('click', () => {
                const modal = btn.closest('.modal');
                closeModal(modal);
            });
        });
        
        dom.closeRoutePanel.addEventListener('click', () => {
            dom.routePlannerPanel.classList.add('hidden');
            dom.directionsPanel.style.display = 'none';
            app.state.routePanelOpen = false;
        });
        
        if (dom.settings.darkMode) {
            dom.settings.darkMode.addEventListener('change', toggleDarkMode);
        }
        if (dom.settings.autoRefresh) {
            dom.settings.autoRefresh.addEventListener('change', toggleAutoRefresh);
        }
        if (dom.settings.notifications) {
            dom.settings.notifications.addEventListener('change', saveSettings);
        }
        if (dom.settings.predictions) {
            dom.settings.predictions.addEventListener('change', () => {
                saveSettings();
                updateAllLightStatus();
            });
        }
        if (dom.settings.autoMeasure) {
            dom.settings.autoMeasure.addEventListener('change', saveSettings);
        }
        if (dom.settings.defaultZoom) {
            dom.settings.defaultZoom.addEventListener('change', saveSettings);
        }
        
        dom.buttons.routeTypeCar.addEventListener('click', () => setTravelMode('DRIVING'));
        dom.buttons.routeTypeBike.addEventListener('click', () => setTravelMode('BICYCLING'));
        dom.buttons.routeTypeWalk.addEventListener('click', () => setTravelMode('WALKING'));
        dom.calculateRoute.addEventListener('click', calculateRoute);
        dom.startNavigation.addEventListener('click', startRouteNavigation);
        dom.buttons.endNavigation.addEventListener('click', endNavigation);
        
        dom.buttons.resetApp.addEventListener('click', resetAppData);
        dom.buttons.measureRed.addEventListener('click', () => startMeasure('red'));
        dom.buttons.measureGreen.addEventListener('click', () => startMeasure('green'));
        dom.buttons.startTimer.addEventListener('click', startMeasureTimer);
        dom.buttons.stopTimer.addEventListener('click', stopMeasureTimer);
        dom.buttons.saveTimer.addEventListener('click', saveMeasureTimer);
        
        if (dom.auth.showLogin) dom.auth.showLogin.addEventListener('click', showLoginForm);
        if (dom.auth.showRegister) dom.auth.showRegister.addEventListener('click', showRegisterForm);
        if (dom.auth.login) dom.auth.login.addEventListener('click', () => openModal(dom.modals.login));
        if (dom.form.login) dom.form.login.addEventListener('submit', handleLogin);
        if (dom.form.register) dom.form.register.addEventListener('submit', handleRegister);
        
        if (dom.buttons.accountLogin) dom.buttons.accountLogin.addEventListener('click', () => openModal(dom.modals.login));
        if (dom.buttons.accountLogout) dom.buttons.accountLogout.addEventListener('click', handleLogout);
        if (dom.buttons.accountSettings) dom.buttons.accountSettings.addEventListener('click', () => openModal(dom.modals.settings));
        if (dom.buttons.accountRoutes) dom.buttons.accountRoutes.addEventListener('click', () => {
            toggleAccountPanel();
            showSavedRoutes();
        });
        
        dom.buttons.skipWelcome.addEventListener('click', () => closeModal(dom.modals.welcome));
        dom.buttons.nextSlide.addEventListener('click', nextOnboardingSlide);
        dom.welcome.dots.forEach(dot => {
            dot.addEventListener('click', () => {
                const slide = parseInt(dot.getAttribute('data-dot'));
                goToSlide(slide);
            });
        });
        
        dom.form.addLight.addEventListener('submit', handleAddLight);
        
        window.addEventListener('online', handleOnline);
        window.addEventListener('offline', handleOffline);
        document.addEventListener('visibilitychange', handleVisibilityChange);
        
        document.addEventListener('click', (e) => {
            if (app.state.accountPanelOpen && !dom.accountPanel.contains(e.target) && e.target !== dom.buttons.account) {
                toggleAccountPanel(false);
            }
        });
    }

    function toggleSidebar() {
        dom.sidebar.classList.toggle('active');
        dom.buttons.menuToggle.innerHTML = dom.sidebar.classList.contains('active') 
            ? '<i class="fas fa-times"></i>' 
            : '<i class="fas fa-bars"></i>';
    }
    
    function toggleAccountPanel(show) {
        const panel = dom.accountPanel;
        if (show === undefined) {
            show = !app.state.accountPanelOpen;
        }
        
        if (show) {
            panel.style.transform = 'translateX(0)';
            app.state.accountPanelOpen = true;
        } else {
            panel.style.transform = 'translateX(100%)';
            app.state.accountPanelOpen = false;
        }
    }

    function locateUser() {
        if (app.userMarker) {
            app.map.panTo(app.userMarker.getPosition());
            app.map.setZoom(16);
        } else {
            startGeolocation();
        }
    }

    function refreshLights() {
        dom.buttons.refresh.innerHTML = '<i class="fas fa-sync-alt fa-spin"></i>';
        loadTrafficLights(true);
        setTimeout(() => {
            dom.buttons.refresh.innerHTML = '<i class="fas fa-sync-alt"></i>';
            if (app.settings.notifications) {
                showNotification('Traffic lights refreshed', 'success');
            }
        }, 800);
    }

    function findNearbyLights() {
        if (app.userMarker) {
            app.map.panTo(app.userMarker.getPosition());
            app.map.setZoom(16);
            highlightNearbyLights();
        } else {
            startGeolocation();
        }
    }
    
    function toggleRoutePanel() {
        if (app.state.routePanelOpen) {
            dom.routePlannerPanel.classList.add('hidden');
            dom.directionsPanel.style.display = 'none';
            app.state.routePanelOpen = false;
        } else {
            if (app.userMarker) {
                const position = app.userMarker.getPosition();
                
                reverseGeocode(position.lng(), position.lat())
                    .then(address => {
                        dom.form.startLocation.value = address || `${position.lat().toFixed(6)}, ${position.lng().toFixed(6)}`;
                        app.navigation.startPosition = {
                            lat: position.lat(),
                            lng: position.lng(),
                            formatted_address: address || `${position.lat().toFixed(6)}, ${position.lng().toFixed(6)}`
                        };
                    });
            }
            
            dom.routePlannerPanel.classList.remove('hidden');
            app.state.routePanelOpen = true;
        }
    }
    
    function reverseGeocode(lng, lat) {
        return new Promise((resolve, reject) => {
            const geocoder = new google.maps.Geocoder();
            const latlng = { lat: parseFloat(lat), lng: parseFloat(lng) };
            
            geocoder.geocode({ location: latlng }, (results, status) => {
                if (status === 'OK') {
                    if (results[0]) {
                        resolve(results[0].formatted_address);
                    } else {
                        resolve(null);
                    }
                } else {
                    console.error('Geocoder failed due to: ' + status);
                    resolve(null);
                }
            });
        });
    }
    
    function showSavedRoutes() {
        fetch('db.php?action=getUserRoutes')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showRoutesListInSidebar(data.routes);
                } else {
                    if (app.settings.notifications) {
                        showNotification(data.message, 'error');
                    }
                }
            })
            .catch(error => {
                console.error('Error fetching routes:', error);
                if (app.settings.notifications) {
                    showNotification('Error loading saved routes', 'error');
                }
            });
    }
    
    function showRoutesListInSidebar(routes) {
        dom.lightsList.innerHTML = '';
        
        if (routes.length === 0) {
            dom.noLightsMessage.querySelector('p').textContent = 'No saved routes';
            dom.noLightsMessage.classList.remove('hidden');
            return;
        }
        
        dom.noLightsMessage.classList.add('hidden');
        
        routes.forEach(route => {
            const card = document.createElement('div');
            card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg rounded-xl';
            
            const routeData = route.route_data;
            const startName = routeData.start_point?.formatted_address || 'Start';
            const endName = routeData.end_point?.formatted_address || 'Destination';
            const distance = routeData.distance ? `${(routeData.distance / 1000).toFixed(1)} km` : '';
            const duration = routeData.duration ? `${Math.round(routeData.duration / 60)} min` : '';
            
            card.innerHTML = `
                <div class="flex justify-between items-center">
                    <h3 class="font-medium">${route.name}</h3>
                    <span class="px-2.5 py-1 text-xs font-medium rounded-full bg-primary-100 text-primary-800">
                        <i class="fas fa-route mr-1"></i>${distance}
                    </span>
                </div>
                <div class="flex items-center text-sm text-gray-500 mt-2">
                    <i class="fas fa-map-marker-alt text-primary-500 mr-2"></i>
                    <span class="truncate">${startName} → ${endName}</span>
                </div>
                <div class="mt-2 flex justify-between text-xs">
                    <span class="text-gray-500">${duration}</span>
                    <span class="text-gray-500">${formatDate(route.created_at)}</span>
                </div>
            `;
            
            card.addEventListener('click', () => {
                loadSavedRoute(route);
            });
            
            dom.lightsList.appendChild(card);
        });
    }
    
    function formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString();
    }
    
    function loadSavedRoute(route) {
        dom.form.startLocation.value = route.route_data.start_point.formatted_address || '';
        dom.form.endLocation.value = route.route_data.end_point.formatted_address || '';
        
        app.navigation.startPosition = route.route_data.start_point;
        app.navigation.endPosition = route.route_data.end_point;
        app.navigation.travelMode = route.route_data.travel_mode || 'DRIVING';
        
        setTravelMode(app.navigation.travelMode, false);
        
        dom.routePlannerPanel.classList.remove('hidden');
        app.state.routePanelOpen = true;
        
        if (window.innerWidth <= 768) {
            dom.sidebar.classList.remove('active');
            dom.buttons.menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
        }
        
        calculateRoute(true);
    }

    function openModal(modal) {
        if (modal === dom.modals.add) {
            app.state.selectingLocation = true;
            if (app.userMarker) {
                const pos = app.userMarker.getPosition();
                document.getElementById('latitude').value = pos.lat().toFixed(6);
                document.getElementById('longitude').value = pos.lng().toFixed(6);
                
                if (app.tempMarker) app.tempMarker.setMap(null);
                
                app.tempMarker = new google.maps.Marker({
                    position: pos,
                    map: app.map,
                    draggable: true,
                    animation: google.maps.Animation.DROP,
                    icon: {
                        url: 'data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="36" height="36" viewBox="0 0 24 24" fill="%2310b981" stroke="%23ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="12" y1="8" x2="12" y2="16"/><line x1="8" y1="12" x2="16" y2="12"/></svg>',
                        size: new google.maps.Size(36, 36),
                        scaledSize: new google.maps.Size(36, 36),
                        anchor: new google.maps.Point(18, 18)
                    }
                });
                
                app.tempMarker.addListener('dragend', function() {
                    const newPos = app.tempMarker.getPosition();
                    document.getElementById('latitude').value = newPos.lat().toFixed(6);
                    document.getElementById('longitude').value = newPos.lng().toFixed(6);
                });
            }
        }
        
        modal.style.display = 'flex';
        const modalBody = modal.querySelector('.modal-body');
        modalBody.style.transform = 'translateY(0)';
    }

    function closeModal(modal) {
        const modalBody = modal.querySelector('.modal-body');
        modalBody.style.transition = 'transform 0.3s ease';
        modalBody.style.transform = 'translateY(100%)';
        
        setTimeout(() => {
            modal.style.display = 'none';
            modalBody.style.transform = 'translateY(0)';
            
            if (modal === dom.modals.add) {
                app.state.selectingLocation = false;
                if (app.tempMarker) {
                    app.tempMarker.setMap(null);
                    app.tempMarker = null;
                }
            }
        }, 300);
    }

    function handleMapClick(e) {
        if (app.state.selectingLocation) {
            const lat = e.latLng.lat();
            const lng = e.latLng.lng();
            
            document.getElementById('latitude').value = lat.toFixed(6);
            document.getElementById('longitude').value = lng.toFixed(6);
            
            if (app.tempMarker) app.tempMarker.setMap(null);
            
            app.tempMarker = new google.maps.Marker({
                position: { lat, lng },
                map: app.map,
                draggable: true,
                animation: google.maps.Animation.DROP,
                icon: {
                    url: 'data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="36" height="36" viewBox="0 0 24 24" fill="%2310b981" stroke="%23ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="12" y1="8" x2="12" y2="16"/><line x1="8" y1="12" x2="16" y2="12"/></svg>',
                    size: new google.maps.Size(36, 36),
                    scaledSize: new google.maps.Size(36, 36),
                    anchor: new google.maps.Point(18, 18)
                }
            });
            
            app.tempMarker.addListener('dragend', function() {
                const newPos = app.tempMarker.getPosition();
                document.getElementById('latitude').value = newPos.lat().toFixed(6);
                document.getElementById('longitude').value = newPos.lng().toFixed(6);
            });
        }
    }

    function handleSearch() {
        const searchTerm = dom.form.searchInput.value.toLowerCase();
        const lights = document.querySelectorAll('.light-card');
        
        let hasResults = false;
        
        lights.forEach(light => {
            const name = light.getAttribute('data-name')?.toLowerCase() || '';
            const visible = name.includes(searchTerm);
            light.style.display = visible ? 'block' : 'none';
            if (visible) hasResults = true;
        });
        
        if (lights.length > 0 && !hasResults) {
            dom.noLightsMessage.classList.remove('hidden');
            dom.noLightsMessage.querySelector('p').textContent = 'No results found';
        } else {
            dom.noLightsMessage.classList.add('hidden');
        }
    }

    function toggleDarkMode() {
        app.settings.darkMode = dom.settings.darkMode.checked;
        
        if (app.settings.darkMode) {
            document.documentElement.classList.add('dark');
            document.body.classList.add('dark-mode');
            
            if (app.map) {
                app.map.setMapTypeId(google.maps.MapTypeId.SATELLITE);
            }
        } else {
            document.documentElement.classList.remove('dark');
            document.body.classList.remove('dark-mode');
            
            if (app.map) {
                setMapStyle();
            }
        }
        
        saveSettings();
    }

    function toggleAutoRefresh() {
        app.settings.autoRefresh = dom.settings.autoRefresh.checked;
        
        if (app.settings.autoRefresh) {
            startStatusUpdates();
        } else {
            stopStatusUpdates();
        }
        
        saveSettings();
    }

    function resetAppData() {
        if (confirm('Are you sure you want to reset all app data? This will clear all your settings and cache.')) {
            localStorage.clear();
            showNotification('App data reset. Refreshing...', 'info');
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        }
    }

    function navigateToSelectedLight() {
        if (app.selectedLightId && app.markers[app.selectedLightId]) {
            const light = app.markers[app.selectedLightId].data;
            
            dom.form.startLocation.value = '';
            dom.form.endLocation.value = light.name;
            app.navigation.endPosition = {
                lat: parseFloat(light.latitude),
                lng: parseFloat(light.longitude),
                formatted_address: light.name
            };
            
            if (app.userMarker) {
                const userPosition = app.userMarker.getPosition();
                reverseGeocode(userPosition.lng(), userPosition.lat())
                    .then(address => {
                        dom.form.startLocation.value = address || `${userPosition.lat().toFixed(6)}, ${userPosition.lng().toFixed(6)}`;
                        app.navigation.startPosition = {
                            lat: userPosition.lat(),
                            lng: userPosition.lng(),
                            formatted_address: address || `${userPosition.lat().toFixed(6)}, ${userPosition.lng().toFixed(6)}`
                        };
                        
                        dom.lightPopup.classList.add('hidden');
                        dom.routePlannerPanel.classList.remove('hidden');
                        app.state.routePanelOpen = true;
                        
                        setTimeout(() => {
                            calculateRoute();
                        }, 500);
                    });
            } else {
                dom.lightPopup.classList.add('hidden');
                dom.routePlannerPanel.classList.remove('hidden');
                app.state.routePanelOpen = true;
                
                startGeolocation();
            }
        }
        
        function setTravelMode(mode, calculate = true) {
            app.navigation.travelMode = mode;
            
            document.querySelectorAll('.travel-type-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            
            let activeBtn;
            switch (mode) {
                case 'BICYCLING':
                    activeBtn = dom.buttons.routeTypeBike;
                    break;
                case 'WALKING':
                    activeBtn = dom.buttons.routeTypeWalk;
                    break;
                default:
                    activeBtn = dom.buttons.routeTypeCar;
            }
            
            if (activeBtn) activeBtn.classList.add('active');
            
            if (calculate && app.navigation.startPosition && app.navigation.endPosition) {
                calculateRoute();
            }
        }
        
        function calculateRoute(isSavedRoute = false) {
            if (!dom.form.startLocation.value || !dom.form.endLocation.value) {
                if (app.settings.notifications) {
                    showNotification('Please enter start and end locations', 'error');
                }
                return;
            }
            
            const promises = [];
            
            if (!app.navigation.startPosition || (!app.navigation.startPosition.lat && !isSavedRoute)) {
                promises.push(
                    geocodeLocation(dom.form.startLocation.value)
                        .then(result => {
                            if (result) app.navigation.startPosition = result;
                        })
                );
            }
            
            if (!app.navigation.endPosition || (!app.navigation.endPosition.lat && !isSavedRoute)) {
                promises.push(
                    geocodeLocation(dom.form.endLocation.value)
                        .then(result => {
                            if (result) app.navigation.endPosition = result;
                        })
                );
            }
            
            dom.calculateRoute.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Calculating...';
            dom.calculateRoute.disabled = true;
            
            Promise.all(promises)
                .then(() => {
                    if (!app.navigation.startPosition || !app.navigation.endPosition) {
                        throw new Error('Could not geocode locations');
                    }
                    
                    const request = {
                        origin: { 
                            lat: app.navigation.startPosition.lat, 
                            lng: app.navigation.startPosition.lng 
                        },
                        destination: { 
                            lat: app.navigation.endPosition.lat, 
                            lng: app.navigation.endPosition.lng 
                        },
                        travelMode: google.maps.TravelMode[app.navigation.travelMode],
                        provideRouteAlternatives: false
                    };
                    
                    return new Promise((resolve, reject) => {
                        app.directionsService.route(request, (result, status) => {
                            if (status === 'OK') {
                                resolve(result);
                            } else {
                                reject(new Error('Directions request failed: ' + status));
                            }
                        });
                    });
                })
                .then(result => {
                    app.navigation.route = result;
                    app.directionsRenderer.setDirections(result);
                    
                    const route = result.routes[0];
                    const leg = route.legs[0];
                    
                    app.navigation.duration = leg.duration.value;
                    app.navigation.distance = leg.distance.value;
                    
                    // Extract coordinates from route
                    app.navigation.coordinates = [];
                    route.overview_path.forEach(point => {
                        app.navigation.coordinates.push({
                            lat: point.lat(),
                            lng: point.lng()
                        });
                    });
                    
                    // Show directions panel with steps
                    dom.directionsPanel.style.display = 'block';
                    
                    return fetch('db.php?action=getLightsOnRoute', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/json'},
                        body: JSON.stringify({
                            coordinates: app.navigation.coordinates
                        })
                    });
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        app.navigation.lightsOnRoute = data.lights;
                        
                        // Display traffic lights on the map
                        displayTrafficLightsOnRoute();
                        
                        // Show route details
                        displayRouteDetails();
                        dom.startNavigation.classList.remove('hidden');
                        
                        // Fit map to show the entire route
                        const bounds = new google.maps.LatLngBounds();
                        app.navigation.route.routes[0].overview_path.forEach(point => {
                            bounds.extend(point);
                        });
                        app.map.fitBounds(bounds);
                        
                        calculateTrafficLightImpact();
                    } else {
                        throw new Error(data.message || 'Error fetching traffic lights on route');
                    }
                })
                .catch(error => {
                    console.error('Route calculation error:', error);
                    if (app.settings.notifications) {
                        showNotification(error.message || 'Error calculating route', 'error');
                    }
                })
                .finally(() => {
                    dom.calculateRoute.innerHTML = '<i class="fas fa-route"></i> Calculate Route';
                    dom.calculateRoute.disabled = false;
                });
        }
        
        function geocodeLocation(query) {
            return new Promise((resolve, reject) => {
                const geocoder = new google.maps.Geocoder();
                
                geocoder.geocode({ address: query }, (results, status) => {
                    if (status === 'OK' && results[0]) {
                        resolve({
                            lat: results[0].geometry.location.lat(),
                            lng: results[0].geometry.location.lng(),
                            formatted_address: results[0].formatted_address
                        });
                    } else {
                        resolve(null);
                    }
                });
            });
        }
        
        function calculateTrafficLightImpact() {
            if (!app.navigation.lightsOnRoute || app.navigation.lightsOnRoute.length === 0) {
                return;
            }
            
            const now = new Date();
            const estimatedTimeWithoutLights = app.navigation.duration;
            let additionalTime = 0;
            
            app.navigation.lightsOnRoute.forEach(light => {
                const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
                const timeInCycle = Math.floor(now.getTime() / 1000) % totalCycle;
                
                if (timeInCycle < light.red_duration) {
                    const remainingRed = light.red_duration - timeInCycle;
                    additionalTime += remainingRed / 2; // Average waiting time if we hit this light
                }
            });
            
            app.navigation.estimatedDurationWithLights = estimatedTimeWithoutLights + additionalTime;
            
            // Calculate estimated arrival time
            const arrivalTime = new Date(now.getTime() + app.navigation.estimatedDurationWithLights * 1000);
            const formattedArrival = arrivalTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
            
            dom.arrivalTime.textContent = formattedArrival;
            dom.routeTime.textContent = `${Math.round(app.navigation.estimatedDurationWithLights / 60)} min`;
        }
        
        function displayTrafficLightsOnRoute() {
            // Clear existing markers first
            for (const id in app.markers) {
                if (app.markers[id].isRouteLight) {
                    app.markers[id].marker.setMap(null);
                    delete app.markers[id];
                }
            }
            
            if (!app.navigation.lightsOnRoute || app.navigation.lightsOnRoute.length === 0) {
                return;
            }
            
            app.navigation.lightsOnRoute.forEach(light => {
                const status = getLightStatus(light);
                const position = { lat: parseFloat(light.latitude), lng: parseFloat(light.longitude) };
                
                const marker = new google.maps.Marker({
                    position: position,
                    map: app.map,
                    icon: {
                        url: `data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="${status.color === 'red' ? '%23ef4444' : '%2310b981'}" stroke="%23ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/></svg>`,
                        size: new google.maps.Size(24, 24),
                        scaledSize: new google.maps.Size(24, 24),
                        anchor: new google.maps.Point(12, 12)
                    },
                    zIndex: 100
                });
                
                marker.addListener('click', () => showLightDetail(light));
                
                app.markers[light.id] = {
                    marker: marker,
                    data: light,
                    isRouteLight: true
                };
            });
        }
        
        function displayRouteDetails() {
            dom.routeDetails.classList.remove('hidden');
            
            const distance = app.navigation.distance ? (app.navigation.distance / 1000).toFixed(1) : 0;
            const duration = app.navigation.estimatedDurationWithLights ? 
                Math.round(app.navigation.estimatedDurationWithLights / 60) : 
                Math.round(app.navigation.duration / 60);
            
            dom.routeDistance.textContent = `${distance} km`;
            dom.routeTime.textContent = `${duration} min`;
            
            let redLights = 0;
            let greenLights = 0;
            
            if (app.navigation.lightsOnRoute) {
                app.navigation.lightsOnRoute.forEach(light => {
                    const status = getLightStatus(light);
                    if (status.isRed) {
                        redLights++;
                    } else {
                        greenLights++;
                    }
                });
            }
            
            dom.routeLights.innerHTML = `
                <span class="inline-block px-2 py-1 bg-traffic-red/20 text-traffic-red rounded text-xs font-medium">${redLights} red</span>
                <span class="inline-block px-2 py-1 bg-traffic-green/20 text-traffic-green rounded text-xs font-medium">${greenLights} green</span>
            `;
            
            const now = new Date();
            const arrivalTime = new Date(now.getTime() + (app.navigation.estimatedDurationWithLights || app.navigation.duration) * 1000);
            const formattedArrival = arrivalTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
            
            dom.arrivalTime.textContent = formattedArrival;
        }
        
        function startRouteNavigation() {
            if (!app.navigation.route) {
                if (app.settings.notifications) {
                    showNotification('Please calculate a route first', 'error');
                }
                return;
            }
            
            dom.routePlannerPanel.classList.add('hidden');
            dom.navigationPanel.classList.remove('hidden');
            
            app.isNavigating = true;
            app.navigation.startTime = new Date();
            app.navigation.currentStep = 0;
            
            updateNavigationInstructions();
            
            if (app.userWatchId) {
                navigator.geolocation.clearWatch(app.userWatchId);
            }
            
            app.userWatchId = navigator.geolocation.watchPosition(
                updateNavigationPosition,
                handleGeolocationError,
                { enableHighAccuracy: true, maximumAge: 2000, timeout: 5000 }
            );
            
            if (app.settings.autoMeasure) {
                startAutoDetectStops();
            }
            
            if (app.settings.notifications) {
                showNotification('Navigation started. Follow the blue route.', 'success');
            }
        }
        
        function updateNavigationInstructions() {
            if (!app.navigation.route) return;
            
            const leg = app.navigation.route.routes[0].legs[0];
            if (!leg || !leg.steps || app.navigation.currentStep >= leg.steps.length) {
                return;
            }
            
            const step = leg.steps[app.navigation.currentStep];
            
            dom.navigation.nextDirection.innerHTML = step.instructions || 'Continue';
            
            const distance = step.distance.value;
            let formattedDistance;
            
            if (distance < 100) {
                formattedDistance = `${Math.round(distance)} m`;
            } else if (distance < 1000) {
                formattedDistance = `${Math.round(distance / 10) * 10} m`;
            } else {
                formattedDistance = `${(distance / 1000).toFixed(1)} km`;
            }
            
            dom.navigation.nextDistance.textContent = formattedDistance;
            
            const now = new Date();
            const elapsedSeconds = (now - app.navigation.startTime) / 1000;
            const remainingSeconds = Math.max(0, app.navigation.estimatedDurationWithLights - elapsedSeconds);
            
            if (remainingSeconds > 0) {
                const arrivalTime = new Date(now.getTime() + (remainingSeconds * 1000));
                const formattedArrival = arrivalTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
                dom.navigation.etaText.textContent = `ETA ${formattedArrival}`;
            } else {
                dom.navigation.etaText.textContent = 'Arriving';
            }
            
            // Street name from next step if available, otherwise blank
            dom.navigation.nextStreet.textContent = '';
            if (step.instructions) {
                const streetMatch = step.instructions.match(/<b>(.*?)<\/b>/);
                if (streetMatch && streetMatch[1]) {
                    dom.navigation.nextStreet.textContent = streetMatch[1];
                }
            }
            
            const totalDistance = app.navigation.distance;
            const remainingDistance = calculateRemainingDistance(app.navigation.currentStep);
            
            const coveredDistance = totalDistance - remainingDistance;
            dom.navigation.totalDistance.textContent = `${(coveredDistance / 1000).toFixed(1)} / ${(totalDistance / 1000).toFixed(1)} km`;
            
            const totalMinutes = Math.round(app.navigation.estimatedDurationWithLights / 60);
            const elapsedMinutes = Math.round(elapsedSeconds / 60);
            dom.navigation.totalTime.textContent = `${elapsedMinutes} / ${totalMinutes} min`;
        }
        
        function calculateRemainingDistance(fromStep) {
            if (!app.navigation.route) return 0;
            
            const leg = app.navigation.route.routes[0].legs[0];
            let distance = 0;
            
            for (let i = fromStep; i < leg.steps.length; i++) {
                distance += leg.steps[i].distance.value;
            }
            
            return distance;
        }
        
        function updateNavigationPosition(position) {
            const lat = position.coords.latitude;
            const lng = position.coords.longitude;
            const speed = position.coords.speed || 0;
            const currentPosition = { lat, lng };
            
            if (!app.userMarker) {
                app.userMarker = new google.maps.Marker({
                    position: currentPosition,
                    map: app.map,
                    icon: {
                        path: google.maps.SymbolPath.CIRCLE,
                        scale: 8,
                        fillColor: '#0ea5e9',
                        fillOpacity: 1,
                        strokeColor: '#ffffff',
                        strokeWeight: 2
                    }
                });
            } else {
                app.userMarker.setPosition(currentPosition);
            }
            
            if (app.isNavigating) {
                // Center map on user with forward facing direction
                app.map.panTo(currentPosition);
                
                if (position.coords.heading !== null) {
                    app.map.setHeading(position.coords.heading);
                }
                
                // Update speed display
                const speedKmh = speed ? Math.round(speed * 3.6) : 0;
                dom.navigation.currentSpeed.textContent = `${speedKmh} km/h`;
                app.navigation.currentSpeed = speedKmh;
                
                checkNavigationProgress(currentPosition);
                updateETA(currentPosition);
                
                if (app.settings.autoMeasure && speed < 0.5) {
                    handleNavigationStop(currentPosition);
                } else if (speed >= 0.5 && app.navigation.lastStop) {
                    handleNavigationResume();
                }
                
                app.navigation.lastPosition = currentPosition;
            }
        }
        
        function updateETA(currentPosition) {
            if (!app.navigation.route) return;
            
            const now = new Date();
            const elapsedSeconds = (now - app.navigation.startTime) / 1000;
            
            // Request updated duration from Google
            const request = {
                origin: new google.maps.LatLng(currentPosition.lat, currentPosition.lng),
                destination: new google.maps.LatLng(
                    app.navigation.endPosition.lat, 
                    app.navigation.endPosition.lng
                ),
                travelMode: google.maps.TravelMode[app.navigation.travelMode]
            };
            
            app.directionsService.route(request, (result, status) => {
                if (status === 'OK') {
                    const estimatedDuration = result.routes[0].legs[0].duration.value;
                    
                    // Account for traffic lights ahead
                    const lightsAhead = getTrafficLightsAhead(currentPosition);
                    let additionalTime = 0;
                    
                    lightsAhead.forEach(light => {
                        const status = getLightStatus(light);
                        if (status.isRed) {
                            additionalTime += Math.min(status.timeLeft, 30);
                        }
                    });
                    
                    const totalEstimatedTime = estimatedDuration + additionalTime;
                    const arrivalTime = new Date(now.getTime() + (totalEstimatedTime * 1000));
                    const formattedArrival = arrivalTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
                    
                    dom.navigation.etaText.textContent = `ETA ${formattedArrival}`;
                    
                    // Update total time in navigation panel
                    const totalTimeMinutes = Math.round((elapsedSeconds + totalEstimatedTime) / 60);
                    const elapsedMinutes = Math.round(elapsedSeconds / 60);
                    dom.navigation.totalTime.textContent = `${elapsedMinutes} / ${totalTimeMinutes} min`;
                }
            });
        }
        
        function getTrafficLightsAhead(currentPosition) {
            if (!app.navigation.lightsOnRoute || app.navigation.lightsOnRoute.length === 0) {
                return [];
            }
            
            const lightsAhead = [];
            
            app.navigation.lightsOnRoute.forEach(light => {
                const lightPosition = { 
                    lat: parseFloat(light.latitude), 
                    lng: parseFloat(light.longitude) 
                };
                
                // Calculate distance from current position to light
                const distance = getDistance(currentPosition, lightPosition);
                
                // If light is ahead within 1km
                if (distance > 20 && distance < 1000) {
                    lightsAhead.push({
                        ...light,
                        distance
                    });
                }
            });
            
            // Sort by distance
            return lightsAhead.sort((a, b) => a.distance - b.distance);
        }
        
        function checkNavigationProgress(currentPosition) {
            if (!app.navigation.route) return;
            
            const leg = app.navigation.route.routes[0].legs[0];
            if (!leg || !leg.steps || app.navigation.currentStep >= leg.steps.length) {
                return;
            }
            
            const currentStep = leg.steps[app.navigation.currentStep];
            const nextStep = leg.steps[app.navigation.currentStep + 1];
            
            if (nextStep) {
                const endLocation = currentStep.end_location;
                const distanceToNextStep = getDistanceFromLatLng(
                    currentPosition.lat, 
                    currentPosition.lng,
                    endLocation.lat(), 
                    endLocation.lng()
                );
                
                if (distanceToNextStep < 30) { // Within 30 meters
                    app.navigation.currentStep++;
                    updateNavigationInstructions();
                    
                    if (app.settings.notifications) {
                        showNotification(nextStep.instructions, 'info');
                    }
                }
            }
            
            // Check if reached destination
            const endLocation = leg.end_location;
            const distanceToEnd = getDistanceFromLatLng(
                currentPosition.lat, 
                currentPosition.lng,
                endLocation.lat(), 
                endLocation.lng()
            );
            
            if (distanceToEnd < 50) { // Within 50 meters of destination
                handleNavigationComplete();
            }
        }
        
        function getDistance(point1, point2) {
            return getDistanceFromLatLng(point1.lat, point1.lng, point2.lat, point2.lng);
        }
        
        function getDistanceFromLatLng(lat1, lng1, lat2, lng2) {
            const R = 6371000; // Earth radius in meters
            const dLat = (lat2 - lat1) * Math.PI / 180;
            const dLng = (lng2 - lng1) * Math.PI / 180;
            const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                      Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) * 
                      Math.sin(dLng/2) * Math.sin(dLng/2);
            const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
            return R * c;
        }
        
        function handleNavigationComplete() {
            if (app.settings.notifications) {
                showNotification('You have reached your destination!', 'success');
            }
            
            endNavigation();
            
            // Save the route if user is logged in
            if (document.cookie.includes('user_id=')) {
                const routeName = `${app.navigation.startPosition.formatted_address.split(',')[0]} to ${app.navigation.endPosition.formatted_address.split(',')[0]}`;
                
                fetch('db.php?action=saveRoute', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        name: routeName,
                        start_point: app.navigation.startPosition,
                        end_point: app.navigation.endPosition,
                        coordinates: app.navigation.coordinates,
                        distance: app.navigation.distance,
                        duration: app.navigation.duration,
                        travel_mode: app.navigation.travelMode
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success && app.settings.notifications) {
                        showNotification('Route saved to your account', 'success');
                    }
                })
                .catch(error => console.error('Error saving route:', error));
            }
        }
        
        function startAutoDetectStops() {
            console.log('Auto-detect stops enabled');
        }
        
        function handleNavigationStop(position) {
            const now = new Date();
            
            if (!app.navigation.lastStop) {
                app.navigation.lastStop = {
                    position: position,
                    time: now,
                    duration: 0
                };
            } else {
                app.navigation.stopDuration = (now - app.navigation.lastStop.time) / 1000;
                
                if (app.navigation.stopDuration > 2 && app.settings.autoMeasure) {
                    checkForTrafficLight(position);
                }
            }
        }
        
        function handleNavigationResume() {
            if (app.navigation.lastStop && app.navigation.stopDuration > 2) {
                app.navigation.lastStop = null;
                app.navigation.stopDuration = 0;
            }
        }
        
        function checkForTrafficLight(position) {
            if (!app.navigation.route) return;
            
            const leg = app.navigation.route.routes[0].legs[0];
            if (!leg) return;
            
            const distanceFromStart = getDistance(position, {
                lat: leg.start_location.lat(),
                lng: leg.start_location.lng()
            });
            
            const distanceFromEnd = getDistance(position, {
                lat: leg.end_location.lat(),
                lng: leg.end_location.lng()
            });
            
            if (distanceFromStart < 1000 || distanceFromEnd < 1000) {
                return;
            }
            
            // Check if we're near an existing traffic light
            let nearestLight = null;
            let minDistance = Infinity;
            
            for (const id in app.markers) {
                const marker = app.markers[id];
                if (!marker.isRouteLight) continue;
                
                const light = marker.data;
                const lightPosition = { 
                    lat: parseFloat(light.latitude), 
                    lng: parseFloat(light.longitude) 
                };
                
                const distance = getDistance(position, lightPosition);
                
                if (distance < 50 && distance < minDistance) {
                    minDistance = distance;
                    nearestLight = light;
                }
            }
            
            // Check if we're near an intersection
            let isNearIntersection = false;
            
            for (let i = 0; i < leg.steps.length; i++) {
                const step = leg.steps[i];
                if (step.maneuver && (
                    step.maneuver.includes('turn') || 
                    step.maneuver.includes('fork') ||
                    step.maneuver.includes('ramp') ||
                    step.maneuver.includes('exit')
                )) {
                    const intersectionPos = { 
                        lat: step.start_location.lat(), 
                        lng: step.start_location.lng() 
                    };
                    
                    const distance = getDistance(position, intersectionPos);
                    
                    if (distance < 50) {
                        isNearIntersection = true;
                        break;
                    }
                }
            }
            
            if ((nearestLight || isNearIntersection) && app.navigation.stopDuration > 5) {
                // Likely at a traffic light, send data to server
                autoDetectTrafficLight(position, app.navigation.stopDuration);
            }
        }
        
        function autoDetectTrafficLight(position, duration) {
            let direction = 'unknown';
            
            if (app.navigation.lastPosition) {
                const dx = position.lng - app.navigation.lastPosition.lng;
                const dy = position.lat - app.navigation.lastPosition.lat;
                
                if (Math.abs(dx) > Math.abs(dy)) {
                    direction = dx > 0 ? 'east' : 'west';
                } else {
                    direction = dy > 0 ? 'north' : 'south';
                }
            }
            
            fetch('db.php?action=autoDetectLight', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    latitude: position.lat,
                    longitude: position.lng,
                    duration: duration,
                    duration_type: 'red',
                    direction: direction,
                    speed: app.navigation.currentSpeed || 0,
                    timestamp: new Date().toISOString()
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && app.settings.notifications) {
                    showNotification(data.message, 'success');
                    loadTrafficLights(true);
                }
            })
            .catch(error => console.error('Auto-detect light error:', error));
        }
        
        function endNavigation() {
            app.isNavigating = false;
            
            dom.navigationPanel.classList.add('hidden');
            
            app.navigation.startTime = null;
            app.navigation.currentStep = 0;
            app.navigation.lastStop = null;
            app.navigation.stopDuration = 0;
            
            if (app.userMarker) {
                app.map.panTo(app.userMarker.getPosition());
                app.map.setZoom(parseInt(app.settings.defaultZoom));
                app.map.setHeading(0);
                app.map.setTilt(0);
            }
            
            app.directionsRenderer.setDirections({ routes: [] });
            
            // Remove route lights markers
            for (const id in app.markers) {
                if (app.markers[id].isRouteLight) {
                    app.markers[id].marker.setMap(null);
                    delete app.markers[id];
                }
            }
            
            if (app.userWatchId) {navigator.geolocation.clearWatch(app.userWatchId);
            }
            startGeolocation();
        }
    
        function handleAddLight(e) {
            e.preventDefault();
            
            const submitBtn = dom.form.addLight.querySelector('[type="submit"]');
            const originalText = submitBtn.textContent;
            submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Adding...';
            submitBtn.disabled = true;
            
            const lightData = {
                name: document.getElementById('lightName').value,
                latitude: document.getElementById('latitude').value,
                longitude: document.getElementById('longitude').value,
                direction: document.getElementById('direction').value,
                red_duration: document.getElementById('redDuration').value,
                green_duration: document.getElementById('greenDuration').value
            };
            
            fetch('db.php?action=addTrafficLight', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify(lightData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    addLightToMap(data.light);
                    addLightToSidebar(data.light);
                    
                    if (app.settings.notifications) {
                        showNotification(data.message || 'Traffic light added successfully and pending review by Imators team.', 'success');
                    }
                    
                    closeModal(dom.modals.add);
                    dom.form.addLight.reset();
                    
                    if (app.tempMarker) {
                        app.tempMarker.setMap(null);
                        app.tempMarker = null;
                    }
                    
                    app.state.selectingLocation = false;
                    dom.noLightsMessage.classList.add('hidden');
                    
                    app.map.panTo({
                        lat: parseFloat(data.light.latitude),
                        lng: parseFloat(data.light.longitude)
                    });
                    app.map.setZoom(16);
                } else {
                    if (app.settings.notifications) {
                        showNotification('Error: ' + data.message, 'error');
                    }
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            })
            .catch(error => {
                console.error('Error:', error);
                if (app.settings.notifications) {
                    showNotification('Connection error', 'error');
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            });
        }
        
        function showLoginForm() {
            dom.form.login.classList.remove('hidden');
            dom.form.register.classList.add('hidden');
            dom.auth.showLogin.classList.add('border-primary-500', 'text-primary-500');
            dom.auth.showLogin.classList.remove('border-transparent', 'text-gray-500');
            dom.auth.showRegister.classList.add('border-transparent', 'text-gray-500');
            dom.auth.showRegister.classList.remove('border-primary-500', 'text-primary-500');
            dom.auth.authTitle.textContent = 'Login';
        }
        
        function showRegisterForm() {
            dom.form.login.classList.add('hidden');
            dom.form.register.classList.remove('hidden');
            dom.auth.showRegister.classList.add('border-primary-500', 'text-primary-500');
            dom.auth.showRegister.classList.remove('border-transparent', 'text-gray-500');
            dom.auth.showLogin.classList.add('border-transparent', 'text-gray-500');
            dom.auth.showLogin.classList.remove('border-primary-500', 'text-primary-500');
            dom.auth.authTitle.textContent = 'Register';
        }
        
        function handleLogin(e) {
            e.preventDefault();
            
            const formData = new FormData(dom.form.login);
            const submitBtn = dom.form.login.querySelector('[type="submit"]');
            const originalText = submitBtn.textContent;
            
            submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Logging in...';
            submitBtn.disabled = true;
            
            fetch('db.php?action=login', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    email: formData.get('email'),
                    password: formData.get('password'),
                    remember: formData.get('remember') === 'on'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (app.settings.notifications) {
                        showNotification('Login successful', 'success');
                    }
                    
                    updateUIForLoggedInUser(data.user);
                    closeModal(dom.modals.login);
                } else {
                    if (app.settings.notifications) {
                        showNotification('Error: ' + data.message, 'error');
                    }
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            })
            .catch(error => {
                console.error('Login error:', error);
                if (app.settings.notifications) {
                    showNotification('Connection error', 'error');
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            });
        }
        
        function handleRegister(e) {
            e.preventDefault();
            
            const formData = new FormData(dom.form.register);
            const submitBtn = dom.form.register.querySelector('[type="submit"]');
            const originalText = submitBtn.textContent;
            
            if (formData.get('password') !== formData.get('password_confirm')) {
                if (app.settings.notifications) {
                    showNotification('Passwords do not match', 'error');
                }
                return;
            }
            
            submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Registering...';
            submitBtn.disabled = true;
            
            fetch('db.php?action=register', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    username: formData.get('username'),
                    email: formData.get('email'),
                    password: formData.get('password'),
                    remember: true
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (app.settings.notifications) {
                        showNotification('Registration successful', 'success');
                    }
                    
                    updateUIForLoggedInUser(data.user);
                    closeModal(dom.modals.login);
                } else {
                    if (app.settings.notifications) {
                        showNotification('Error: ' + data.message, 'error');
                    }
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            })
            .catch(error => {
                console.error('Registration error:', error);
                if (app.settings.notifications) {
                    showNotification('Connection error', 'error');
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            });
        }
        
        function updateUIForLoggedInUser(user) {
            const loginSection = document.querySelector('.sidebar > .p-4.border-t');
            if (loginSection) {
                loginSection.innerHTML = `
                    <div class="flex items-center mb-3">
                        <div class="w-10 h-10 rounded-full bg-primary-500 text-white flex items-center justify-center mr-3">
                            <i class="fas fa-user"></i>
                        </div>
                        <div>
                            <p class="font-medium text-sm">${user.username}</p>
                            <p class="text-xs text-gray-500">Logged in</p>
                        </div>
                        <button id="logoutBtn" class="ml-auto text-gray-500 hover:text-gray-700">
                            <i class="fas fa-sign-out-alt"></i>
                        </button>
                    </div>
                    <button id="settingsBtn" class="w-full bg-gray-100 hover:bg-gray-200 text-gray-800 font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center justify-center gap-2">
                        <i class="fas fa-cog"></i> Settings
                    </button>
                `;
                
                document.getElementById('logoutBtn').addEventListener('click', handleLogout);
                document.getElementById('settingsBtn').addEventListener('click', () => openModal(dom.modals.settings));
            }
            
            // Update account panel
            const accountPanel = document.getElementById('accountPanel');
            if (accountPanel) {
                const content = accountPanel.querySelector('.p-5');
                content.innerHTML = `
                    <h3 class="font-semibold text-lg mb-4">Account</h3>
                    <div class="mb-4">
                        <div class="flex items-center mb-3">
                            <div class="w-12 h-12 rounded-full bg-primary-500 text-white flex items-center justify-center mr-3">
                                <i class="fas fa-user"></i>
                            </div>
                            <div>
                                <p class="font-medium">${user.username}</p>
                                <p class="text-xs text-gray-500">${user.email}</p>
                            </div>
                        </div>
                        
                        <div class="bg-gray-50 p-3 rounded-xl mb-4">
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-sm text-gray-600">Last login</span>
                                <span class="text-sm font-medium">${formatDate(user.last_login || new Date())}</span>
                            </div>
                            <div class="flex justify-between items-center">
                                <span class="text-sm text-gray-600">Contributions</span>
                                <span class="text-sm font-medium">${user.contributions || 0} traffic lights</span>
                            </div>
                        </div>
                        
                        <div class="space-y-2">
                            <button id="accountSettings" class="w-full bg-gray-100 hover:bg-gray-200 text-gray-800 font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center gap-2">
                                <i class="fas fa-user-cog"></i> Account Settings
                            </button>
                            <button id="accountRoutes" class="w-full bg-gray-100 hover:bg-gray-200 text-gray-800 font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center gap-2">
                                <i class="fas fa-route"></i> My Saved Routes
                            </button>
                            <button id="accountLights" class="w-full bg-gray-100 hover:bg-gray-200 text-gray-800 font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center gap-2">
                                <i class="fas fa-traffic-light"></i> My Traffic Lights
                            </button>
                            <button id="accountLogoutBtn" class="w-full bg-traffic-red hover:bg-red-600 text-white font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center justify-center gap-2 mt-4">
                                <i class="fas fa-sign-out-alt"></i> Logout
                            </button>
                        </div>
                    </div>
                `;
                
                document.getElementById('accountLogoutBtn').addEventListener('click', handleLogout);
                document.getElementById('accountSettings').addEventListener('click', () => {
                    toggleAccountPanel(false);
                    openModal(dom.modals.settings);
                });
                document.getElementById('accountRoutes').addEventListener('click', () => {
                    toggleAccountPanel(false);
                    showSavedRoutes();
                });
                document.getElementById('accountLights').addEventListener('click', () => {
                    toggleAccountPanel(false);
                    showUserLights();
                });
            }
            
            // Add logged in indicator to account button
            const accountBtn = document.getElementById('accountBtn');
            if (accountBtn && !accountBtn.querySelector('span')) {
                const indicator = document.createElement('span');
                indicator.className = 'absolute top-0 right-0 w-3 h-3 bg-green-500 rounded-full border border-white';
                accountBtn.appendChild(indicator);
            }
        }
        
        function showUserLights() {
            fetch('db.php?action=getUserLights')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showLightsListInSidebar(data.lights, true);
                    } else {
                        if (app.settings.notifications) {
                            showNotification(data.message, 'error');
                        }
                    }
                })
                .catch(error => {
                    console.error('Error fetching user lights:', error);
                    if (app.settings.notifications) {
                        showNotification('Error loading your traffic lights', 'error');
                    }
                });
        }
        
        function showLightsListInSidebar(lights, isUserLights = false) {
            dom.lightsList.innerHTML = '';
            
            if (lights.length === 0) {
                dom.noLightsMessage.querySelector('p').textContent = isUserLights ? 
                    'You haven\'t added any traffic lights yet' : 'No traffic lights found';
                dom.noLightsMessage.classList.remove('hidden');
                return;
            }
            
            dom.noLightsMessage.classList.add('hidden');
            
            lights.forEach(light => {
                const status = getLightStatus(light);
                
                const card = document.createElement('div');
                card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg rounded-xl';
                card.setAttribute('data-id', light.id);
                card.setAttribute('data-name', light.name);
                
                let statusBadge = `<span class="px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white">
                    ${status.label}
                </span>`;
                
                if (light.status === 'pending') {
                    statusBadge = `<span class="px-2.5 py-1 text-xs font-medium rounded-full bg-amber-100 text-amber-800">
                        Pending
                    </span>`;
                }
                
                card.innerHTML = `
                    <div class="flex justify-between items-center">
                        <h3 class="font-medium">${light.name}</h3>
                        ${statusBadge}
                    </div>
                    <div class="flex items-center text-sm text-gray-500 mt-2">
                        <i class="fas fa-location-dot mr-2"></i>
                        <span class="capitalize">${light.direction}</span>
                        <div class="ml-auto flex items-center" id="timer-${light.id}">
                            <i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s
                        </div>
                    </div>
                    ${app.settings.predictions && light.status !== 'pending' ? `
                    <div class="mt-2 text-xs">
                        <div class="w-full bg-gray-100 rounded-full h-1.5">
                            <div class="bg-traffic-${status.color} h-1.5 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                        </div>
                        <div class="flex justify-between mt-1 text-gray-400">
                            <span>0s</span>
                            <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                        </div>
                    </div>` : ''}
                `;
                
                card.addEventListener('click', function() {
                    const allCards = document.querySelectorAll('.light-card');
                    allCards.forEach(c => c.classList.remove('active'));
                    this.classList.add('active');
                    
                    showLightDetail(light);
                    
                    if (window.innerWidth <= 768) {
                        dom.sidebar.classList.remove('active');
                        dom.buttons.menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
                    }
                });
                
                dom.lightsList.insertBefore(card, dom.loadingLights);
            });
        }
        
        function handleLogout() {
            fetch('db.php?action=logout')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        if (app.settings.notifications) {
                            showNotification('Logout successful', 'success');
                        }
                        
                        // Update UI for logged out state
                        const loginSection = document.querySelector('.sidebar > .p-4.border-t');
                        if (loginSection) {
                            loginSection.innerHTML = `
                                <button id="loginBtn" class="w-full bg-primary-500 hover:bg-primary-600 text-white font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center justify-center gap-2 shadow-sm">
                                    <i class="fas fa-sign-in-alt"></i> Login / Sign Up
                                </button>
                                <button id="settingsBtn" class="w-full bg-gray-100 hover:bg-gray-200 text-gray-800 font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center justify-center gap-2 mt-2">
                                    <i class="fas fa-cog"></i> Settings
                                </button>
                            `;
                            
                            document.getElementById('loginBtn').addEventListener('click', () => openModal(dom.modals.login));
                            document.getElementById('settingsBtn').addEventListener('click', () => openModal(dom.modals.settings));
                        }
                        
                        // Update account panel
                        const accountPanel = document.getElementById('accountPanel');
                        if (accountPanel) {
                            const content = accountPanel.querySelector('.p-5');
                            content.innerHTML = `
                                <h3 class="font-semibold text-lg mb-4">Account</h3>
                                <div class="text-center p-4">
                                    <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-3">
                                        <i class="fas fa-user text-2xl text-gray-400"></i>
                                    </div>
                                    <p class="font-medium mb-4">Not logged in</p>
                                    <button id="accountLoginBtn" class="w-full bg-primary-500 hover:bg-primary-600 text-white font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center justify-center gap-2 shadow-sm">
                                        <i class="fas fa-sign-in-alt"></i> Login / Sign Up
                                    </button>
                                </div>
                            `;
                            
                            document.getElementById('accountLoginBtn').addEventListener('click', () => {
                                toggleAccountPanel(false);
                                openModal(dom.modals.login);
                            });
                        }
                        
                        // Remove logged in indicator from account button
                        const accountBtn = document.getElementById('accountBtn');
                        if (accountBtn) {
                            const indicator = accountBtn.querySelector('span');
                            if (indicator) indicator.remove();
                        }
                        
                        // Refresh traffic lights
                        loadTrafficLights(true);
                    } else {
                        if (app.settings.notifications) {
                            showNotification('Error: ' + data.message, 'error');
                        }
                    }
                })
                .catch(error => {
                    console.error('Logout error:', error);
                    if (app.settings.notifications) {
                        showNotification('Connection error', 'error');
                    }
                });
        }
    
        function nextOnboardingSlide() {
            if (app.state.currentSlide < dom.welcome.slides.length) {
                goToSlide(app.state.currentSlide + 1);
            } else {
                closeModal(dom.modals.welcome);
            }
        }
    
        function goToSlide(slideNumber) {
            app.state.currentSlide = slideNumber;
            
            dom.welcome.slides.forEach(slide => {
                slide.classList.add('hidden');
            });
            
            dom.welcome.dots.forEach(dot => {
                dot.classList.remove('active');
            });
            
            document.querySelector(`.onboarding-slide[data-slide="${slideNumber}"]`)?.classList.remove('hidden');
            document.querySelector(`.onboarding-dot[data-dot="${slideNumber}"]`)?.classList.add('active');
            
            dom.buttons.nextSlide.textContent = slideNumber === dom.welcome.slides.length ? 'Get Started' : 'Next';
        }
    
        function startMeasure(mode) {
            app.measure.mode = mode;
            dom.measure.container.classList.remove('hidden');
            
            if (mode === 'red') {
                dom.measure.instructions.textContent = 'Press "Start" when the light turns red, then "Stop" when it turns green.';
                dom.buttons.startTimer.className = 'bg-traffic-red text-white py-2.5 rounded-xl hover:bg-red-600 transition-colors shadow-sm';
                dom.buttons.stopTimer.className = 'bg-traffic-green text-white py-2.5 rounded-xl hover:bg-green-600 transition-colors opacity-50 shadow-sm';
            } else {
                dom.measure.instructions.textContent = 'Press "Start" when the light turns green, then "Stop" when it turns red.';
                dom.buttons.startTimer.className = 'bg-traffic-green text-white py-2.5 rounded-xl hover:bg-green-600 transition-colors shadow-sm';
                dom.buttons.stopTimer.className = 'bg-traffic-red text-white py-2.5 rounded-xl hover:bg-red-600 transition-colors opacity-50 shadow-sm';
            }
            
            resetMeasureTimer();
        }
    
        function startMeasureTimer() {
            app.measure.startTime = performance.now();
            app.measure.timer = 0;
            
            if (app.measure.rafId) cancelAnimationFrame(app.measure.rafId);
            
            function updateTimer(timestamp) {
                const elapsed = Math.floor((timestamp - app.measure.startTime) / 1000);
                if (elapsed !== app.measure.timer) {
                    app.measure.timer = elapsed;
                    const minutes = Math.floor(elapsed / 60);
                    const seconds = elapsed % 60;
                    dom.measure.display.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                }
                app.measure.rafId = requestAnimationFrame(updateTimer);
            }
            
            app.measure.rafId = requestAnimationFrame(updateTimer);
            dom.measure.display.textContent = '00:00';
            dom.measure.result.textContent = 'Measuring...';
            
            dom.buttons.startTimer.disabled = true;
            dom.buttons.startTimer.classList.add('opacity-50');
            dom.buttons.stopTimer.disabled = false;
            dom.buttons.stopTimer.classList.remove('opacity-50');
            dom.buttons.saveTimer.disabled = true;
            dom.buttons.saveTimer.classList.add('opacity-50');
        }
    
        function stopMeasureTimer() {
            if (app.measure.rafId) {
                cancelAnimationFrame(app.measure.rafId);
                app.measure.rafId = null;
                
                dom.measure.result.textContent = `Measured duration: ${app.measure.timer} seconds. Click Save to confirm.`;
                
                dom.buttons.stopTimer.disabled = true;
                dom.buttons.stopTimer.classList.add('opacity-50');
                dom.buttons.saveTimer.disabled = false;
                dom.buttons.saveTimer.classList.remove('opacity-50');
            }
        }
    
        function resetMeasureTimer() {
            if (app.measure.rafId) {
                cancelAnimationFrame(app.measure.rafId);
                app.measure.rafId = null;
            }
            app.measure.timer = 0;
            app.measure.startTime = 0;
            dom.measure.display.textContent = '00:00';
            dom.measure.result.textContent = '';
            dom.buttons.startTimer.disabled = false;
            dom.buttons.startTimer.classList.remove('opacity-50');
            dom.buttons.stopTimer.disabled = true;
            dom.buttons.stopTimer.classList.add('opacity-50');
            dom.buttons.saveTimer.disabled = true;
            dom.buttons.saveTimer.classList.add('opacity-50');
        }
    
        function saveMeasureTimer() {
            const lightId = dom.measure.title.getAttribute('data-id');
            
            if (!lightId || app.measure.timer <= 0 || !app.measure.mode) return;
            
            dom.measure.result.textContent = 'Saving...';
            
            fetch('db.php?action=updateTiming', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    id: lightId,
                    duration_type: app.measure.mode,
                    duration: app.measure.timer
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    dom.measure.result.innerHTML = '<i class="fas fa-check text-green-500 mr-1"></i> Timing updated!';
                    
                    app.markers[lightId].data = data.light;
                    updateLightStatus(data.light);
                    
                    dom.buttons.saveTimer.disabled = true;
                    dom.buttons.saveTimer.classList.add('opacity-50');
                    
                    setTimeout(() => {
                        dom.measure.container.classList.add('hidden');
                        app.measure.mode = null;
                    }, 1500);
                } else {
                    dom.measure.result.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Error: ' + data.message;
                }
            })
            .catch(error => {
                console.error('Error:', error);
                dom.measure.result.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Connection error';
            });
        }
    
        function openMeasureModal(lightId) {
            const light = app.markers[lightId].data;
            app.measure.lightId = lightId;
            
            dom.measure.title.textContent = light.name;
            dom.measure.title.setAttribute('data-id', lightId);
            
            const status = getLightStatus(light);
            dom.measure.status.innerHTML = `
                <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                    ${status.label} (${status.timeLeft}s)
                </span>
            `;
            
            dom.measure.container.classList.add('hidden');
            app.measure.mode = null;
            resetMeasureTimer();
            
            openModal(dom.modals.measure);
        }
    
        function loadTrafficLights(isRefresh = false) {
            if (!isRefresh) {
                dom.loadingLights.style.display = 'block';
                dom.noLightsMessage.classList.add('hidden');
            }
            
            fetch('db.php?action=getTrafficLights')
                .then(response => response.json())
                .then(data => {
                    dom.loadingLights.style.display = 'none';
                    
                    if (isRefresh) {
                        for (let id in app.markers) {
                            if (!app.markers[id].isRouteLight) {
                                app.markers[id].marker.setMap(null);
                                delete app.markers[id];
                            }
                        }
                        
                        const lightCards = document.querySelectorAll('.light-card');
                        lightCards.forEach(card => card.remove());
                    }
                    
                    if (Array.isArray(data) && data.length > 0) {
                        data.forEach(light => {
                            addLightToMap(light);
                            addLightToSidebar(light);
                        });
                    } else {
                        dom.noLightsMessage.classList.remove('hidden');
                    }
                    
                    if (app.settings.autoRefresh) {
                        startStatusUpdates();
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    dom.loadingLights.style.display = 'none';
                    dom.noLightsMessage.classList.remove('hidden');
                    if (app.settings.notifications) {
                        showNotification('Error loading traffic lights', 'error');
                    }
                });
        }
    
        function addLightToMap(light) {
            const status = getLightStatus(light);
            const position = { lat: parseFloat(light.latitude), lng: parseFloat(light.longitude) };
            
            const marker = new google.maps.Marker({
                position: position,
                map: app.map,
                icon: {
                    url: `data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="${status.color === 'red' ? '%23ef4444' : '%2310b981'}" stroke="%23ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><path d="M12 6v6l4 2"/></svg>`,
                    size: new google.maps.Size(36, 36),
                    scaledSize: new google.maps.Size(36, 36),
                    anchor: new google.maps.Point(18, 18)
                },
                title: light.name
            });
            
            marker.addListener('click', () => showLightDetail(light));
            
            app.markers[light.id] = {
                marker: marker,
                data: light,
                isRouteLight: false
            };
        }
    
        function addLightToSidebar(light) {
            const status = getLightStatus(light);
            
            const card = document.createElement('div');
            card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg rounded-xl';
        card.setAttribute('data-id', light.id);
        card.setAttribute('data-name', light.name);
        
        card.innerHTML = `
            <div class="flex justify-between items-center">
                <h3 class="font-medium">${light.name}</h3>
                <span id="status-${light.id}" class="px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white">${status.label}</span>
            </div>
            <div class="flex items-center text-sm text-gray-500 mt-2">
                <i class="fas fa-location-dot mr-2"></i>
                <span class="capitalize">${light.direction}</span>
                <div class="ml-auto flex items-center" id="timer-${light.id}">
                    <i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s
                </div>
            </div>
            ${app.settings.predictions ? `
            <div class="mt-2 text-xs">
                <div class="w-full bg-gray-100 rounded-full h-1.5">
                    <div class="bg-traffic-${status.color} h-1.5 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                </div>
                <div class="flex justify-between mt-1 text-gray-400">
                    <span>0s</span>
                    <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                </div>
            </div>` : ''}
        `;
        
        card.addEventListener('click', function() {
            const allCards = document.querySelectorAll('.light-card');
            allCards.forEach(c => c.classList.remove('active'));
            this.classList.add('active');
            
            showLightDetail(light);
            
            if (window.innerWidth <= 768) {
                dom.sidebar.classList.remove('active');
                dom.buttons.menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
            }
        });
        
        dom.lightsList.insertBefore(card, dom.loadingLights);
    }

    function getLightStatus(light) {
        const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
        const currentTime = Math.floor(Date.now() / 1000);
        const timeInCycle = currentTime % totalCycle;
        
        if (timeInCycle < light.red_duration) {
            return {
                isRed: true,
                color: 'red',
                label: 'RED',
                timeLeft: light.red_duration - timeInCycle
            };
        } else {
            return {
                isRed: false,
                color: 'green',
                label: 'GREEN',
                timeLeft: totalCycle - timeInCycle
            };
        }
    }

    function updateLightStatus(light) {
        const status = getLightStatus(light);
        
        if (app.markers[light.id]) {
            const marker = app.markers[light.id].marker;
            
            // Update marker icon
            marker.setIcon({
                url: `data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="${status.color === 'red' ? '%23ef4444' : '%2310b981'}" stroke="%23ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><path d="M12 6v6l4 2"/></svg>`,
                size: new google.maps.Size(36, 36),
                scaledSize: new google.maps.Size(36, 36),
                anchor: new google.maps.Point(18, 18)
            });
        }
        
        const statusElement = document.getElementById(`status-${light.id}`);
        const timerElement = document.getElementById(`timer-${light.id}`);
        
        if (statusElement) {
            statusElement.className = `px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white`;
            statusElement.textContent = status.label;
        }
        
        if (timerElement) {
            timerElement.innerHTML = `<i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s`;
        }
        
        const card = document.querySelector(`.light-card[data-id="${light.id}"]`);
        if (card && app.settings.predictions) {
            let predictionEl = card.querySelector('.mt-2.text-xs');
            
            if (predictionEl) {
                const progressBar = predictionEl.querySelector('.bg-traffic-red, .bg-traffic-green');
                if (progressBar) {
                    progressBar.className = `bg-traffic-${status.color} h-1.5 rounded-full`;
                    progressBar.style.width = `${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%`;
                }
            } else {
                predictionEl = document.createElement('div');
                predictionEl.className = 'mt-2 text-xs';
                predictionEl.innerHTML = `
                    <div class="w-full bg-gray-100 rounded-full h-1.5">
                        <div class="bg-traffic-${status.color} h-1.5 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                    </div>
                    <div class="flex justify-between mt-1 text-gray-400">
                        <span>0s</span>
                        <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                    </div>
                `;
                card.appendChild(predictionEl);
            }
        } else if (card && !app.settings.predictions) {
            const predictionEl = card.querySelector('.mt-2.text-xs');
            if (predictionEl) {
                predictionEl.remove();
            }
        }
        
        if (app.selectedLightId === light.id && !dom.lightPopup.classList.contains('hidden')) {
            const popupStatus = dom.popupStatus;
            popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
            popupStatus.innerHTML = `
                <div class="text-xl font-semibold mb-1">${status.label}</div>
                <div class="flex justify-center items-center gap-2">
                    <i class="fas fa-clock"></i>
                    <span>Changes in ${status.timeLeft} seconds</span>
                </div>
                ${app.settings.predictions ? `
                <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                    <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                </div>
                ` : ''}
            `;
        }
        
        if (app.measure.lightId === light.id && dom.modals.measure.style.display === 'flex') {
            dom.measure.status.innerHTML = `
                <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                    ${status.label} (${status.timeLeft}s)
                </span>
            `;
        }
    }

    function updateAllLightStatus() {
        for (const id in app.markers) {
            if (!app.markers[id].isRouteLight) {
                updateLightStatus(app.markers[id].data);
            }
        }
    }

    function startStatusUpdates() {
        stopStatusUpdates();
        
        app.refreshTimerId = setInterval(() => {
            updateAllLightStatus();
        }, 1000);
    }

    function stopStatusUpdates() {
        if (app.refreshTimerId) {
            clearInterval(app.refreshTimerId);
            app.refreshTimerId = null;
        }
    }

    function showLightDetail(light) {
        app.selectedLightId = light.id;
        
        app.map.panTo({
            lat: parseFloat(light.latitude),
            lng: parseFloat(light.longitude)
        });
        app.map.setZoom(17);
        
        dom.popupTitle.textContent = light.name;
        dom.popupContent.innerHTML = `
            <div class="grid grid-cols-2 gap-3">
                <div class="text-gray-500">Direction:</div>
                <div class="font-medium capitalize">${light.direction}</div>
                
                <div class="text-gray-500">Red duration:</div>
                <div class="font-medium">${light.red_duration} seconds</div>
                
                <div class="text-gray-500">Green duration:</div>
                <div class="font-medium">${light.green_duration} seconds</div>
                
                <div class="text-gray-500">Total cycle:</div>
                <div class="font-medium">${parseInt(light.red_duration) + parseInt(light.green_duration)} seconds</div>
                
                <div class="text-gray-500">Location:</div>
                <div class="font-medium truncate">${light.latitude.substring(0, 8)}, ${light.longitude.substring(0, 8)}</div>
            </div>
        `;
        
        const status = getLightStatus(light);
        
        dom.popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
        dom.popupStatus.innerHTML = `
            <div class="text-xl font-semibold mb-1">${status.label}</div>
            <div class="flex justify-center items-center gap-2">
                <i class="fas fa-clock"></i>
                <span>Changes in ${status.timeLeft} seconds</span>
            </div>
            ${app.settings.predictions ? `
            <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
            </div>
            ` : ''}
        `;
        
        dom.lightPopup.classList.remove('hidden');
    }

    function startGeolocation() {
        if (navigator.geolocation) {
            if (app.userWatchId) navigator.geolocation.clearWatch(app.userWatchId);
            
            dom.buttons.headerLocate.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>';
            
            app.userWatchId = navigator.geolocation.watchPosition(
                updateUserLocation,
                handleGeolocationError,
                { enableHighAccuracy: true, maximumAge: 10000, timeout: 5000 }
            );
            
            navigator.geolocation.getCurrentPosition(
                updateUserLocation,
                handleGeolocationError,
                { enableHighAccuracy: true }
            );
        }
    }

    function updateUserLocation(position) {
        const lat = position.coords.latitude;
        const lng = position.coords.longitude;
        const pos = { lat, lng };
        
        dom.buttons.headerLocate.innerHTML = '<i class="fas fa-location-crosshairs"></i>';
        
        if (!app.userMarker) {
            app.userMarker = new google.maps.Marker({
                position: pos,
                map: app.map,
                icon: {
                    path: google.maps.SymbolPath.CIRCLE,
                    scale: 8,
                    fillColor: '#0ea5e9',
                    fillOpacity: 1,
                    strokeColor: '#ffffff',
                    strokeWeight: 2
                },
                zIndex: 1000
            });
            
            app.map.panTo(pos);
            app.map.setZoom(16);
        } else {
            app.userMarker.setPosition(pos);
        }
    }

    function handleGeolocationError(error) {
        console.error('Geolocation error:', error.message);
        
        dom.buttons.headerLocate.innerHTML = '<i class="fas fa-location-crosshairs"></i>';
        
        if (app.settings.notifications) {
            showNotification('Location access denied', 'error');
        }
    }

    function highlightNearbyLights() {
        if (!app.userMarker) return;
        
        const userPos = app.userMarker.getPosition();
        const nearbyLights = [];
        
        for (const id in app.markers) {
            if (app.markers[id].isRouteLight) continue;
            
            const markerPos = app.markers[id].marker.getPosition();
            const distance = getDistanceFromLatLng(
                userPos.lat(),
                userPos.lng(),
                markerPos.lat(),
                markerPos.lng()
            );
            
            if (distance < 1000) {
                nearbyLights.push({
                    id: id,
                    distance: distance
                });
            }
        }
        
        nearbyLights.sort((a, b) => a.distance - b.distance);
        
        document.querySelectorAll('.light-card').forEach(card => {
            card.classList.remove('active');
        });
        
        if (nearbyLights.length > 0) {
            if (app.settings.notifications) {
                showNotification(`${nearbyLights.length} traffic lights found within 1km`, 'info');
            }
            
            nearbyLights.slice(0, 5).forEach(light => {
                const card = document.querySelector(`.light-card[data-id="${light.id}"]`);
                if (card) {
                    card.classList.add('active');
                    card.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                }
            });
        } else {
            if (app.settings.notifications) {
                showNotification('No traffic lights nearby', 'info');
            }
        }
    }

    function showNotification(message, type) {
        const colors = {
            success: 'bg-traffic-green',
            error: 'bg-traffic-red',
            info: 'bg-primary-500'
        };
        
        const icons = {
            success: 'check-circle',
            error: 'exclamation-circle',
            info: 'info-circle'
        };
        
        const notification = document.createElement('div');
        notification.className = `fixed top-20 left-1/2 transform -translate-x-1/2 ${colors[type]} text-white px-4 py-3 rounded-lg shadow-lg z-50 fade-in flex items-center`;
        notification.innerHTML = `<i class="fas fa-${icons[type]} mr-2"></i>${message}`;
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translate(-50%, -10px)';
            notification.style.transition = 'opacity 0.5s, transform 0.5s';
            setTimeout(() => notification.remove(), 500);
        }, 3000);
    }

    function handleOnline() {
        if (app.settings.notifications) {
            showNotification('You are back online', 'success');
        }
        loadTrafficLights(true);
    }

    function handleOffline() {
        if (app.settings.notifications) {
            showNotification('You are offline. Some features may be limited', 'error');
        }
    }

    function handleVisibilityChange() {
        if (document.visibilityState === 'visible') {
            loadTrafficLights(true);
        }
    }

    function handleInstallPrompt() {
        let deferredPrompt;
        
        window.addEventListener('beforeinstallprompt', (e) => {
            e.preventDefault();
            deferredPrompt = e;
            
            const installBanner = document.createElement('div');
            installBanner.className = 'fixed bottom-0 left-0 right-0 bg-primary-500 text-white py-3 px-4 shadow-lg z-40 fade-in';
            installBanner.innerHTML = `
                <div class="flex items-center justify-between max-w-md mx-auto">
                    <div>
                        <p class="font-medium">Add TrafficLight to Home Screen</p>
                        <p class="text-sm">Get quick access to traffic lights data</p>
                    </div>
                    <div class="flex space-x-2">
                        <button id="dismissInstall" class="px-3 py-1.5 bg-primary-600 hover:bg-primary-700 rounded font-medium">Later</button>
                        <button id="installApp" class="px-3 py-1.5 bg-white text-primary-600 hover:bg-gray-100 rounded font-medium">Install</button>
                    </div>
                </div>
            `;
            
           document.body.appendChild(installBanner);
            
            document.getElementById('dismissInstall').addEventListener('click', () => {
                installBanner.style.opacity = '0';
                installBanner.style.transform = 'translateY(100%)';
                installBanner.style.transition = 'opacity 0.5s, transform 0.5s';
                setTimeout(() => installBanner.remove(), 500);
            });
            
            document.getElementById('installApp').addEventListener('click', () => {
                deferredPrompt.prompt();
                deferredPrompt.userChoice.then((choiceResult) => {
                    if (choiceResult.outcome === 'accepted') {
                        console.log('User accepted the install prompt');
                        installBanner.style.opacity = '0';
                        installBanner.style.transform = 'translateY(100%)';
                        installBanner.style.transition = 'opacity 0.5s, transform 0.5s';
                        setTimeout(() => installBanner.remove(), 500);
                    }
                    deferredPrompt = null;
                });
            });
        });
    }
    
    handleInstallPrompt();
});