<?php
session_start();
require_once 'db.php';

// Limite de tentatives de connexion
$max_attempts = 5;
$lockout_time = 15 * 60; // 15 minutes

function recordLoginAttempt($ip) {
    $database = new Database();
    $conn = $database->getConnection();
    
    $stmt = $conn->prepare("INSERT INTO login_attempts (ip_address, attempt_time) VALUES (:ip, NOW())");
    $stmt->bindParam(':ip', $ip);
    $stmt->execute();
}

function checkLoginAttempts($ip) {
    $database = new Database();
    $conn = $database->getConnection();
    
    $stmt = $conn->prepare("
        SELECT COUNT(*) as attempts 
        FROM login_attempts 
        WHERE ip_address = :ip 
        AND attempt_time > DATE_SUB(NOW(), INTERVAL 15 MINUTE)
    ");
    $stmt->bindParam(':ip', $ip);
    $stmt->execute();
    
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result['attempts'];
}

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $ip_address = $_SERVER['REMOTE_ADDR'];
    $login_attempts = checkLoginAttempts($ip_address);
    
    if ($login_attempts >= 5) {
        $error = "Trop de tentatives. Veuillez attendre 15 minutes.";
    } else {
        $code = $_POST['code'] ?? '';
        
        $database = new Database();
        $conn = $database->getConnection();
        
        $stmt = $conn->prepare("SELECT * FROM users WHERE code = :code");
        $stmt->bindParam(':code', $code);
        $stmt->execute();
        
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user) {
            // Enregistrement de la connexion réussie
            $stmt = $conn->prepare("
                INSERT INTO user_logins (user_id, login_time, ip_address) 
                VALUES (:user_id, NOW(), :ip_address)
            ");
            $stmt->bindParam(':user_id', $user['id']);
            $stmt->bindParam(':ip_address', $ip_address);
            $stmt->execute();
            
            // Nettoyer les tentatives de connexion précédentes
            $stmt = $conn->prepare("
                DELETE FROM login_attempts 
                WHERE ip_address = :ip
            ");
            $stmt->bindParam(':ip', $ip_address);
            $stmt->execute();
            
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['login_time'] = time();
            
            // Régénérer l'ID de session pour prévenir la fixation de session
            session_regenerate_id(true);
            
            header("Location: dashboard.php");
            exit();
        } else {
            // Enregistrer la tentative de connexion échouée
            recordLoginAttempt($ip_address);
            $error = "Code invalide";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <title>Peerkinton - Connexion</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300&display=swap" rel="stylesheet">
    <style>
        body { 
            font-family: 'Poppins', sans-serif; 
            background: linear-gradient(135deg, #000000 0%, #434343 100%);
        }
        .login-container {
            background: rgba(0,0,0,0.7);
            border: 1px solid rgba(255,255,255,0.1);
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
    </style>
</head>
<body class="flex items-center justify-center min-h-screen">
    <div class="w-full max-w-md p-8">
        <div class="login-container p-8 rounded-lg">
            <h2 class="text-3xl mb-6 text-center text-white">Peerkinton</h2>
            
            <?php if(isset($error)): ?>
                <div class="bg-red-600 text-white p-4 rounded mb-4">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <form method="post" class="space-y-4">
                <input type="text" name="code" required 
                    pattern="\d{12}"
                    maxlength="12"
                    class="w-full p-3 bg-gray-800 text-white rounded focus:outline-none focus:ring-2 focus:ring-gray-600" 
                    placeholder="Code de connexion (12 chiffres)"
                    title="Entrez un code de 12 chiffres">
                
                <button type="submit" 
                    class="w-full bg-white text-black p-3 rounded hover:bg-gray-200 transition duration-300">
                    Se connecter
                </button>
            </form>
        </div>
        
        <div class="text-center text-white mt-4 text-sm">
            <p>&copy; 2025 Peerkinton. Tous droits réservés.</p>
        </div>
    </div>

    <script>
    document.querySelector('form').addEventListener('submit', function(e) {
        const codeInput = this.querySelector('input[name="code"]');
        const code = codeInput.value.replace(/\D/g, '');
        
        if (code.length !== 12) {
            e.preventDefault();
            alert('Le code doit contenir exactement 12 chiffres.');
            codeInput.value = code;
            codeInput.focus();
        }
    });
    </script>
</body>
</html>