<?php
session_start();
require_once 'db.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: search.php');
    exit;
}

$profile_id = $_GET['id'];
$profile = null;
$gallery_images = [];
$alertMessage = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'report_sighting') {
    $latitude = $_POST['latitude'];
    $longitude = $_POST['longitude'];
    $description = $_POST['description'];
    
    try {
        $stmt = $pdo->prepare("INSERT INTO profile_alerts (user_id, profile_id, latitude, longitude, description, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $stmt->execute([$user_id, $profile_id, $latitude, $longitude, $description]);
        
        $alertMessage = 'Sighting reported successfully';
    } catch (PDOException $e) {
        $alertMessage = 'Error reporting sighting: ' . $e->getMessage();
    }
}

try {
    $stmt = $pdo->prepare("SELECT p.*, u.username as created_by_name 
                          FROM profiles p 
                          JOIN users u ON p.created_by = u.id 
                          WHERE p.id = ?");
    $stmt->execute([$profile_id]);
    $profile = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$profile) {
        header('Location: search.php');
        exit;
    }
    
    $stmt = $pdo->prepare("SELECT image_url FROM profile_images WHERE profile_id = ? ORDER BY created_at");
    $stmt->execute([$profile_id]);
    $gallery_images = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $stmt = $pdo->prepare("SELECT pa.*, u.username 
                          FROM profile_alerts pa 
                          JOIN users u ON pa.user_id = u.id 
                          WHERE pa.profile_id = ? 
                          ORDER BY pa.created_at DESC 
                          LIMIT 10");
    $stmt->execute([$profile_id]);
    $recent_sightings = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    $error = "Database error: " . $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="en-GB">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($profile['name'] ?? 'Profile'); ?> - Peerkinton</title>
    <link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.3/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.9.3/dist/leaflet.js"></script>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        jost: ['Jost', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style>
        body {
            font-family: 'Jost', sans-serif;
        }
        #map {
            height: 300px;
            width: 100%;
        }
        .gallery-img {
            aspect-ratio: 1/1;
            object-fit: cover;
        }
    </style>
</head>
<body class="bg-gray-50 text-black font-jost">
    <div class="min-h-screen flex flex-col">
        <header class="bg-white py-4 px-6 border-b border-gray-200">
            <div class="flex justify-between items-center">
                <h1 class="text-2xl font-light">Peerkinton</h1>
                <div class="flex items-center space-x-4">
                    <span><?php echo htmlspecialchars($user['username']); ?></span>
                    <a href="logout" class="text-sm text-gray-600">Logout</a>
                </div>
            </div>
        </header>
        
        <nav class="bg-white py-3 px-6 border-b border-gray-200">
            <div class="flex space-x-6">
                <a href="dashboard" class="text-gray-500 hover:text-gray-800">Map</a>
                <a href="search" class="text-gray-500 hover:text-gray-800">Library</a>
                <a href="create-profile" class="text-gray-500 hover:text-gray-800">Create Profile</a>
                <a href="dashboard?tab=settings" class="text-gray-500 hover:text-gray-800">Settings</a>
            </div>
        </nav>
        
        <?php if (!empty($alertMessage)): ?>
        <div class="bg-green-100 text-green-800 px-6 py-3">
            <?php echo htmlspecialchars($alertMessage); ?>
        </div>
        <?php endif; ?>
        
        <main class="flex-1 p-6">
            <div class="max-w-4xl mx-auto">
                <?php if (isset($error)): ?>
                <div class="bg-red-100 text-red-800 p-4 mb-6">
                    <?php echo $error; ?>
                </div>
                <?php endif; ?>
                
                <div class="mb-4">
                    <a href="search" class="text-gray-600 hover:text-black flex items-center">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-4 w-4 mr-1" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                        </svg>
                        Back to Library
                    </a>
                </div>
                
                <div class="bg-white border border-gray-200 rounded-lg overflow-hidden mb-6">
                    <div class="md:flex">
                        <div class="md:w-1/3">
                            <?php if (!empty($profile['profile_image'])): ?>
                            <img src="<?php echo htmlspecialchars($profile['profile_image']); ?>" 
                                alt="<?php echo htmlspecialchars($profile['name']); ?>" 
                                class="w-full h-full object-cover aspect-square">
                            <?php else: ?>
                            <div class="flex items-center justify-center h-full aspect-square bg-gray-100 text-gray-500">
                                No profile image
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="p-6 md:w-2/3">
                            <h1 class="text-3xl font-medium mb-2"><?php echo htmlspecialchars($profile['name']); ?></h1>
                            
                            <div class="flex items-center text-gray-500 text-sm mb-4">
                                <span>Added by <?php echo htmlspecialchars($profile['created_by_name']); ?></span>
                                <span class="mx-2">•</span>
                                <span><?php echo date('F j, Y', strtotime($profile['created_at'])); ?></span>
                            </div>
                            
                            <div class="prose max-w-none mb-6">
                                <p><?php echo nl2br(htmlspecialchars($profile['description'])); ?></p>
                            </div>
                            
                            <div class="flex space-x-4">
                                <button id="reportSightingBtn" class="px-4 py-2 bg-black text-white">Report Sighting</button>
                                <?php if ($profile['created_by'] == $user_id): ?>
                                <a href="edit-profile?id=<?php echo $profile_id; ?>" class="px-4 py-2 bg-gray-200 text-gray-800">Edit Profile</a>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <?php if (!empty($gallery_images)): ?>
                    <div class="border-t border-gray-200 p-6">
                        <h2 class="text-xl font-medium mb-4">Gallery</h2>
                        <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-4">
                            <?php foreach ($gallery_images as $image): ?>
                            <a href="<?php echo htmlspecialchars($image); ?>" target="_blank">
                                <img src="<?php echo htmlspecialchars($image); ?>" alt="Gallery image" class="gallery-img rounded-lg shadow-sm hover:shadow-md transition-shadow">
                            </a>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div class="bg-white border border-gray-200 rounded-lg overflow-hidden">
                        <div class="p-6">
                            <h2 class="text-xl font-medium mb-4">Recent Sightings</h2>
                            
                            <?php if (!empty($recent_sightings)): ?>
                            <div class="space-y-4">
                                <?php foreach ($recent_sightings as $sighting): ?>
                                <div class="border-b border-gray-100 pb-4 last:border-0 last:pb-0">
                                    <div class="flex justify-between">
                                        <div>
                                            <p class="font-medium"><?php echo htmlspecialchars($sighting['username']); ?></p>
                                            <p class="text-sm text-gray-600"><?php echo htmlspecialchars($sighting['description']); ?></p>
                                        </div>
                                        <div class="text-sm text-gray-500">
                                            <?php echo date('M j, g:ia', strtotime($sighting['created_at'])); ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <?php else: ?>
                            <p class="text-gray-500">No sightings reported yet</p>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="bg-white border border-gray-200 rounded-lg overflow-hidden">
                        <div class="p-6">
                            <h2 class="text-xl font-medium mb-4">Locations Map</h2>
                            <div id="map"></div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
        
        <div id="reportSightingModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center hidden z-50">
            <div class="bg-white rounded-lg max-w-md w-full p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 class="text-xl font-medium">Report Sighting</h3>
                    <button id="closeModalBtn" class="text-gray-500 hover:text-gray-700">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                        </svg>
                    </button>
                </div>
                
                <form method="POST">
                    <input type="hidden" name="action" value="report_sighting">
                    <input type="hidden" name="latitude" id="sightingLat">
                    <input type="hidden" name="longitude" id="sightingLng">
                    
                    <div class="mb-4">
                        <label class="block text-sm text-gray-700 mb-1">Location</label>
                        <div class="flex space-x-2">
                            <input type="text" id="sightingLocation" class="w-full px-3 py-2 border border-gray-300 focus:outline-none" readonly placeholder="Click on map to set location">
                            <button type="button" id="useCurrentLocationBtn" class="px-3 py-2 text-sm bg-gray-200">Current</button>
                        </div>
                    </div>
                    
                    <div class="mb-4">
                        <div id="modalMap" class="h-48 bg-gray-100 w-full mb-2"></div>
                        <p class="text-xs text-gray-500">Click on the map to set the sighting location</p>
                    </div>
                    
                    <div class="mb-4">
                        <label for="description" class="block text-sm text-gray-700 mb-1">Description</label>
                        <textarea name="description" id="description" rows="3" class="w-full px-3 py-2 border border-gray-300 focus:outline-none" placeholder="Describe the sighting..."></textarea>
                    </div>
                    
                    <button type="submit" class="w-full bg-black text-white py-2">Report Sighting</button>
                </form>
            </div>
        </div>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize main map
            const map = L.map('map');
            
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
            }).addTo(map);
            
            // Add sighting markers to map
            const sightings = <?php echo json_encode($recent_sightings ?? []); ?>;
            const bounds = [];
            
            if (sightings.length > 0) {
                sightings.forEach(sighting => {
                    const marker = L.marker([sighting.latitude, sighting.longitude]).addTo(map);
                    marker.bindPopup(`
                        <b>${sighting.username}</b><br>
                        ${sighting.description}<br>
                        <small>${new Date(sighting.created_at).toLocaleString()}</small>
                    `);
                    bounds.push([sighting.latitude, sighting.longitude]);
                });
                
                if (bounds.length > 0) {
                    map.fitBounds(bounds);
                } else {
                    map.setView([0, 0], 2);
                }
            } else {
                map.setView([0, 0], 2);
            }
            
            // Report sighting modal
            const reportSightingBtn = document.getElementById('reportSightingBtn');
            const closeModalBtn = document.getElementById('closeModalBtn');
            const reportSightingModal = document.getElementById('reportSightingModal');
            
            reportSightingBtn.addEventListener('click', function() {
                reportSightingModal.classList.remove('hidden');
                
                // Initialize modal map
                setTimeout(() => {
                    const modalMap = L.map('modalMap').setView([0, 0], 2);
                    
                    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                        attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
                    }).addTo(modalMap);
                    
                    let modalMarker = null;
                    
                    modalMap.on('click', function(e) {
                        if (modalMarker) {
                            modalMap.removeLayer(modalMarker);
                        }
                        
                        modalMarker = L.marker(e.latlng).addTo(modalMap);
                        
                        document.getElementById('sightingLat').value = e.latlng.lat;
                        document.getElementById('sightingLng').value = e.latlng.lng;
                        document.getElementById('sightingLocation').value = `${e.latlng.lat.toFixed(6)}, ${e.latlng.lng.toFixed(6)}`;
                    });
                    
                    document.getElementById('useCurrentLocationBtn').addEventListener('click', function() {
                        if (navigator.geolocation) {
                            navigator.geolocation.getCurrentPosition(function(position) {
                                const lat = position.coords.latitude;
                                const lng = position.coords.longitude;
                                
                                if (modalMarker) {
                                    modalMap.removeLayer(modalMarker);
                                }
                                
                                modalMarker = L.marker([lat, lng]).addTo(modalMap);
                                modalMap.setView([lat, lng], 15);
                                
                                document.getElementById('sightingLat').value = lat;
                                document.getElementById('sightingLng').value = lng;
                                document.getElementById('sightingLocation').value = `${lat.toFixed(6)}, ${lng.toFixed(6)}`;
                            });
                        }
                    });
                    
                    if (bounds.length > 0) {
                        modalMap.fitBounds(bounds);
                    }
                }, 100);
            });
            
            closeModalBtn.addEventListener('click', function() {
                reportSightingModal.classList.add('hidden');
            });
            
            reportSightingModal.addEventListener('click', function(e) {
                if (e.target === reportSightingModal) {
                    reportSightingModal.classList.add('hidden');
                }
            });
        });
    </script>
</body>
</html>