<?php
require_once 'db.php';
session_start();

$errors = [];
$success = false;

if (isset($_FILES['profile-picture'])) {
    $uploadDir = '/../cdn.imators.com/uploads/';
    $cdnPrefix = 'https://cdn.imators.com/uploads/';
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    $maxFileSize = 5 * 1024 * 1024;

    if ($_FILES['profile-picture']['error'] === UPLOAD_ERR_OK) {
        $fileType = mime_content_type($_FILES['profile-picture']['tmp_name']);
        $fileSize = $_FILES['profile-picture']['size'];

        if (in_array($fileType, $allowedTypes) && $fileSize <= $maxFileSize) {
            $fileName = uniqid() . '_' . bin2hex(random_bytes(8)) . '.' . pathinfo($_FILES['profile-picture']['name'], PATHINFO_EXTENSION);
            $uploadPath = $uploadDir . $fileName;
            $cdnUrl = $cdnPrefix . $fileName;

            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            if (move_uploaded_file($_FILES['profile-picture']['tmp_name'], $uploadPath)) {
                $_SESSION['profile-picture'] = $cdnUrl;
            } else {
                $errors[] = "Error uploading image";
            }
        } else {
            $errors[] = "Invalid file type or size";
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db = new Database();
    $conn = $db->connect();

    $username = htmlspecialchars(trim($_POST['username']));
    $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
    $password = $_POST['password'];
    $profilePicture = $_SESSION['profile-picture'] ?? null;
    
    $errors = [];

    if (empty($username) || strlen($username) < 3) {
        $errors[] = "Username must be at least 3 characters long";
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Invalid email address";
    }

    $stmt = $conn->prepare("SELECT id FROM utilisateurs WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->rowCount() > 0) {
        $errors[] = "Email already exists";
    }

    $passwordErrors = [];
    if (strlen($password) < 8) {
        $passwordErrors[] = "Password must be at least 8 characters long";
    }
    if (!preg_match("/[A-Z]/", $password)) {
        $passwordErrors[] = "Password must contain at least one uppercase letter";
    }
    if (!preg_match("/[a-z]/", $password)) {
        $passwordErrors[] = "Password must contain at least one lowercase letter";
    }
    if (!preg_match("/[0-9]/", $password)) {
        $passwordErrors[] = "Password must contain at least one number";
    }
    $errors = array_merge($errors, $passwordErrors);

    if (empty($errors)) {
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $verificationCode = sprintf("%06d", mt_rand(0, 999999));
        
        try {
            $stmt = $conn->prepare("INSERT INTO utilisateurs (username, email, password, verification_code, `profile-picture`) VALUES (?, ?, ?, ?, ?)");
            if($stmt->execute([$username, $email, $hashedPassword, $verificationCode, $profilePicture])) {
                $_SESSION['pending_email'] = $email;
                header('Location: verify.php');
                exit;
            } else {
                $errors[] = "Registration failed";
            }
        } catch (PDOException $e) {
            $errors[] = "Registration failed: " . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Imators</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            font-family: 'Poppins', sans-serif;
        }
        .password-check.valid {
            color: #22c55e;
        }
        .password-check.invalid {
            color: #ef4444;
        }
    </style>
</head>
<body class="bg-black text-white min-h-screen flex items-center justify-center">
    <div class="w-full max-w-md p-8">
        <div class="text-center mb-8">
            <img src="https://cdn.imators.com/logo.png" alt="Logo" class="mx-left mb-4" style="height: 60px;">
            <h1 class="text-3xl text-left text-green-200">Creating your account</h1>
            <p class="mt-2 text-xs text-left">By creating an Imators account, you agree to be aware of <a class="underline text-green-300" href="https://imators.com/terms-of-use">our terms and condition</a></p>
        </div>

        <?php if (!empty($errors)): ?>
            <div class="bg-red-500 bg-opacity-20 border border-red-500 text-red-500 p-4 rounded-lg mb-6">
                <ul class="list-disc pl-4">
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <form method="POST" enctype="multipart/form-data" class="space-y-6" id="registerForm">
            <div>
                <label class="block text-sm font-medium mb-2">Username</label>
                <input type="text" name="username" required 
                    value="<?php echo isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''; ?>"
                    class="w-full px-4 py-2 rounded-lg bg-gray-900 border border-gray-800 focus:border-white focus:outline-none 
                    <?php echo isset($errors['username']) ? 'border-red-500' : ''; ?>">
            </div>

            <div>
                <label class="block text-sm font-medium mb-2">Email</label>
                <input type="email" name="email" required 
                    value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>"
                    class="w-full px-4 py-2 rounded-lg bg-gray-900 border border-gray-800 focus:border-white focus:outline-none
                    <?php echo isset($errors['email']) ? 'border-red-500' : ''; ?>">
            </div>

            <div>
                <label class="block text-sm font-medium mb-2">Profile Picture (Optional)</label>
                <input type="file" name="profile_picture" accept="image/*"
                    class="w-full px-4 py-2 rounded-lg bg-gray-900 border border-gray-800 focus:border-white focus:outline-none">
                
                <?php if (!empty($_SESSION['profile_picture'])): ?>
                    <img src="<?php echo $_SESSION['profile_picture']; ?>" alt="Profile Preview" 
                         class="mt-4 rounded-full w-32 h-32 object-cover mx-auto">
                <?php endif; ?>
            </div>

            <div>
                <label class="block text-sm font-medium mb-2">Password</label>
                <input type="password" name="password" id="password" required 
                    class="w-full px-4 py-2 rounded-lg bg-gray-900 border border-gray-800 focus:border-white focus:outline-none
                    <?php echo isset($errors['password']) ? 'border-red-500' : ''; ?>">
                
                <ul class="text-gray-400 text-xs mt-2 space-y-1">
                    <li class="password-check" id="length-check">
                        <i class="fas fa-times-circle"></i> At least 8 characters
                    </li>
                    <li class="password-check" id="uppercase-check">
                        <i class="fas fa-times-circle"></i> One uppercase letter
                    </li>
                    <li class="password-check" id="lowercase-check">
                        <i class="fas fa-times-circle"></i> One lowercase letter
                    </li>
                    <li class="password-check" id="number-check">
                        <i class="fas fa-times-circle"></i> One number
                    </li>
                </ul>
            </div>
                      

            <button type="submit" id="submitBtn" 
                class="w-full bg-white text-black py-2 px-4 rounded-lg font-medium hover:bg-gray-100 transition-colors">
                Register
            </button>
        </form>

        <p class="mt-4 text-center text-sm">
            <a href="login.php" class="text-gray-400 hover:text-white">Already have an account? Login</a>
        </p>
    </div>

    <script>
        const password = document.getElementById('password');
        const lengthCheck = document.getElementById('length-check');
        const uppercaseCheck = document.getElementById('uppercase-check');
        const lowercaseCheck = document.getElementById('lowercase-check');
        const numberCheck = document.getElementById('number-check');

        const requirements = {
            length: false,
            uppercase: false,
            lowercase: false,
            number: false
        };

        function updateRequirement(element, valid) {
            element.classList.toggle('valid', valid);
            element.classList.toggle('invalid', !valid);
            element.querySelector('i').className = valid ? 'fas fa-check-circle' : 'fas fa-times-circle';
        }

        function checkPassword(password) {
            requirements.length = password.length >= 8;
            requirements.uppercase = /[A-Z]/.test(password);
            requirements.lowercase = /[a-z]/.test(password);
            requirements.number = /[0-9]/.test(password);

            updateRequirement(lengthCheck, requirements.length);
            updateRequirement(uppercaseCheck, requirements.uppercase);
            updateRequirement(lowercaseCheck, requirements.lowercase);
            updateRequirement(numberCheck, requirements.number);
        }

        password.addEventListener('input', (e) => {
            checkPassword(e.target.value);
        });

        checkPassword(password.value);
    </script>
</body>
</html>