<?php 
require_once 'config.php';

$query = $pdo->query("
    SELECT 
        c.*,
        u.username as author_name,
        COUNT(DISTINCT CASE WHEN i.type = 'view' THEN i.id END) as view_count,
        COUNT(DISTINCT CASE WHEN i.type = 'like' THEN i.id END) as like_count,
        COUNT(DISTINCT CASE WHEN i.type = 'dislike' THEN i.id END) as dislike_count,
        COUNT(DISTINCT CASE WHEN i.type = 'comment' THEN i.id END) as comment_count
    FROM content c
    LEFT JOIN users u ON c.author_id = u.id
    LEFT JOIN interactions i ON c.id = i.content_id
    GROUP BY c.id
    ORDER BY c.created_at DESC
");

$videos = $query->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ForMore</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Titillium+Web:wght@200;300;400;600;700;900&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Titillium Web', sans-serif;
        }
        .formore {
            font-weight: 900;
            background: linear-gradient(to right, #3b82f6, #1d4ed8);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        .video-card {
            opacity: 0;
            transform: translateY(20px);
            animation: fadeInUp 0.6s ease forwards;
        }
        @keyframes fadeInUp {
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        .menu {
            transition: transform 0.3s ease-in-out;
        }
        .menu.active {
            transform: translateX(0);
        }
        @media (max-width: 768px) {
            .menu {
                transform: translateX(100%);
            }
        }
        .hamburger span {
            transition: all 0.3s ease-in-out;
        }
        .hamburger.active span:nth-child(1) {
            transform: rotate(45deg) translate(5px, 5px);
        }
        .hamburger.active span:nth-child(2) {
            opacity: 0;
        }
        .hamburger.active span:nth-child(3) {
            transform: rotate(-45deg) translate(5px, -5px);
        }
    </style>
</head>
<body class="bg-gradient-to-br from-black to-blue-900 min-h-screen text-white">
    <!-- Header -->
    <header class="fixed top-0 left-0 right-0 bg-black/50 backdrop-blur-sm z-40">
        <div class="max-w-7xl mx-auto px-4 py-4">
            <nav class="flex justify-between items-center">
                <!-- Logo -->
                <a href="index.php" class="text-4xl formore z-50">ForMore</a>

                <!-- Mobile Menu Button -->
                <button class="hamburger lg:hidden flex flex-col gap-1.5 z-50" onclick="toggleMenu()">
                    <span class="block w-8 h-0.5 bg-white"></span>
                    <span class="block w-8 h-0.5 bg-white"></span>
                    <span class="block w-8 h-0.5 bg-white"></span>
                </button>

                <!-- Desktop/Mobile Menu -->
                <div class="menu fixed lg:relative top-0 right-0 h-screen lg:h-auto w-3/4 lg:w-auto 
                           bg-black/90 lg:bg-transparent backdrop-blur-sm lg:backdrop-blur-none
                           flex flex-col lg:flex-row items-center justify-center lg:justify-end gap-8 lg:gap-6
                           lg:translate-x-0">
                    <?php if(isset($_SESSION['user_id'])): ?>
                        <a href="profile.php" class="text-gray-300 hover:text-white transition-colors">
                            <?= htmlspecialchars($_SESSION['username']) ?>
                        </a>
                        <a href="upload.php" class="bg-blue-600 hover:bg-blue-700 px-6 py-2 rounded-xl transition-all">
                            Upload
                        </a>
                        <a href="logout.php" class="text-red-400 hover:text-red-300 transition-colors">
                            Logout
                        </a>
                    <?php else: ?>
                        <a href="login.php" class="text-blue-400 hover:text-blue-300 transition-colors">
                            Login
                        </a>
                        <a href="signup.php" class="bg-blue-600 hover:bg-blue-700 px-6 py-2 rounded-xl transition-all">
                            Sign Up
                        </a>
                    <?php endif; ?>
                </div>
            </nav>
        </div>
    </header>

    <!-- Main Content -->
    <main class="max-w-7xl mx-auto px-4 pt-24 pb-8">
        <!-- Search Bar -->
        <div class="mb-8">
            <div class="relative max-w-xl mx-auto">
                <input type="text" 
                       placeholder="Search videos..." 
                       class="w-full bg-gray-800/50 backdrop-blur-sm border-0 rounded-xl py-3 px-4 pl-11 text-white focus:ring-2 focus:ring-blue-500">
                <svg class="w-5 h-5 absolute left-4 top-1/2 -translate-y-1/2 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                </svg>
            </div>
        </div>

        <!-- Videos Grid -->
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            <?php foreach($videos as $index => $video): ?>
                <div class="video-card bg-gray-800/50 backdrop-blur-sm rounded-2xl overflow-hidden cursor-pointer hover:scale-105 transition-all duration-300" 
                     onclick="openModal(<?= $video['id'] ?>)"
                     style="animation-delay: <?= $index * 0.1 ?>s">
                    <div class="relative aspect-video bg-gray-900">
                        <img src="<?= htmlspecialchars($video['picture_cover']) ?>" alt="" class="w-full h-full object-cover">
                        <div class="absolute bottom-2 right-2 bg-black/70 backdrop-blur-sm px-3 py-1 rounded-lg text-sm">
                            <?= htmlspecialchars($video['time']) ?>
                        </div>
                    </div>
                    <div class="p-4">
                        <h2 class="text-lg font-bold line-clamp-1"><?= htmlspecialchars($video['name']) ?></h2>
                        <p class="text-gray-400 text-sm mt-2 line-clamp-2"><?= htmlspecialchars($video['description']) ?></p>
                        <div class="flex items-center justify-between mt-4">
                            <span class="text-sm text-blue-400 hover:text-blue-300 cursor-pointer">
                                <?= htmlspecialchars($video['author_name']) ?>
                            </span>
                            <span class="text-sm text-gray-400">
                                <?= number_format($video['view_count']) ?> views
                            </span>
                        </div>
                        <div class="flex flex-wrap gap-2 mt-3">
                            <?php foreach(explode(',', $video['tags']) as $tag): ?>
                                <span class="bg-blue-900/50 backdrop-blur-sm px-3 py-1 rounded-full text-xs">
                                    <?= htmlspecialchars(trim($tag)) ?>
                                </span>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </main>

    <!-- Modal -->
    <div id="videoModal" class="hidden fixed inset-0 bg-black/95 backdrop-blur-sm z-50 overflow-y-auto">
        <div class="min-h-screen flex items-center justify-center p-4">
            <div class="bg-gray-800/50 backdrop-blur-sm rounded-2xl w-full max-w-5xl">
                <div id="modalContent" class="p-6">
                    <div class="flex justify-between items-center mb-4">
                        <h3 class="text-2xl font-bold" id="videoTitle"></h3>
                        <button onclick="closeModal()" class="text-gray-400 hover:text-white transition-colors">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                        </button>
                    </div>
                    
                    <div class="relative bg-gray-900 rounded-xl overflow-hidden aspect-video mb-6">
                        <div id="videoContainer" class="w-full h-full">
                            <div class="flex items-center justify-center h-full">
                                <div class="animate-pulse">
                                    <svg class="w-12 h-12 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 3l14 9-14 9V3z"></path>
                                    </svg>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="space-y-6">
                        <div id="videoInfo" class="space-y-4"></div>

                        <?php if(isset($_SESSION['user_id'])): ?>
                            <div class="flex items-center gap-4">
                                <button onclick="handleInteraction('like')" class="px-4 py-2 rounded-xl bg-gray-700 hover:bg-gray-600 transition-colors">
                                    <span id="likeCount">0</span> Likes
                                </button>
                                <button onclick="handleInteraction('dislike')" class="px-4 py-2 rounded-xl bg-gray-700 hover:bg-gray-600 transition-colors">
                                    <span id="dislikeCount">0</span> Dislikes
                                </button>
                            </div>

                            <div class="border-t border-gray-700 pt-6">
                                <form onsubmit="handleComment(event)" class="space-y-4">
                                    <textarea id="commentText" rows="3" 
                                        class="w-full bg-gray-900/50 backdrop-blur-sm border-0 rounded-xl p-4 text-white focus:ring-2 focus:ring-blue-500"
                                        placeholder="Add a comment..."></textarea>
                                    <button type="submit" 
                                        class="px-6 py-2 bg-blue-600 hover:bg-blue-700 rounded-xl transition-colors">
                                        Comment
                                    </button>
                                </form>
                            </div>
                        <?php endif; ?>

                        <div id="comments" class="space-y-4 border-t border-gray-700 pt-6">
                            <h4 class="text-xl font-bold">Comments</h4>
                            <div id="commentsList" class="space-y-4"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Menu Toggle
        function toggleMenu() {
            document.querySelector('.menu').classList.toggle('active');
            document.querySelector('.hamburger').classList.toggle('active');
        }

        // Close menu on outside click
        document.addEventListener('click', function(e) {
            const menu = document.querySelector('.menu');
            const hamburger = document.querySelector('.hamburger');
            if (window.innerWidth < 1024 && !menu.contains(e.target) && !hamburger.contains(e.target)) {
                menu.classList.remove('active');
                hamburger.classList.remove('active');
            }
        });

        // Modal Functions
        let currentVideoId = null;

        function openModal(videoId) {
            currentVideoId = videoId;
            document.getElementById('videoModal').classList.remove('hidden');
            document.body.style.overflow = 'hidden';
            
            fetch('get_video.php?id=' + videoId)
                .then(response => response.json())
                .then(data => {
                    document.getElementById('videoTitle').textContent = data.name;
                    document.getElementById('videoInfo').innerHTML = `
                        <div class="flex items-center gap-4">
                            <a href="profile.php?id=${data.author_id}" class="flex items-center gap-3 text-blue-400 hover:text-blue-300">
                                <div class="w-10 h-10 rounded-full bg-gray-700 flex items-center justify-center">
                                    ${data.author_name.charAt(0).toUpperCase()}
                                </div>
                                <span class="font-semibold">${data.author_name}</span>
                            </a>
                            <div class="flex gap-4 text-gray-400 text-sm">
                                <span>${data.view_count} views</span>
                                <span>${data.like_count} likes</span>
                                <span>${data.dislike_count} dislikes</span>
                            </div>
                        </div>
                        <div class="text-gray-300 whitespace-pre-wrap">${data.description}</div>
                        <div class="flex flex-wrap gap-2">
                            ${data.tags.split(',').map(tag => 
                                `<span class="bg-blue-900/50 backdrop-blur-sm px-3 py-1 rounded-full text-sm">${tag.trim()}</span>`
                            ).join('')}
                        </div>
                    `;
                    
                    document.getElementById('videoContainer').innerHTML = `
                        <video src="${data.url_video}" controls class="w-full h-full" autoplay>
                            Your browser does not support the video tag.
                        </video>
                    `;

                    loadComments(videoId);
                });
        }

        function closeModal() {
            document.getElementById('videoModal').classList.add('hidden');
            document.body.style.overflow = 'auto';
            currentVideoId = null;
        }

        function handleInteraction(type) {
            if (!currentVideoId) return;
            
            fetch('get_video.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=${type}&video_id=${currentVideoId}`
            }).then(() => {
                openModal(currentVideoId);
            });
        }

        function loadComments(videoId) {
            fetch('get_comments.php?video_id=' + videoId)
                .then(response => response.json())
                .then(comments => {
                    document.getElementById('commentsList').innerHTML = comments.map(comment => `
                        <div class="bg-gray-900/50 backdrop-blur-sm rounded-xl p-4">
                            <div class="flex justify-between mb-2">
                                <a href="profile.php?id=${comment.user_id}" class="text-blue-400 hover:text-blue-300 font-semibold">
                                    ${comment.username}
                                </a>
                                <span class="text-gray-400 text-sm">${comment.created_at}</span>
                            </div>
                            <p class="text-gray-300">${comment.comment_text}</p>
                        </div>
                    `).join('');
                });
        }

        function handleComment(event) {
            event.preventDefault();
            if (!currentVideoId) return;
            
            const comment = document.getElementById('commentText').value;
            if (!comment.trim()) return;

            fetch('get_video.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `comment=${encodeURIComponent(comment)}&video_id=${currentVideoId}`
            }).then(() => {
                document.getElementById('commentText').value = '';
                loadComments(currentVideoId);
            });
        }

        // Close modal on escape key
        document.addEventListener('keydown', e => {
            if (e.key === 'Escape') closeModal();
        });
    </script>
</body>
</html>