<?php
session_start();
require_once 'db.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    
    if (empty($name)) {
        $message = 'Profile name is required';
        $messageType = 'error';
    } else {
        try {
            $pdo->beginTransaction();
            
            $profile_image = '';
            
            if (isset($_FILES['profile_image']) && $_FILES['profile_image']['error'] === UPLOAD_ERR_OK) {
                $tmp_name = $_FILES['profile_image']['tmp_name'];
                $image_info = getimagesize($tmp_name);
                
                if ($image_info !== false) {
                    $image_type = $image_info[2];
                    
                    if ($image_type === IMAGETYPE_JPEG || $image_type === IMAGETYPE_PNG || $image_type === IMAGETYPE_GIF) {
                        $unique_name = uniqid('profile_') . '.jpg';
                        $upload_path = __DIR__ . '/image/' . $unique_name;
                        
                        if ($image_type === IMAGETYPE_JPEG) {
                            $image = imagecreatefromjpeg($tmp_name);
                        } elseif ($image_type === IMAGETYPE_PNG) {
                            $image = imagecreatefrompng($tmp_name);
                        } else {
                            $image = imagecreatefromgif($tmp_name);
                        }
                        
                        imagejpeg($image, $upload_path, 90);
                        imagedestroy($image);
                        
                        $profile_image = 'https://peerkinton.com/image/' . $unique_name;
                    } else {
                        $message = 'Invalid image format. Only JPEG, PNG, and GIF are allowed.';
                        $messageType = 'error';
                        throw new Exception('Invalid image format');
                    }
                } else {
                    $message = 'Invalid image file';
                    $messageType = 'error';
                    throw new Exception('Invalid image file');
                }
            }
            
            $gallery_images = [];
            
            if (isset($_FILES['gallery_images'])) {
                $file_count = count($_FILES['gallery_images']['name']);
                
                for ($i = 0; $i < $file_count; $i++) {
                    if ($_FILES['gallery_images']['error'][$i] === UPLOAD_ERR_OK) {
                        $tmp_name = $_FILES['gallery_images']['tmp_name'][$i];
                        $image_info = getimagesize($tmp_name);
                        
                        if ($image_info !== false) {
                            $image_type = $image_info[2];
                            
                            if ($image_type === IMAGETYPE_JPEG || $image_type === IMAGETYPE_PNG || $image_type === IMAGETYPE_GIF) {
                                $unique_name = uniqid('gallery_') . '.jpg';
                                $upload_path = __DIR__ . '/image/' . $unique_name;
                                
                                if ($image_type === IMAGETYPE_JPEG) {
                                    $image = imagecreatefromjpeg($tmp_name);
                                } elseif ($image_type === IMAGETYPE_PNG) {
                                    $image = imagecreatefrompng($tmp_name);
                                } else {
                                    $image = imagecreatefromgif($tmp_name);
                                }
                                
                                imagejpeg($image, $upload_path, 90);
                                imagedestroy($image);
                                
                                $gallery_images[] = 'https://peerkinton.com/image/' . $unique_name;
                            }
                        }
                    }
                }
            }
            
            $stmt = $pdo->prepare("INSERT INTO profiles (name, description, profile_image, created_by, created_at) VALUES (?, ?, ?, ?, NOW())");
            $stmt->execute([$name, $description, $profile_image, $user_id]);
            $profile_id = $pdo->lastInsertId();
            
            if (!empty($gallery_images)) {
                $stmt = $pdo->prepare("INSERT INTO profile_images (profile_id, image_url, created_at) VALUES (?, ?, NOW())");
                
                foreach ($gallery_images as $image_url) {
                    $stmt->execute([$profile_id, $image_url]);
                }
            }
            
            $pdo->commit();
            
            $message = 'Profile created successfully';
            $messageType = 'success';
            
            header("Location: view-profile.php?id=$profile_id");
            exit;
            
        } catch (Exception $e) {
            $pdo->rollBack();
            
            if (empty($message)) {
                $message = 'Error creating profile: ' . $e->getMessage();
                $messageType = 'error';
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en-GB">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Profile - Peerkinton</title>
    <link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        jost: ['Jost', 'sans-serif'],
                    },
                }
            }
        }
    </script>
    <style>
        body {
            font-family: 'Jost', sans-serif;
        }
        .image-preview {
            aspect-ratio: 1/1;
            object-fit: cover;
        }
    </style>
</head>
<body class="bg-gray-50 text-black font-jost">
    <div class="min-h-screen flex flex-col">
        <header class="bg-white py-4 px-6 border-b border-gray-200">
            <div class="flex justify-between items-center">
                <h1 class="text-2xl font-light">Peerkinton</h1>
                <div class="flex items-center space-x-4">
                    <span><?php echo htmlspecialchars($user['username']); ?></span>
                    <a href="logout.php" class="text-sm text-gray-600">Logout</a>
                </div>
            </div>
        </header>
        
        <nav class="bg-white py-3 px-6 border-b border-gray-200">
            <div class="flex space-x-6">
                <a href="dashboard.php" class="text-gray-500 hover:text-gray-800">Map</a>
                <a href="search.php" class="text-gray-500 hover:text-gray-800">Library</a>
                <span class="text-gray-800 font-medium">Create Profile</span>
                <a href="dashboard.php?tab=settings" class="text-gray-500 hover:text-gray-800">Settings</a>
            </div>
        </nav>
        
        <main class="flex-1 p-6">
            <div class="max-w-3xl mx-auto">
                <h2 class="text-2xl font-medium mb-6">Create New Profile</h2>
                
                <?php if (!empty($message)): ?>
                <div class="mb-6 p-4 <?php echo $messageType === 'success' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                    <?php echo $message; ?>
                </div>
                <?php endif; ?>
                
                <form method="POST" enctype="multipart/form-data" class="bg-white p-6 rounded-lg border border-gray-200">
                    <div class="mb-4">
                        <label for="name" class="block text-sm text-gray-700 mb-1">Profile Name</label>
                        <input type="text" id="name" name="name" 
                            class="w-full px-3 py-2 border border-gray-300 focus:outline-none" 
                            placeholder="Enter profile name" required>
                    </div>
                    
                    <div class="mb-4">
                        <label for="description" class="block text-sm text-gray-700 mb-1">Description</label>
                        <textarea id="description" name="description" rows="4"
                            class="w-full px-3 py-2 border border-gray-300 focus:outline-none" 
                            placeholder="Enter profile description"></textarea>
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-sm text-gray-700 mb-1">Profile Image</label>
                        <div class="flex flex-col items-center justify-center border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                            <div id="profilePreviewContainer" class="mb-4 hidden">
                                <img id="profilePreview" class="image-preview w-32 h-32 rounded-full" src="#" alt="Profile preview">
                            </div>
                            
                            <label for="profile_image" class="cursor-pointer">
                                <div class="flex flex-col items-center justify-center">
                                    <svg xmlns="http://www.w3.org/2000/svg" class="h-10 w-10 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6" />
                                    </svg>
                                    <span class="mt-2 text-sm text-gray-600">Select profile image</span>
                                </div>
                                <input type="file" id="profile_image" name="profile_image" class="hidden" accept="image/jpeg, image/png, image/gif">
                            </label>
                        </div>
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-sm text-gray-700 mb-1">Gallery Images</label>
                        <div class="flex flex-col items-center justify-center border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                            <div id="galleryPreviewContainer" class="grid grid-cols-2 sm:grid-cols-3 gap-2 w-full mb-4 hidden">
                            </div>
                            
                            <label for="gallery_images" class="cursor-pointer">
                                <div class="flex flex-col items-center justify-center">
                                    <svg xmlns="http://www.w3.org/2000/svg" class="h-10 w-10 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                    </svg>
                                    <span class="mt-2 text-sm text-gray-600">Select gallery images (multiple allowed)</span>
                                </div>
                                <input type="file" id="gallery_images" name="gallery_images[]" class="hidden" multiple accept="image/jpeg, image/png, image/gif">
                            </label>
                        </div>
                    </div>
                    
                    <div class="flex justify-end">
                        <button type="submit" class="bg-black text-white px-6 py-2">Create Profile</button>
                    </div>
                </form>
            </div>
        </main>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const profileImage = document.getElementById('profile_image');
            const profilePreview = document.getElementById('profilePreview');
            const profilePreviewContainer = document.getElementById('profilePreviewContainer');
            
            const galleryImages = document.getElementById('gallery_images');
            const galleryPreviewContainer = document.getElementById('galleryPreviewContainer');
            
            profileImage.addEventListener('change', function() {
                if (this.files && this.files[0]) {
                    const reader = new FileReader();
                    
                    reader.onload = function(e) {
                        profilePreview.src = e.target.result;
                        profilePreviewContainer.classList.remove('hidden');
                    }
                    
                    reader.readAsDataURL(this.files[0]);
                }
            });
            
            galleryImages.addEventListener('change', function() {
                galleryPreviewContainer.innerHTML = '';
                
                if (this.files && this.files.length > 0) {
                    for (let i = 0; i < this.files.length; i++) {
                        const reader = new FileReader();
                        const file = this.files[i];
                        
                        reader.onload = function(e) {
                            const div = document.createElement('div');
                            div.className = 'relative';
                            
                            const img = document.createElement('img');
                            img.src = e.target.result;
                            img.className = 'image-preview w-full h-32 object-cover rounded';
                            img.alt = 'Gallery preview';
                            
                            div.appendChild(img);
                            galleryPreviewContainer.appendChild(div);
                        }
                        
                        reader.readAsDataURL(file);
                    }
                    
                    galleryPreviewContainer.classList.remove('hidden');
                } else {
                    galleryPreviewContainer.classList.add('hidden');
                }
            });
        });
    </script>
</body>
</html>