<?php
require_once 'config.php';

// Vérifier si on regarde un profil spécifique ou son propre profil
$profile_id = isset($_GET['id']) ? (int)$_GET['id'] : ($_SESSION['user_id'] ?? null);

if(!$profile_id) {
    header('Location: login.php');
    exit;
}

// Récupérer les infos du profil
$stmt = $pdo->prepare("
    SELECT id, username, email, created_at, avatar_url 
    FROM users 
    WHERE id = ?
");
$stmt->execute([$profile_id]);
$user = $stmt->fetch();

if(!$user) {
    header('Location: index.php');
    exit;
}

// Vérifier si c'est notre propre profil
$isOwnProfile = isset($_SESSION['user_id']) && $_SESSION['user_id'] == $profile_id;

// Si c'est notre propil et qu'on update la photo
if($isOwnProfile && isset($_FILES['avatar'])) {
    $uploadDir = 'uploads/avatars/';
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }

    $allowedTypes = ['image/jpeg', 'image/png', 'image/webp'];
    $fileType = mime_content_type($_FILES['avatar']['tmp_name']);

    if(in_array($fileType, $allowedTypes)) {
        $fileName = uniqid() . '_' . basename($_FILES['avatar']['name']);
        $uploadPath = $uploadDir . $fileName;

        if(move_uploaded_file($_FILES['avatar']['tmp_name'], $uploadPath)) {
            $stmt = $pdo->prepare("UPDATE users SET avatar_url = ? WHERE id = ?");
            $stmt->execute([$uploadPath, $profile_id]);
            $user['avatar_url'] = $uploadPath;
        }
    }
}

// Récupérer les stats des vidéos
$stmt = $pdo->prepare("
    SELECT 
        COUNT(DISTINCT c.id) as total_videos,
        SUM(
            CASE WHEN i.type = 'view' THEN 1 ELSE 0 END
        ) as total_views,
        SUM(
            CASE WHEN i.type = 'like' THEN 1 ELSE 0 END
        ) as total_likes,
        SUM(
            CASE WHEN i.type = 'comment' THEN 1 ELSE 0 END
        ) as total_comments
    FROM content c
    LEFT JOIN interactions i ON c.id = i.content_id
    WHERE c.author_id = ?
");
$stmt->execute([$profile_id]);
$stats = $stmt->fetch();

// Récupérer les vidéos
$stmt = $pdo->prepare("
    SELECT 
        c.*,
        COUNT(DISTINCT CASE WHEN i.type = 'view' THEN i.id END) as views,
        COUNT(DISTINCT CASE WHEN i.type = 'like' THEN i.id END) as likes,
        COUNT(DISTINCT CASE WHEN i.type = 'comment' THEN i.id END) as comments
    FROM content c
    LEFT JOIN interactions i ON c.id = i.content_id
    WHERE c.author_id = ?
    GROUP BY c.id
    ORDER BY c.created_at DESC
");
$stmt->execute([$profile_id]);
$videos = $stmt->fetchAll();

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($user['username']) ?> - ForMore</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Titillium+Web:wght@200;300;400;600;700;900&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Titillium Web', sans-serif;
        }
        .formore {
            font-weight: 900;
            background: linear-gradient(to right, #3b82f6, #1d4ed8);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
    </style>
</head>
<body class="bg-gradient-to-br from-black to-blue-900 min-h-screen text-white">
    <nav class="max-w-7xl mx-auto flex justify-between items-center p-8">
        <a href="index.php" class="text-5xl formore">ForMore</a>
        <div class="flex items-center gap-6">
            <?php if(isset($_SESSION['user_id'])): ?>
                <?php if(!$isOwnProfile): ?>
                    <a href="profile.php" class="text-gray-300 hover:text-white">My Profile</a>
                <?php endif; ?>
                <a href="upload.php" class="bg-blue-600 hover:bg-blue-700 px-6 py-2 rounded-xl transition-all">Upload</a>
                <a href="logout.php" class="text-red-400 hover:text-red-300">Logout</a>
            <?php else: ?>
                <a href="login.php" class="text-blue-400 hover:text-blue-300">Login</a>
                <a href="signup.php" class="bg-blue-600 hover:bg-blue-700 px-6 py-2 rounded-xl transition-all">Sign Up</a>
            <?php endif; ?>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto p-8">
        <!-- Profile Header -->
        <div class="bg-gray-800/50 backdrop-blur-sm rounded-2xl p-8 mb-8">
            <div class="flex items-center gap-8">
                <div class="relative group">
                    <div class="w-32 h-32 rounded-full overflow-hidden bg-gray-700">
                        <?php if($user['avatar_url']): ?>
                            <img src="<?= htmlspecialchars($user['avatar_url']) ?>" alt="Profile" class="w-full h-full object-cover">
                        <?php endif; ?>
                    </div>
                    <?php if($isOwnProfile): ?>
                        <form id="avatarForm" class="hidden">
                            <input type="file" name="avatar" id="avatarInput" accept="image/*" class="hidden">
                        </form>
                        <button onclick="document.getElementById('avatarInput').click()" 
                            class="absolute inset-0 bg-black/50 rounded-full opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
                            Change Photo
                        </button>
                    <?php endif; ?>
                </div>

                <div>
                    <h1 class="text-3xl font-bold mb-2"><?= htmlspecialchars($user['username']) ?></h1>
                    <p class="text-gray-400">Member since <?= (new DateTime($user['created_at']))->format('F Y') ?></p>
                    
                    <div class="flex gap-8 mt-4">
                        <div>
                            <div class="text-2xl font-bold"><?= number_format($stats['total_videos']) ?></div>
                            <div class="text-gray-400">Videos</div>
                        </div>
                        <div>
                            <div class="text-2xl font-bold"><?= number_format($stats['total_views']) ?></div>
                            <div class="text-gray-400">Views</div>
                        </div>
                        <div>
                            <div class="text-2xl font-bold"><?= number_format($stats['total_likes']) ?></div>
                            <div class="text-gray-400">Likes</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Videos Grid -->
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            <?php foreach($videos as $video): ?>
                <div class="bg-gray-800/50 backdrop-blur-sm rounded-2xl overflow-hidden cursor-pointer" onclick="openModal(<?= $video['id'] ?>)">
                    <div class="relative aspect-video">
                        <img src="<?= htmlspecialchars($video['picture_cover']) ?>" alt="" class="w-full h-full object-cover">
                        <div class="absolute bottom-2 right-2 bg-black/70 backdrop-blur-sm px-3 py-1 rounded-lg text-sm">
                            <?= htmlspecialchars($video['time']) ?>
                        </div>
                    </div>
                    <div class="p-4">
                        <h2 class="text-lg font-bold mb-2 line-clamp-1"><?= htmlspecialchars($video['name']) ?></h2>
                        <p class="text-gray-400 text-sm mb-3 line-clamp-2"><?= htmlspecialchars($video['description']) ?></p>
                        <div class="flex justify-between text-sm text-gray-400">
                            <span><?= number_format($video['views']) ?> views</span>
                            <span><?= number_format($video['likes']) ?> likes</span>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>

            <?php if(empty($videos)): ?>
                <div class="col-span-full text-center py-12 text-gray-400">
                    No videos uploaded yet.
                    <?php if($isOwnProfile): ?>
                        <br>
                        <a href="upload.php" class="text-blue-400 hover:text-blue-300">Upload your first video!</a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <?php if($isOwnProfile): ?>
    <script>
    document.getElementById('avatarInput').addEventListener('change', function() {
        const formData = new FormData(document.getElementById('avatarForm'));
        fetch('profile.php', {
            method: 'POST',
            body: formData
        }).then(() => {
            location.reload();
        });
    });
    </script>
    <?php endif; ?>
</body>
</html>