<?php
require_once '../search/db.php';
session_start();

// Vérification de la connexion
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$message = '';
$error = '';

// Gestion de l'upload de fichier
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['document'])) {
    $file = $_FILES['document'];
    $allowed_types = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 
                     'text/plain', 'image/jpeg', 'image/png'];
    $max_size = 5 * 1024 * 1024; // 5MB

    if ($file['error'] === 0) {
        if ($file['size'] <= $max_size) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mime_type = finfo_file($finfo, $file['tmp_name']);
            
            if (in_array($mime_type, $allowed_types)) {
                // Création du dossier sécurisé pour l'utilisateur
                $upload_dir = 'uploads/user_' . $user_id;
                if (!file_exists($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }

                // Génération d'un nom de fichier unique
                $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $new_filename = uniqid() . '.' . $file_extension;
                $upload_path = $upload_dir . '/' . $new_filename;

                if (move_uploaded_file($file['tmp_name'], $upload_path)) {
                    try {
                        $query = "INSERT INTO user_documents (user_id, filename, original_filename, file_path, file_type, file_size) 
                                 VALUES (:user_id, :filename, :original_filename, :file_path, :file_type, :file_size)";
                        $stmt = $db->prepare($query);
                        $stmt->execute([
                            ':user_id' => $user_id,
                            ':filename' => $new_filename,
                            ':original_filename' => $file['name'],
                            ':file_path' => $upload_path,
                            ':file_type' => $mime_type,
                            ':file_size' => $file['size']
                        ]);
                        $message = 'File uploaded successfully.';
                    } catch (PDOException $e) {
                        $error = 'Error saving file information.';
                    }
                } else {
                    $error = 'Error uploading file.';
                }
            } else {
                $error = 'Invalid file type.';
            }
        } else {
            $error = 'File is too large (max 5MB).';
        }
    } else {
        $error = 'Error uploading file.';
    }
}

// Création/Modification de note
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'create_note' || $_POST['action'] === 'update_note') {
        $title = trim($_POST['title'] ?? '');
        $content = trim($_POST['content'] ?? '');
        
        if (!empty($title) && !empty($content)) {
            try {
                if ($_POST['action'] === 'create_note') {
                    $query = "INSERT INTO user_notes (user_id, title, content) VALUES (:user_id, :title, :content)";
                    $params = [':user_id' => $user_id, ':title' => $title, ':content' => $content];
                } else {
                    $note_id = (int)$_POST['note_id'];
                    $query = "UPDATE user_notes SET title = :title, content = :content WHERE id = :note_id AND user_id = :user_id";
                    $params = [':user_id' => $user_id, ':title' => $title, ':content' => $content, ':note_id' => $note_id];
                }
                
                $stmt = $db->prepare($query);
                $stmt->execute($params);
                $message = 'Note saved successfully.';
            } catch (PDOException $e) {
                $error = 'Error saving note.';
            }
        } else {
            $error = 'Title and content are required.';
        }
    }
}

// Suppression de fichier/note
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $type = $_POST['type'] ?? '';
    $id = (int)$_POST['id'] ?? 0;

    if ($type === 'document') {
        try {
            // Récupérer le chemin du fichier
            $query = "SELECT file_path FROM user_documents WHERE id = :id AND user_id = :user_id";
            $stmt = $db->prepare($query);
            $stmt->execute([':id' => $id, ':user_id' => $user_id]);
            $file = $stmt->fetch();

            if ($file && file_exists($file['file_path'])) {
                unlink($file['file_path']); // Supprimer le fichier physique
            }

            // Supprimer l'entrée de la base de données
            $query = "DELETE FROM user_documents WHERE id = :id AND user_id = :user_id";
            $stmt = $db->prepare($query);
            $stmt->execute([':id' => $id, ':user_id' => $user_id]);
            $message = 'File deleted successfully.';
        } catch (PDOException $e) {
            $error = 'Error deleting file.';
        }
    } elseif ($type === 'note') {
        try {
            $query = "DELETE FROM user_notes WHERE id = :id AND user_id = :user_id";
            $stmt = $db->prepare($query);
            $stmt->execute([':id' => $id, ':user_id' => $user_id]);
            $message = 'Note deleted successfully.';
        } catch (PDOException $e) {
            $error = 'Error deleting note.';
        }
    }
}

// Récupération des documents
$docs_query = "SELECT * FROM user_documents WHERE user_id = :user_id ORDER BY created_at DESC";
$stmt = $db->prepare($docs_query);
$stmt->execute([':user_id' => $user_id]);
$documents = $stmt->fetchAll();

// Récupération des notes
$notes_query = "SELECT * FROM user_notes WHERE user_id = :user_id ORDER BY updated_at DESC";
$stmt = $db->prepare($notes_query);
$stmt->execute([':user_id' => $user_id]);
$notes = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en" class="h-full">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Casier membre - Peerkinton</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://api.fontshare.com/v2/css?f[]=satoshi@400,500,700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Satoshi', sans-serif;
            background: #fafafa;
        }

        .nav-link {
            transition: all 0.2s ease;
        }

        .nav-link:hover {
            background-color: rgba(0, 0, 0, 0.05);
        }

        .nav-link.active {
            background-color: #f3f4f6;
            color: #111827;
        }

        .file-card {
            transition: all 0.3s ease;
        }

        .file-card:hover {
            transform: translateY(-2px);
        }

        .note-editor {
            min-height: 200px;
            max-height: 400px;
            overflow-y: auto;
        }
    </style>
</head>
<body class="h-full">
    <?php include '../header.php'; ?>

    <div class="flex min-h-screen pt-20">
        <!-- Sidebar -->
        <aside class="w-64 border-r border-gray-200 px-4 py-6">
            <nav class="space-y-2">
                <a href="dashboard.php" class="nav-link flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"/>
                    </svg>
                    Dashboard
                </a>
                <a href="dashboard-files.php" class="nav-link active flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 8h14M5 8a2 2 0 110-4h14a2 2 0 110 4M5 8v10a2 2 0 002 2h10a2 2 0 002-2V8m-9 4h4"/>
                    </svg>
                    Member records
                </a>
                <a href="dashboard-profile.php" class="nav-link flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                    </svg>
                    Profil
                </a>
                <?php if ($user['role'] === 'writer'): ?>
                <a href="dashboard-write.php" class="nav-link flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                    </svg>
                    Editorial
                </a>
                <?php endif; ?>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="flex-1 px-8 py-6">
            <?php if ($message): ?>
                <div class="bg-green-50 border border-green-200 text-green-800 rounded-lg p-4 mb-6">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div class="bg-red-50 border border-red-200 text-red-800 rounded-lg p-4 mb-6">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
                <!-- Documents Section -->
                <div>
                    <div class="bg-white rounded-xl p-6 border border-gray-200">
                        <h2 class="text-xl font-medium text-gray-900 mb-6">Documents</h2>

                        <!-- Upload Form -->
                        <form method="POST" enctype="multipart/form-data" class="mb-6">
                            <div class="flex items-center space-x-4">
                                <input 
                                    type="file" 
                                    name="document" 
                                    class="flex-1 text-sm text-gray-600 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-medium file:bg-gray-100 file:text-gray-700 hover:file:bg-gray-200"
                                    required
                                >
                                <button type="submit" class="px-4 py-2 bg-gray-900 text-white rounded-full text-sm hover:transform hover:-translate-y-0.5 transition-all duration-200">
                                    Upload
                                </button>
                            </div>
                        </form>

                        <!-- Documents List -->
                        <div class="space-y-4">
                            <?php foreach ($documents as $doc): ?>
                                <div class="file-card flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center space-x-4">
                                        <svg class="w-8 h-8 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z"/>
                                        </svg>
                                        <div>
                                            <p class="text-sm font-medium text-gray-900">
                                                <?php echo htmlspecialchars($doc['original_filename']); ?>
                                            </p>
                                            <p class="text-sm text-gray-500">
                                                <?php echo date('j M Y', strtotime($doc['created_at'])); ?> · 
                                                <?php echo formatBytes($doc['file_size']); ?>
                                            </p>
                                        </div>
                                    </div>
                                    <div class="flex items-center space-x-2">
                                        <a href="download.php?id=<?php echo $doc['id']; ?>" 
                                           class="p-2 text-gray-500 hover:text-gray-900">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4h8"/>
                                            </svg>
                                        </a>
                                        <form method="POST" class="inline">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="type" value="document">
                                            <input type="hidden" name="id" value="<?php echo $doc['id']; ?>">
                                            <button type="submit" 
                                                    class="p-2 text-gray-500 hover:text-red-600"
                                                    onclick="return confirm('Are you sure you want to delete this file?')">
                                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                                                </svg>
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            <?php endforeach; ?>

                            <?php if (empty($documents)): ?>
                                <div class="text-center py-12">
                                    <svg class="w-12 h-12 mx-auto text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z"/>
                                    </svg>
                                    <p class="mt-4 text-gray-500">No documents yet</p>
                                    <p class="text-sm text-gray-400">Upload your first document using the form above</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Notes Section -->
                <div>
                    <div class="bg-white rounded-xl p-6 border border-gray-200">
                        <div class="flex justify-between items-center mb-6">
                            <h2 class="text-xl font-medium text-gray-900">Notes</h2>
                            <button 
                                type="button" 
                                onclick="openNoteEditor()"
                                class="px-4 py-2 bg-gray-900 text-white rounded-full text-sm hover:transform hover:-translate-y-0.5 transition-all duration-200"
                            >
                                New Note
                            </button>
                        </div>

                        <!-- Notes List -->
                        <div class="space-y-4">
                            <?php foreach ($notes as $note): ?>
                                <div class="p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center justify-between mb-2">
                                        <h3 class="text-sm font-medium text-gray-900">
                                            <?php echo htmlspecialchars($note['title']); ?>
                                        </h3>
                                        <div class="flex items-center space-x-2">
                                            <button 
                                                type="button" 
                                                onclick="editNote(<?php echo htmlspecialchars(json_encode($note)); ?>)"
                                                class="p-2 text-gray-500 hover:text-gray-900"
                                            >
                                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                                                </svg>
                                            </button>
                                            <form method="POST" class="inline">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="type" value="note">
                                                <input type="hidden" name="id" value="<?php echo $note['id']; ?>">
                                                <button type="submit" 
                                                        class="p-2 text-gray-500 hover:text-red-600"
                                                        onclick="return confirm('Are you sure you want to delete this note?')">
                                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                                                    </svg>
                                                </button>
                                            </form>
                                        </div>
                                    </div>
                                    <div class="prose prose-sm max-w-none text-gray-600">
                                        <?php echo nl2br(htmlspecialchars($note['content'])); ?>
                                    </div>
                                    <div class="mt-2 text-xs text-gray-500">
                                        Last updated: <?php echo date('j M Y, H:i', strtotime($note['updated_at'])); ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>

                            <?php if (empty($notes)): ?>
                                <div class="text-center py-12">
                                    <svg class="w-12 h-12 mx-auto text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                                    </svg>
                                    <p class="mt-4 text-gray-500">No notes yet</p>
                                    <p class="text-sm text-gray-400">Create your first note using the button above</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        let currentNoteId = null;

        function openNoteEditor(note = null) {
            currentNoteId = note ? note.id : null;
            document.getElementById('noteModal').classList.remove('hidden');
            
            if (note) {
                document.getElementById('noteTitle').value = note.title;
                document.getElementById('noteContent').value = note.content;
                document.querySelector('[name="action"]').value = 'update_note';
                document.querySelector('[name="note_id"]').value = note.id;
            } else {
                document.getElementById('noteForm').reset();
                document.querySelector('[name="action"]').value = 'create_note';
                document.querySelector('[name="note_id"]').value = '';
            }
        }

        function closeNoteEditor() {
            document.getElementById('noteModal').classList.add('hidden');
            document.getElementById('noteForm').reset();
        }

        // Ajustement automatique de la hauteur du textarea
        document.getElementById('noteContent').addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
        });

        // Fermeture du modal par clic en dehors
        document.getElementById('noteModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeNoteEditor();
            }
        });
    </script>
</body>
</html>