<?php
session_start();
require_once 'config.php';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $data = json_decode(file_get_contents('stock.json'), true) ?: [];
    $trains = $data['trains'];
    
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'add') {
            $trains[] = [
                'id' => uniqid(),
                'line' => $_POST['line'],
                'type' => $_POST['type'],
                'station' => $_POST['station'],
                'arrival' => $_POST['arrival'],
                'platform' => $_POST['platform'],
                'status' => [
                    'code' => 'on-time',
                    'delay' => 0,
                    'message' => ''
                ],
                'info' => [
                    'message' => $_POST['info'],
                    'category' => $_POST['info_category']
                ],
                'composition' => [
                    'first' => isset($_POST['first']),
                    'second' => isset($_POST['second']),
                    'bar' => isset($_POST['bar'])
                ]
            ];
        } 
        elseif ($_POST['action'] === 'update') {
            $id = $_POST['id'];
            foreach ($trains as &$train) {
                if ($train['id'] === $id) {
                    $train['status'] = [
                        'code' => $_POST['status'],
                        'delay' => intval($_POST['delay'] ?? 0),
                        'message' => $_POST['status_message']
                    ];
                    $train['platform'] = $_POST['platform'];
                    $train['info'] = [
                        'message' => $_POST['info'],
                        'category' => $_POST['info_category']
                    ];
                    break;
                }
            }
        }
        elseif ($_POST['action'] === 'delete') {
            $id = $_POST['id'];
            $trains = array_filter($trains, fn($t) => $t['id'] !== $id);
        }
    }
    
    $data['trains'] = array_values($trains);
    $data['last_update'] = date('Y-m-d H:i:s');
    
    file_put_contents('stock.json', json_encode($data, JSON_PRETTY_PRINT));
    header('Location: admin.php');
    exit;
}

$data = json_decode(file_get_contents('stock.json'), true) ?: ['trains' => [], 'last_update' => date('Y-m-d H:i:s')];
$trains = $data['trains'];
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Administration des Trains</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-b from-gray-900 to-gray-800 text-white min-h-screen">
    <div class="container mx-auto p-8">
        <header class="mb-12">
            <div class="flex justify-between items-center">
                <h1 class="text-5xl font-bold bg-gradient-to-r from-blue-500 to-purple-500 text-transparent bg-clip-text">
                    Administration des Trains
                </h1>
                <a href="index.php" class="bg-blue-500 px-4 py-2 rounded-lg hover:bg-blue-600 transition-colors">
                    Voir l'affichage public
                </a>
            </div>
            <p class="text-gray-400 mt-2">Dernière mise à jour: <?= $data['last_update'] ?></p>
        </header>

        <!-- Formulaire d'ajout -->
        <div x-data="{ open: false }" class="mb-12">
            <button 
                @click="open = !open" 
                class="bg-gradient-to-r from-green-500 to-green-700 px-6 py-3 rounded-lg font-medium hover:from-green-600 hover:to-green-800 transition-colors mb-4">
                + Ajouter un train
            </button>

            <form 
                x-show="open"
                x-transition
                method="POST" 
                class="bg-gray-800 bg-opacity-50 backdrop-blur-lg p-8 rounded-xl shadow-lg">
                <input type="hidden" name="action" value="add">
                
                <div class="grid grid-cols-3 gap-6 mb-6">
                    <div class="space-y-2">
                        <label class="block text-sm font-medium">Ligne</label>
                        <input type="text" name="line" class="w-full bg-gray-700 p-3 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none" required>
                    </div>
                    <div class="space-y-2">
                        <label class="block text-sm font-medium">Type de train</label>
                        <select name="type" class="w-full bg-gray-700 p-3 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none">
                            <option value="TGV">TGV</option>
                            <option value="TER">TER</option>
                            <option value="Intercités">Intercités</option>
                        </select>
                    </div>
                    <div class="space-y-2">
                        <label class="block text-sm font-medium">Destination</label>
                        <input type="text" name="station" class="w-full bg-gray-700 p-3 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none" required>
                    </div>
                </div>

                <div class="grid grid-cols-3 gap-6 mb-6">
                    <div class="space-y-2">
                        <label class="block text-sm font-medium">Heure d'arrivée</label>
                        <input type="datetime-local" name="arrival" class="w-full bg-gray-700 p-3 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none" required>
                    </div>
                    <div class="space-y-2">
                        <label class="block text-sm font-medium">Voie</label>
                        <input type="number" name="platform" min="1" max="99" class="w-full bg-gray-700 p-3 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none" required>
                    </div>
                    <div class="space-y-2">
                        <label class="block text-sm font-medium">Catégorie d'info</label>
                        <select name="info_category" class="w-full bg-gray-700 p-3 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none">
                            <option value="none">Aucune</option>
                            <option value="work">Travaux</option>
                            <option value="traffic">Perturbation</option>
                            <option value="info">Information</option>
                        </select>
                    </div>
                </div>

                <div class="mb-6">
                    <label class="block text-sm font-medium mb-2">Message d'information</label>
                    <input type="text" name="info" class="w-full bg-gray-700 p-3 rounded-lg focus:ring-2 focus:ring-blue-500 focus:outline-none">
                </div>

                <div class="mb-6">
                    <label class="block text-sm font-medium mb-2">Composition du train</label>
                    <div class="flex gap-4">
                        <label class="flex items-center">
                            <input type="checkbox" name="first" class="mr-2"> 1ère classe
                        </label>
                        <label class="flex items-center">
                            <input type="checkbox" name="second" class="mr-2"> 2nde classe
                        </label>
                        <label class="flex items-center">
                            <input type="checkbox" name="bar" class="mr-2"> Voiture bar
                        </label>
                    </div>
                </div>

                <button type="submit" class="bg-gradient-to-r from-green-500 to-green-700 px-6 py-3 rounded-lg font-medium hover:from-green-600 hover:to-green-800 transition-colors">
                    Ajouter ce train
                </button>
            </form>
        </div>

        <!-- Liste des trains -->
        <div class="space-y-6">
            <?php foreach ($trains as $train): ?>
            <div class="bg-gray-800 bg-opacity-50 backdrop-blur-lg p-6 rounded-xl shadow-lg">
                <div class="flex justify-between items-start">
                    <div class="flex gap-6">
                        <div class="flex flex-col items-center">
                            <div class="w-16 h-16 rounded-full bg-blue-500 flex items-center justify-center text-2xl font-bold mb-1">
                                <?= htmlspecialchars($train['line']) ?>
                            </div>
                            <span class="text-sm text-gray-400"><?= htmlspecialchars($train['type']) ?></span>
                        </div>
                        <div>
                            <div class="text-xl font-medium">
                                <?= htmlspecialchars($train['station']) ?>
                                <?php if ($train['composition']['first'] || $train['composition']['second']): ?>
                                <span class="text-sm text-gray-400">
                                    (<?= implode(', ', array_filter([
                                        $train['composition']['first'] ? '1ère' : '',
                                        $train['composition']['second'] ? '2nde' : '',
                                        $train['composition']['bar'] ? 'Bar' : ''
                                    ])) ?>)
                                </span>
                                <?php endif; ?>
                            </div>
                            <div class="text-gray-400">
                                Voie <?= htmlspecialchars($train['platform']) ?> - 
                                <?= (new DateTime($train['arrival']))->format('H:i') ?>
                            </div>
                        </div>
                    </div>

                    <div class="flex gap-4">
                        <form method="POST" class="flex gap-4">
                            <input type="hidden" name="action" value="update">
                            <input type="hidden" name="id" value="<?= $train['id'] ?>">
                            
                            <div>
                                <select name="status" class="bg-gray-700 p-2 rounded-lg text-sm">
                                    <option value="on-time" <?= $train['status']['code'] === 'on-time' ? 'selected' : '' ?>>À l'heure</option>
                                    <option value="delayed" <?= $train['status']['code'] === 'delayed' ? 'selected' : '' ?>>Retardé</option>
                                    <option value="cancelled" <?= $train['status']['code'] === 'cancelled' ? 'selected' : '' ?>>Supprimé</option>
                                </select>
                            </div>
                            
                            <div>
                                <input type="number" name="delay" value="<?= $train['status']['delay'] ?>" 
                                    class="w-20 bg-gray-700 p-2 rounded-lg text-sm"
                                    placeholder="Retard (min)">
                            </div>
                            
                            <div>
                                <input type="text" name="status_message" value="<?= htmlspecialchars($train['status']['message']) ?>" 
                                    class="w-48 bg-gray-700 p-2 rounded-lg text-sm"
                                    placeholder="Message de statut">
                            </div>
                            
                            <button type="submit" class="bg-blue-500 px-4 py-2 rounded-lg text-sm hover:bg-blue-600 transition-colors">
                                Mettre à jour
                            </button>
                        </form>
                        
                        <form method="POST" class="flex items-start" onsubmit="return confirm('Confirmer la suppression ?')">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="id" value="<?= $train['id'] ?>">
                            <button type="submit" class="bg-red-500 px-4 py-2 rounded-lg text-sm hover:bg-red-600 transition-colors">
                                Supprimer
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</body>
</html>