<?php
$data = json_decode(file_get_contents('stock.json'), true);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Horaires des Trains</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/howler/2.2.3/howler.min.js"></script>
</head>
<body class="bg-gradient-to-b from-gray-900 to-gray-800 text-white min-h-screen">
    <div class="container mx-auto p-8">
        <div class="flex items-center justify-between mb-8">
            <h1 class="text-5xl font-bold bg-gradient-to-r from-blue-500 to-purple-500 text-transparent bg-clip-text">Horaires des Trains</h1>
            <div id="clock" class="text-3xl font-mono"></div>
        </div>
        <div id="trains" class="space-y-6"></div>
    </div>
    <div id="announcement" class="fixed bottom-0 left-0 w-full bg-gradient-to-r from-blue-600 to-blue-800 transform translate-y-full transition-transform duration-500 p-6 text-xl font-bold"></div>

    <script>
    class SpeechManager {
        constructor() {
            this.synth = window.speechSynthesis;
            this.voice = null;
            this.initialized = false;
        }

        async init() {
            if (this.initialized) return;

            return new Promise((resolve) => {
                let voices = this.synth.getVoices();
                if (voices.length) {
                    this.setupVoice(voices);
                    resolve();
                } else {
                    this.synth.onvoiceschanged = () => {
                        voices = this.synth.getVoices();
                        this.setupVoice(voices);
                        resolve();
                    };
                }
            });
        }

        setupVoice(voices) {
            this.voice = voices.find(voice => voice.lang === 'fr-FR') || voices[0];
            this.initialized = true;
        }

        speak(text) {
            if (!this.initialized) return;
            
            const utterance = new SpeechSynthesisUtterance(text);
            utterance.voice = this.voice;
            utterance.rate = 0.9;
            utterance.pitch = 1;
            utterance.volume = 1;
            this.synth.speak(utterance);
        }
    }

    const speechManager = new SpeechManager();
    speechManager.init();

    const sound = new Howl({
        src: ['https://assets.mixkit.co/active_storage/sfx/2571/2571-preview.mp3'],
        volume: 0.5
    });

    function updateClock() {
        const now = new Date();
        document.getElementById('clock').textContent = now.toLocaleTimeString('fr-FR');
    }

    function formatTimeLeft(minutes) {
        if (minutes === 0) return 'À quai';
        if (minutes === 1) return '1 minute';
        return `${minutes} minutes`;
    }

    function updateTrains() {
        fetch('stock.json?' + new Date().getTime())
        .then(response => response.json())
        .then(data => {
            const trainsDiv = document.getElementById('trains');
            trainsDiv.innerHTML = '';
            
            data.forEach(train => {
                const arrival = new Date(train.arrival);
                const now = new Date();
                const timeLeft = Math.max(0, arrival - now);
                const minutes = Math.floor(timeLeft / 60000);
                
                const statusColor = train.status === 'on-time' ? 'green' : 'red';
                const statusText = train.status === 'on-time' ? 'À l\'heure' : 'Retardé';
                
                const div = document.createElement('div');
                div.className = 'bg-gray-800 bg-opacity-50 backdrop-blur-lg p-8 rounded-xl shadow-lg transform transition-all duration-300 hover:scale-102 hover:bg-opacity-70';
                div.innerHTML = `
                    <div class="grid grid-cols-4 gap-8 items-center">
                        <div class="flex items-center gap-4">
                            <div class="w-12 h-12 rounded-full bg-blue-500 flex items-center justify-center text-xl font-bold">${train.line}</div>
                            <div class="text-xl">${train.station}</div>
                        </div>
                        <div class="text-2xl font-mono">${arrival.toLocaleTimeString('fr-FR')}</div>
                        <div class="text-xl font-medium">${formatTimeLeft(minutes)}</div>
                        <div class="text-${statusColor}-500 text-xl font-medium">${statusText}</div>
                    </div>
                    <div class="mt-4 h-1 bg-gray-700 rounded-full overflow-hidden">
                        <div class="h-full bg-blue-500 transition-all duration-500" style="width: ${Math.min(100, (minutes / 60) * 100)}%"></div>
                    </div>
                `;
                
                if (minutes === 0) {
                    const announcement = `Train ligne ${train.line} à destination de ${train.station} entre en gare`;
                    showAnnouncement(announcement);
                    speechManager.speak(announcement);
                    sound.play();
                }
                
                trainsDiv.appendChild(div);
            });
        });
    }

    function showAnnouncement(text) {
        const announcement = document.getElementById('announcement');
        announcement.textContent = text;
        announcement.classList.remove('translate-y-full');
        setTimeout(() => {
            announcement.classList.add('translate-y-full');
        }, 7000);
    }

    updateClock();
    setInterval(updateClock, 1000);
    updateTrains();
    setInterval(updateTrains, 5000);
    </script>
</body>
</html>