<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

$host = 'localhost:3306';
$dbname = 'gqdcvggs_traffic';
$username = 'gqdcvggs';
$password = 'imaors_management.346980*#@-onlyforcpanel;forchange';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Database connection failed']);
    exit;
}

$request_method = $_SERVER['REQUEST_METHOD'];
$endpoint = isset($_GET['endpoint']) ? $_GET['endpoint'] : '';
$api_key = isset($_GET['api_key']) ? $_GET['api_key'] : '';

if (!validateApiKey($pdo, $api_key)) {
    http_response_code(401);
    echo json_encode(['status' => 'error', 'message' => 'Invalid or missing API key']);
    exit;
}

switch ($request_method) {
    case 'GET':
        switch ($endpoint) {
            case 'lights':
                getAllLights($pdo);
                break;
            case 'light':
                $id = isset($_GET['id']) ? $_GET['id'] : null;
                getLightById($pdo, $id);
                break;
            case 'nearby':
                $lat = isset($_GET['lat']) ? $_GET['lat'] : null;
                $lng = isset($_GET['lng']) ? $_GET['lng'] : null;
                $radius = isset($_GET['radius']) ? $_GET['radius'] : 1;
                getNearbyLights($pdo, $lat, $lng, $radius);
                break;
            default:
                http_response_code(404);
                echo json_encode(['status' => 'error', 'message' => 'Invalid endpoint']);
                break;
        }
        break;
    
    case 'POST':
        switch ($endpoint) {
            case 'light':
                addLight($pdo);
                break;
            case 'update':
                updateLight($pdo);
                break;
            default:
                http_response_code(404);
                echo json_encode(['status' => 'error', 'message' => 'Invalid endpoint']);
                break;
        }
        break;
    
    case 'OPTIONS':
        http_response_code(200);
        exit;
        
    default:
        http_response_code(405);
        echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
        break;
}

function validateApiKey($pdo, $api_key) {
    if (empty($api_key)) {
        return false;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM api_keys WHERE api_key = ? AND active = 1");
        $stmt->execute([$api_key]);
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        return false;
    }
}

function getAllLights($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM traffic_lights");
        $lights = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($lights as &$light) {
            $light['status'] = calculateLightStatus($light);
        }
        
        echo json_encode(['status' => 'success', 'data' => $lights]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Failed to fetch traffic lights']);
    }
}

function getLightById($pdo, $id) {
    if (!$id) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'ID is required']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM traffic_lights WHERE id = ?");
        $stmt->execute([$id]);
        $light = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($light) {
            $light['status'] = calculateLightStatus($light);
            echo json_encode(['status' => 'success', 'data' => $light]);
        } else {
            http_response_code(404);
            echo json_encode(['status' => 'error', 'message' => 'Traffic light not found']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Failed to fetch traffic light']);
    }
}

function getNearbyLights($pdo, $lat, $lng, $radius) {
    if (!$lat || !$lng) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Latitude and longitude are required']);
        return;
    }
    
    try {
        $earth_radius = 6371; // Earth's radius in km
        
        $stmt = $pdo->prepare("
            SELECT *, 
            ($earth_radius * acos(cos(radians(?)) * cos(radians(latitude)) * 
            cos(radians(longitude) - radians(?)) + sin(radians(?)) * 
            sin(radians(latitude)))) AS distance 
            FROM traffic_lights 
            HAVING distance <= ? 
            ORDER BY distance
        ");
        $stmt->execute([$lat, $lng, $lat, $radius]);
        $lights = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($lights as &$light) {
            $light['status'] = calculateLightStatus($light);
        }
        
        echo json_encode(['status' => 'success', 'data' => $lights]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Failed to fetch nearby traffic lights']);
    }
}

function addLight($pdo) {
    $json = file_get_contents('php://input');
    $data = json_decode($json, true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Invalid data format']);
        return;
    }
    
    $required = ['name', 'latitude', 'longitude', 'direction', 'red_duration', 'green_duration'];
    foreach ($required as $field) {
        if (!isset($data[$field])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => "$field is required"]);
            return;
        }
    }
    
    try {
        $stmt = $pdo->prepare("INSERT INTO traffic_lights (name, latitude, longitude, direction, red_duration, green_duration) 
                              VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $data['name'],
            $data['latitude'],
            $data['longitude'],
            $data['direction'],
            $data['red_duration'],
            $data['green_duration']
        ]);
        
        $id = $pdo->lastInsertId();
        
        $stmt = $pdo->prepare("SELECT * FROM traffic_lights WHERE id = ?");
        $stmt->execute([$id]);
        $light = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $light['status'] = calculateLightStatus($light);
        
        echo json_encode(['status' => 'success', 'data' => $light]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Failed to add traffic light']);
    }
}

function updateLight($pdo) {
    $json = file_get_contents('php://input');
    $data = json_decode($json, true);
    
    if (!$data || !isset($data['id'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Invalid data format or missing ID']);
        return;
    }
    
    $id = $data['id'];
    unset($data['id']);
    
    $allowedFields = ['name', 'latitude', 'longitude', 'direction', 'red_duration', 'green_duration'];
    $fields = [];
    $values = [];
    
    foreach ($data as $key => $value) {
        if (in_array($key, $allowedFields)) {
            $fields[] = "$key = ?";
            $values[] = $value;
        }
    }
    
    if (empty($fields)) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'No valid fields to update']);
        return;
    }
    
    $values[] = $id;
    
    try {
        $stmt = $pdo->prepare("UPDATE traffic_lights SET " . implode(', ', $fields) . " WHERE id = ?");
        $stmt->execute($values);
        
        if ($stmt->rowCount() === 0) {
            http_response_code(404);
            echo json_encode(['status' => 'error', 'message' => 'Traffic light not found or no changes made']);
            return;
        }
        
        $stmt = $pdo->prepare("SELECT * FROM traffic_lights WHERE id = ?");
        $stmt->execute([$id]);
        $light = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $light['status'] = calculateLightStatus($light);
        
        echo json_encode(['status' => 'success', 'data' => $light]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Failed to update traffic light']);
    }
}

function calculateLightStatus($light) {
    $totalCycle = (int)$light['red_duration'] + (int)$light['green_duration'];
    $currentTime = time();
    $timeInCycle = $currentTime % $totalCycle;
    
    if ($timeInCycle < $light['red_duration']) {
        return [
            'current' => 'RED',
            'next_change' => $light['red_duration'] - $timeInCycle,
            'next_state' => 'GREEN',
            'cycle_position' => ($timeInCycle / $totalCycle) * 100
        ];
    } else {
        return [
            'current' => 'GREEN',
            'next_change' => $totalCycle - $timeInCycle,
            'next_state' => 'RED',
            'cycle_position' => ($timeInCycle / $totalCycle) * 100
        ];
    }
}
?>