document.addEventListener('DOMContentLoaded', () => {
    // Main application object to store all states and data
    const app = {
        map: null,
        userMarker: null,
        tempMarker: null,
        directionsControl: null,
        mapboxKey: null,
        markers: {},
        selectedLightId: null,
        pendingLights: [],
        userLocation: null,
        isMenuOpen: false,
        isAddingLight: false,
        isNavigating: false,
        currentRouteData: null,
        measureMode: null,
        measureTimer: 0,
        measureInterval: null,
        currentTransportMode: 'driving',
        statusUpdateInterval: null,
        
        // Settings with default values
        settings: {
            darkMode: false,
            notifications: true,
            autoRefresh: true,
            autoMeasure: true,
            mapStyle: 'navigation-day-v1',
            defaultTransport: 'driving'
        },
        
        // User data
        user: {
            isLoggedIn: false,
            firstName: '',
            lastName: '',
            email: '',
            preferredVehicle: 'car'
        }
    };

    // Initialize the application
    initializeApp();

    function initializeApp() {
        loadSettings();
        checkUserSession();
        setupMapbox();
        bindEvents();
        checkFirstVisit();
    }

    // Load user settings from localStorage
    function loadSettings() {
        try {
            const savedSettings = localStorage.getItem('trafficLightSettings');
            if (savedSettings) {
                const parsed = JSON.parse(savedSettings);
                Object.assign(app.settings, parsed);
                
                // Apply UI settings
                const darkModeToggle = document.getElementById('darkModeToggle');
                const notificationsToggle = document.getElementById('notificationsToggle');
                const autoRefreshToggle = document.getElementById('autoRefreshToggle');
                const autoMeasureToggle = document.getElementById('autoMeasureToggle');
                const mapStyleSelect = document.getElementById('mapStyle');
                const defaultTransportSelect = document.getElementById('defaultTransport');
                
                if (darkModeToggle) darkModeToggle.checked = app.settings.darkMode;
                if (notificationsToggle) notificationsToggle.checked = app.settings.notifications;
                if (autoRefreshToggle) autoRefreshToggle.checked = app.settings.autoRefresh;
                if (autoMeasureToggle) autoMeasureToggle.checked = app.settings.autoMeasure;
                if (mapStyleSelect) mapStyleSelect.value = app.settings.mapStyle;
                if (defaultTransportSelect) defaultTransportSelect.value = app.settings.defaultTransport;
                
                // Apply dark mode if enabled
                if (app.settings.darkMode) {
                    document.documentElement.classList.add('dark');
                    document.body.classList.add('dark-mode');
                }
            }
        } catch (error) {
            console.error('Error loading settings:', error);
        }
    }

    // Save settings to localStorage
    function saveSettings() {
        const darkModeToggle = document.getElementById('darkModeToggle');
        const notificationsToggle = document.getElementById('notificationsToggle');
        const autoRefreshToggle = document.getElementById('autoRefreshToggle');
        const autoMeasureToggle = document.getElementById('autoMeasureToggle');
        const mapStyleSelect = document.getElementById('mapStyle');
        const defaultTransportSelect = document.getElementById('defaultTransport');
        
        app.settings.darkMode = darkModeToggle ? darkModeToggle.checked : app.settings.darkMode;
        app.settings.notifications = notificationsToggle ? notificationsToggle.checked : app.settings.notifications;
        app.settings.autoRefresh = autoRefreshToggle ? autoRefreshToggle.checked : app.settings.autoRefresh;
        app.settings.autoMeasure = autoMeasureToggle ? autoMeasureToggle.checked : app.settings.autoMeasure;
        app.settings.mapStyle = mapStyleSelect ? mapStyleSelect.value : app.settings.mapStyle;
        app.settings.defaultTransport = defaultTransportSelect ? defaultTransportSelect.value : app.settings.defaultTransport;
        
        try {
            localStorage.setItem('trafficLightSettings', JSON.stringify(app.settings));
        } catch (error) {
            console.error('Error saving settings:', error);
        }
    }

    // Check for existing user session
    function checkUserSession() {
        const userId = localStorage.getItem('userId') || sessionStorage.getItem('userId');
        if (userId) {
            fetch(`api/user-info.php?id=${userId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        app.user.isLoggedIn = true;
                        app.user.firstName = data.user.first_name;
                        app.user.lastName = data.user.last_name;
                        app.user.email = data.user.email;
                        app.user.preferredVehicle = data.user.preferred_vehicle;
                        updateUserDisplay();
                    } else {
                        clearUserSession();
                    }
                })
                .catch(error => {
                    console.error('Error fetching user info:', error);
                    clearUserSession();
                });
        } else {
            updateUserDisplay();
        }
    }

    // Update user information display
    function updateUserDisplay() {
        const nameElement = document.getElementById('accountName');
        const emailElement = document.getElementById('accountEmail');
        const loginButtons = document.getElementById('loginButtons');
        const logoutButton = document.getElementById('logoutButton');
        
        if (app.user.isLoggedIn) {
            if (nameElement) nameElement.textContent = `${app.user.firstName} ${app.user.lastName}`;
            if (emailElement) emailElement.textContent = app.user.email;
            if (loginButtons) loginButtons.classList.add('hidden');
            if (logoutButton) logoutButton.classList.remove('hidden');
        } else {
            if (nameElement) nameElement.textContent = 'Guest User';
            if (emailElement) emailElement.textContent = 'Not signed in';
            if (loginButtons) loginButtons.classList.remove('hidden');
            if (logoutButton) logoutButton.classList.add('hidden');
        }
    }

    // Setup Mapbox with proper error handling
    function setupMapbox() {
        fetch('get-mapbox-key.php')
            .then(response => response.json())
            .then(data => {
                if (!data.success || !data.key) {
                    throw new Error('Invalid MapBox API key response');
                }
                
                app.mapboxKey = data.key;
                mapboxgl.accessToken = app.mapboxKey;
                
                initMap();
            })
            .catch(error => {
                console.error('Error loading MapBox key:', error);
                showNotification('Error', 'Failed to load map service. Please refresh the page.', 'error');
                showMapLoadError();
            });
    }

    // Initialize the map with proper error handling
    function initMap() {
        try {
            // Create the map instance
            app.map = new mapboxgl.Map({
                container: 'map',
                style: `mapbox://styles/mapbox/${app.settings.mapStyle}`,
                center: [0, 0], // Will be updated by geolocation
                zoom: 15,
                maxZoom: 19,
                minZoom: 3,
                attributionControl: false,
                trackResize: true,
                transformRequest: (url, resourceType) => {
                    if (resourceType === 'Tile' || resourceType === 'Style' || resourceType === 'Source') {
                        return {
                            url: url,
                            headers: {
                                'Cache-Control': 'max-age=3600'
                            }
                        };
                    }
                }
            });
            
            // Add attribution in bottom-left
            app.map.addControl(new mapboxgl.AttributionControl(), 'bottom-left');
            
            // Setup event handlers
            app.map.on('load', onMapLoad);
            app.map.on('error', handleMapError);
            app.map.on('click', handleMapClick);
            
            // Handle style changes (which can happen when toggling dark mode)
            app.map.on('style.load', () => {
                console.log('Map style loaded');
                loadTrafficLights();
            });
        } catch (error) {
            console.error('Error initializing map:', error);
            showNotification('Error', 'Failed to initialize map. Please refresh the page.', 'error');
            showMapLoadError();
        }
    }

    // Handle map load event
    function onMapLoad() {
        console.log('Map loaded');
        
        // Add navigation controls
        app.map.addControl(new mapboxgl.NavigationControl({
            showCompass: true,
            visualizePitch: true
        }), 'bottom-right');
        
        // Add geolocate control - this will handle user location
        const geolocateControl = new mapboxgl.GeolocateControl({
            positionOptions: {
                enableHighAccuracy: true,
                timeout: 6000
            },
            trackUserLocation: true,
            showUserHeading: true,
            showAccuracyCircle: true
        });
        
        app.map.addControl(geolocateControl, 'bottom-right');
        
        // Set up directions control
        try {
            app.directionsControl = new MapboxDirections({
                accessToken: mapboxgl.accessToken,
                unit: 'metric',
                profile: 'mapbox/driving',
                alternatives: false,
                geometries: 'geojson',
                controls: {
                    inputs: false,
                    instructions: false,
                    profileSwitcher: false
                },
                flyTo: false
            });
            
            app.map.addControl(app.directionsControl, 'top-left');
            
            // Hide directions UI by default
            const directionsElement = document.querySelector('.mapboxgl-ctrl-directions');
            if (directionsElement) directionsElement.style.display = 'none';
            
            // Handle route updates
            app.directionsControl.on('route', handleRouteUpdate);
        } catch (error) {
            console.error('Error setting up directions:', error);
        }
        
        // Setup search boxes
        setupSearchBoxes();
        
        // Load traffic lights
        loadTrafficLights();
        
        // Auto-locate user with a slight delay to ensure map is ready
        setTimeout(() => {
            try {
                geolocateControl.trigger();
                
                // Listen for user position updates
                geolocateControl.on('geolocate', (e) => {
                    app.userLocation = [e.coords.longitude, e.coords.latitude];
                    console.log('User located at:', app.userLocation);
                    
                    // If we're adding a light and don't have coords yet, use user location
                    if (app.isAddingLight && !app.tempMarker) {
                        setAddLightPosition(app.userLocation[0], app.userLocation[1]);
                    }
                });
            } catch (error) {
                console.error('Error triggering geolocation:', error);
                // Fall back to a default location if geolocation fails
                app.map.flyTo({
                    center: [2.3522, 48.8566], // Paris as a fallback
                    zoom: 13
                });
            }
        }, 1000);
        
        // Start regular status updates if enabled
        if (app.settings.autoRefresh) {
            startStatusUpdates();
        }
    }

    // Handle map errors
    function handleMapError(e) {
        console.error('Map error:', e);
    }

    // Show error message when map fails to load
    function showMapLoadError() {
        const mapContainer = document.getElementById('map');
        if (mapContainer) {
            mapContainer.innerHTML = `
                <div class="h-full w-full flex items-center justify-center bg-gray-100">
                    <div class="text-center p-5">
                        <i class="fas fa-map-marked-alt text-4xl text-gray-400 mb-3"></i>
                        <h3 class="text-lg font-medium text-gray-700">Map could not be loaded</h3>
                        <p class="text-gray-500 mt-1">Please check your internet connection and refresh the page</p>
                        <button onclick="window.location.reload()" class="mt-4 bg-primary-500 text-white px-4 py-2 rounded-lg">
                            <i class="fas fa-sync-alt mr-2"></i> Refresh Page
                        </button>
                    </div>
                </div>
            `;
        }
    }

    // Set up search boxes for address lookup
    function setupSearchBoxes() {
        try {
            // Main search box
            const searchBox = new MapboxGeocoder({
                accessToken: mapboxgl.accessToken,
                mapboxgl: mapboxgl,
                marker: false,
                placeholder: 'Search locations or traffic lights...',
                proximity: 'ip' // Prioritize results near user's location
            });
            
            const searchBoxContainer = document.getElementById('searchBox');
            if (searchBoxContainer) {
                searchBoxContainer.appendChild(searchBox.onAdd(app.map));
            }
            
            // Route modal search boxes
            const startLocationSearch = new MapboxGeocoder({
                accessToken: mapboxgl.accessToken,
                mapboxgl: mapboxgl,
                marker: false,
                placeholder: 'Enter starting point',
                proximity: 'ip'
            });
            
            const endLocationSearch = new MapboxGeocoder({
                accessToken: mapboxgl.accessToken,
                mapboxgl: mapboxgl,
                marker: false,
                placeholder: 'Enter destination',
                proximity: 'ip'
            });
            
            const startLocationContainer = document.getElementById('startLocationSearch');
            const endLocationContainer = document.getElementById('endLocationSearch');
            
            if (startLocationContainer && endLocationContainer) {
                startLocationContainer.appendChild(startLocationSearch.onAdd(app.map));
                endLocationContainer.appendChild(endLocationSearch.onAdd(app.map));
                
                // Store search results
                startLocationSearch.on('result', (e) => {
                    if (app.directionsControl && e.result && e.result.center) {
                        app.directionsControl.setOrigin(e.result.center);
                    }
                });
                
                endLocationSearch.on('result', (e) => {
                    if (app.directionsControl && e.result && e.result.center) {
                        app.directionsControl.setDestination(e.result.center);
                    }
                });
            }
        } catch (error) {
            console.error('Error setting up search boxes:', error);
        }
    }

    // Bind all application event listeners
    function bindEvents() {
        // Menu toggle button
        const fabMenuBtn = document.getElementById('fabMenuBtn');
        if (fabMenuBtn) {
            fabMenuBtn.addEventListener('click', toggleMenu);
        }
        
        // Menu items
        document.querySelectorAll('.menu-item').forEach(item => {
            const button = item.querySelector('button');
            if (button) {
                button.addEventListener('click', () => {
                    const id = item.id;
                    handleMenuItemClick(id);
                });
            }
        });
        
        // Modal close buttons
        document.querySelectorAll('.close-modal').forEach(button => {
            button.addEventListener('click', () => {
                const modal = button.closest('.modal');
                if (modal) closeModal(modal);
            });
        });
        
        // Modal overlays
        document.querySelectorAll('.modal-overlay').forEach(overlay => {
            overlay.addEventListener('click', () => {
                const modal = overlay.closest('.modal');
                if (modal) closeModal(modal);
            });
        });
        
        // Info panel
        const infoPanel = document.getElementById('infoPanel');
        const closeInfoBtn = document.getElementById('closeInfoPanel');
        if (closeInfoBtn) {
            closeInfoBtn.addEventListener('click', () => {
                if (infoPanel) infoPanel.classList.remove('active');
            });
        }
        
        // Info panel navigate button
        const infoPanelNavigateBtn = document.getElementById('infoPanelNavigate');
        if (infoPanelNavigateBtn) {
            infoPanelNavigateBtn.addEventListener('click', () => {
                if (app.selectedLightId && app.markers[app.selectedLightId]) {
                    const light = app.markers[app.selectedLightId].data;
                    if (infoPanel) infoPanel.classList.remove('active');
                    navigateToLight(light);
                }
            });
        }
        
        // Info panel measure button
        const infoPanelMeasureBtn = document.getElementById('infoPanelMeasure');
        if (infoPanelMeasureBtn) {
            infoPanelMeasureBtn.addEventListener('click', () => {
                if (app.selectedLightId) {
                    if (infoPanel) infoPanel.classList.remove('active');
                    openMeasureModal(app.selectedLightId);
                }
            });
        }
        
        // Route panel buttons
        const cancelRouteBtn = document.getElementById('cancelRouteBtn');
        if (cancelRouteBtn) {
            cancelRouteBtn.addEventListener('click', cancelRoute);
        }
        
        const startNavigationBtn = document.getElementById('startNavigationBtn');
        if (startNavigationBtn) {
            startNavigationBtn.addEventListener('click', toggleNavigation);
        }
        
        // Transport mode buttons in route panel
        document.querySelectorAll('.transport-mode-btn').forEach(button => {
            button.addEventListener('click', () => {
                updateTransportMode(button);
            });
        });
        
        // Transport mode choices in route search modal
        document.querySelectorAll('.transport-mode-choice').forEach(button => {
            button.addEventListener('click', () => {
                document.querySelectorAll('.transport-mode-choice').forEach(btn => {
                    btn.classList.remove('active', 'bg-primary-50', 'border-primary-200', 'text-primary-600');
                    btn.classList.add('bg-gray-50', 'border-gray-200', 'text-gray-600');
                });
                
                button.classList.add('active', 'bg-primary-50', 'border-primary-200', 'text-primary-600');
                button.classList.remove('bg-gray-50', 'border-gray-200', 'text-gray-600');
                
                app.currentTransportMode = button.getAttribute('data-mode');
            });
        });
        
        // Calculate route button
        const calculateRouteBtn = document.getElementById('calculateRouteBtn');
        if (calculateRouteBtn) {
            calculateRouteBtn.addEventListener('click', () => {
                if (app.directionsControl) {
                    const origin = app.directionsControl.getOrigin();
                    const destination = app.directionsControl.getDestination();
                    
                    if (origin && destination) {
                        calculateRoute(origin.geometry.coordinates, destination.geometry.coordinates);
                        closeModal(document.getElementById('routeSearchModal'));
                    } else {
                        showNotification('Error', 'Please select start and destination points', 'error');
                    }
                }
            });
        }
        
        // Settings toggles and inputs
        const darkModeToggle = document.getElementById('darkModeToggle');
        if (darkModeToggle) {
            darkModeToggle.addEventListener('change', () => {
                toggleDarkMode();
                saveSettings();
            });
        }
        
        const notificationsToggle = document.getElementById('notificationsToggle');
        if (notificationsToggle) {
            notificationsToggle.addEventListener('change', saveSettings);
        }
        
        const autoRefreshToggle = document.getElementById('autoRefreshToggle');
        if (autoRefreshToggle) {
            autoRefreshToggle.addEventListener('change', () => {
                app.settings.autoRefresh = autoRefreshToggle.checked;
                if (app.settings.autoRefresh) {
                    startStatusUpdates();
                } else {
                    stopStatusUpdates();
                }
                saveSettings();
            });
        }
        
        const autoMeasureToggle = document.getElementById('autoMeasureToggle');
        if (autoMeasureToggle) {
            autoMeasureToggle.addEventListener('change', saveSettings);
        }
        
        const mapStyleSelect = document.getElementById('mapStyle');
        if (mapStyleSelect) {
            mapStyleSelect.addEventListener('change', () => {
                app.settings.mapStyle = mapStyleSelect.value;
                if (app.map) {
                    app.map.setStyle(`mapbox://styles/mapbox/${app.settings.mapStyle}`);
                }
                saveSettings();
            });
        }
        
        const defaultTransportSelect = document.getElementById('defaultTransport');
        if (defaultTransportSelect) {
            defaultTransportSelect.addEventListener('change', saveSettings);
        }
        
        // Add traffic light form
        const addLightForm = document.getElementById('addLightForm');
        if (addLightForm) {
            addLightForm.addEventListener('submit', handleAddLight);
        }
        
        // Measurement buttons
        const measureRedBtn = document.getElementById('measureRedBtn');
        if (measureRedBtn) {
            measureRedBtn.addEventListener('click', () => startMeasure('red'));
        }
        
        const measureGreenBtn = document.getElementById('measureGreenBtn');
        if (measureGreenBtn) {
            measureGreenBtn.addEventListener('click', () => startMeasure('green'));
        }
        
        const startTimerBtn = document.getElementById('startTimer');
        if (startTimerBtn) {
            startTimerBtn.addEventListener('click', startMeasureTimer);
        }
        
        const stopTimerBtn = document.getElementById('stopTimer');
        if (stopTimerBtn) {
            stopTimerBtn.addEventListener('click', stopMeasureTimer);
        }
        
        const saveTimerBtn = document.getElementById('saveTimer');
        if (saveTimerBtn) {
            saveTimerBtn.addEventListener('click', saveMeasureTimer);
        }
        
        // Account buttons
        const loginBtn = document.getElementById('loginBtn');
        if (loginBtn) {
            loginBtn.addEventListener('click', () => {
                showLoginForm();
                openModal(document.getElementById('accountModal'));
            });
        }
        
        const signupBtn = document.getElementById('signupBtn');
        if (signupBtn) {
            signupBtn.addEventListener('click', () => {
                showSignupForm();
                openModal(document.getElementById('accountModal'));
            });
        }
        
        const logoutBtn = document.getElementById('logoutBtn');
        if (logoutBtn) {
            logoutBtn.addEventListener('click', logoutUser);
        }
        
        const switchToSignupBtn = document.getElementById('switchToSignupBtn');
        if (switchToSignupBtn) {
            switchToSignupBtn.addEventListener('click', (e) => {
                e.preventDefault();
                showSignupForm();
            });
        }
        
        const switchToLoginBtn = document.getElementById('switchToLoginBtn');
        if (switchToLoginBtn) {
            switchToLoginBtn.addEventListener('click', (e) => {
                e.preventDefault();
                showLoginForm();
            });
        }
        
        // Account forms
        const loginForm = document.getElementById('loginForm');
        if (loginForm) {
            loginForm.addEventListener('submit', (e) => {
                e.preventDefault();
                loginUser();
            });
        }
        
        const signupForm = document.getElementById('signupForm');
        if (signupForm) {
            signupForm.addEventListener('submit', (e) => {
                e.preventDefault();
                signupUser();
            });
        }
        
        const onboardingForm = document.getElementById('onboardingForm');
        if (onboardingForm) {
            onboardingForm.addEventListener('submit', (e) => {
                e.preventDefault();
                createAccount(true);
            });
        }
        
        // Reset app button
        const resetAppBtn = document.getElementById('resetAppBtn');
        if (resetAppBtn) {
            resetAppBtn.addEventListener('click', resetApp);
        }
        
        // Notification close button
        const closeNotification = document.getElementById('closeNotification');
        if (closeNotification) {
            closeNotification.addEventListener('click', () => {
                const banner = document.getElementById('notificationBanner');
                if (banner) banner.style.opacity = '0';
            });
        }
        
        // Certification banner dismiss button
        const dismissCertBanner = document.getElementById('dismissCertBanner');
        if (dismissCertBanner) {
            dismissCertBanner.addEventListener('click', () => {
                const certBanner = document.getElementById('certificationBanner');
                if (certBanner) certBanner.classList.add('hidden');
            });
        }
    }

    // Check if this is the first visit
    function checkFirstVisit() {
        if (!localStorage.getItem('trafficLightOnboardingCompleted') && !app.user.isLoggedIn) {
            const onboardingModal = document.getElementById('onboardingModal');
            if (onboardingModal) openModal(onboardingModal);
        }
    }

    // Toggle the FAB menu
    function toggleMenu() {
        app.isMenuOpen = !app.isMenuOpen;
        
        const fabMenuBtn = document.getElementById('fabMenuBtn');
        if (!fabMenuBtn) return;
        
        if (app.isMenuOpen) {
            fabMenuBtn.innerHTML = '<i class="fas fa-times"></i>';
            
            document.querySelectorAll('.menu-item').forEach(item => {
                const position = parseInt(item.getAttribute('data-position'));
                setTimeout(() => {
                    item.classList.add('active');
                    item.style.transform = `translateY(-${position * 70}px)`;
                }, position * 50);
            });
        } else {
            fabMenuBtn.innerHTML = '<i class="fas fa-plus"></i>';
            
            document.querySelectorAll('.menu-item').forEach(item => {
                item.classList.remove('active');
                item.style.transform = '';
            });
        }
    }

    // Handle menu item clicks
    function handleMenuItemClick(id) {
        // Close the menu first
        toggleMenu();
        
        switch (id) {
            case 'accountMenuItem':
                if (app.user.isLoggedIn) {
                    showAccountForm();
                } else {
                    showLoginForm();
                }
                openModal(document.getElementById('accountModal'));
                break;
                
            case 'addLightMenuItem':
                app.isAddingLight = true;
                if (app.userLocation) {
                    setAddLightPosition(app.userLocation[0], app.userLocation[1]);
                }
                openModal(document.getElementById('addLightModal'));
                break;
                
            case 'searchMenuItem':
                openModal(document.getElementById('routeSearchModal'));
                break;
                
            case 'locateMenuItem':
                locateUser();
                break;
                
            case 'settingsMenuItem':
                openModal(document.getElementById('settingsModal'));
                break;
        }
    }

    // Open a modal with proper animation
    function openModal(modal) {
        if (!modal) return;
        
        modal.style.display = 'flex';
        setTimeout(() => {
            const content = modal.querySelector('.modal-content');
            if (content) content.classList.add('active');
        }, 10);
    }

    // Close a modal with proper animation
    function closeModal(modal) {
        if (!modal) return;
        
        const content = modal.querySelector('.modal-content');
        if (content) content.classList.remove('active');
        
        setTimeout(() => {
            modal.style.display = 'none';
            
            // Clean up when closing add light modal
            if (modal.id === 'addLightModal' && app.isAddingLight) {
                app.isAddingLight = false;
                if (app.tempMarker) {
                    app.tempMarker.remove();
                    app.tempMarker = null;
                }
            }
        }, 300);
    }

    // Locate user on the map
    function locateUser() {
        if (app.userLocation) {
            app.map.flyTo({
                center: app.userLocation,
                zoom: 16,
                speed: 1.5
            });
        } else {
            // Try to trigger the geolocate control
            const geolocateBtn = document.querySelector('.mapboxgl-ctrl-geolocate');
            if (geolocateBtn) geolocateBtn.click();
            
            showNotification('Location', 'Getting your location...', 'info');
        }
    }

    // Set position for adding a new traffic light
    function setAddLightPosition(longitude, latitude) {
        const latInput = document.getElementById('latitude');
        const lngInput = document.getElementById('longitude');
        
        if (latInput && lngInput) {
            latInput.value = latitude.toFixed(6);
            lngInput.value = longitude.toFixed(6);
        }
        
        if (app.tempMarker) app.tempMarker.remove();
        
        app.tempMarker = new mapboxgl.Marker({
            color: '#10b981',
            draggable: true
        })
        .setLngLat([longitude, latitude])
        .addTo(app.map);
        
        app.tempMarker.on('dragend', () => {
            const position = app.tempMarker.getLngLat();
            if (latInput && lngInput) {
                latInput.value = position.lat.toFixed(6);
                lngInput.value = position.lng.toFixed(6);
            }
        });
        
        // Center map on the marker
        app.map.flyTo({
            center: [longitude, latitude],
            zoom: 17
        });
    }

    // Handle map click events
    function handleMapClick(e) {
        if (app.isAddingLight) {
            setAddLightPosition(e.lngLat.lng, e.lngLat.lat);
        }
    }

    // Handle adding a new traffic light
    function handleAddLight(e) {
        e.preventDefault();
        
        const submitBtn = e.target.querySelector('[type="submit"]');
        if (!submitBtn) return;
        
        const originalText = submitBtn.textContent;
        submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Adding...';
        submitBtn.disabled = true;
        
        const lightName = document.getElementById('lightName');
        const latitude = document.getElementById('latitude');
        const longitude = document.getElementById('longitude');
        const direction = document.getElementById('direction');
        const redDuration = document.getElementById('redDuration');
        const greenDuration = document.getElementById('greenDuration');
        
        if (!lightName || !latitude || !longitude || !direction || !redDuration || !greenDuration) {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
            showNotification('Error', 'Missing form fields', 'error');
            return;
        }
        
        const lightData = {
            name: lightName.value,
            latitude: latitude.value,
            longitude: longitude.value,
            direction: direction.value,
            red_duration: redDuration.value,
           green_duration: greenDuration.value,
           user_id: app.user.isLoggedIn ? (localStorage.getItem('userId') || sessionStorage.getItem('userId')) : null
       };
       
       fetch('api/add-traffic-light.php', {
           method: 'POST',
           headers: { 'Content-Type': 'application/json' },
           body: JSON.stringify(lightData)
       })
       .then(response => {
           if (!response.ok) throw new Error('Network response was not ok');
           return response.json();
       })
       .then(data => {
           if (data.success) {
               app.pendingLights.push(data.light.id);
               
               // Show certification banner
               const certBanner = document.getElementById('certificationBanner');
               if (certBanner) certBanner.classList.remove('hidden');
               
               showNotification('Success', 'Traffic light added successfully! Pending certification.', 'success');
               
               // Close modal and reset form
               closeModal(document.getElementById('addLightModal'));
               e.target.reset();
               
               // Remove temporary marker
               if (app.tempMarker) {
                   app.tempMarker.remove();
                   app.tempMarker = null;
               }
               
               app.isAddingLight = false;
               
               // Add the new light to the map
               addLightToMap(data.light);
           } else {
               showNotification('Error', data.message || 'Failed to add traffic light', 'error');
           }
           
           // Reset button
           submitBtn.innerHTML = originalText;
           submitBtn.disabled = false;
       })
       .catch(error => {
           console.error('Error adding traffic light:', error);
           showNotification('Error', 'Connection error. Please try again later.', 'error');
           
           // Reset button
           submitBtn.innerHTML = originalText;
           submitBtn.disabled = false;
       });
   }

   // Load traffic lights from the server
   function loadTrafficLights() {
       fetch('api/get-traffic-lights.php')
           .then(response => {
               if (!response.ok) throw new Error('Network response was not ok');
               return response.json();
           })
           .then(data => {
               // Clear existing markers
               for (const id in app.markers) {
                   if (app.markers[id].marker) app.markers[id].marker.remove();
               }
               app.markers = {};
               
               if (Array.isArray(data) && data.length > 0) {
                   data.forEach(light => {
                       addLightToMap(light);
                   });
                   showNotification('Traffic Lights', `Loaded ${data.length} traffic lights`, 'info');
               } else {
                   showNotification('Traffic Lights', 'No traffic lights found in this area', 'info');
               }
           })
           .catch(error => {
               console.error('Error loading traffic lights:', error);
               showNotification('Error', 'Failed to load traffic lights', 'error');
           });
   }

   // Add a traffic light to the map
   function addLightToMap(light) {
       if (!app.map) return;
       
       const status = getLightStatus(light);
       
       // Create custom HTML element for the marker
       const el = document.createElement('div');
       el.className = `w-8 h-8 bg-traffic-${status.color} rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white`;
       el.innerHTML = '<i class="fas fa-traffic-light"></i>';
       
       // Create marker
       const marker = new mapboxgl.Marker(el)
           .setLngLat([parseFloat(light.longitude), parseFloat(light.latitude)])
           .addTo(app.map);
       
       // Add click handler
       marker.getElement().addEventListener('click', () => {
           showLightDetail(light);
       });
       
       // Store marker and data
       app.markers[light.id] = {
           marker: marker,
           element: el,
           data: light
       };
   }

   // Get traffic light status based on current time
   function getLightStatus(light) {
       const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
       const currentTime = Math.floor(Date.now() / 1000);
       const timeInCycle = currentTime % totalCycle;
       
       if (timeInCycle < light.red_duration) {
           return {
               isRed: true,
               color: 'red',
               label: 'RED',
               timeLeft: light.red_duration - timeInCycle,
               percentage: Math.round(((light.red_duration - timeInCycle) / totalCycle) * 100)
           };
       } else {
           return {
               isRed: false,
               color: 'green',
               label: 'GREEN',
               timeLeft: totalCycle - timeInCycle,
               percentage: Math.round(((totalCycle - timeInCycle) / totalCycle) * 100)
           };
       }
   }

   // Update the visual status of a traffic light
   function updateLightStatus(light) {
       const lightObj = app.markers[light.id];
       if (!lightObj || !lightObj.element) return;
       
       const status = getLightStatus(light);
       
       // Update marker appearance
       lightObj.element.className = `w-8 h-8 bg-traffic-${status.color} rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white`;
       
       // Update info panel if the light is selected
       if (app.selectedLightId === light.id) {
           const lightStatusText = document.getElementById('lightStatusText');
           const lightStatusTime = document.getElementById('lightStatusTime');
           const lightStatusBar = document.getElementById('lightStatusBar');
           
           if (lightStatusText) lightStatusText.textContent = status.label;
           if (lightStatusTime) lightStatusTime.textContent = `${status.timeLeft}s until change`;
           
           if (lightStatusBar) {
               lightStatusBar.className = `bg-traffic-${status.color} h-2 rounded-full`;
               lightStatusBar.style.width = `${status.percentage}%`;
           }
       }
       
       // Update measure modal if open
       const measureTitle = document.getElementById('measureTitle');
       const measureStatus = document.getElementById('measureStatus');
       
       if (measureTitle && measureStatus && measureTitle.getAttribute('data-id') === light.id) {
           measureStatus.innerHTML = `
               <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                   ${status.label} (${status.timeLeft}s)
               </span>
           `;
       }
   }

   // Update all traffic lights status
   function updateAllLightStatus() {
       for (const id in app.markers) {
           updateLightStatus(app.markers[id].data);
       }
   }

   // Start regular status updates
   function startStatusUpdates() {
       // Clear any existing interval
       stopStatusUpdates();
       
       // Update every second
       app.statusUpdateInterval = setInterval(() => {
           updateAllLightStatus();
           
           // Update navigation if active
           if (app.isNavigating) {
               updateNavigationInfo();
           }
       }, 1000);
   }

   // Stop status updates
   function stopStatusUpdates() {
       if (app.statusUpdateInterval) {
           clearInterval(app.statusUpdateInterval);
           app.statusUpdateInterval = null;
       }
   }

   // Show traffic light details
   function showLightDetail(light) {
       app.selectedLightId = light.id;
       
       // Center map on light
       app.map.flyTo({
           center: [parseFloat(light.longitude), parseFloat(light.latitude)],
           zoom: 17,
           speed: 1.2
       });
       
       // Update info panel
       const infoPanel = document.getElementById('infoPanel');
       const infoPanelTitle = document.getElementById('infoPanelTitle');
       const status = getLightStatus(light);
       
       if (infoPanelTitle) infoPanelTitle.textContent = light.name;
       
       const lightStatusText = document.getElementById('lightStatusText');
       const lightStatusTime = document.getElementById('lightStatusTime');
       const lightStatusBar = document.getElementById('lightStatusBar');
       
       if (lightStatusText) lightStatusText.textContent = status.label;
       if (lightStatusTime) lightStatusTime.textContent = `${status.timeLeft}s until change`;
       
       if (lightStatusBar) {
           lightStatusBar.className = `bg-traffic-${status.color} h-2 rounded-full`;
           lightStatusBar.style.width = `${status.percentage}%`;
       }
       
       // Show the panel
       if (infoPanel) infoPanel.classList.add('active');
   }

   // Calculate a route
   function calculateRoute(origin, destination) {
       if (!app.directionsControl || !origin || !destination) {
           showNotification('Error', 'Cannot calculate route', 'error');
           return;
       }
       
       try {
           // Set origin and destination
           app.directionsControl.setOrigin(origin);
           app.directionsControl.setDestination(destination);
           
           // Set transport profile
           app.directionsControl.setProfile(`mapbox/${app.currentTransportMode}`);
           
           // Show directions UI
           const directionsElement = document.querySelector('.mapboxgl-ctrl-directions');
           if (directionsElement) directionsElement.style.display = 'block';
           
           // Show route panel
           const routePanel = document.querySelector('.route-panel');
           const searchContainer = document.querySelector('.search-container');
           
           if (routePanel) routePanel.classList.add('active');
           if (searchContainer) searchContainer.classList.add('route-active');
       } catch (error) {
           console.error('Error calculating route:', error);
           showNotification('Error', 'Failed to calculate route', 'error');
       }
   }

   // Handle route update from directions control
   function handleRouteUpdate(e) {
       if (!e.route || !e.route[0]) return;
       
       const route = e.route[0];
       app.currentRouteData = route;
       
       // Update estimated time display
       const estimatedDuration = document.getElementById('estimatedDuration');
       const arrivalTime = document.getElementById('arrivalTime');
       
       if (estimatedDuration) {
           const minutes = Math.round(route.duration / 60);
           estimatedDuration.textContent = `${minutes} min`;
       }
       
       if (arrivalTime) {
           const now = new Date();
           const arrival = new Date(now.getTime() + route.duration * 1000);
           arrivalTime.textContent = `Arrival at ${arrival.getHours().toString().padStart(2, '0')}:${arrival.getMinutes().toString().padStart(2, '0')}`;
       }
       
       // Find traffic lights along route
       findTrafficLightsOnRoute(route.geometry.coordinates);
   }

   // Find traffic lights along the calculated route
   function findTrafficLightsOnRoute(coordinates) {
       app.lightsOnRoute = [];
       
       for (const id in app.markers) {
           const light = app.markers[id].data;
           const lightCoords = [parseFloat(light.longitude), parseFloat(light.latitude)];
           
           for (let i = 0; i < coordinates.length; i++) {
               const distance = calculateDistance(coordinates[i], lightCoords);
               
               if (distance < 50) { // Within 50 meters
                   app.lightsOnRoute.push({
                       id: id,
                       light: light,
                       routePointIndex: i,
                       distance: distance
                   });
                   break;
               }
           }
       }
       
       if (app.lightsOnRoute.length > 0) {
           showNotification('Route Info', `Found ${app.lightsOnRoute.length} traffic lights on your route`, 'info');
       }
   }

   // Calculate distance between two points
   function calculateDistance(point1, point2) {
       const [lon1, lat1] = point1;
       const [lon2, lat2] = point2;
       
       // Haversine formula
       const R = 6371000; // Earth radius in meters
       const φ1 = lat1 * Math.PI / 180;
       const φ2 = lat2 * Math.PI / 180;
       const Δφ = (lat2 - lat1) * Math.PI / 180;
       const Δλ = (lon2 - lon1) * Math.PI / 180;
       
       const a = Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
                 Math.cos(φ1) * Math.cos(φ2) *
                 Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
       const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
       
       return R * c;
   }

   // Navigate to a specific traffic light
   function navigateToLight(light) {
       if (!app.userLocation) {
           showNotification('Error', 'Your location is not available', 'error');
           return;
       }
       
       calculateRoute(
           app.userLocation,
           [parseFloat(light.longitude), parseFloat(light.latitude)]
       );
   }

   // Cancel the current route
   function cancelRoute() {
       app.isNavigating = false;
       
       // Hide route panel
       const routePanel = document.querySelector('.route-panel');
       const searchContainer = document.querySelector('.search-container');
       
       if (routePanel) routePanel.classList.remove('active');
       if (searchContainer) searchContainer.classList.remove('route-active');
       
       // Hide directions UI
       const directionsElement = document.querySelector('.mapboxgl-ctrl-directions');
       if (directionsElement) directionsElement.style.display = 'none';
       
       // Clear routes
       if (app.directionsControl) app.directionsControl.removeRoutes();
       
       // Reset navigation button
       const startNavigationBtn = document.getElementById('startNavigationBtn');
       if (startNavigationBtn) startNavigationBtn.innerHTML = '<i class="fas fa-play"></i>';
       
       app.currentRouteData = null;
       app.lightsOnRoute = [];
   }

   // Toggle navigation mode
   function toggleNavigation() {
       const startNavigationBtn = document.getElementById('startNavigationBtn');
       
       if (app.isNavigating) {
           app.isNavigating = false;
           if (startNavigationBtn) startNavigationBtn.innerHTML = '<i class="fas fa-play"></i>';
           showNotification('Navigation', 'Navigation stopped', 'info');
       } else {
           if (!app.currentRouteData) {
               showNotification('Error', 'No route calculated', 'error');
               return;
           }
           
           app.isNavigating = true;
           if (startNavigationBtn) startNavigationBtn.innerHTML = '<i class="fas fa-stop"></i>';
           showNotification('Navigation', 'Navigation started', 'success');
           
           // Start tracking for auto-measure if enabled
           if (app.settings.autoMeasure) {
               startAutoMeasureDetection();
           }
       }
   }

   // Update navigation information
   function updateNavigationInfo() {
       if (!app.isNavigating || !app.userLocation || !app.currentRouteData) return;
       
       try {
           // Find closest point on route
           const coordinates = app.currentRouteData.geometry.coordinates;
           let closestPointIndex = 0;
           let minDistance = Infinity;
           
           for (let i = 0; i < coordinates.length; i++) {
               const distance = calculateDistance(
                   [app.userLocation[0], app.userLocation[1]],
                   coordinates[i]
               );
               
               if (distance < minDistance) {
                   minDistance = distance;
                   closestPointIndex = i;
               }
           }
           
           // Calculate remaining distance and time
           const remainingCoordinates = coordinates.slice(closestPointIndex);
           const remainingDistance = calculatePathDistance(remainingCoordinates);
           const remainingTime = (remainingDistance / app.currentRouteData.distance) * app.currentRouteData.duration;
           
           // Update UI
           const estimatedDuration = document.getElementById('estimatedDuration');
           const arrivalTime = document.getElementById('arrivalTime');
           
           if (estimatedDuration) {
               const minutes = Math.round(remainingTime / 60);
               estimatedDuration.textContent = `${minutes} min`;
           }
           
           if (arrivalTime) {
               const now = new Date();
               const arrival = new Date(now.getTime() + remainingTime * 1000);
               arrivalTime.textContent = `Arrival at ${arrival.getHours().toString().padStart(2, '0')}:${arrival.getMinutes().toString().padStart(2, '0')}`;
           }
           
           // Check for upcoming traffic lights
           checkUpcomingTrafficLights(closestPointIndex);
       } catch (error) {
           console.error('Error updating navigation info:', error);
       }
   }

   // Calculate total distance of a path
   function calculatePathDistance(coordinates) {
       let totalDistance = 0;
       
       for (let i = 0; i < coordinates.length - 1; i++) {
           totalDistance += calculateDistance(coordinates[i], coordinates[i + 1]);
       }
       
       return totalDistance;
   }

   // Check for upcoming traffic lights
   function checkUpcomingTrafficLights(currentRouteIndex) {
       if (!app.lightsOnRoute || app.lightsOnRoute.length === 0) return;
       
       // Find upcoming lights
       const upcomingLights = app.lightsOnRoute.filter(item => 
           item.routePointIndex > currentRouteIndex
       ).sort((a, b) => a.routePointIndex - b.routePointIndex);
       
       if (upcomingLights.length > 0) {
           const nextLight = upcomingLights[0];
           const lightStatus = getLightStatus(nextLight.light);
           
           // Calculate distance to light
           const coordinates = app.currentRouteData.geometry.coordinates;
           const distanceToLight = calculatePathDistance(
               coordinates.slice(currentRouteIndex, nextLight.routePointIndex + 1)
           );
           
           // Estimate time to reach light based on travel mode
           const speedKmh = app.currentTransportMode === 'driving' ? 50 : 
                           app.currentTransportMode === 'cycling' ? 15 : 5;
           const timeToLightSec = (distanceToLight / 1000) / speedKmh * 3600;
           
           // Only notify if light is close enough (within 60 seconds)
           if (timeToLightSec < 60) {
               // Calculate if we'll hit a red or green light
               const willHitRed = lightStatus.isRed && lightStatus.timeLeft > timeToLightSec;
               const willHitGreen = !lightStatus.isRed && lightStatus.timeLeft > timeToLightSec;
               
               let statusMessage = '';
               
               if (willHitRed) {
                   statusMessage = `Light will be RED when you arrive. Wait time: ~${Math.round(lightStatus.timeLeft - timeToLightSec)}s`;
               } else if (willHitGreen) {
                   statusMessage = `Light will be GREEN when you arrive. Remaining green time: ~${Math.round(lightStatus.timeLeft - timeToLightSec)}s`;
               } else if (lightStatus.isRed) {
                   statusMessage = `Light will turn GREEN in ${lightStatus.timeLeft}s`;
               } else {
                   statusMessage = `Light will turn RED in ${lightStatus.timeLeft}s`;
               }
               
               const timeSinceLastNotification = Date.now() - (app.lastLightNotification || 0);
               
               // Avoid too frequent notifications
               if (timeSinceLastNotification > 10000) {
                   app.lastLightNotification = Date.now();
                   showNotification(
                       `Approaching: ${nextLight.light.name}`,
                       `${Math.round(distanceToLight)}m ahead. ${statusMessage}`,
                       willHitRed ? 'error' : willHitGreen ? 'success' : 'info'
                   );
               }
           }
       }
   }

   // Start auto-measure detection
   function startAutoMeasureDetection() {
       app.lastPosition = app.userLocation;
       app.lastPositionTime = Date.now();
       app.stoppedAt = null;
       app.measuringLightId = null;
       
       // Auto-measure detection is handled in updateNavigationInfo
   }

   // Detect traffic stops for auto-measurement
   function detectTrafficStop() {
       if (!app.settings.autoMeasure || !app.isNavigating || !app.userLocation || !app.lastPosition) return;
       
       const now = Date.now();
       const distance = calculateDistance(
           [app.lastPosition[0], app.lastPosition[1]],
           [app.userLocation[0], app.userLocation[1]]
       );
       
       // Check if we're stopped (less than 2 meters movement)
       if (distance < 2) {
           if (!app.stoppedAt) {
               app.stoppedAt = now;
           } else if (now - app.stoppedAt > 2000 && !app.measuringLightId) {
               // Stopped for more than 2 seconds, check for nearby traffic light
               checkForTrafficLightAtStop();
           }
       } else {
           if (app.stoppedAt && app.measuringLightId) {
               // We were stopped but now moving - save measurement
               const stopDuration = (now - app.stoppedAt) / 1000; // in seconds
               
               if (stopDuration > 5 && stopDuration < 120) {
                   // Only save if stop was between 5 and 120 seconds
                   autoSaveTrafficLightMeasurement(app.measuringLightId, stopDuration);
               }
           }
           
           // Update last position
           app.lastPosition = app.userLocation;
           app.stoppedAt = null;
           app.measuringLightId = null;
       }
   }

   // Check for a traffic light at the stopping position
   function checkForTrafficLightAtStop() {
       let nearestLightId = null;
       let minDistance = Infinity;
       
       // Check all traffic lights
       for (const id in app.markers) {
           const light = app.markers[id].data;
           const distance = calculateDistance(
               [app.userLocation[0], app.userLocation[1]],
               [parseFloat(light.longitude), parseFloat(light.latitude)]
           );
           
           // Consider lights within 50 meters
           if (distance < 50 && distance < minDistance) {
               minDistance = distance;
               nearestLightId = id;
           }
       }
       
       if (nearestLightId) {
           app.measuringLightId = nearestLightId;
           const light = app.markers[nearestLightId].data;
           const status = getLightStatus(light);
           
           if (status.isRed) {
               showNotification(
                   'Auto Measurement',
                   `Measuring red light time for "${light.name}"`,
                   'info'
               );
           }
       }
   }

   // Auto-save traffic light measurement
   function autoSaveTrafficLightMeasurement(lightId, duration) {
       const light = app.markers[lightId].data;
       const status = getLightStatus(light);
       
       // Only measure red lights for now
       if (!status.isRed) return;
       
       fetch('api/update-traffic-light-timing.php', {
           method: 'POST',
           headers: { 'Content-Type': 'application/json' },
           body: JSON.stringify({
               id: lightId,
               duration_type: 'red',
               duration: Math.round(duration),
               auto_measured: true,
               user_id: app.user.isLoggedIn ? localStorage.getItem('userId') || sessionStorage.getItem('userId') : null
           })
       })
       .then(response => response.json())
       .then(data => {
           if (data.success) {
               app.markers[lightId].data = data.light;
               showNotification(
                   'Auto Measurement',
                   `Red light time for "${light.name}" updated (${Math.round(duration)}s)`,
                   'success'
               );
           }
       })
       .catch(error => {
           console.error('Error saving auto measurement:', error);
       });
   }

   // Open the measure modal for a specific light
   function openMeasureModal(lightId) {
       const measureModal = document.getElementById('measureModal');
       if (!measureModal) return;
       
       const light = app.markers[lightId].data;
       const measureTitle = document.getElementById('measureTitle');
       const measureStatus = document.getElementById('measureStatus');
       
       if (measureTitle) {
           measureTitle.textContent = light.name;
           measureTitle.setAttribute('data-id', lightId);
       }
       
       if (measureStatus) {
           const status = getLightStatus(light);
           measureStatus.innerHTML = `
               <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                   ${status.label} (${status.timeLeft}s)
               </span>
           `;
       }
       
       // Hide timer container
       const timerContainer = document.getElementById('timerContainer');
       if (timerContainer) timerContainer.classList.add('hidden');
       
       app.measureMode = null;
       resetMeasureTimer();
       
       openModal(measureModal);
   }

   // Start measure timing
   function startMeasure(mode) {
       app.measureMode = mode;
       
       const timerContainer = document.getElementById('timerContainer');
       const timerInstructions = document.getElementById('timerInstructions');
       const startTimer = document.getElementById('startTimer');
       const stopTimer = document.getElementById('stopTimer');
       
       if (timerContainer) timerContainer.classList.remove('hidden');
       
       if (timerInstructions) {
           timerInstructions.textContent = mode === 'red' ?
               'Press "Start" when the light turns red, then "Stop" when it turns green.' :
               'Press "Start" when the light turns green, then "Stop" when it turns red.';
       }
       
       if (startTimer) {
           startTimer.className = mode === 'red' ?
               'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors' :
               'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors';
       }
       
       if (stopTimer) {
           stopTimer.className = mode === 'red' ?
               'bg-traffic-green text-white py-2.5 rounded-lg hover:bg-green-600 transition-colors opacity-50' :
               'bg-traffic-red text-white py-2.5 rounded-lg hover:bg-red-600 transition-colors opacity-50';
       }
       
       resetMeasureTimer();
   }

   // Start the measurement timer
   function startMeasureTimer() {
       clearInterval(app.measureInterval);
       
       app.measureTimer = 0;
       const startTime = Date.now();
       
       const timerDisplay = document.getElementById('timerDisplay');
       const measureResult = document.getElementById('measureResult');
       const startTimer = document.getElementById('startTimer');
       const stopTimer = document.getElementById('stopTimer');
       const saveTimer = document.getElementById('saveTimer');
       
       if (timerDisplay) timerDisplay.textContent = '00:00';
       if (measureResult) measureResult.textContent = 'Measuring...';
       
       if (startTimer) {
           startTimer.disabled = true;
           startTimer.classList.add('opacity-50');
       }
       
       if (stopTimer) {
           stopTimer.disabled = false;
           stopTimer.classList.remove('opacity-50');
       }
       
       if (saveTimer) {
           saveTimer.disabled = true;
           saveTimer.classList.add('opacity-50');
       }
       
       app.measureInterval = setInterval(() => {
           const elapsed = Math.floor((Date.now() - startTime) / 1000);
           app.measureTimer = elapsed;
           
           if (timerDisplay) {
               const minutes = Math.floor(elapsed / 60);
               const seconds = elapsed % 60;
               timerDisplay.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
           }
       }, 1000);
   }

   // Stop the measurement timer
   function stopMeasureTimer() {
       clearInterval(app.measureInterval);
       app.measureInterval = null;
       
       const measureResult = document.getElementById('measureResult');
       const stopTimer = document.getElementById('stopTimer');
       const saveTimer = document.getElementById('saveTimer');
       
       if (measureResult) {
           measureResult.textContent = `Measured duration: ${app.measureTimer} seconds. Click Save to confirm.`;
       }
       
       if (stopTimer) {
           stopTimer.disabled = true;
           stopTimer.classList.add('opacity-50');
       }
       
       if (saveTimer) {
           saveTimer.disabled = false;
           saveTimer.classList.remove('opacity-50');
       }
   }

   // Reset the measurement timer
   function resetMeasureTimer() {
       clearInterval(app.measureInterval);
       app.measureInterval = null;
       app.measureTimer = 0;
       
       const timerDisplay = document.getElementById('timerDisplay');
       const measureResult = document.getElementById('measureResult');
       const startTimer = document.getElementById('startTimer');
       const stopTimer = document.getElementById('stopTimer');
       const saveTimer = document.getElementById('saveTimer');
       
       if (timerDisplay) timerDisplay.textContent = '00:00';
       if (measureResult) measureResult.textContent = '';
       
       if (startTimer) {
           startTimer.disabled = false;
           startTimer.classList.remove('opacity-50');
       }
       
       if (stopTimer) {
           stopTimer.disabled = true;
           stopTimer.classList.add('opacity-50');
       }
       
       if (saveTimer) {
           saveTimer.disabled = true;
           saveTimer.classList.add('opacity-50');
       }
   }

   // Save the measurement timer
   function saveMeasureTimer() {
       const measureTitle = document.getElementById('measureTitle');
       if (!measureTitle) return;
       
       const lightId = measureTitle.getAttribute('data-id');
       
       if (!lightId || app.measureTimer <= 0 || !app.measureMode) {
           showNotification('Error', 'Invalid measurement data', 'error');
           return;
       }
       
       const measureResult = document.getElementById('measureResult');
       if (measureResult) measureResult.textContent = 'Saving...';
       
       fetch('api/update-traffic-light-timing.php', {
           method: 'POST',
           headers: { 'Content-Type': 'application/json' },
           body: JSON.stringify({
               id: lightId,
               duration_type: app.measureMode,
               duration: app.measureTimer,
               user_id: app.user.isLoggedIn ? (localStorage.getItem('userId') || sessionStorage.getItem('userId')) : null
           })
       })
       .then(response => response.json())
       .then(data => {
           if (data.success) {
               const measureResult = document.getElementById('measureResult');
               const saveTimer = document.getElementById('saveTimer');
               const timerContainer = document.getElementById('timerContainer');
               
               if (measureResult) {
                   measureResult.innerHTML = '<i class="fas fa-check text-green-500 mr-1"></i> Timing updated!';
               }
               
               if (saveTimer) {
                   saveTimer.disabled = true;
                   saveTimer.classList.add('opacity-50');
               }
               
               // Update the light data and status
               if (app.markers[lightId]) {
                   app.markers[lightId].data = data.light;
                   updateLightStatus(data.light);
               }
               
               setTimeout(() => {
                   if (timerContainer) timerContainer.classList.add('hidden');
                   app.measureMode = null;
               }, 1500);
               
               showNotification('Success', 'Traffic light timing updated successfully', 'success');
           } else {
               const measureResult = document.getElementById('measureResult');
               if (measureResult) {
                   measureResult.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Error: ' + (data.message || 'Unknown error');
               }
               
               showNotification('Error', data.message || 'Failed to update traffic light timing', 'error');
           }
       })
       .catch(error => {
           console.error('Error saving measurement:', error);
           
           const measureResult = document.getElementById('measureResult');
           if (measureResult) {
               measureResult.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Connection error';
           }
           
           showNotification('Error', 'Connection error. Please try again later.', 'error');
       });
   }

   // Toggle dark mode
   function toggleDarkMode() {
       const darkModeToggle = document.getElementById('darkModeToggle');
       if (!darkModeToggle) return;
       
       app.settings.darkMode = darkModeToggle.checked;
       
       if (app.settings.darkMode) {
           document.documentElement.classList.add('dark');
           document.body.classList.add('dark-mode');
           
           if (app.map && app.settings.mapStyle.includes('day')) {
               app.settings.mapStyle = app.settings.mapStyle.replace('day', 'night');
               
               const mapStyleSelect = document.getElementById('mapStyle');
               if (mapStyleSelect) mapStyleSelect.value = app.settings.mapStyle;
               
               app.map.setStyle(`mapbox://styles/mapbox/${app.settings.mapStyle}`);
           }
       } else {
           document.documentElement.classList.remove('dark');
           document.body.classList.remove('dark-mode');
           
           if (app.map && app.settings.mapStyle.includes('night')) {
               app.settings.mapStyle = app.settings.mapStyle.replace('night', 'day');
               
               const mapStyleSelect = document.getElementById('mapStyle');
               if (mapStyleSelect) mapStyleSelect.value = app.settings.mapStyle;
               
               app.map.setStyle(`mapbox://styles/mapbox/${app.settings.mapStyle}`);
           }
       }
   }

   // Update transport mode based on button click
   function updateTransportMode(button) {
       if (!button) return;
       
       document.querySelectorAll('.transport-mode-btn').forEach(btn => {
           btn.classList.remove('active', 'bg-white', 'text-primary-600');
           btn.classList.add('text-gray-600');
       });
       
       button.classList.add('active', 'bg-white', 'text-primary-600');
       button.classList.remove('text-gray-600');
       
       let mode = 'driving';
       if (button.id === 'bikingModeBtn') mode = 'cycling';
       if (button.id === 'walkingModeBtn') mode = 'walking';
       
       app.currentTransportMode = mode;
       
       if (app.directionsControl && app.currentRouteData) {
           app.directionsControl.setProfile(`mapbox/${mode}`);
       }
   }

   // Reset app data
   function resetApp() {
       if (confirm('Are you sure you want to reset all app data? This will clear all your settings and log you out.')) {
           localStorage.removeItem('trafficLightSettings');
           localStorage.removeItem('trafficLightOnboardingCompleted');
           localStorage.removeItem('userId');
           sessionStorage.removeItem('userId');
           
           showNotification('Reset', 'App data reset. Refreshing...', 'info');
           
           setTimeout(() => {
               window.location.reload();
           }, 1500);
       }
   }

   // Show account form
   function showAccountForm() {
       const accountForm = document.getElementById('accountForm');
       const loginForm = document.getElementById('loginForm');
       const signupForm = document.getElementById('signupForm');
       
       if (accountForm) accountForm.classList.remove('hidden');
       if (loginForm) loginForm.classList.add('hidden');
       if (signupForm) signupForm.classList.add('hidden');
       
       if (app.user.isLoggedIn) {
           const firstName = document.getElementById('firstName');
           const lastName = document.getElementById('lastName');
           const email = document.getElementById('email');
           const preferredVehicle = document.getElementById('preferredVehicle');
           
           if (firstName) firstName.value = app.user.firstName;
           if (lastName) lastName.value = app.user.lastName;
           if (email) email.value = app.user.email;
           if (preferredVehicle) preferredVehicle.value = app.user.preferredVehicle;
       }
   }

   // Show login form
   function showLoginForm() {
       const accountForm = document.getElementById('accountForm');
       const loginForm = document.getElementById('loginForm');
       const signupForm = document.getElementById('signupForm');
       
       if (accountForm) accountForm.classList.add('hidden');
       if (loginForm) loginForm.classList.remove('hidden');
       if (signupForm) signupForm.classList.add('hidden');
   }

   // Show signup form
   function showSignupForm() {
       const accountForm = document.getElementById('accountForm');
       const loginForm = document.getElementById('loginForm');
       const signupForm = document.getElementById('signupForm');
       
       if (accountForm) accountForm.classList.add('hidden');
       if (loginForm) loginForm.classList.add('hidden');
       if (signupForm) signupForm.classList.remove('hidden');
   }

   // Login user
   function loginUser() {
       const loginEmail = document.getElementById('loginEmail');
       const loginPassword = document.getElementById('loginPassword');
       const rememberMe = document.getElementById('rememberMe');
       
       if (!loginEmail || !loginPassword) return;
       
       fetch('api/login.php', {
           method: 'POST',
           headers: { 'Content-Type': 'application/json' },
           body: JSON.stringify({
               email: loginEmail.value,
               password: loginPassword.value
           })
       })
       .then(response => response.json())
       .then(data => {
           if (data.success) {
               app.user.isLoggedIn = true;
               app.user.firstName = data.user.first_name;
               app.user.lastName = data.user.last_name;
               app.user.email = data.user.email;
               app.user.preferredVehicle = data.user.preferred_vehicle || 'car';
               
               if (rememberMe && rememberMe.checked) {
                   localStorage.setItem('userId', data.user.id);
               } else {
                   sessionStorage.setItem('userId', data.user.id);
               }
               
               updateUserDisplay();
               closeModal(document.getElementById('accountModal'));
               showNotification('Success', 'Logged in successfully', 'success');
           } else {
               showNotification('Error', data.message || 'Login failed', 'error');
           }
       })
       .catch(error => {
           console.error('Error logging in:', error);
           showNotification('Error', 'Connection error. Please try again later.', 'error');
       });
   }

   // Signup user
   function signupUser() {
       createAccount(false);
   }

   // Create a new user account
   function createAccount(isOnboarding = false) {
       const prefix = isOnboarding ? 'onboarding' : 'signup';
       
       const firstName = document.getElementById(`${prefix}FirstName`);
       const lastName = document.getElementById(`${prefix}LastName`);
       const email = document.getElementById(`${prefix}Email`);
       const password = document.getElementById(`${prefix}Password`);
       const preferredVehicle = document.getElementById(`${prefix}Vehicle`);
       
       if (!firstName || !lastName || !email || !password || !preferredVehicle) return;
       
       fetch('api/signup.php', {
           method: 'POST',
           headers: { 'Content-Type': 'application/json' },
           body: JSON.stringify({
               firstName: firstName.value,
               lastName: lastName.value,
               email: email.value,
               password: password.value,
               preferredVehicle: preferredVehicle.value
           })
       })
       .then(response => response.json())
       .then(data => {
           if (data.success) {
               app.user.isLoggedIn = true;
               app.user.firstName = firstName.value;
               app.user.lastName = lastName.value;
               app.user.email = email.value;
               app.user.preferredVehicle = preferredVehicle.value;
               
               localStorage.setItem('userId', data.user.id);
               
               if (isOnboarding) {
                   localStorage.setItem('trafficLightOnboardingCompleted', 'true');
                   closeModal(document.getElementById('onboardingModal'));
               } else {
                   closeModal(document.getElementById('accountModal'));
               }
               
               updateUserDisplay();
               showNotification('Success', 'Account created successfully', 'success');
               
               // Update settings based on preferred vehicle
               app.settings.defaultTransport = preferredVehicle.value === 'car' ? 'driving' : 
                                             preferredVehicle.value === 'bicycle' ? 'cycling' : 'walking';
               saveSettings();
           } else {
               showNotification('Error', data.message || 'Failed to create account', 'error');
           }
       })
       .catch(error => {
           console.error('Error creating account:', error);
           showNotification('Error', 'Connection error. Please try again later.', 'error');
       });
   }

   // Logout user
   function logoutUser() {
       clearUserSession();
       closeModal(document.getElementById('accountModal'));
       showNotification('Logout', 'You have been logged out', 'info');
   }

   // Clear user session data
   function clearUserSession() {
       app.user.isLoggedIn = false;
       app.user.firstName = '';
       app.user.lastName = '';
       app.user.email = '';
       app.user.preferredVehicle = 'car';
       
       localStorage.removeItem('userId');
       sessionStorage.removeItem('userId');
       
       updateUserDisplay();
   }

   // Show notification
   function showNotification(title, message, type) {
       if (!app.settings.notifications && type !== 'error') return;
       
       const banner = document.getElementById('notificationBanner');
       const titleElement = document.getElementById('notificationTitle');
       const messageElement = document.getElementById('notificationMessage');
       const iconElement = document.getElementById('notificationIcon');
       
       if (!banner || !titleElement || !messageElement || !iconElement) return;
       
       titleElement.textContent = title;
       messageElement.textContent = message;
       
       let bgColor, iconClass;
       
       switch (type) {
           case 'success':
               bgColor = 'bg-traffic-green';
               iconClass = 'fas fa-check';
               break;
           case 'error':
               bgColor = 'bg-traffic-red';
               iconClass = 'fas fa-exclamation-circle';
               break;
           default:
               bgColor = 'bg-primary-500';
               iconClass = 'fas fa-info-circle';
       }
       
       iconElement.className = `w-8 h-8 ${bgColor} text-white rounded-full flex items-center justify-center mr-3`;
       iconElement.innerHTML = `<i class="${iconClass}"></i>`;
       
       banner.style.display = 'block';
       setTimeout(() => {
           banner.style.opacity = '1';
       }, 10);
       
       setTimeout(() => {
           banner.style.opacity = '0';
           setTimeout(() => {
               banner.style.display = 'none';
           }, 300);
       }, 4000);
   }
});