document.addEventListener('DOMContentLoaded', () => {
    const app = {
        map: null,
        mapboxKey: '',
        mapboxDirections: null,
        markers: {},
        userMarker: null,
        tempMarker: null,
        selectedLightId: null,
        userWatchId: null,
        refreshTimerId: null,
        rafId: null,
        lastTimestamp: 0,
        currentRoute: null,
        isNavigating: false,
        routeLineLayer: null,
        trafficLayer: null,
        currentSpeedLimit: null,
        speedAlertPlayed: false,
        speedAudio: new Audio('speed-alert.mp3'),
        destinationWeather: null,
        settings: {
            darkMode: false,
            notifications: true,
            autoRefresh: true,
            predictions: true,
            autoMeasure: true,
            defaultZoom: 15,
            mapStyle: 'satellite',
            voiceAlerts: true,
            trafficView: true,
            weatherDisplay: true
        },
        measure: {
            timer: 0,
            startTime: 0,
            rafId: null,
            mode: null,
            lightId: null
        },
        navigation: {
            startPosition: null,
            endPosition: null,
            coordinates: [],
            lightsOnRoute: [],
            travelMode: 'driving',
            distance: null,
            duration: null,
            startTime: null,
            estimatedArrival: null,
            currentStep: 0,
            lastStop: null,
            stopDuration: 0,
            currentSpeed: 0,
            lastPosition: null,
            steps: [],
            speedLimits: [],
            currentSpeedLimit: null,
            nextTurn: null,
            trafficDelay: 0
        },
        state: {
            selectingLocation: false,
            currentSlide: 1,
            routePanelOpen: false,
            navigationActive: false,
            lightsPanelOpen: false,
            accountPanelOpen: false
        },

        init() {
            Promise.all([
                this.fetchMapboxKey(),
                this.registerServiceWorker()
            ])
            .then(() => {
                this.loadSettings();
                this.initMap();
                this.bindEvents();
                this.setupDraggableModals();
                this.checkFirstVisit();
                this.checkLoggedInUser();
                
                setTimeout(() => {
                    const loadingScreen = document.getElementById('loadingScreen');
                    if (loadingScreen) {
                        loadingScreen.style.opacity = '0';
                        setTimeout(() => {
                            loadingScreen.style.display = 'none';
                        }, 500);
                    }
                }, 800);
            })
            .catch(error => {
                console.error('Failed to initialize app:', error);
                this.showNotification('Failed to initialize app. Please refresh.', 'error');
                const loadingScreen = document.getElementById('loadingScreen');
                if (loadingScreen) {
                    loadingScreen.innerHTML = '<div class="text-center"><p class="text-xl text-red-500 mb-2">Failed to load app</p><p>Please refresh the page</p></div>';
                }
            });
        },
        
        fetchMapboxKey() {
            return fetch('get-mapbox-key.php')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        this.mapboxKey = data.key;
                    } else {
                        throw new Error('Failed to get Mapbox API key');
                    }
                });
        },

        async registerServiceWorker() {
            if ('serviceWorker' in navigator) {
                try {
                    await navigator.serviceWorker.register('/service-worker.js');
                    console.log('Service Worker registered');
                } catch (error) {
                    console.error('Service Worker registration failed:', error);
                }
            }
        },

        loadSettings() {
            try {
                const saved = localStorage.getItem('trafficLightSettings');
                if (saved) {
                    const parsed = JSON.parse(saved);
                    this.settings = { ...this.settings, ...parsed };
                } else {
                    this.settings.mapStyle = 'satellite';
                }
                this.applySettings();
            } catch (e) {
                console.error('Error loading settings');
                this.settings.mapStyle = 'satellite';
                this.applySettings();
            }
        },

        applySettings() {
            if (document.getElementById('darkModeToggle')) document.getElementById('darkModeToggle').checked = this.settings.darkMode;
            if (document.getElementById('notificationsToggle')) document.getElementById('notificationsToggle').checked = this.settings.notifications;
            if (document.getElementById('autoRefreshToggle')) document.getElementById('autoRefreshToggle').checked = this.settings.autoRefresh;
            if (document.getElementById('predictionToggle')) document.getElementById('predictionToggle').checked = this.settings.predictions;
            if (document.getElementById('autoMeasureToggle')) document.getElementById('autoMeasureToggle').checked = this.settings.autoMeasure;
            if (document.getElementById('defaultZoom')) document.getElementById('defaultZoom').value = this.settings.defaultZoom;
            if (document.getElementById('voiceAlertsToggle')) document.getElementById('voiceAlertsToggle').checked = this.settings.voiceAlerts;
            if (document.getElementById('trafficViewToggle')) document.getElementById('trafficViewToggle').checked = this.settings.trafficView;
            if (document.getElementById('weatherDisplayToggle')) document.getElementById('weatherDisplayToggle').checked = this.settings.weatherDisplay;
            
            if (this.settings.darkMode) {
                document.documentElement.classList.add('dark');
                document.body.classList.add('dark-mode');
            }
        },

        saveSettings() {
            if (document.getElementById('darkModeToggle')) this.settings.darkMode = document.getElementById('darkModeToggle').checked;
            if (document.getElementById('notificationsToggle')) this.settings.notifications = document.getElementById('notificationsToggle').checked;
            if (document.getElementById('autoRefreshToggle')) this.settings.autoRefresh = document.getElementById('autoRefreshToggle').checked;
            if (document.getElementById('predictionToggle')) this.settings.predictions = document.getElementById('predictionToggle').checked;
            if (document.getElementById('autoMeasureToggle')) this.settings.autoMeasure = document.getElementById('autoMeasureToggle').checked;
            if (document.getElementById('defaultZoom')) this.settings.defaultZoom = document.getElementById('defaultZoom').value;
            if (document.getElementById('voiceAlertsToggle')) this.settings.voiceAlerts = document.getElementById('voiceAlertsToggle').checked;
            if (document.getElementById('trafficViewToggle')) this.settings.trafficView = document.getElementById('trafficViewToggle').checked;
            if (document.getElementById('weatherDisplayToggle')) this.settings.weatherDisplay = document.getElementById('weatherDisplayToggle').checked;
            
            try {
                localStorage.setItem('trafficLightSettings', JSON.stringify(this.settings));
            } catch (e) {
                console.error('Error saving settings');
            }
        },

        checkFirstVisit() {
            if (!localStorage.getItem('trafficLightAppVisited')) {
                const welcomeModal = document.getElementById('welcomeModal');
                if (welcomeModal) {
                    welcomeModal.style.display = 'flex';
                }
                localStorage.setItem('trafficLightAppVisited', 'true');
            } else {
                const welcomeModal = document.getElementById('welcomeModal');
                if (welcomeModal) {
                    welcomeModal.style.display = 'none';
                }
            }
            
            const modals = document.querySelectorAll('.modal');
            modals.forEach(modal => {
                if (modal.id !== 'welcomeModal') modal.style.display = 'none';
            });
        },
        
        checkLoggedInUser() {
            const userData = this.getCookie('user_data');
            
            if (userData) {
                try {
                    const user = JSON.parse(userData);
                    this.updateUIForLoggedInUser(user);
                } catch (e) {
                    console.error('Error parsing user data from cookie');
                }
            }
        },
        
        getCookie(name) {
            const value = `; ${document.cookie}`;
            const parts = value.split(`; ${name}=`);
            if (parts.length === 2) return parts.pop().split(';').shift();
            return null;
        },

        initMap() {
            if (!this.mapboxKey) {
                console.error('MapBox API key not available');
                return;
            }
            
            mapboxgl.accessToken = this.mapboxKey;
            
            this.map = new mapboxgl.Map({
                container: 'map',
                style: this.getMapStyle(),
                center: [0, 0],
                zoom: parseInt(this.settings.defaultZoom),
                pitch: 60,
                bearing: 0
            });
            
            this.map.addControl(new mapboxgl.NavigationControl({
                showCompass: true,
                showZoom: true,
                visualizePitch: true
            }), 'bottom-right');
            
            this.map.on('load', () => {
                if (this.settings.trafficView) {
                    this.addTrafficLayer();
                }
                this.map.on('click', this.handleMapClick.bind(this));
                this.loadTrafficLights();
                this.startGeolocation();
                
                this.mapboxDirections = new MapboxDirections({
                    accessToken: mapboxgl.accessToken,
                    unit: 'metric',
                    profile: 'mapbox/driving',
                    alternatives: false,
                    controls: false,
                    interactive: false
                });
            });
        },

        getMapStyle() {
            switch (this.settings.mapStyle) {
                case 'streets':
                    return 'mapbox://styles/mapbox/streets-v12';
                case 'navigation':
                    return 'mapbox://styles/mapbox/navigation-day-v1';
                case 'dark':
                    return 'mapbox://styles/mapbox/dark-v11';
                case 'satellite':
                default:
                    return 'mapbox://styles/mapbox/satellite-streets-v12';
            }
        },

        addTrafficLayer() {
            this.map.addLayer({
                id: 'traffic',
                type: 'line',
                source: {
                    type: 'vector',
                    url: 'mapbox://mapbox.mapbox-traffic-v1'
                },
                'source-layer': 'traffic',
                paint: {
                    'line-width': 3,
                    'line-color': [
                        'match',
                        ['get', 'congestion'],
                        'low', '#10b981',
                        'moderate', '#f59e0b',
                        'heavy', '#ef4444',
                        'severe', '#991b1b',
                        '#6b7280'
                    ],
                    'line-opacity': 0.7
                }
            });
            this.trafficLayer = 'traffic';
        },

        setupDraggableModals() {
            const draggableHandles = document.querySelectorAll('.draggable-handle');
            draggableHandles.forEach(handle => {
                const modal = handle.closest('.modal-body');
                if (!modal) return;
                
                let startY, currentY, initialY;
                let isDragging = false;
                
                handle.addEventListener('touchstart', e => {
                    startY = e.touches[0].clientY;
                    initialY = 0;
                    isDragging = true;
                    modal.style.transition = 'none';
                }, { passive: true });
                
                handle.addEventListener('touchmove', e => {
                    if (!isDragging) return;
                    currentY = e.touches[0].clientY;
                    const diffY = currentY - startY;
                    if (diffY < 0) return;
                    
                    modal.style.transform = `translateY(${diffY}px)`;
                }, { passive: true });
                
                handle.addEventListener('touchend', e => {
                    modal.style.transition = 'transform 0.3s ease';
                    if (!isDragging) return;
                    
                    const diffY = currentY - startY;
                    if (diffY > 100) {
                        modal.style.transform = 'translateY(100%)';
                        const modalContainer = modal.closest('.modal');
                        setTimeout(() => {
                            if (modalContainer) {
                                modalContainer.style.display = 'none';
                                modal.style.transform = 'translateY(0)';
                            }
                        }, 300);
                    } else {
                        modal.style.transform = 'translateY(0)';
                    }
                    
                    isDragging = false;
                }, { passive: true });
            });
            
            const overlays = document.querySelectorAll('.modal-overlay');
            overlays.forEach(overlay => {
                overlay.addEventListener('click', () => {
                    const modal = overlay.closest('.modal');
                    if (modal) this.closeModal(modal);
                });
            });
        },

        bindEvents() {
            const dom = {
                menuToggle: document.getElementById('menuToggle'),
                headerLocate: document.getElementById('headerLocateBtn'),
                headerAdd: document.getElementById('headerAddBtn'),
                route: document.getElementById('routeBtn'),
                refresh: document.getElementById('refreshBtn'),
                settings: document.getElementById('settingsBtn'),
                nearby: document.getElementById('nearbyBtn'),
                routes: document.getElementById('routesBtn'),
                toggleTraffic: document.getElementById('toggleTraffic'),
                closePopup: document.getElementById('closePopup'),
                popupNavigate: document.getElementById('popupNavigate'),
                popupMeasure: document.getElementById('popupMeasure'),
                closeRoutePanel: document.getElementById('closeRoutePanel'),
                calculateRoute: document.getElementById('calculateRoute'),
                startNavigation: document.getElementById('startNavigation'),
                routeTypeCar: document.getElementById('routeTypeCar'),
                routeTypeBike: document.getElementById('routeTypeBike'),
                routeTypeWalk: document.getElementById('routeTypeWalk'),
                endNavigation: document.getElementById('endNavigation'),
                resetApp: document.getElementById('resetAppBtn'),
                measureRed: document.getElementById('measureRedBtn'),
                measureGreen: document.getElementById('measureGreenBtn'),
                startTimer: document.getElementById('startTimer'),
                stopTimer: document.getElementById('stopTimer'),
                saveTimer: document.getElementById('saveTimer'),
                skipWelcome: document.getElementById('skipWelcome'),
                nextSlide: document.getElementById('nextSlide'),
                searchInput: document.getElementById('searchInput'),
                startLocation: document.getElementById('startLocation'),
                endLocation: document.getElementById('endLocation'),
                loginForm: document.getElementById('loginForm'),
                registerForm: document.getElementById('registerForm'),
                showLogin: document.getElementById('showLoginForm'),
                showRegister: document.getElementById('showRegisterForm'),
                authTitle: document.getElementById('authTitle'),
                login: document.getElementById('loginBtn'),
                closeButtons: document.querySelectorAll('.close-modal'),
                darkModeToggle: document.getElementById('darkModeToggle'),
                autoRefreshToggle: document.getElementById('autoRefreshToggle'),
                notificationsToggle: document.getElementById('notificationsToggle'),
                predictionToggle: document.getElementById('predictionToggle'),
                autoMeasureToggle: document.getElementById('autoMeasureToggle'),
                defaultZoom: document.getElementById('defaultZoom'),
                voiceAlertsToggle: document.getElementById('voiceAlertsToggle'),
                trafficViewToggle: document.getElementById('trafficViewToggle'),
                weatherDisplayToggle: document.getElementById('weatherDisplayToggle')
            };
            
            if (dom.menuToggle) dom.menuToggle.addEventListener('click', this.toggleSidebar.bind(this));
            if (dom.headerLocate) dom.headerLocate.addEventListener('click', this.locateUser.bind(this));
            if (dom.headerAdd) dom.headerAdd.addEventListener('click', () => this.openModal('addModal'));
            if (dom.settings) dom.settings.addEventListener('click', () => this.openModal('settingsModal'));
            if (dom.refresh) dom.refresh.addEventListener('click', this.refreshLights.bind(this));
            if (dom.nearby) dom.nearby.addEventListener('click', this.findNearbyLights.bind(this));
            if (dom.route) dom.route.addEventListener('click', this.toggleRoutePanel.bind(this));
            if (dom.routes) dom.routes.addEventListener('click', this.showSavedRoutes.bind(this));
            if (dom.toggleTraffic) dom.toggleTraffic.addEventListener('click', this.toggleTrafficLayer.bind(this));
            
            if (dom.searchInput) {
                dom.searchInput.addEventListener('input', this.handleSearch.bind(this));
            }
            
            if (dom.closePopup) dom.closePopup.addEventListener('click', () => {
                const lightPopup = document.getElementById('lightPopup');
                if (lightPopup) lightPopup.classList.add('hidden');
            });
            
            if (dom.popupNavigate) dom.popupNavigate.addEventListener('click', this.navigateToSelectedLight.bind(this));
            if (dom.popupMeasure) dom.popupMeasure.addEventListener('click', () => {
                const lightPopup = document.getElementById('lightPopup');
                if (lightPopup) lightPopup.classList.add('hidden');
                this.openMeasureModal(this.selectedLightId);
            });
            
            if (dom.closeButtons) {
                dom.closeButtons.forEach(btn => {
                    btn.addEventListener('click', () => {
                        const modal = btn.closest('.modal');
                        if (modal) this.closeModal(modal);
                    });
                });
            }
            
            if (dom.closeRoutePanel) {
                dom.closeRoutePanel.addEventListener('click', () => {
                    const routePlannerPanel = document.getElementById('routePlannerPanel');
                    if (routePlannerPanel) {
                        routePlannerPanel.classList.add('hidden');
                        this.state.routePanelOpen = false;
                    }
                });
            }
            
            if (dom.darkModeToggle) {
                dom.darkModeToggle.addEventListener('change', this.toggleDarkMode.bind(this));
            }
            if (dom.autoRefreshToggle) {
                dom.autoRefreshToggle.addEventListener('change', this.toggleAutoRefresh.bind(this));
            }
            if (dom.notificationsToggle) {
                dom.notificationsToggle.addEventListener('change', this.saveSettings.bind(this));
            }
            if (dom.predictionToggle) {
                dom.predictionToggle.addEventListener('change', () => {
                    this.saveSettings();
                    this.updateAllLightStatus();
                });
            }
            if (dom.autoMeasureToggle) {
                dom.autoMeasureToggle.addEventListener('change', this.saveSettings.bind(this));
            }
            if (dom.defaultZoom) {
                dom.defaultZoom.addEventListener('change', this.saveSettings.bind(this));
            }
            if (dom.voiceAlertsToggle) {
                dom.voiceAlertsToggle.addEventListener('change', this.saveSettings.bind(this));
            }
            if (dom.trafficViewToggle) {
                dom.trafficViewToggle.addEventListener('change', this.toggleTrafficLayer.bind(this));
            }
            if (dom.weatherDisplayToggle) {
                dom.weatherDisplayToggle.addEventListener('change', this.saveSettings.bind(this));
            }
            
            if (dom.routeTypeCar) dom.routeTypeCar.addEventListener('click', () => this.setTravelMode('driving'));
            if (dom.routeTypeBike) dom.routeTypeBike.addEventListener('click', () => this.setTravelMode('cycling'));
            if (dom.routeTypeWalk) dom.routeTypeWalk.addEventListener('click', () => this.setTravelMode('walking'));
            if (dom.calculateRoute) dom.calculateRoute.addEventListener('click', this.calculateRoute.bind(this));
            if (dom.startNavigation) dom.startNavigation.addEventListener('click', this.startRouteNavigation.bind(this));
            if (dom.endNavigation) dom.endNavigation.addEventListener('click', this.endNavigation.bind(this));
            
            if (dom.resetApp) dom.resetApp.addEventListener('click', this.resetAppData.bind(this));
            if (dom.measureRed) dom.measureRed.addEventListener('click', () => this.startMeasure('red'));
            if (dom.measureGreen) dom.measureGreen.addEventListener('click', () => this.startMeasure('green'));
            if (dom.startTimer) dom.startTimer.addEventListener('click', this.startMeasureTimer.bind(this));
            if (dom.stopTimer) dom.stopTimer.addEventListener('click', this.stopMeasureTimer.bind(this));
            if (dom.saveTimer) dom.saveTimer.addEventListener('click', this.saveMeasureTimer.bind(this));
            
            if (dom.showLogin) dom.showLogin.addEventListener('click', this.showLoginForm.bind(this));
            if (dom.showRegister) dom.showRegister.addEventListener('click', this.showRegisterForm.bind(this));
            if (dom.login) dom.login.addEventListener('click', () => this.openModal('loginModal'));
            if (dom.loginForm) dom.loginForm.addEventListener('submit', this.handleLogin.bind(this));
            if (dom.registerForm) dom.registerForm.addEventListener('submit', this.handleRegister.bind(this));
            
            if (dom.skipWelcome) dom.skipWelcome.addEventListener('click', () => this.closeModal('welcomeModal'));
            if (dom.nextSlide) dom.nextSlide.addEventListener('click', this.nextOnboardingSlide.bind(this));
            
            const welcomeDots = document.querySelectorAll('.onboarding-dot');
            if (welcomeDots) {
                welcomeDots.forEach(dot => {
                    dot.addEventListener('click', () => {
                        const slide = parseInt(dot.getAttribute('data-dot'));
                        this.goToSlide(slide);
                    });
                });
            }
            
            const addLightForm = document.getElementById('addLightForm');
            if (addLightForm) addLightForm.addEventListener('submit', this.handleAddLight.bind(this));
            
            if (dom.startLocation) this.setupAddressAutocomplete(dom.startLocation);
            if (dom.endLocation) this.setupAddressAutocomplete(dom.endLocation);
            
            window.addEventListener('online', this.handleOnline.bind(this));
            window.addEventListener('offline', this.handleOffline.bind(this));
            document.addEventListener('visibilitychange', this.handleVisibilityChange.bind(this));
        },

        toggleSidebar() {
            const sidebar = document.querySelector('.sidebar');
            if (!sidebar) return;
            
            sidebar.classList.toggle('active');
            
            const menuToggle = document.getElementById('menuToggle');
            if (menuToggle) {
                menuToggle.innerHTML = sidebar.classList.contains('active') 
                    ? '<i class="fas fa-times"></i>' 
                    : '<i class="fas fa-bars"></i>';
            }
        },

        locateUser() {
            if (this.userMarker) {
                this.map.flyTo({
                    center: this.userMarker.getLngLat(),
                    zoom: 16,
                    essential: true
                });
            } else {
                this.startGeolocation();
            }
        },

        refreshLights() {
            const refreshBtn = document.getElementById('refreshBtn');
            if (refreshBtn) {
                refreshBtn.innerHTML = '<i class="fas fa-sync-alt fa-spin"></i>';
            }
            
            this.loadTrafficLights(true);
            
            setTimeout(() => {
                if (refreshBtn) {
                    refreshBtn.innerHTML = '<i class="fas fa-sync-alt"></i>';
                }
                if (this.settings.notifications) {
                    this.showNotification('Traffic lights refreshed', 'success');
                }
            }, 800);
        },

        findNearbyLights() {
            if (this.userMarker) {
                this.map.flyTo({
                    center: this.userMarker.getLngLat(),
                    zoom: 16,
                    essential: true
                });
                this.highlightNearbyLights();
            } else {
                this.startGeolocation();
            }
        },
        
        toggleRoutePanel() {
            const routePlannerPanel = document.getElementById('routePlannerPanel');
            if (!routePlannerPanel) return;
            
            if (this.state.routePanelOpen) {
                routePlannerPanel.classList.add('hidden');
                this.state.routePanelOpen = false;
            } else {
                if (this.userMarker) {
                    const position = this.userMarker.getLngLat();
                    this.reverseGeocode(position.lng, position.lat)
                        .then(placeName => {
                            const startLocation = document.getElementById('startLocation');
                            if (startLocation) {
                                startLocation.value = placeName || `${position.lat.toFixed(6)}, ${position.lng.toFixed(6)}`;
                                this.navigation.startPosition = {
                                    center: [position.lng, position.lat],
                                    place_name: placeName || `${position.lat.toFixed(6)}, ${position.lng.toFixed(6)}`
                                };
                            }
                        });
                }
                
                routePlannerPanel.classList.remove('hidden');
                this.state.routePanelOpen = true;
            }
        },
        
        showSavedRoutes() {
            fetch('db.php?action=getUserRoutes')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        this.showRoutesListInSidebar(data.routes);
                    } else {
                        if (this.settings.notifications) {
                            this.showNotification(data.message, 'error');
                        }
                    }
                })
                .catch(error => {
                    console.error('Error fetching routes:', error);
                    if (this.settings.notifications) {
                        this.showNotification('Error loading saved routes', 'error');
                    }
                });
        },
        
        showRoutesListInSidebar(routes) {
            const lightsList = document.getElementById('lightsList');
            const noLightsMessage = document.getElementById('noLightsMessage');
            
            if (!lightsList || !noLightsMessage) return;
            
            lightsList.innerHTML = '';
            
            if (routes.length === 0) {
                noLightsMessage.querySelector('p').textContent = 'No saved routes';
                noLightsMessage.classList.remove('hidden');
                return;
            }
            
            noLightsMessage.classList.add('hidden');
            
            routes.forEach(route => {
                const card = document.createElement('div');
                card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg rounded-xl';
                
                const startName = route.route_data.start_point.place_name || 'Start';
                const endName = route.route_data.end_point.place_name || 'Destination';
                const distance = route.route_data.distance ? `${(route.route_data.distance / 1000).toFixed(1)} km` : '';
                const duration = route.route_data.duration ? `${Math.round(route.route_data.duration / 60)} min` : '';
                
                card.innerHTML = `
                    <div class="flex justify-between items-center">
                        <h3 class="font-medium">${route.name}</h3>
                        <span class="px-2.5 py-1 text-xs font-medium rounded-full bg-primary-100 text-primary-800">
                            <i class="fas fa-route mr-1"></i>${distance}
                        </span>
                    </div>
                    <div class="flex items-center text-sm text-gray-500 mt-2">
                        <i class="fas fa-map-marker-alt text-primary-500 mr-2"></i>
                        <span class="truncate">${startName} → ${endName}</span>
                    </div>
                    <div class="mt-2 flex justify-between text-xs">
                        <span class="text-gray-500">${duration}</span>
                        <span class="text-gray-500">${this.formatDate(route.created_at)}</span>
                    </div>
                `;
                
                card.addEventListener('click', () => {
                    this.loadSavedRoute(route);
                });
                
                lightsList.appendChild(card);
            });
        },
        
        formatDate(dateString) {
            const date = new Date(dateString);
            return date.toLocaleDateString();
        },
        
        loadSavedRoute(route) {
            const startLocation = document.getElementById('startLocation');
            const endLocation = document.getElementById('endLocation');
            const routePlannerPanel = document.getElementById('routePlannerPanel');
            
            if (!startLocation || !endLocation || !routePlannerPanel) return;
            
            startLocation.value = route.route_data.start_point.place_name || '';
            endLocation.value = route.route_data.end_point.place_name || '';
            
            this.navigation.startPosition = route.route_data.start_point;
            this.navigation.endPosition = route.route_data.end_point;
            this.navigation.travelMode = route.route_data.travel_mode || 'driving';
            
            this.setTravelMode(this.navigation.travelMode, false);
            
            routePlannerPanel.classList.remove('hidden');
            this.state.routePanelOpen = true;
            
            if (window.innerWidth <= 768) {
                const sidebar = document.querySelector('.sidebar');
                if (sidebar) {
                    sidebar.classList.remove('active');
                    const menuToggle = document.getElementById('menuToggle');
                    if (menuToggle) {
                        menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
                    }
                }
            }
            
            this.displayRouteDetails(route.route_data);
            
            this.drawRoute(route.route_data.coordinates);
        },

        openModal(modal) {
            const modalElement = document.getElementById(modal);
            if (!modalElement) return;
            
            if (modal === 'addModal') {
                this.state.selectingLocation = true;
                if (this.userMarker) {
                    const pos = this.userMarker.getLngLat();
                    const latitudeInput = document.getElementById('latitude');
                    const longitudeInput = document.getElementById('longitude');
                    
                    if (latitudeInput && longitudeInput) {
                        latitudeInput.value = pos.lat.toFixed(6);
                        longitudeInput.value = pos.lng.toFixed(6);
                    }
                    
                    if (this.tempMarker) this.tempMarker.remove();
                    this.tempMarker = new mapboxgl.Marker({ draggable: true })
                        .setLngLat([pos.lng, pos.lat])
                        .addTo(this.map);
                    
                    this.tempMarker.on('dragend', this.updateMarkerPosition.bind(this));
                }
            }
            
            modalElement.style.display = 'flex';
            const modalBody = modalElement.querySelector('.modal-body');
            if (modalBody) {
                modalBody.style.transform = 'translateY(0)';
            }
        },

        closeModal(modal) {
            const modalElement = typeof modal === 'string' ? document.getElementById(modal) : modal;
            if (!modalElement) return;
            
            const modalBody = modalElement.querySelector('.modal-body');
            if (modalBody) {
                modalBody.style.transition = 'transform 0.3s ease';
                modalBody.style.transform = 'translateY(100%)';
            }
            
            setTimeout(() => {
                modalElement.style.display = 'none';
                if (modalBody) {
                    modalBody.style.transform = 'translateY(0)';
                }
                
                if (modalElement.id === 'addModal') {
                    this.state.selectingLocation = false;
                    if (this.tempMarker) {
                        this.tempMarker.remove();
                        this.tempMarker = null;
                    }
                }
            }, 300);
        },

        handleMapClick(e) {
            if (this.state.selectingLocation) {
                const latitudeInput = document.getElementById('latitude');
                const longitudeInput = document.getElementById('longitude');
                
                if (latitudeInput && longitudeInput) {
                    latitudeInput.value = e.lngLat.lat.toFixed(6);
                    longitudeInput.value = e.lngLat.lng.toFixed(6);
                }
                
                if (this.tempMarker) this.tempMarker.remove();
                this.tempMarker = new mapboxgl.Marker({ draggable: true })
                    .setLngLat([e.lngLat.lng, e.lngLat.lat])
                    .addTo(this.map);
                
                this.tempMarker.on('dragend', this.updateMarkerPosition.bind(this));
            }
        },

        updateMarkerPosition() {
            if (!this.tempMarker) return;
            
            const pos = this.tempMarker.getLngLat();
            const latitudeInput = document.getElementById('latitude');
            const longitudeInput = document.getElementById('longitude');
            
            if (latitudeInput && longitudeInput) {
                latitudeInput.value = pos.lat.toFixed(6);
                longitudeInput.value = pos.lng.toFixed(6);
            }
        },

        handleSearch() {
            const searchInput = document.getElementById('searchInput');
            if (!searchInput) return;
            
            const searchTerm = searchInput.value.toLowerCase();
            const lights = document.querySelectorAll('.light-card');
            
            let hasResults = false;
            
            lights.forEach(light => {
                const name = light.getAttribute('data-name')?.toLowerCase() || '';
                const visible = name.includes(searchTerm);
                light.style.display = visible ? 'block' : 'none';
                if (visible) hasResults = true;
            });
            
            const noLightsMessage = document.getElementById('noLightsMessage');
            if (noLightsMessage) {
                if (lights.length > 0 && !hasResults) {
                    noLightsMessage.classList.remove('hidden');
                    const messageParagraph = noLightsMessage.querySelector('p');
                    if (messageParagraph) {
                        messageParagraph.textContent = 'No results found';
                    }
                } else {
                    noLightsMessage.classList.add('hidden');
                }
            }
        },

        toggleDarkMode() {
            const darkModeToggle = document.getElementById('darkModeToggle');
            if (!darkModeToggle) return;
            
            this.settings.darkMode = darkModeToggle.checked;
            
            if (this.settings.darkMode) {
                document.documentElement.classList.add('dark');
                document.body.classList.add('dark-mode');
            } else {
                document.documentElement.classList.remove('dark');
                document.body.classList.remove('dark-mode');
            }
            
            this.saveSettings();
        },

        toggleAutoRefresh() {
            const autoRefreshToggle = document.getElementById('autoRefreshToggle');
            if (!autoRefreshToggle) return;
            
            this.settings.autoRefresh = autoRefreshToggle.checked;
            
            if (this.settings.autoRefresh) {
                this.startStatusUpdates();
            } else {
                this.stopStatusUpdates();
            }
            
            this.saveSettings();
        },

        resetAppData() {
            if (confirm('Are you sure you want to reset all app data? This will clear all your settings and cache.')) {
                localStorage.clear();
                this.showNotification('App data reset. Refreshing...', 'info');
                setTimeout(() => {
                    window.location.reload();
                }, 1500);
            }
        },

        navigateToSelectedLight() {
            if (!this.selectedLightId || !this.markers[this.selectedLightId]) return;
            
            const light = this.markers[this.selectedLightId].data;
            
            const startLocation = document.getElementById('startLocation');
            const endLocation = document.getElementById('endLocation');
            const lightPopup = document.getElementById('lightPopup');
            const routePlannerPanel = document.getElementById('routePlannerPanel');
            
            if (!startLocation || !endLocation || !lightPopup || !routePlannerPanel) return;
            
            startLocation.value = '';
            endLocation.value = light.name;
            this.navigation.endPosition = {
                center: [parseFloat(light.longitude), parseFloat(light.latitude)],
                place_name: light.name
            };
            
            if (this.userMarker) {
                const userPosition = this.userMarker.getLngLat();
                this.reverseGeocode(userPosition.lng, userPosition.lat)
                    .then(placeName => {
                        startLocation.value = placeName || `${userPosition.lat.toFixed(6)}, ${userPosition.lng.toFixed(6)}`;
                        this.navigation.startPosition = {
                            center: [userPosition.lng, userPosition.lat],
                            place_name: placeName || `${userPosition.lat.toFixed(6)}, ${userPosition.lng.toFixed(6)}`
                        };
                        
                        lightPopup.classList.add('hidden');
                        routePlannerPanel.classList.remove('hidden');
                        this.state.routePanelOpen = true;
                    });
            } else {
                lightPopup.classList.add('hidden');
                routePlannerPanel.classList.remove('hidden');
                this.state.routePanelOpen = true;
                
                this.startGeolocation();
            }
        },

        reverseGeocode(lng, lat) {
            return fetch(`https://api.mapbox.com/geocoding/v5/mapbox.places/${lng},${lat}.json?access_token=${this.mapboxKey}`)
                .then(response => response.json())
                .then(data => {
                    if (data.features && data.features.length > 0) {
                        return data.features[0].place_name;
                    }
                    return null;
                })
                .catch(error => {
                    console.error('Reverse geocoding error:', error);
                    return null;
                });
        },
        
        setTravelMode(mode, calculate = true) {
            this.navigation.travelMode = mode;
            
            const carBtn = document.getElementById('routeTypeCar');
            const bikeBtn = document.getElementById('routeTypeBike');
            const walkBtn = document.getElementById('routeTypeWalk');
            
            if (carBtn) carBtn.classList.remove('active');
            if (bikeBtn) bikeBtn.classList.remove('active');
            if (walkBtn) walkBtn.classList.remove('active');
            
            let activeBtn;
            switch (mode) {
                case 'cycling':
                    activeBtn = bikeBtn;
                    break;
                case 'walking':
                    activeBtn = walkBtn;
                    break;
                default:
                    activeBtn = carBtn;
            }
            
            if (activeBtn) activeBtn.classList.add('active');
            
            if (calculate && this.navigation.startPosition && this.navigation.endPosition) {
                this.calculateRoute();
            }
        },
        
        calculateRoute() {
            const startLocation = document.getElementById('startLocation');
            const endLocation = document.getElementById('endLocation');
            
            if (!startLocation || !endLocation) return;
            
            if (!startLocation.value || !endLocation.value) {
                if (this.settings.notifications) {
                    this.showNotification('Please enter start and end locations', 'error');
                }
                return;
            }
            
            const promises = [];
            
            if (!this.navigation.startPosition || !this.navigation.startPosition.center) {
                promises.push(
                    this.geocodeLocation(startLocation.value)
                        .then(result => {
                            if (result) this.navigation.startPosition = result;
                        })
                );
            }
            
            if (!this.navigation.endPosition || !this.navigation.endPosition.center) {
                promises.push(
                    this.geocodeLocation(endLocation.value)
                        .then(result => {
                            if (result) this.navigation.endPosition = result;
                        })
                );
            }
            
            const calculateRouteBtn = document.getElementById('calculateRoute');
            if (calculateRouteBtn) {
                calculateRouteBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Calculating...';
                calculateRouteBtn.disabled = true;
            }
            
            Promise.all(promises)
                .then(() => {
                    if (!this.navigation.startPosition || !this.navigation.endPosition) {
                        throw new Error('Could not geocode locations');
                    }
                    
                    return fetch(`https://api.mapbox.com/directions/v5/mapbox/${this.navigation.travelMode}/${this.navigation.startPosition.center[0]},${this.navigation.startPosition.center[1]};${this.navigation.endPosition.center[0]},${this.navigation.endPosition.center[1]}?geometries=geojson&overview=full&steps=true&access_token=${this.mapboxKey}`);
                })
                .then(response => response.json())
                .then(data => {
                    if (data.code !== 'Ok' || !data.routes || data.routes.length === 0) {
                        throw new Error('No route found');
                    }
                    
                    const route = data.routes[0];
                    this.navigation.coordinates = route.geometry.coordinates;
                    this.navigation.distance = route.distance;
                    this.navigation.duration = route.duration;
                    this.navigation.steps = route.legs[0].steps;
                    
                    this.drawRoute(this.navigation.coordinates);
                    
                    return Promise.all([
                        fetch('db.php?action=getLightsOnRoute', {
                            method: 'POST',
                            headers: {'Content-Type': 'application/json'},
                            body: JSON.stringify({
                                coordinates: this.navigation.coordinates
                            })
                        }).then(res => res.json()),
                        this.analyzeRouteTraffic(route),
                        this.fetchWeatherForDestination()
                    ]);
                })
                .then(([lightsData, trafficData, weatherData]) => {
                    if (lightsData.success) {
                        this.navigation.lightsOnRoute = lightsData.lights;
                        this.navigation.trafficDelay = trafficData.delay;
                        this.destinationWeather = weatherData;
                        
                        this.displayRouteDetails();
                        
                        const startNavigation = document.getElementById('startNavigation');
                        if (startNavigation) {
                            startNavigation.classList.remove('hidden');
                        }
                        
                        this.fitMapToRoute(this.navigation.coordinates);
                        
                        this.calculateTrafficLightImpact();
                    } else {
                        throw new Error(lightsData.message || 'Error fetching traffic lights on route');
                    }
                })
                .catch(error => {
                    console.error('Route calculation error:', error);
                    if (this.settings.notifications) {
                        this.showNotification(error.message || 'Error calculating route', 'error');
                    }
                })
                .finally(() => {
                    if (calculateRouteBtn) {
                        calculateRouteBtn.innerHTML = '<i class="fas fa-route"></i> Calculate Route';
                        calculateRouteBtn.disabled = false;
                    }
                });
        },
        
        analyzeRouteTraffic(route) {
            return new Promise((resolve) => {
                let delay = 0;
                
                const bounds = this.getBoundingBox(route.geometry.coordinates);
                
                fetch(`https://api.mapbox.com/v4/mapbox.mapbox-traffic-v1/${bounds[0]},${bounds[1]},${bounds[2]},${bounds[3]}.json?access_token=${this.mapboxKey}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.features) {
                            data.features.forEach(feature => {
                                const congestion = feature.properties.congestion;
                                if (congestion === 'moderate') delay += 5;
                                if (congestion === 'heavy') delay += 10;
                                if (congestion === 'severe') delay += 15;
                            });
                        }
                        resolve({ delay });
                    })
                    .catch(error => {
                        console.error('Traffic analysis error:', error);
                        resolve({ delay: 0 });
                    });
            });
        },
        
        getBoundingBox(coordinates) {
            const lngs = coordinates.map(c => c[0]);
            const lats = coordinates.map(c => c[1]);
            return [
                Math.min(...lngs),
                Math.min(...lats),
                Math.max(...lngs),
                Math.max(...lats)
            ];
        },
        
        fetchWeatherForDestination() {
            if (!this.settings.weatherDisplay || !this.navigation.endPosition) {
                return Promise.resolve(null);
            }
            
            const end = this.navigation.endPosition.center;
            
            return fetch(`https://api.openweathermap.org/data/2.5/weather?lat=${end[1]}&lon=${end[0]}&units=metric&appid=YOUR_OPENWEATHER_API_KEY`)
                .then(response => response.json())
                .then(data => {
                    return {
                        temperature: Math.round(data.main.temp),
                        condition: data.weather[0].main,
                        description: data.weather[0].description
                    };
                })
                .catch(error => {
                    console.error('Weather fetch error:', error);
                    return null;
                });
        },

        geocodeLocation(query) {
            return fetch(`https://api.mapbox.com/geocoding/v5/mapbox.places/${encodeURIComponent(query)}.json?access_token=${this.mapboxKey}`)
                .then(response => response.json())
                .then(data => {
                    if (data.features && data.features.length > 0) {
                        return {
                            center: data.features[0].center,
                            place_name: data.features[0].place_name
                        };
                    }
                    return null;
                });
        },
        
        calculateTrafficLightImpact() {
            if (!this.navigation.lightsOnRoute || this.navigation.lightsOnRoute.length === 0) {
                return;
            }
            
            const now = new Date();
            const estimatedTimeWithLights = this.navigation.duration;
            let additionalTime = 0;
            
            this.navigation.lightsOnRoute.forEach(light => {
                const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
                const timeInCycle = Math.floor(now.getTime() / 1000) % totalCycle;
                
                if (timeInCycle < light.red_duration) {
                    const remainingRed = light.red_duration - timeInCycle;
                    additionalTime += remainingRed / 2;
                }
            });
            
            additionalTime += this.navigation.trafficDelay * 60;
            
            this.navigation.estimatedDurationWithLights = estimatedTimeWithLights + additionalTime;
            
            const arrivalTime = new Date(now.getTime() + this.navigation.estimatedDurationWithLights * 1000);
            const formattedArrival = arrivalTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
            
            const arrivalTimeEl = document.getElementById('arrivalTime');
            const routeTimeEl = document.getElementById('routeTime');
            const trafficDelayDisplayEl = document.getElementById('trafficDelayDisplay');
            
            if (arrivalTimeEl) arrivalTimeEl.textContent = formattedArrival;
            if (routeTimeEl) routeTimeEl.textContent = `${Math.round(this.navigation.estimatedDurationWithLights / 60)} min`;
            
            if (trafficDelayDisplayEl) {
                if (this.navigation.trafficDelay > 0) {
                    trafficDelayDisplayEl.textContent = `+${this.navigation.trafficDelay} min delay`;
                    trafficDelayDisplayEl.classList.remove('hidden');
                } else {
                    trafficDelayDisplayEl.classList.add('hidden');
                }
            }
        },
        
        drawRoute(coordinates) {
            if (this.map.getSource('route')) {
                this.map.getSource('route').setData({
                    type: 'Feature',
                    properties: {},
                    geometry: {
                        type: 'LineString',
                        coordinates: coordinates
                    }
                });
            } else {
                this.map.addSource('route', {
                    type: 'geojson',
                    data: {
                        type: 'Feature',
                        properties: {},
                        geometry: {
                            type: 'LineString',
                            coordinates: coordinates
                        }
                    }
                });
                
                this.map.addLayer({
                    id: 'route',
                    type: 'line',
                    source: 'route',
                    layout: {
                        'line-join': 'round',
                        'line-cap': 'round'
                    },
                    paint: {
                        'line-color': '#0ea5e9',
                        'line-width': 8,
                        'line-opacity': 0.8
                    }
                });
            }
            
            if (this.map.getLayer('route-lights')) {
                this.map.removeLayer('route-lights');
                this.map.removeSource('route-lights');
            }
            
            if (this.navigation.lightsOnRoute && this.navigation.lightsOnRoute.length > 0) {
                const lightsGeoJson = {
                    type: 'FeatureCollection',
                    features: this.navigation.lightsOnRoute.map(light => {
                        const status = this.getLightStatus(light);
                        return {
                            type: 'Feature',
                            geometry: {
                                type: 'Point',
                                coordinates: [parseFloat(light.longitude), parseFloat(light.latitude)]
                            },
                            properties: {
                                id: light.id,
                                color: status.color
                            }
                        };
                    })
                };
                
                this.map.addSource('route-lights', {
                    type: 'geojson',
                    data: lightsGeoJson
                });
                
                this.map.addLayer({
                    id: 'route-lights',
                    type: 'circle',
                    source: 'route-lights',
                    paint: {
                        'circle-radius': 8,
                        'circle-color': ['match', ['get', 'color'], 'red', '#ef4444', 'green', '#10b981', '#6b7280'],
                        'circle-opacity': 0.8,
                        'circle-stroke-width': 2,
                        'circle-stroke-color': '#ffffff'
                    }
                });
            }
        },
        
        fitMapToRoute(coordinates) {
            if (!coordinates || coordinates.length === 0) return;
            
            const bounds = coordinates.reduce((bounds, coord) => {
                return bounds.extend(coord);
            }, new mapboxgl.LngLatBounds(coordinates[0], coordinates[0]));
            
            this.map.fitBounds(bounds, {
                padding: 50,
                maxZoom: 15,
                duration: 1000
            });
        },
        
        displayRouteDetails() {
            const routeDetails = document.getElementById('routeDetails');
            const routeDistance = document.getElementById('routeDistance');
            const routeTime = document.getElementById('routeTime');
            const routeLights = document.getElementById('routeLights');
            const arrivalTime = document.getElementById('arrivalTime');
            const weatherInfo = document.getElementById('weatherInfo');
            const weatherTemp = document.getElementById('weatherTemp');
            const weatherDesc = document.getElementById('weatherDesc');
            const weatherIcon = document.getElementById('weatherIcon');
            
            if (!routeDetails) return;
            
            routeDetails.classList.remove('hidden');
            
            const distance = this.navigation.distance ? (this.navigation.distance / 1000).toFixed(1) : 0;
            const duration = this.navigation.duration ? Math.round(this.navigation.duration / 60) : 0;
            
            if (routeDistance) routeDistance.textContent = `${distance} km`;
            if (routeTime) routeTime.textContent = `${duration} min`;
            
            let redLights = 0;
            let greenLights = 0;
            
            if (this.navigation.lightsOnRoute) {
                this.navigation.lightsOnRoute.forEach(light => {
                    const status = this.getLightStatus(light);
                    if (status.isRed) {
                        redLights++;
                    } else {
                        greenLights++;
                    }
                });
            }
            
            if (routeLights) {
                routeLights.innerHTML = `
                    <span class="inline-block px-2 py-1 bg-traffic-red/20 text-traffic-red rounded text-xs font-medium">${redLights} red</span>
                    <span class="inline-block px-2 py-1 bg-traffic-green/20 text-traffic-green rounded text-xs font-medium">${greenLights} green</span>
                `;
            }
            
            const now = new Date();
            const arrivalTimeObj = new Date(now.getTime() + (this.navigation.duration * 1000));
            const formattedArrival = arrivalTimeObj.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
            
            if (arrivalTime) arrivalTime.textContent = formattedArrival;
            
            if (this.destinationWeather && weatherInfo) {
                weatherInfo.classList.remove('hidden');
                if (weatherTemp) weatherTemp.textContent = `${this.destinationWeather.temperature}°C`;
                if (weatherDesc) weatherDesc.textContent = this.destinationWeather.description;
                if (weatherIcon) weatherIcon.innerHTML = this.getWeatherIcon(this.destinationWeather.condition);
            }
        },
        
        getWeatherIcon(condition) {
            const iconMap = {
                'Clear': '<i class="fas fa-sun text-yellow-400"></i>',
                'Cloudy': '<i class="fas fa-cloud text-gray-400"></i>',
                'Rain': '<i class="fas fa-cloud-rain text-blue-400"></i>',
                'Snow': '<i class="fas fa-snowflake text-blue-200"></i>',
                'Thunderstorm': '<i class="fas fa-bolt text-yellow-500"></i>',
                'Fog': '<i class="fas fa-smog text-gray-300"></i>'
            };
            return iconMap[condition] || '<i class="fas fa-cloud text-gray-400"></i>';
        },

        startRouteNavigation() {
            if (!this.navigation.coordinates || this.navigation.coordinates.length === 0) {
                if (this.settings.notifications) {
                    this.showNotification('Please calculate a route first', 'error');
                }
                return;
            }
            
            const routePlannerPanel = document.getElementById('routePlannerPanel');
            const navigationPanel = document.getElementById('navigationPanel');
            
            if (routePlannerPanel) routePlannerPanel.classList.add('hidden');
            if (navigationPanel) navigationPanel.classList.remove('hidden');
            
            this.isNavigating = true;
            this.navigation.startTime = new Date();
            this.navigation.currentStep = 0;
            
            this.updateNavigationInstructions();
            
            if (this.userWatchId) {
                navigator.geolocation.clearWatch(this.userWatchId);
            }
            
            this.userWatchId = navigator.geolocation.watchPosition(
                this.updateNavigationPosition.bind(this),
                this.handleGeolocationError.bind(this),
                { enableHighAccuracy: true, maximumAge: 2000, timeout: 5000 }
            );
            
            if (this.settings.autoMeasure) {
                this.startAutoDetectStops();
            }
            
            if (this.settings.voiceAlerts) {
                this.startSpeedLimitMonitoring();
            }
            
            if (this.settings.notifications) {
                this.showNotification('Navigation started. Follow the blue route.', 'success');
            }
        },
        
        updateNavigationInstructions() {
            if (!this.navigation.steps || this.navigation.steps.length === 0 || 
                this.navigation.currentStep >= this.navigation.steps.length) {
                return;
            }
            
            const step = this.navigation.steps[this.navigation.currentStep];
            
            const nextDirection = document.getElementById('nextDirection');
            const nextStreet = document.getElementById('nextStreet');
            const nextDistance = document.getElementById('nextDistance');
            const etaText = document.getElementById('etaText');
            
            if (nextDirection) nextDirection.textContent = step.maneuver.instruction || 'Continue';
            
            if (nextStreet) {
                if (step.name) {
                    nextStreet.textContent = `on ${step.name}`;
                } else {
                    nextStreet.textContent = '';
                }
            }
            
            const distance = step.distance;
            let formattedDistance;
            
            if (distance < 100) {
                formattedDistance = `${Math.round(distance)} m`;
            } else if (distance < 1000) {
                formattedDistance = `${Math.round(distance / 10) * 10} m`;
            } else {
                formattedDistance = `${(distance / 1000).toFixed(1)} km`;
            }
            
            if (nextDistance) nextDistance.textContent = formattedDistance;
            
            const now = new Date();
            const elapsedSeconds = (now - this.navigation.startTime) / 1000;
            const remainingSeconds = Math.max(0, this.navigation.estimatedDurationWithLights - elapsedSeconds);
            
            if (etaText && remainingSeconds > 0) {
                const arrivalTime = new Date(now.getTime() + (remainingSeconds * 1000));
                const formattedArrival = arrivalTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
                etaText.textContent = `ETA ${formattedArrival}`;
            } else if (etaText) {
                etaText.textContent = 'Arriving';
            }
            
            const totalDistance = this.navigation.distance;
            const remainingDistance = this.navigation.steps.slice(this.navigation.currentStep)
                .reduce((total, step) => total + step.distance, 0);
            
            const coveredDistance = totalDistance - remainingDistance;
            
            const totalDistanceEl = document.getElementById('totalDistance');
            if (totalDistanceEl) {
                totalDistanceEl.textContent = `${(coveredDistance / 1000).toFixed(1)} / ${(totalDistance / 1000).toFixed(1)} km`;
            }
            
            const totalMinutes = Math.round(this.navigation.estimatedDurationWithLights / 60);
            const elapsedMinutes = Math.round(elapsedSeconds / 60);
            
            const totalTimeEl = document.getElementById('totalTime');
            if (totalTimeEl) {
                totalTimeEl.textContent = `${elapsedMinutes} / ${totalMinutes} min`;
            }
        },
        
        updateNavigationPosition(position) {
            const lat = position.coords.latitude;
            const lng = position.coords.longitude;
            const speed = position.coords.speed || 0;
            const currentPosition = [lng, lat];
            
            if (!this.userMarker) {
                const el = document.createElement('div');
                el.className = 'location-dot';
                
                this.userMarker = new mapboxgl.Marker(el)
                    .setLngLat(currentPosition)
                    .addTo(this.map);
            } else {
                this.userMarker.setLngLat(currentPosition);
            }
            
            if (this.isNavigating) {
                this.map.flyTo({
                    center: currentPosition,
                    zoom: 17,
                    bearing: position.coords.heading || 0,
                    pitch: 60,
                    essential: true
                });
                
                const speedKmh = speed ? Math.round(speed * 3.6) : 0;
                
                const currentSpeedEl = document.getElementById('currentSpeed');
                if (currentSpeedEl) {
                    currentSpeedEl.textContent = `${speedKmh} km/h`;
                }
                
                this.navigation.currentSpeed = speedKmh;
                
                const currentSpeedDisplayEl = document.getElementById('currentSpeedDisplay');
                if (speedKmh > 0 && currentSpeedDisplayEl) {
                    currentSpeedDisplayEl.textContent = `${speedKmh} km/h`;
                }
                
                this.checkNavigationProgress(currentPosition);
                this.updateETA(currentPosition);
                
                if (this.settings.voiceAlerts) {
                    this.checkSpeedLimit(speedKmh);
                }
                
                if (this.settings.autoMeasure && speed < 0.5) {
                    this.handleNavigationStop(currentPosition);
                } else if (speed >= 0.5 && this.navigation.lastStop) {
                    this.handleNavigationResume();
                }
                
                this.navigation.lastPosition = currentPosition;
            }
        },
        
        startSpeedLimitMonitoring() {
            this.speedCheckInterval = setInterval(() => {
                if (this.isNavigating && this.navigation.currentStep < this.navigation.steps.length) {
                    const step = this.navigation.steps[this.navigation.currentStep];
                    const maxspeed = step.maxspeed || step.speed_limit;
                    
                    const speedLimitDisplayEl = document.getElementById('speedLimitDisplay');
                    if (maxspeed && speedLimitDisplayEl) {
                        this.currentSpeedLimit = maxspeed;
                        speedLimitDisplayEl.textContent = `${maxspeed} km/h`;
                    }
                }
            }, 1000);
        },
        
        checkSpeedLimit(currentSpeed) {
            if (this.currentSpeedLimit && currentSpeed > this.currentSpeedLimit) {
                if (!this.speedAlertPlayed) {
                    this.speedAudio.play().catch(() => {});
                    this.speedAlertPlayed = true;
                }
            } else {
                this.speedAlertPlayed = false;
            }
        },
        
        updateETA(currentPosition) {
            if (!this.navigation.coordinates || !this.navigation.lightsOnRoute) return;
            
            const now = new Date();
            const elapsedSeconds = (now - this.navigation.startTime) / 1000;
            
            let closestPointIndex = this.findClosestPointOnRoute(currentPosition);
            if (closestPointIndex === -1) return;
            
            let remainingDistance = this.calculateRemainingDistance(closestPointIndex);
            let estimatedRemainingTime = 0;
            
            if (this.navigation.currentSpeed > 5) {
                estimatedRemainingTime = remainingDistance / (this.navigation.currentSpeed * 1000 / 3600);
            } else {
                estimatedRemainingTime = remainingDistance / (30 * 1000 / 3600);
            }
            
            const lightsAhead = this.getTrafficLightsAhead(closestPointIndex);
            let additionalTime = 0;
            
            lightsAhead.forEach(light => {
                const status = this.getLightStatus(light);
                if (status.isRed) {
                    additionalTime += Math.min(status.timeLeft, 30);
                }
            });
            
            additionalTime += this.navigation.trafficDelay * 60;
            
            const totalEstimatedTime = estimatedRemainingTime + additionalTime;
            const arrivalTime = new Date(now.getTime() + (totalEstimatedTime * 1000));
            const formattedArrival = arrivalTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
            
            const etaTextEl = document.getElementById('etaText');
            if (etaTextEl) {
                etaTextEl.textContent = `ETA ${formattedArrival}`;
            }
            
            const totalTimeMinutes = Math.round((elapsedSeconds + totalEstimatedTime) / 60);
            const elapsedMinutes = Math.round(elapsedSeconds / 60);
            
            const totalTimeEl = document.getElementById('totalTime');
            if (totalTimeEl) {
                totalTimeEl.textContent = `${elapsedMinutes} / ${totalTimeMinutes} min`;
            }
        },
        
        findClosestPointOnRoute(position) {
            let closestIndex = -1;
            let minDistance = Infinity;
            
            for (let i = 0; i < this.navigation.coordinates.length; i++) {
                const point = this.navigation.coordinates[i];
                const distance = this.getDistance(position, point);
                
                if (distance < minDistance) {
                    minDistance = distance;
                    closestIndex = i;
                }
            }
            
            return closestIndex;
        },

        calculateRemainingDistance(fromIndex) {
            let distance = 0;
            
            for (let i = fromIndex; i < this.navigation.coordinates.length - 1; i++) {
                const point1 = this.navigation.coordinates[i];
                const point2 = this.navigation.coordinates[i + 1];
                distance += this.getDistance(point1, point2);
            }
            
            return distance;
        },
        
        getTrafficLightsAhead(fromIndex) {
            if (!this.navigation.lightsOnRoute || this.navigation.lightsOnRoute.length === 0) {
                return [];
            }
            
            const lightsAhead = [];
            const currentPosition = this.navigation.coordinates[fromIndex];
            
            this.navigation.lightsOnRoute.forEach(light => {
                const lightPosition = [parseFloat(light.longitude), parseFloat(light.latitude)];
                
                let isAhead = false;
                let minDistanceToRoute = Infinity;
                
                for (let i = fromIndex; i < this.navigation.coordinates.length; i++) {
                    const routePoint = this.navigation.coordinates[i];
                    const distance = this.getDistance(lightPosition, routePoint);
                    
                    if (distance < minDistanceToRoute) {
                        minDistanceToRoute = distance;
                    }
                }
                
                if (minDistanceToRoute < 50) {
                    const distanceFromCurrent = this.getDistance(currentPosition, lightPosition);
                    
                    if (distanceFromCurrent > 20) {
                        lightsAhead.push({
                            ...light,
                            distanceFromCurrent
                        });
                    }
                }
            });
            
            return lightsAhead.sort((a, b) => a.distanceFromCurrent - b.distanceFromCurrent);
        },
        
        checkNavigationProgress(userPosition) {
            if (!this.navigation.steps || this.navigation.currentStep >= this.navigation.steps.length) {
                return;
            }
            
            const currentStep = this.navigation.steps[this.navigation.currentStep];
            const nextStep = this.navigation.steps[this.navigation.currentStep + 1];
            
            if (nextStep) {
                const maneuverPoint = nextStep.maneuver.location;
                const distanceToNextStep = this.getDistance(userPosition, maneuverPoint);
                
                if (distanceToNextStep < 30) {
                    this.navigation.currentStep++;
                    this.updateNavigationInstructions();
                    
                    if (this.settings.notifications) {
                        this.showNotification(nextStep.maneuver.instruction, 'info');
                    }
                }
            }
            
            if (this.navigation.currentStep === this.navigation.steps.length - 1) {
                const finalPoint = this.navigation.coordinates[this.navigation.coordinates.length - 1];
                const distanceToEnd = this.getDistance(userPosition, finalPoint);
                
                if (distanceToEnd < 50) {
                    this.handleNavigationComplete();
                }
            }
        },
        
        getDistance(point1, point2) {
            const lat1 = point1[1] * Math.PI / 180;
            const lon1 = point1[0] * Math.PI / 180;
            const lat2 = point2[1] * Math.PI / 180;
            const lon2 = point2[0] * Math.PI / 180;
            
            const dlon = lon2 - lon1;
            const dlat = lat2 - lat1;
            const a = Math.sin(dlat/2)**2 + Math.cos(lat1) * Math.cos(lat2) * Math.sin(dlon/2)**2;
            const c = 2 * Math.asin(Math.sqrt(a));
            const r = 6371000;
            
            return c * r;
        },
        
        handleNavigationComplete() {
            if (this.settings.notifications) {
                this.showNotification('You have reached your destination!', 'success');
            }
            
            this.endNavigation();
            
            // Check if user is logged in
            if (document.cookie.includes('user_data')) {
                const routeName = `${this.navigation.startPosition.place_name.split(',')[0]} to ${this.navigation.endPosition.place_name.split(',')[0]}`;
                
                fetch('db.php?action=saveRoute', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        name: routeName,
                        start_point: this.navigation.startPosition,
                        end_point: this.navigation.endPosition,
                        coordinates: this.navigation.coordinates,
                        distance: this.navigation.distance,
                        duration: this.navigation.duration,
                        travel_mode: this.navigation.travelMode
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success && this.settings.notifications) {
                        this.showNotification('Route saved to your account', 'success');
                    }
                })
                .catch(error => console.error('Error saving route:', error));
            }
        },
        
        startAutoDetectStops() {
            console.log('Auto-detect stops enabled');
        },
        
        handleNavigationStop(position) {
            const now = new Date();
            
            if (!this.navigation.lastStop) {
                this.navigation.lastStop = {
                    position: position,
                    time: now,
                    duration: 0
                };
            } else {
                this.navigation.stopDuration = (now - this.navigation.lastStop.time) / 1000;
                
                if (this.navigation.stopDuration > 2 && this.settings.autoMeasure) {
                    this.checkForTrafficLight(position);
                }
            }
        },
        
        handleNavigationResume() {
            if (this.navigation.lastStop && this.navigation.stopDuration > 2) {
                this.navigation.lastStop = null;
                this.navigation.stopDuration = 0;
            }
        },
        
        checkForTrafficLight(position) {
            const routeLength = this.navigation.coordinates.length;
            
            if (routeLength <= 2) return;
            
            const startPoint = this.navigation.coordinates[0];
            const endPoint = this.navigation.coordinates[routeLength - 1];
            
            const distanceFromStart = this.getDistance(position, startPoint);
            const distanceFromEnd = this.getDistance(position, endPoint);
            
            if (distanceFromStart < 1000 || distanceFromEnd < 1000) {
                return;
            }
            
            let isNearIntersection = false;
            let nearestIntersectionIndex = -1;
            let minDistance = Infinity;
            
            for (let i = 1; i < routeLength - 1; i++) {
                const prevPoint = this.navigation.coordinates[i - 1];
                const currPoint = this.navigation.coordinates[i];
                const nextPoint = this.navigation.coordinates[i + 1];
                
                const v1 = [currPoint[0] - prevPoint[0], currPoint[1] - prevPoint[1]];
                const v2 = [nextPoint[0] - currPoint[0], nextPoint[1] - currPoint[1]];
                
                const len1 = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1]);
                const len2 = Math.sqrt(v2[0]*v2[0] + v2[1]*v2[1]);
                
                if (len1 > 0 && len2 > 0) {
                    const nv1 = [v1[0]/len1, v1[1]/len1];
                    const nv2 = [v2[0]/len2, v2[1]/len2];
                    
                    const dotProduct = nv1[0]*nv2[0] + nv1[1]*nv2[1];
                    const angle = Math.acos(Math.min(Math.max(dotProduct, -1), 1)) * (180/Math.PI);
                    
                    if (angle > 30) {
                        const dist = this.getDistance(position, currPoint);
                        if (dist < 50 && dist < minDistance) {
                            minDistance = dist;
                            nearestIntersectionIndex = i;
                            isNearIntersection = true;
                        }
                    }
                }
            }
            
            let isNearExistingLight = false;
            let lightDistance = Infinity;

            for (const id in this.markers) {
                const light = this.markers[id].data;
                const lightPos = [parseFloat(light.longitude), parseFloat(light.latitude)];
                const dist = this.getDistance(position, lightPos);
                
                if (dist < 30 && dist < lightDistance) {
                    lightDistance = dist;
                    isNearExistingLight = true;
                }
            }
            
            if ((isNearIntersection || isNearExistingLight) && this.navigation.stopDuration > 5) {
                this.autoDetectTrafficLight(position, this.navigation.stopDuration);
            }
        },
        
        autoDetectTrafficLight(position, duration) {
            let direction = 'unknown';
            
            if (this.navigation.lastPosition && position) {
                const dx = position[0] - this.navigation.lastPosition[0];
                const dy = position[1] - this.navigation.lastPosition[1];
                
                if (Math.abs(dx) > Math.abs(dy)) {
                    direction = dx > 0 ? 'east' : 'west';
                } else {
                    direction = dy > 0 ? 'north' : 'south';
                }
            }
            
            fetch('db.php?action=autoDetectLight', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    latitude: position[1],
                    longitude: position[0],
                    duration: duration,
                    duration_type: 'red',
                    direction: direction,
                    speed: this.navigation.currentSpeed || 0,
                    timestamp: new Date().toISOString()
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && this.settings.notifications) {
                    this.showNotification(data.message, 'success');
                    this.loadTrafficLights(true);
                }
            })
            .catch(error => console.error('Auto-detect light error:', error));
        },
        
        endNavigation() {
            this.isNavigating = false;
            
            const navigationPanel = document.getElementById('navigationPanel');
            if (navigationPanel) navigationPanel.classList.add('hidden');
            
            this.navigation.startTime = null;
            this.navigation.currentStep = 0;
            this.navigation.lastStop = null;
            this.navigation.stopDuration = 0;
            
            if (this.userMarker) {
                this.map.flyTo({
                    center: this.userMarker.getLngLat(),
                    zoom: parseInt(this.settings.defaultZoom),
                    pitch: 0,
                    bearing: 0,
                    essential: true
                });
            }
            
            if (this.map.getLayer('route')) {
                this.map.removeLayer('route');
                this.map.removeSource('route');
            }
            
            if (this.map.getLayer('route-lights')) {
                this.map.removeLayer('route-lights');
                this.map.removeSource('route-lights');
            }
            
            if (this.userWatchId) {
                navigator.geolocation.clearWatch(this.userWatchId);
            }
            
            if (this.speedCheckInterval) {
                clearInterval(this.speedCheckInterval);
            }
            
            this.startGeolocation();
        },

        handleAddLight(e) {
            e.preventDefault();
            
            const submitBtn = e.target.querySelector('[type="submit"]');
            const originalText = submitBtn.textContent;
            submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Adding...';
            submitBtn.disabled = true;
            
            const formData = new FormData(e.target);
            const lightData = {
                name: formData.get('name'),
                latitude: formData.get('latitude'),
                longitude: formData.get('longitude'),
                direction: formData.get('direction'),
                red_duration: formData.get('red_duration'),
                green_duration: formData.get('green_duration')
            };
            
            fetch('db.php?action=addTrafficLight', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify(lightData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    this.addLightToMap(data.light);
                    this.addLightToSidebar(data.light);
                    
                    if (this.settings.notifications) {
                        this.showNotification(data.message || 'Traffic light added successfully and pending review by Imators team.', 'success');
                    }
                    
                    this.closeModal('addModal');
                    e.target.reset();
                    
                    if (this.tempMarker) {
                        this.tempMarker.remove();
                        this.tempMarker = null;
                    }
                    
                    this.state.selectingLocation = false;
                    
                    const noLightsMessage = document.getElementById('noLightsMessage');
                    if (noLightsMessage) noLightsMessage.classList.add('hidden');
                    
                    this.map.flyTo({
                        center: [parseFloat(data.light.longitude), parseFloat(data.light.latitude)],
                        zoom: 16,
                        essential: true
                    });
                } else {
                    if (this.settings.notifications) {
                        this.showNotification('Error: ' + data.message, 'error');
                    }
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            })
            .catch(error => {
                console.error('Error:', error);
                if (this.settings.notifications) {
                    this.showNotification('Connection error', 'error');
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            });
        },
        
        showLoginForm() {
            const loginForm = document.getElementById('loginForm');
            const registerForm = document.getElementById('registerForm');
            const showLogin = document.getElementById('showLoginForm');
            const showRegister = document.getElementById('showRegisterForm');
            const authTitle = document.getElementById('authTitle');
            
            if (!loginForm || !registerForm || !showLogin || !showRegister || !authTitle) return;
            
            loginForm.classList.remove('hidden');
            registerForm.classList.add('hidden');
            
            showLogin.classList.add('border-primary-500', 'text-primary-500');
            showLogin.classList.remove('border-transparent', 'text-gray-500');
            
            showRegister.classList.add('border-transparent', 'text-gray-500');
            showRegister.classList.remove('border-primary-500', 'text-primary-500');
            
            authTitle.textContent = 'Login';
        },
        
        showRegisterForm() {
            const loginForm = document.getElementById('loginForm');
            const registerForm = document.getElementById('registerForm');
            const showLogin = document.getElementById('showLoginForm');
            const showRegister = document.getElementById('showRegisterForm');
            const authTitle = document.getElementById('authTitle');
            
            if (!loginForm || !registerForm || !showLogin || !showRegister || !authTitle) return;
            
            loginForm.classList.add('hidden');
            registerForm.classList.remove('hidden');
            
            showRegister.classList.add('border-primary-500', 'text-primary-500');
            showRegister.classList.remove('border-transparent', 'text-gray-500');
            
            showLogin.classList.add('border-transparent', 'text-gray-500');
            showLogin.classList.remove('border-primary-500', 'text-primary-500');
            
            authTitle.textContent = 'Register';
        },

        handleLogin(e) {
            e.preventDefault();
            
            const form = e.target;
            const formData = new FormData(form);
            const submitBtn = form.querySelector('[type="submit"]');
            const originalText = submitBtn.textContent;
            
            submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Logging in...';
            submitBtn.disabled = true;
            
            fetch('db.php?action=login', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    email: formData.get('email'),
                    password: formData.get('password'),
                    remember: formData.get('remember') === 'on'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (formData.get('remember') === 'on') {
                        document.cookie = `user_data=${JSON.stringify(data.user)}; max-age=31536000; path=/`;
                    }
                    
                    if (this.settings.notifications) {
                        this.showNotification('Login successful', 'success');
                    }
                    
                    this.updateUIForLoggedInUser(data.user);
                    this.closeModal('loginModal');
                } else {
                    if (this.settings.notifications) {
                        this.showNotification('Error: ' + data.message, 'error');
                    }
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            })
            .catch(error => {
                console.error('Login error:', error);
                if (this.settings.notifications) {
                    this.showNotification('Connection error', 'error');
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            });
        },
        
        handleRegister(e) {
            e.preventDefault();
            
            const form = e.target;
            const formData = new FormData(form);
            const submitBtn = form.querySelector('[type="submit"]');
            const originalText = submitBtn.textContent;
            
            if (formData.get('password') !== formData.get('password_confirm')) {
                if (this.settings.notifications) {
                    this.showNotification('Passwords do not match', 'error');
                }
                return;
            }
            
            submitBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div> Registering...';
            submitBtn.disabled = true;
            
            fetch('db.php?action=register', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    username: formData.get('username'),
                    email: formData.get('email'),
                    password: formData.get('password'),
                    remember: true
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.cookie = `user_data=${JSON.stringify(data.user)}; max-age=31536000; path=/`;
                    
                    if (this.settings.notifications) {
                        this.showNotification('Registration successful', 'success');
                    }
                    
                    this.updateUIForLoggedInUser(data.user);
                    this.closeModal('loginModal');
                } else {
                    if (this.settings.notifications) {
                        this.showNotification('Error: ' + data.message, 'error');
                    }
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            })
            .catch(error => {
                console.error('Registration error:', error);
                if (this.settings.notifications) {
                    this.showNotification('Connection error', 'error');
                }
                
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            });
        },

        updateUIForLoggedInUser(user) {
            const loginSection = document.querySelector('.sidebar > .p-4.border-t');
            if (loginSection) {
                loginSection.innerHTML = `
                    <div class="flex items-center mb-3">
                        <div class="w-10 h-10 rounded-full bg-primary-500 text-white flex items-center justify-center mr-3">
                            <i class="fas fa-user"></i>
                        </div>
                        <div>
                            <p class="font-medium text-sm">${user.username}</p>
                            <p class="text-xs text-gray-500">Logged in</p>
                        </div>
                        <button id="logoutBtn" class="ml-auto text-gray-500 hover:text-gray-700">
                            <i class="fas fa-sign-out-alt"></i>
                        </button>
                    </div>
                    <button id="settingsBtn" class="w-full bg-gray-100 hover:bg-gray-200 text-gray-800 font-medium px-4 py-2.5 rounded-xl transition-colors flex items-center justify-center gap-2">
                        <i class="fas fa-cog"></i> Settings
                    </button>
                `;
                
                document.getElementById('logoutBtn')?.addEventListener('click', this.handleLogout.bind(this));
                document.getElementById('settingsBtn')?.addEventListener('click', () => this.openModal('settingsModal'));
            }
        },
        
        handleLogout() {
            fetch('db.php?action=logout')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        document.cookie = 'user_data=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;';
                        
                        if (this.settings.notifications) {
                            this.showNotification('Logout successful', 'success');
                        }
                        
                        window.location.reload();
                    } else {
                        if (this.settings.notifications) {
                            this.showNotification('Error: ' + data.message, 'error');
                        }
                    }
                })
                .catch(error => {
                    console.error('Logout error:', error);
                    if (this.settings.notifications) {
                        this.showNotification('Connection error', 'error');
                    }
                });
        },

        nextOnboardingSlide() {
            const welcomeSlides = document.querySelectorAll('.onboarding-slide');
            if (this.state.currentSlide < welcomeSlides.length) {
                this.goToSlide(this.state.currentSlide + 1);
            } else {
                this.closeModal('welcomeModal');
            }
        },

        goToSlide(slideNumber) {
            this.state.currentSlide = slideNumber;
            
            const welcomeSlides = document.querySelectorAll('.onboarding-slide');
            const welcomeDots = document.querySelectorAll('.onboarding-dot');
            const nextSlideBtn = document.getElementById('nextSlide');
            
            welcomeSlides.forEach(slide => {
                slide.classList.add('hidden');
            });
            
            welcomeDots.forEach(dot => {
                dot.classList.remove('active');
            });
            
            const currentSlide = document.querySelector(`.onboarding-slide[data-slide="${slideNumber}"]`);
            const currentDot = document.querySelector(`.onboarding-dot[data-dot="${slideNumber}"]`);
            
            if (currentSlide) currentSlide.classList.remove('hidden');
            if (currentDot) currentDot.classList.add('active');
            
            if (nextSlideBtn) {
                nextSlideBtn.textContent = slideNumber === welcomeSlides.length ? 'Get Started' : 'Next';
            }
        },

        startMeasure(mode) {
            this.measure.mode = mode;
            
            const timerContainer = document.getElementById('timerContainer');
            const timerInstructions = document.getElementById('timerInstructions');
            const startTimerBtn = document.getElementById('startTimer');
            const stopTimerBtn = document.getElementById('stopTimer');
            
            if (timerContainer) timerContainer.classList.remove('hidden');
            
            if (timerInstructions) {
                if (mode === 'red') {
                    timerInstructions.textContent = 'Press "Start" when the light turns red, then "Stop" when it turns green.';
                } else {
                    timerInstructions.textContent = 'Press "Start" when the light turns green, then "Stop" when it turns red.';
                }
            }
            
            if (startTimerBtn) {
                if (mode === 'red') {
                    startTimerBtn.className = 'bg-traffic-red text-white py-2.5 rounded-xl hover:bg-red-600 transition-colors shadow-sm';
                } else {
                    startTimerBtn.className = 'bg-traffic-green text-white py-2.5 rounded-xl hover:bg-green-600 transition-colors shadow-sm';
                }
            }
            
            if (stopTimerBtn) {
                if (mode === 'red') {
                    stopTimerBtn.className = 'bg-traffic-green text-white py-2.5 rounded-xl hover:bg-green-600 transition-colors opacity-50 shadow-sm';
                } else {
                    stopTimerBtn.className = 'bg-traffic-red text-white py-2.5 rounded-xl hover:bg-red-600 transition-colors opacity-50 shadow-sm';
                }
            }
            
            this.resetMeasureTimer();
        },

        startMeasureTimer() {
            this.measure.startTime = performance.now();
            this.measure.timer = 0;
            
            if (this.measure.rafId) cancelAnimationFrame(this.measure.rafId);
            
            const updateTimer = (timestamp) => {
                const elapsed = Math.floor((timestamp - this.measure.startTime) / 1000);
                if (elapsed !== this.measure.timer) {
                    this.measure.timer = elapsed;
                    const minutes = Math.floor(elapsed / 60);
                    const seconds = elapsed % 60;
                    
                    const timerDisplay = document.getElementById('timerDisplay');
                    if (timerDisplay) {
                        timerDisplay.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                    }
                }
                this.measure.rafId = requestAnimationFrame(updateTimer);
            };
            
            this.measure.rafId = requestAnimationFrame(updateTimer);
            
            const timerDisplay = document.getElementById('timerDisplay');
            const measureResult = document.getElementById('measureResult');
            const startTimerBtn = document.getElementById('startTimer');
            const stopTimerBtn = document.getElementById('stopTimer');
            const saveTimerBtn = document.getElementById('saveTimer');
            
            if (timerDisplay) timerDisplay.textContent = '00:00';
            if (measureResult) measureResult.textContent = 'Measuring...';
            
            if (startTimerBtn) {
                startTimerBtn.disabled = true;
                startTimerBtn.classList.add('opacity-50');
            }
            
            if (stopTimerBtn) {
                stopTimerBtn.disabled = false;
                stopTimerBtn.classList.remove('opacity-50');
            }
            
            if (saveTimerBtn) {
                saveTimerBtn.disabled = true;
                saveTimerBtn.classList.add('opacity-50');
            }
        },

        stopMeasureTimer() {
            if (this.measure.rafId) {
                cancelAnimationFrame(this.measure.rafId);
                this.measure.rafId = null;
                
                const measureResult = document.getElementById('measureResult');
                const stopTimerBtn = document.getElementById('stopTimer');
                const saveTimerBtn = document.getElementById('saveTimer');
                
                if (measureResult) {
                    measureResult.textContent = `Measured duration: ${this.measure.timer} seconds. Click Save to confirm.`;
                }
                
                if (stopTimerBtn) {
                    stopTimerBtn.disabled = true;
                    stopTimerBtn.classList.add('opacity-50');
                }
                
                if (saveTimerBtn) {
                    saveTimerBtn.disabled = false;
                    saveTimerBtn.classList.remove('opacity-50');
                }
            }
        },

        resetMeasureTimer() {
            if (this.measure.rafId) {
                cancelAnimationFrame(this.measure.rafId);
                this.measure.rafId = null;
            }
            
            this.measure.timer = 0;
            this.measure.startTime = 0;
            
            const timerDisplay = document.getElementById('timerDisplay');
            const measureResult = document.getElementById('measureResult');
            const startTimerBtn = document.getElementById('startTimer');
            const stopTimerBtn = document.getElementById('stopTimer');
            const saveTimerBtn = document.getElementById('saveTimer');
            
            if (timerDisplay) timerDisplay.textContent = '00:00';
            if (measureResult) measureResult.textContent = '';
            
            if (startTimerBtn) {
                startTimerBtn.disabled = false;
                startTimerBtn.classList.remove('opacity-50');
            }
            
            if (stopTimerBtn) {
                stopTimerBtn.disabled = true;
                stopTimerBtn.classList.add('opacity-50');
            }
            
            if (saveTimerBtn) {
                saveTimerBtn.disabled = true;
                saveTimerBtn.classList.add('opacity-50');
            }
        },

        saveMeasureTimer() {
            const measureTitle = document.getElementById('measureTitle');
            const measureResult = document.getElementById('measureResult');
            const saveTimerBtn = document.getElementById('saveTimer');
            const timerContainer = document.getElementById('timerContainer');
            
            if (!measureTitle || this.measure.timer <= 0 || !this.measure.mode) return;
            
            const lightId = measureTitle.getAttribute('data-id');
            if (!lightId) return;
            
            if (measureResult) measureResult.textContent = 'Saving...';
            
            fetch('db.php?action=updateTiming', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    id: lightId,
                    duration_type: this.measure.mode,
                    duration: this.measure.timer
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (measureResult) {
                        measureResult.innerHTML = '<i class="fas fa-check text-green-500 mr-1"></i> Timing updated!';
                    }
                    
                    if (this.markers[lightId]) {
                        this.markers[lightId].data = data.light;
                        this.updateLightStatus(data.light);
                    }
                    
                    if (saveTimerBtn) {
                        saveTimerBtn.disabled = true;
                        saveTimerBtn.classList.add('opacity-50');
                    }
                    
                    setTimeout(() => {
                        if (timerContainer) timerContainer.classList.add('hidden');
                        this.measure.mode = null;
                    }, 1500);
                } else {
                    if (measureResult) {
                        measureResult.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Error: ' + data.message;
                    }
                }
            })
            .catch(error => {
                console.error('Error:', error);
                if (measureResult) {
                    measureResult.innerHTML = '<i class="fas fa-times text-red-500 mr-1"></i> Connection error';
                }
            });
        },

        openMeasureModal(lightId) {
            if (!this.markers[lightId]) return;
            
            const light = this.markers[lightId].data;
            this.measure.lightId = lightId;
            
            const measureTitle = document.getElementById('measureTitle');
            const measureStatus = document.getElementById('measureStatus');
            const timerContainer = document.getElementById('timerContainer');
            
            if (measureTitle) {
                measureTitle.textContent = light.name;
                measureTitle.setAttribute('data-id', lightId);
            }
            
            const status = this.getLightStatus(light);
            
            if (measureStatus) {
                measureStatus.innerHTML = `
                    <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                        ${status.label} (${status.timeLeft}s)
                    </span>
                `;
            }
            
            if (timerContainer) timerContainer.classList.add('hidden');
            
            this.measure.mode = null;
            this.resetMeasureTimer();
            
            this.openModal('measureModal');
        },

        loadTrafficLights(isRefresh = false) {
            const lightsList = document.getElementById('lightsList');
            const loadingLights = document.getElementById('loadingLights');
            const noLightsMessage = document.getElementById('noLightsMessage');
            
            if (!isRefresh && loadingLights) {
                loadingLights.style.display = 'block';
            }
            
            if (noLightsMessage) noLightsMessage.classList.add('hidden');
            
            fetch('db.php?action=getTrafficLights')
                .then(response => response.json())
                .then(data => {
                    if (loadingLights) loadingLights.style.display = 'none';
                    
                    if (isRefresh) {
                        for (let id in this.markers) {
                            this.markers[id].marker.remove();
                            delete this.markers[id];
                        }
                        
                        if (lightsList) {
                            const lightCards = lightsList.querySelectorAll('.light-card');
                            lightCards.forEach(card => card.remove());
                        }
                    }
                    
                    if (Array.isArray(data) && data.length > 0) {
                        data.forEach(light => {
                            this.addLightToMap(light);
                            this.addLightToSidebar(light);
                        });
                    } else if (noLightsMessage) {
                        noLightsMessage.classList.remove('hidden');
                    }
                    
                    if (this.settings.autoRefresh) {
                        this.startStatusUpdates();
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    if (loadingLights) loadingLights.style.display = 'none';
                    if (noLightsMessage) noLightsMessage.classList.remove('hidden');
                    
                    if (this.settings.notifications) {
                        this.showNotification('Error loading traffic lights', 'error');
                    }
                });
        },

        addLightToMap(light) {
            const status = this.getLightStatus(light);
            
            const el = document.createElement('div');
            el.className = `w-8 h-8 bg-traffic-${status.color} rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white`;
            el.innerHTML = '<i class="fas fa-traffic-light"></i>';
            
            const marker = new mapboxgl.Marker(el)
                .setLngLat([parseFloat(light.longitude), parseFloat(light.latitude)])
                .addTo(this.map);
            
            marker.getElement().addEventListener('click', () => this.showLightDetail(light));
            
            this.markers[light.id] = {
                marker: marker,
                data: light
            };
        },

        addLightToSidebar(light) {
            const lightsList = document.getElementById('lightsList');
            if (!lightsList) return;
            
            const status = this.getLightStatus(light);
            
            const card = document.createElement('div');
            card.className = 'light-card bg-white shadow-md p-4 mb-3 cursor-pointer hover:shadow-lg rounded-xl';
            card.setAttribute('data-id', light.id);
            card.setAttribute('data-name', light.name);
            
            const showPrediction = this.settings.predictions;
            
            card.innerHTML = `
                <div class="flex justify-between items-center">
                    <h3 class="font-medium">${light.name}</h3>
                    <span id="status-${light.id}" class="px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white">${status.label}</span>
                </div>
                <div class="flex items-center text-sm text-gray-500 mt-2">
                    <i class="fas fa-location-dot mr-2"></i>
                    <span class="capitalize">${light.direction}</span>
                    <div class="ml-auto flex items-center" id="timer-${light.id}">
                        <i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s
                    </div>
                </div>
                ${showPrediction ? `
                <div class="mt-2 text-xs">
                    <div class="w-full bg-gray-100 rounded-full h-1.5">
                        <div class="bg-traffic-${status.color} h-1.5 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                    </div>
                    <div class="flex justify-between mt-1 text-gray-400">
                        <span>0s</span>
                        <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                    </div>
                </div>` : ''}
            `;
            
            card.addEventListener('click', () => {
                const allCards = document.querySelectorAll('.light-card');
                allCards.forEach(c => c.classList.remove('active'));
                card.classList.add('active');
                
                this.showLightDetail(light);
                
                if (window.innerWidth <= 768) {
                    const sidebar = document.querySelector('.sidebar');
                    if (sidebar) {
                        sidebar.classList.remove('active');
                        const menuToggle = document.getElementById('menuToggle');
                        if (menuToggle) {
                            menuToggle.innerHTML = '<i class="fas fa-bars"></i>';
                        }
                    }
                }
            });
            
            lightsList.insertBefore(card, document.getElementById('loadingLights'));
        },

        getLightStatus(light) {
            const totalCycle = parseInt(light.red_duration) + parseInt(light.green_duration);
            const currentTime = Math.floor(Date.now() / 1000);
            const timeInCycle = currentTime % totalCycle;
            
            if (timeInCycle < light.red_duration) {
                return {
                    isRed: true,
                    color: 'red',
                    label: 'RED',
                    timeLeft: light.red_duration - timeInCycle
                };
            } else {
                return {
                    isRed: false,
                    color: 'green',
                    label: 'GREEN',
                    timeLeft: totalCycle - timeInCycle
                };
            }
        },

        updateLightStatus(light) {
            const status = this.getLightStatus(light);
            const showPrediction = this.settings.predictions;
            
            if (this.markers[light.id]) {
                const el = this.markers[light.id].marker.getElement().firstChild;
                
                el.className = el.className.replace(/bg-traffic-\w+/, `bg-traffic-${status.color}`);
            }
            
            const statusElement = document.getElementById(`status-${light.id}`);
            const timerElement = document.getElementById(`timer-${light.id}`);
            
            if (statusElement) {
                statusElement.className = `px-2.5 py-1 text-xs font-medium rounded-full bg-traffic-${status.color} text-white`;
                statusElement.textContent = status.label;
            }
            
            if (timerElement) {
                timerElement.innerHTML = `<i class="fas fa-clock mr-1.5"></i> ${status.timeLeft}s`;
            }
            
            const card = document.querySelector(`.light-card[data-id="${light.id}"]`);
            if (card) {
                let predictionEl = card.querySelector('.mt-2.text-xs');
                
                if (showPrediction) {
                    if (predictionEl) {
                        const progressBar = predictionEl.querySelector('.bg-traffic-red, .bg-traffic-green');
                        if (progressBar) {
                            progressBar.className = `bg-traffic-${status.color} h-1.5 rounded-full`;
                            progressBar.style.width = `${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%`;
                        }
                    } else {
                        predictionEl = document.createElement('div');
                        predictionEl.className = 'mt-2 text-xs';
                        predictionEl.innerHTML = `
                            <div class="w-full bg-gray-100 rounded-full h-1.5">
                                <div class="bg-traffic-${status.color} h-1.5 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                            </div>
                            <div class="flex justify-between mt-1 text-gray-400">
                                <span>0s</span>
                                <span>${parseInt(light.red_duration) + parseInt(light.green_duration)}s</span>
                            </div>
                        `;
                        card.appendChild(predictionEl);
                    }
                } else if (!showPrediction && predictionEl) {
                    predictionEl.remove();
                }
            }
            
            const lightPopup = document.getElementById('lightPopup');
            if (this.selectedLightId === light.id && lightPopup && !lightPopup.classList.contains('hidden')) {
                const popupStatus = document.getElementById('popupStatus');
                if (popupStatus) {
                    popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
                    popupStatus.innerHTML = `
                        <div class="text-xl font-semibold mb-1">${status.label}</div>
                        <div class="flex justify-center items-center gap-2">
                            <i class="fas fa-clock"></i>
                            <span>Changes in ${status.timeLeft} seconds</span>
                        </div>
                        ${showPrediction ? `
                        <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                            <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                        </div>
                        ` : ''}
                    `;
                }
            }
            
            const measureStatus = document.getElementById('measureStatus');
            const measureTitle = document.getElementById('measureTitle');
            if (measureTitle && this.measure.lightId === light.id && measureStatus) {
                measureStatus.innerHTML = `
                    <span class="px-3 py-1.5 rounded-full text-sm font-medium bg-traffic-${status.color} text-white">
                        ${status.label} (${status.timeLeft}s)
                    </span>
                `;
            }
        },

        updateAllLightStatus() {
            for (const id in this.markers) {
                this.updateLightStatus(this.markers[id].data);
            }
        },

        startStatusUpdates() {
            this.stopStatusUpdates();
            
            const updateStatuses = (timestamp) => {
                if (!this.lastTimestamp || timestamp - this.lastTimestamp > 1000) {
                    this.lastTimestamp = timestamp;
                    this.updateAllLightStatus();
                }
                this.rafId = requestAnimationFrame(updateStatuses);
            };
            
            this.rafId = requestAnimationFrame(updateStatuses);
        },

        stopStatusUpdates() {
            if (this.rafId) {
                cancelAnimationFrame(this.rafId);
                this.rafId = null;
            }
        },

        showLightDetail(light) {
            this.selectedLightId = light.id;
            
            this.map.flyTo({
                center: [parseFloat(light.longitude), parseFloat(light.latitude)],
                zoom: 17,
                essential: true
            });
            
            const popupTitle = document.getElementById('popupTitle');
            const popupContent = document.getElementById('popupContent');
            const popupStatus = document.getElementById('popupStatus');
            const lightPopup = document.getElementById('lightPopup');
            
            if (!popupTitle || !popupContent || !popupStatus || !lightPopup) return;
            
            popupTitle.textContent = light.name;
            popupContent.innerHTML = `
                <div class="grid grid-cols-2 gap-3">
                    <div class="text-gray-500">Direction:</div>
                    <div class="font-medium capitalize">${light.direction}</div>
                    
                    <div class="text-gray-500">Red duration:</div>
                    <div class="font-medium">${light.red_duration} seconds</div>
                    
                    <div class="text-gray-500">Green duration:</div>
                    <div class="font-medium">${light.green_duration} seconds</div>
                    
                    <div class="text-gray-500">Total cycle:</div>
                    <div class="font-medium">${parseInt(light.red_duration) + parseInt(light.green_duration)} seconds</div>
                    
                    <div class="text-gray-500">Location:</div>
                    <div class="font-medium truncate">${light.latitude.substring(0, 8)}, ${light.longitude.substring(0, 8)}</div>
                </div>
            `;
            
            const status = this.getLightStatus(light);
            const showPrediction = this.settings.predictions;
            
            popupStatus.className = `p-4 rounded-xl text-center mb-4 bg-${status.color}-100 text-${status.color}-800`;
            popupStatus.innerHTML = `
                <div class="text-xl font-semibold mb-1">${status.label}</div>
                <div class="flex justify-center items-center gap-2">
                    <i class="fas fa-clock"></i>
                    <span>Changes in ${status.timeLeft} seconds</span>
                </div>
                ${showPrediction ? `
                <div class="mt-3 w-full bg-gray-200 rounded-full h-2 max-w-xs mx-auto">
                    <div class="bg-traffic-${status.color} h-2 rounded-full" style="width: ${Math.round((status.timeLeft / (parseInt(light.red_duration) + parseInt(light.green_duration))) * 100)}%"></div>
                </div>
                ` : ''}
            `;
            
            lightPopup.classList.remove('hidden');
        },

        startGeolocation() {
            if (navigator.geolocation) {
                if (this.userWatchId) navigator.geolocation.clearWatch(this.userWatchId);
                
                const headerLocateBtn = document.getElementById('headerLocateBtn');
                if (headerLocateBtn) {
                    headerLocateBtn.innerHTML = '<div class="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>';
                }
                
                this.userWatchId = navigator.geolocation.watchPosition(
                    this.updateUserLocation.bind(this),
                    this.handleGeolocationError.bind(this),
                    { enableHighAccuracy: true, maximumAge: 10000, timeout: 5000 }
                );
                
                navigator.geolocation.getCurrentPosition(
                    this.updateUserLocation.bind(this),
                    this.handleGeolocationError.bind(this),
                    { enableHighAccuracy: true }
                );
            }
        },

        highlightNearbyLights() {
            if (!this.userMarker) return;
            
            const userPos = this.userMarker.getLngLat();
            const nearbyLights = [];
            
            for (const id in this.markers) {
                const markerPos = this.markers[id].marker.getLngLat();
                const distance = this.getDistance(
                    [userPos.lng, userPos.lat],
                    [markerPos.lng, markerPos.lat]
                );
                
                if (distance < 1000) {
                    nearbyLights.push({
                        id: id,
                        distance: distance
                    });
                }
            }
            
            nearbyLights.sort((a, b) => a.distance - b.distance);
            
            document.querySelectorAll('.light-card').forEach(card => {
                card.classList.remove('active');
            });
            
            if (nearbyLights.length > 0) {
                if (this.settings.notifications) {
                    this.showNotification(`${nearbyLights.length} traffic lights found within 1km`, 'info');
                }
                
                nearbyLights.slice(0, 5).forEach(light => {
                    const card = document.querySelector(`.light-card[data-id="${light.id}"]`);
                    if (card) {
                        card.classList.add('active');
                        card.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                    }
                });
            } else {
                if (this.settings.notifications) {
                    this.showNotification('No traffic lights nearby', 'info');
                }
            }
        },

        setupAddressAutocomplete(inputElement) {
            if (!inputElement) return;
            
            let timeoutId;
            let suggestionsContainer;
            
            inputElement.setAttribute('autocomplete', 'off');
            
            inputElement.addEventListener('input', async () => {
                clearTimeout(timeoutId);
                
                if (inputElement.value.length < 3) {
                    if (suggestionsContainer) {
                        suggestionsContainer.remove();
                        suggestionsContainer = null;
                    }
                    return;
                }
                
                timeoutId = setTimeout(async () => {
                    const searchValue = inputElement.value;
                    
                    try {
                        const response = await fetch(`https://api.mapbox.com/geocoding/v5/mapbox.places/${encodeURIComponent(searchValue)}.json?access_token=${this.mapboxKey}`);
                        const data = await response.json();
                        
                        if (suggestionsContainer) {
                            suggestionsContainer.remove();
                        }
                        
                        if (!data.features || data.features.length === 0) return;
                        
                        suggestionsContainer = document.createElement('div');
                        suggestionsContainer.className = 'absolute z-50 mt-1 w-full bg-white shadow-lg rounded-xl overflow-hidden border border-gray-200';
                        
                        data.features.forEach(feature => {
                            const item = document.createElement('div');
                            item.className = 'px-3 py-2 hover:bg-gray-100 cursor-pointer text-sm';
                            item.textContent = feature.place_name;
                            
                            item.addEventListener('click', () => {
                                inputElement.value = feature.place_name;
                                
                                const center = feature.center;
                                
                                if (inputElement === document.getElementById('startLocation')) {
                                    this.navigation.startPosition = {
                                        center: center,
                                        place_name: feature.place_name
                                    };
                                } else if (inputElement === document.getElementById('endLocation')) {
                                    this.navigation.endPosition = {
                                        center: center,
                                        place_name: feature.place_name
                                    };
                                }
                                
                                suggestionsContainer.remove();
                                suggestionsContainer = null;
                            });
                            
                            suggestionsContainer.appendChild(item);
                        });
                        
                        inputElement.parentNode.appendChild(suggestionsContainer);
                    } catch (error) {
                        console.error('Geocoding error:', error);
                    }
                }, 300);
            });
            
            document.addEventListener('click', (e) => {
                if (suggestionsContainer && !inputElement.contains(e.target) && !suggestionsContainer.contains(e.target)) {
                    suggestionsContainer.remove();
                    suggestionsContainer = null;
                }
            });
        },

        toggleTrafficLayer() {
            this.settings.trafficView = !this.settings.trafficView;
            
            if (this.settings.trafficView) {
                if (!this.map.getLayer('traffic')) {
                    this.addTrafficLayer();
                }
            } else {
                if (this.map.getLayer('traffic')) {
                    this.map.removeLayer('traffic');
                    this.map.removeSource('traffic');
                }
            }
            
            const trafficViewToggle = document.getElementById('trafficViewToggle');
            if (trafficViewToggle) {
                trafficViewToggle.checked = this.settings.trafficView;
            }
            
            this.saveSettings();
        },

        showNotification(message, type = 'info') {
            if (!this.settings.notifications) return;
            
            const colors = {
                success: 'bg-green-500',
                error: 'bg-red-500',
                info: 'bg-primary-500'
            };
            
            const icons = {
                success: 'check-circle',
                error: 'exclamation-circle',
                info: 'info-circle'
            };
            
            const notification = document.createElement('div');
            notification.className = `fixed top-20 left-1/2 transform -translate-x-1/2 ${colors[type]} text-white px-4 py-3 rounded-lg shadow-lg z-50 fade-in flex items-center`;
            notification.innerHTML = `<i class="fas fa-${icons[type]} mr-2"></i>${message}`;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.opacity = '0';
                notification.style.transform = 'translate(-50%, -10px)';
                notification.style.transition = 'opacity 0.5s, transform 0.5s';
                setTimeout(() => notification.remove(), 500);
            }, 3000);
        },

        handleOnline() {
            if (this.settings.notifications) {
                this.showNotification('You are back online', 'success');
            }
            this.loadTrafficLights(true);
        },

        handleOffline() {
            if (this.settings.notifications) {
                this.showNotification('You are offline. Some features may be limited', 'error');
            }
        },

        handleVisibilityChange() {
            if (document.visibilityState === 'visible') {
                this.loadTrafficLights(true);
            }
        },

        handleInstallPrompt() {
            let deferredPrompt;
            
            window.addEventListener('beforeinstallprompt', (e) => {
                e.preventDefault();
                deferredPrompt = e;
                
                const installBanner = document.createElement('div');
                installBanner.className = 'fixed bottom-0 left-0 right-0 bg-primary-500 text-white py-3 px-4 shadow-lg z-40 fade-in';
                installBanner.innerHTML = `
                    <div class="flex items-center justify-between max-w-md mx-auto">
                        <div>
                            <p class="font-medium">Add TrafficLight to Home Screen</p>
                            <p class="text-sm">Get quick access to traffic lights data</p>
                        </div>
                        <div class="flex space-x-2">
                            <button id="dismissInstall" class="px-3 py-1.5 bg-primary-600 hover:bg-primary-700 rounded font-medium">Later</button>
                            <button id="installApp" class="px-3 py-1.5 bg-white text-primary-600 hover:bg-gray-100 rounded font-medium">Install</button>
                        </div>
                    </div>
                `;
                
                document.body.appendChild(installBanner);
                
                document.getElementById('dismissInstall').addEventListener('click', () => {
                    installBanner.style.opacity = '0';
                    installBanner.style.transform = 'translateY(100%)';
                    installBanner.style.transition = 'opacity 0.5s, transform 0.5s';
                    setTimeout(() => installBanner.remove(), 500);
                });
                
                document.getElementById('installApp').addEventListener('click', () => {
                    deferredPrompt.prompt();
                    deferredPrompt.userChoice.then((choiceResult) => {
                        if (choiceResult.outcome === 'accepted') {
                            console.log('User accepted the install prompt');
                            installBanner.style.opacity = '0';
                            installBanner.style.transform = 'translateY(100%)';
                            installBanner.style.transition = 'opacity 0.5s, transform 0.5s';
                            setTimeout(() => installBanner.remove(), 500);
                        }
                        deferredPrompt = null;
                    });
                });
            });
        }
    };
    
    app.init();
    app.handleInstallPrompt();
});