document.addEventListener('DOMContentLoaded', () => {
    // Global variables
    let selectedTicketId = null;
    let ticketsUpdateInterval = null;

    // DOM Elements
    const searchInput = document.getElementById('global-search');
    const userCards = document.querySelectorAll('.user-card');
    const documentItems = document.querySelectorAll('.document-item');
    const usersCount = document.getElementById('users-count');
    const documentsCount = document.getElementById('documents-count');
    const userModal = document.getElementById('user-modal');
    const profileModal = document.getElementById('profile-modal');
    const closeButtons = document.querySelectorAll('.close-modal, #close-modal');
    const tabs = document.querySelectorAll('[data-tab]');
    const contents = document.querySelectorAll('.tab-content');
    const themeToggle = document.getElementById('theme-toggle');
    const addUserBtn = document.getElementById('add-user-btn');

    // Initialize theme based on localStorage or system preference
    function initTheme() {
        if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }
    }

    // Theme toggle handler
    themeToggle.addEventListener('click', () => {
        document.documentElement.classList.toggle('dark');
        localStorage.theme = document.documentElement.classList.contains('dark') ? 'dark' : 'light';
    });

    // Update counts for users and documents
    function updateCounts() {
        const visibleUsers = document.querySelectorAll('.user-card:not(.hidden)').length;
        const visibleDocs = document.querySelectorAll('.document-item:not(.hidden)').length;
        
        if (usersCount) {
            usersCount.textContent = `Showing ${visibleUsers} user${visibleUsers !== 1 ? 's' : ''}`;
        }
        
        if (documentsCount) {
            documentsCount.textContent = `Showing ${visibleDocs} document${visibleDocs !== 1 ? 's' : ''}`;
        }
    }

    // Highlight search text in elements
    function highlightText(element, searchText) {
        if (!element) return;
        
        const originalText = element.textContent;
        if (searchText && originalText.toLowerCase().includes(searchText.toLowerCase())) {
            const regex = new RegExp(`(${searchText.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')})`, 'gi');
            element.innerHTML = originalText.replace(regex, '<mark class="search-highlight">$1</mark>');
        } else {
            element.textContent = originalText;
        }
    }

    // Load websites monitoring status
    function loadSites() {
        const sitesList = document.getElementById('sites-list');
        if (!sitesList) return;
        
        sitesList.innerHTML = `
            <div class="flex justify-center items-center p-8">
                <div class="animate-spin rounded-full h-8 w-8 border-2 border-gray-500 border-t-black dark:border-t-white"></div>
            </div>
        `;

        fetch('check_sites.php')
            .then(response => response.json())
            .then(data => {
                sitesList.innerHTML = '';
                if (data && data.results) {
                    data.results.forEach(site => {
                        const div = document.createElement('div');
                        div.className = 'flex items-center justify-between p-4 bg-white dark:bg-gray-900 rounded-lg fade-in';
                        
                        const statusClass = site.status ? 'text-green-500' : 'text-red-500';
                        const statusIcon = site.status ? 'fa-check-circle' : 'fa-exclamation-circle';
                        const statusIndicator = site.status ? 'status-online' : 'status-offline';
                        
                        div.innerHTML = `
                            <div class="flex items-center space-x-3">
                                <i class="fas fa-globe text-gray-500 dark:text-gray-400"></i>
                                <a href="${site.url}" target="_blank" class="hover:underline">${site.url}</a>
                            </div>
                            <div class="flex items-center space-x-4">
                                <div class="flex items-center space-x-1 ${statusClass}">
                                    <span class="status-indicator ${statusIndicator}"></span>
                                    <i class="fas ${statusIcon}"></i>
                                    <span>${site.status ? 'Online' : 'Offline'}</span>
                                </div>
                                <span class="text-xs text-gray-500 dark:text-gray-400">
                                    <i class="far fa-clock mr-1"></i>${site.timestamp || ''}
                                </span>
                            </div>
                        `;
                        
                        sitesList.appendChild(div);
                    });
                }
            })
            .catch(error => {
                console.error('Error:', error);
                sitesList.innerHTML = `
                    <div class="bg-red-100 dark:bg-red-900 border-l-4 border-red-500 text-red-700 dark:text-red-300 p-4 rounded">
                        <div class="flex items-center space-x-2">
                            <i class="fas fa-exclamation-circle"></i>
                            <span>Unable to load sites. Please try again later.</span>
                        </div>
                    </div>
                `;
            });
    }

    // Search functionality
    if (searchInput) {
        searchInput.addEventListener('input', (e) => {
            const searchText = e.target.value.toLowerCase();

            userCards.forEach(card => {
                const searchableText = card.dataset.searchable.toLowerCase();
                const visible = searchableText.includes(searchText);
                card.classList.toggle('hidden', !visible);

                if (visible) {
                    card.querySelectorAll('.username, .email, .note, .roleimators').forEach(element => {
                        highlightText(element, searchText);
                    });
                }
            });

            documentItems.forEach(item => {
                const searchableText = item.dataset.searchable.toLowerCase();
                const visible = searchableText.includes(searchText);
                item.classList.toggle('hidden', !visible);

                if (visible) {
                    highlightText(item.querySelector('.document-name'), searchText);
                }
            });

            updateCounts();
        });
    }

    // Tab navigation
    tabs.forEach(tab => {
        tab.addEventListener('click', () => {
            tabs.forEach(t => {
                t.classList.remove('border-black', 'dark:border-white', 'active');
                t.classList.add('border-transparent');
            });
            
            tab.classList.remove('border-transparent');
            tab.classList.add('border-black', 'dark:border-white', 'active');
            
            contents.forEach(content => content.classList.add('hidden'));
            document.getElementById(`${tab.dataset.tab}-content`).classList.remove('hidden');

            if (tab.dataset.tab === 'sites') {
                loadSites();
            }
            
            if (tab.dataset.tab === 'tickets') {
                loadTickets();
                setupTicketsRefresh();
            } else {
                clearInterval(ticketsUpdateInterval);
            }
        });
    });

    // User card click handler - open edit modal
    userCards.forEach(card => {
        card.addEventListener('click', () => {
            userModal.classList.remove('hidden');
            document.getElementById('user-modal-title').innerHTML = `
                <i class="fas fa-user-edit text-gray-600 dark:text-gray-400"></i>
                <span>Edit User</span>
            `;
            document.getElementById('user-id').value = card.dataset.userId;
            document.getElementById('email').value = card.dataset.email;
            document.getElementById('username').value = card.dataset.username;
            document.getElementById('note').value = card.dataset.note;
            document.getElementById('role').value = card.dataset.role || "0";
            document.getElementById('roleimators').value = card.dataset.roleimators || "";
            
            // Show delete button for edit mode
            document.getElementById('delete-user-btn').classList.remove('hidden');
        });
    });

    // Add user button handler
    if (addUserBtn) {
        addUserBtn.addEventListener('click', () => {
            userModal.classList.remove('hidden');
            document.getElementById('user-modal-title').innerHTML = `
                <i class="fas fa-user-plus text-gray-600 dark:text-gray-400"></i>
                <span>Add New User</span>
            `;
            document.getElementById('edit-user-form').reset();
            document.getElementById('user-id').value = '';
            
            // Hide delete button for add mode
            document.getElementById('delete-user-btn').classList.add('hidden');
        });
    }

    // Close modal buttons
    closeButtons.forEach(button => {
        button.addEventListener('click', () => {
            userModal.classList.add('hidden');
            profileModal.classList.add('hidden');
        });
    });

    // User edit/add form submission
    document.getElementById('edit-user-form').addEventListener('submit', async (e) => {
        e.preventDefault();
        const form = e.target;
        const submitButton = form.querySelector('button[type="submit"]');
        const originalContent = submitButton.innerHTML;
        const isNewUser = !document.getElementById('user-id').value;
        
        showLoadingOverlay();
        submitButton.disabled = true;
        submitButton.innerHTML = `
            <div class="flex items-center space-x-2">
                <div class="animate-spin rounded-full h-4 w-4 border-2 border-gray-500 border-t-white"></div>
                <span>${isNewUser ? 'Creating...' : 'Saving...'}</span>
            </div>
        `;

        try {
            const formData = new FormData(form);
            const response = await fetch(isNewUser ? 'add_user.php' : 'update_user.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            if(result.success) {
                userModal.classList.add('hidden');
                showNotification(`User ${isNewUser ? 'created' : 'updated'} successfully!`, 'success');
                setTimeout(() => {
                    window.location.href = '?success=1';
                }, 1000);
            } else {
                throw new Error(result.message || `Failed to ${isNewUser ? 'create' : 'update'} user`);
            }
        } catch(error) {
            console.error('Error:', error);
            showNotification(error.message, 'error');
        } finally {
            hideLoadingOverlay();
            submitButton.disabled = false;
            submitButton.innerHTML = originalContent;
        }
    });

    // Delete user button handler
    document.getElementById('delete-user-btn').addEventListener('click', async () => {
        if (confirm('Are you sure you want to delete this user? This action cannot be undone.')) {
            const userId = document.getElementById('user-id').value;
            const button = document.getElementById('delete-user-btn');
            const originalContent = button.innerHTML;
            
            showLoadingOverlay();
            button.disabled = true;
            button.innerHTML = `
                <div class="flex items-center space-x-2">
                    <div class="animate-spin rounded-full h-4 w-4 border-2 border-gray-500 border-t-white"></div>
                    <span>Deleting...</span>
                </div>
            `;

            try {
                const response = await fetch('delete_user.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `user_id=${userId}`
                });
                
                const result = await response.json();
                if(result.success) {
                    userModal.classList.add('hidden');
                    showNotification('User deleted successfully!', 'success');
                    setTimeout(() => {
                        window.location.href = '?success=1';
                    }, 1000);
                } else {
                    throw new Error(result.message || 'Delete failed');
                }
            } catch(error) {
                console.error('Error:', error);
                showNotification(error.message, 'error');
            } finally {
                hideLoadingOverlay();
                button.disabled = false;
                button.innerHTML = originalContent;
            }
        }
    });

    