<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>ISE Engine - Document Editor</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Urbanist:wght@100;200;300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://use.fontawesome.com/releases/v6.4.0/css/all.css">
    <style>
        body { font-family: 'Urbanist', sans-serif; }
        .editor-content { min-height: 500px; }
        .toolbar-button { @apply px-3 py-2 rounded-lg bg-gray-100 hover:bg-gray-200 transition-colors text-sm; }
        .toolbar-button.active { @apply bg-blue-500 text-white; }
        .ai-panel { max-height: 400px; overflow-y: auto; }
    </style>
</head>
<body class="bg-white text-black">
    <header class="border-b border-gray-200 bg-white sticky top-0 z-40">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center py-4">
                <div class="flex items-center space-x-4">
                    <a href="/workspace" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-arrow-left"></i>
                    </a>
                    <h1 class="text-xl font-semibold">ISE Engine</h1>
                    <span class="text-sm text-gray-500" id="document-title">Nouveau document</span>
                </div>
                <div class="flex items-center space-x-4">
                    <button id="save-btn" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors">
                        <i class="fas fa-save mr-2"></i>Enregistrer
                    </button>
                    <button id="share-btn" class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors">
                        <i class="fas fa-share mr-2"></i>Partager
                    </button>
                    <div class="flex items-center space-x-2">
                        <span class="text-gray-700 text-sm">{{ session('username') }}</span>
                        <a href="/logout" class="text-red-500 hover:text-red-700">
                            <i class="fas fa-sign-out-alt"></i>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
        <div class="grid grid-cols-1 lg:grid-cols-4 gap-6">
            <!-- Éditeur principal -->
            <div class="lg:col-span-3">
                <!-- Barre d'outils -->
                <div class="bg-gray-50 border border-gray-200 rounded-lg p-4 mb-4">
                    <div class="flex flex-wrap gap-2 items-center">
                        <button class="toolbar-button" data-action="bold">
                            <i class="fas fa-bold"></i>
                        </button>
                        <button class="toolbar-button" data-action="italic">
                            <i class="fas fa-italic"></i>
                        </button>
                        <button class="toolbar-button" data-action="underline">
                            <i class="fas fa-underline"></i>
                        </button>
                        <div class="w-px h-6 bg-gray-300 mx-2"></div>
                        <button class="toolbar-button" data-action="h1">H1</button>
                        <button class="toolbar-button" data-action="h2">H2</button>
                        <button class="toolbar-button" data-action="h3">H3</button>
                        <div class="w-px h-6 bg-gray-300 mx-2"></div>
                        <button class="toolbar-button" data-action="bullet-list">
                            <i class="fas fa-list-ul"></i>
                        </button>
                        <button class="toolbar-button" data-action="number-list">
                            <i class="fas fa-list-ol"></i>
                        </button>
                        <div class="w-px h-6 bg-gray-300 mx-2"></div>
                        <button id="claude-assist" class="toolbar-button bg-purple-100 hover:bg-purple-200 text-purple-700">
                            <i class="fas fa-robot mr-1"></i>Claude
                        </button>
                        <button id="perplexity-search" class="toolbar-button bg-orange-100 hover:bg-orange-200 text-orange-700">
                            <i class="fas fa-search mr-1"></i>Recherche
                        </button>
                    </div>
                </div>

                <!-- Zone d'édition -->
                <div class="bg-white border border-gray-200 rounded-lg">
                    <div class="p-6">
                        <input type="text" id="document-title-input" placeholder="Titre du document..." class="w-full text-2xl font-bold border-none outline-none mb-6 placeholder-gray-400">
                        <div id="editor" class="editor-content outline-none" contenteditable="true" data-placeholder="Commencez à écrire votre document...">
                        </div>
                    </div>
                </div>
            </div>

            <!-- Panneau IA -->
            <div class="lg:col-span-1">
                <div class="bg-gray-50 border border-gray-200 rounded-lg p-4 sticky top-24">
                    <h3 class="font-semibold mb-4 flex items-center">
                        <i class="fas fa-brain text-blue-600 mr-2"></i>
                        Assistant IA
                    </h3>

                    <!-- Onglets -->
                    <div class="flex mb-4 bg-white rounded-lg p-1">
                        <button id="tab-claude" class="flex-1 py-2 px-3 rounded-md text-sm font-medium bg-blue-500 text-white">
                            Claude
                        </button>
                        <button id="tab-perplexity" class="flex-1 py-2 px-3 rounded-md text-sm font-medium text-gray-500 hover:text-gray-700">
                            Recherche
                        </button>
                    </div>

                    <!-- Panneau Claude -->
                    <div id="claude-panel" class="space-y-4">
                        <div class="space-y-2">
                            <button data-claude-action="grammar" class="w-full text-left p-3 bg-white rounded-lg hover:bg-gray-50 border text-sm">
                                <i class="fas fa-spell-check text-green-600 mr-2"></i>
                                Corriger la grammaire
                            </button>
                            <button data-claude-action="expand" class="w-full text-left p-3 bg-white rounded-lg hover:bg-gray-50 border text-sm">
                                <i class="fas fa-expand-arrows-alt text-blue-600 mr-2"></i>
                                Développer le texte
                            </button>
                            <button data-claude-action="summarize" class="w-full text-left p-3 bg-white rounded-lg hover:bg-gray-50 border text-sm">
                                <i class="fas fa-compress-arrows-alt text-orange-600 mr-2"></i>
                                Résumer
                            </button>
                            <button data-claude-action="rephrase" class="w-full text-left p-3 bg-white rounded-lg hover:bg-gray-50 border text-sm">
                                <i class="fas fa-magic text-purple-600 mr-2"></i>
                                Reformuler
                            </button>
                        </div>

                        
                        <div class="border-t pt-4">
                            <textarea id="claude-prompt" placeholder="Demandez à Claude de modifier votre texte..." class="w-full p-3 border rounded-lg text-sm resize-none" rows="3"></textarea>
                            <button id="claude-submit" class="w-full mt-2 bg-purple-600 text-white py-2 rounded-lg hover:bg-purple-700 text-sm">
                                Envoyer à Claude
                            </button>
                        </div>

                        <div id="claude-response" class="hidden ai-panel bg-white border rounded-lg p-3 text-sm">
                            <div class="flex items-center justify-between mb-2">
                                <span class="font-medium text-purple-600">Réponse Claude</span>
                                <button id="claude-close" class="text-gray-400 hover:text-gray-600">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                            <div id="claude-content" class="text-gray-700"></div>
                            <button id="claude-apply" class="w-full mt-3 bg-purple-600 text-white py-2 rounded-lg hover:bg-purple-700 text-sm">
                                Appliquer les modifications
                            </button>
                        </div>
                    </div>

                    <!-- Panneau Perplexity -->
                    <div id="perplexity-panel" class="hidden space-y-4">
                        <div class="space-y-2">
                            <input type="text" id="search-query" placeholder="Rechercher des informations..." class="w-full p-3 border rounded-lg text-sm">
                            <button id="search-submit" class="w-full bg-orange-600 text-white py-2 rounded-lg hover:bg-orange-700 text-sm">
                                <i class="fas fa-search mr-2"></i>Rechercher
                            </button>
                        </div>

                        <div id="search-results" class="hidden ai-panel bg-white border rounded-lg p-3 text-sm">
                            <div class="flex items-center justify-between mb-2">
                                <span class="font-medium text-orange-600">Résultats de recherche</span>
                                <button id="search-close" class="text-gray-400 hover:text-gray-600">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                            <div id="search-content" class="text-gray-700 space-y-3"></div>
                            <button id="search-insert" class="w-full mt-3 bg-orange-600 text-white py-2 rounded-lg hover:bg-orange-700 text-sm">
                                Insérer dans le document
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal de sauvegarde -->
    <div id="save-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center">
        <div class="bg-white rounded-lg p-6 w-full max-w-md mx-4">
            <h3 class="text-lg font-semibold mb-4">Enregistrer le document</h3>
            <input type="text" id="save-title" placeholder="Nom du document" class="w-full p-3 border rounded-lg mb-4">
            <div class="flex justify-end space-x-3">
                <button id="save-cancel" class="px-4 py-2 text-gray-600 hover:text-gray-800">Annuler</button>
                <button id="save-confirm" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">Enregistrer</button>
            </div>
        </div>
    </div>

    <!-- Modal de partage -->
    <div id="share-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center">
        <div class="bg-white rounded-lg p-6 w-full max-w-md mx-4">
            <h3 class="text-lg font-semibold mb-4">Partager le document</h3>
            <div class="space-y-4">
                <div>
                    <label class="flex items-center">
                        <input type="radio" name="share-type" value="view" class="mr-2" checked>
                        <span>Lecture seule</span>
                    </label>
                    <label class="flex items-center mt-2">
                        <input type="radio" name="share-type" value="edit" class="mr-2">
                        <span>Édition autorisée</span>
                    </label>
                </div>
                <input type="text" id="share-link" readonly class="w-full p-3 border rounded-lg bg-gray-50" placeholder="Le lien sera généré ici...">
            </div>
            <div class="flex justify-end space-x-3 mt-6">
                <button id="share-cancel" class="px-4 py-2 text-gray-600 hover:text-gray-800">Annuler</button>
                <button id="share-generate" class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">Générer le lien</button>
                <button id="share-copy" class="hidden px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">Copier</button>
            </div>
        </div>
    </div>
<script>
// Variables globales
let currentDocument = {{ isset($document_id) ? $document_id : 'null' }};
let selectedText = '';
let editor = document.getElementById('editor');
let autoSaveInterval;

// Charger le document si ID fourni
if (currentDocument) {
    loadDocument(currentDocument);
}

// Fonction de chargement de document
async function loadDocument(id) {
    try {
        const response = await fetch(`/api/documents/${id}`);
        if (response.ok) {
            const document = await response.json();
            document.getElementById('document-title-input').value = document.title;
            editor.innerHTML = document.content;
            document.getElementById('document-title').textContent = document.title;
        }
    } catch (error) {
        console.error('Erreur chargement document:', error);
    }
}

// Auto-save
function startAutoSave() {
    autoSaveInterval = setInterval(async () => {
        const title = document.getElementById('document-title-input').value;
        const content = editor.innerHTML;
        
        if (title.trim() !== '' || content.trim() !== '') {
            await saveDocument(false);
        }
    }, 30000);
}

async function saveDocument(showNotification = true) {
    const title = document.getElementById('document-title-input').value || 'Document sans titre';
    const content = editor.innerHTML;

    try {
        const response = await fetch('/api/documents/save', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
            },
            body: JSON.stringify({
                title: title,
                content: content,
                document_id: currentDocument
            })
        });

        const result = await response.json();
        
        if (result.success) {
            currentDocument = result.document_id;
            document.getElementById('document-title').textContent = title;
            
            if (showNotification) {
                showNotification('Document sauvegardé !', 'success');
            }
        }
    } catch (error) {
        console.error('Erreur sauvegarde:', error);
        if (showNotification) {
            showNotification('Erreur de sauvegarde', 'error');
        }
    }
}

// Gestion des onglets IA
document.getElementById('tab-claude').addEventListener('click', switchToClaudeTab);
document.getElementById('tab-perplexity').addEventListener('click', switchToPerplexityTab);

function switchToClaudeTab() {
    document.getElementById('tab-claude').classList.add('bg-blue-500', 'text-white');
    document.getElementById('tab-claude').classList.remove('text-gray-500');
    document.getElementById('tab-perplexity').classList.remove('bg-blue-500', 'text-white');
    document.getElementById('tab-perplexity').classList.add('text-gray-500');
    document.getElementById('claude-panel').classList.remove('hidden');
    document.getElementById('perplexity-panel').classList.add('hidden');
}

function switchToPerplexityTab() {
    document.getElementById('tab-perplexity').classList.add('bg-blue-500', 'text-white');
    document.getElementById('tab-perplexity').classList.remove('text-gray-500');
    document.getElementById('tab-claude').classList.remove('bg-blue-500', 'text-white');
    document.getElementById('tab-claude').classList.add('text-gray-500');
    document.getElementById('perplexity-panel').classList.remove('hidden');
    document.getElementById('claude-panel').classList.add('hidden');
}

// Gestion sélection de texte
editor.addEventListener('mouseup', function() {
    selectedText = window.getSelection().toString().trim();
});

// Barre d'outils
document.querySelectorAll('.toolbar-button[data-action]').forEach(button => {
    button.addEventListener('click', function() {
        const action = this.dataset.action;
        executeCommand(action);
    });
});

function executeCommand(action) {
    switch(action) {
        case 'bold':
            document.execCommand('bold');
            break;
        case 'italic':
            document.execCommand('italic');
            break;
        case 'underline':
            document.execCommand('underline');
            break;
        case 'h1':
            document.execCommand('formatBlock', false, 'h1');
            break;
        case 'h2':
            document.execCommand('formatBlock', false, 'h2');
            break;
        case 'h3':
            document.execCommand('formatBlock', false, 'h3');
            break;
        case 'bullet-list':
            document.execCommand('insertUnorderedList');
            break;
        case 'number-list':
            document.execCommand('insertOrderedList');
            break;
    }
    editor.focus();
}

// Fonctions Claude
document.getElementById('claude-submit').addEventListener('click', () => {
    const prompt = document.getElementById('claude-prompt').value;
    if (prompt.trim()) {
        callClaudeAPI(prompt, 'custom');
    }
});

// Boutons Claude prédéfinis
document.querySelector('[data-claude-action="grammar"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'grammar');
    } else {
        showNotification('Sélectionnez du texte à corriger', 'warning');
    }
});

document.querySelector('[data-claude-action="expand"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'expand');
    } else {
        showNotification('Sélectionnez du texte à développer', 'warning');
    }
});

document.querySelector('[data-claude-action="summarize"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'summarize');
    } else {
        showNotification('Sélectionnez du texte à résumer', 'warning');
    }
});

document.querySelector('[data-claude-action="rephrase"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'rephrase');
    } else {
        showNotification('Sélectionnez du texte à reformuler', 'warning');
    }
});

async function callClaudeAPI(prompt, action) {
    try {
        showLoading('claude');
        
        const response = await fetch('/api/claude', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                prompt: prompt,
                selected_text: selectedText,
                action: action
            })
        });

        const result = await response.json();
        
        if (result.success) {
            showClaudeResponse(result.content);
        } else {
            showNotification('Erreur Claude API', 'error');
        }
    } catch (error) {
        console.error('Erreur Claude API:', error);
        showNotification('Erreur de connexion Claude', 'error');
    } finally {
        hideLoading('claude');
    }
}

function showClaudeResponse(content) {
    document.getElementById('claude-content').textContent = content;
    document.getElementById('claude-response').classList.remove('hidden');
}

document.getElementById('claude-close').addEventListener('click', () => {
    document.getElementById('claude-response').classList.add('hidden');
});

document.getElementById('claude-apply').addEventListener('click', () => {
    const claudeContent = document.getElementById('claude-content').textContent;
    if (selectedText && claudeContent) {
        // Remplacer le texte sélectionné par la réponse de Claude
        const selection = window.getSelection();
        if (selection.rangeCount > 0) {
            const range = selection.getRangeAt(0);
            range.deleteContents();
            range.insertNode(document.createTextNode(claudeContent));
        }
        document.getElementById('claude-response').classList.add('hidden');
        selectedText = '';
    }
});

// Fonctions Perplexity
document.getElementById('search-submit').addEventListener('click', () => {
    const query = document.getElementById('search-query').value;
    if (query.trim()) {
        callPerplexityAPI(query);
    }
});

async function callPerplexityAPI(query) {
    try {
        showLoading('search');
        
        const response = await fetch('/api/search', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                query: query
            })
        });

        const result = await response.json();
        
        if (result.success) {
            showSearchResults(result);
        } else {
            showNotification('Erreur recherche API', 'error');
        }
    } catch (error) {
        console.error('Erreur recherche API:', error);
        showNotification('Erreur de connexion recherche', 'error');
    } finally {
        hideLoading('search');
    }
}

function showSearchResults(result) {
    const container = document.getElementById('search-content');
    let html = '';
    
    if (result.summary) {
        html += `<div class="bg-blue-50 p-3 rounded mb-3">
            <div class="font-medium text-blue-900 mb-1">Résumé</div>
            <div class="text-blue-800 text-sm">${result.summary}</div>
        </div>`;
    }
    
    result.results.forEach(item => {
        html += `<div class="border-b pb-2 mb-2 last:border-b-0">
            <div class="font-medium text-sm">${item.title}</div>
            <div class="text-xs text-gray-600 mb-1">${item.source}</div>
            <div class="text-sm text-gray-700">${item.snippet}</div>
        </div>`;
    });
    
    container.innerHTML = html;
    document.getElementById('search-results').classList.remove('hidden');
}

document.getElementById('search-close').addEventListener('click', () => {
    document.getElementById('search-results').classList.add('hidden');
});

document.getElementById('search-insert').addEventListener('click', () => {
    const summary = document.querySelector('#search-content .bg-blue-50 .text-blue-800');
    if (summary) {
        insertAtCursor(summary.textContent);
        document.getElementById('search-results').classList.add('hidden');
    }
});

// Modals
document.getElementById('save-btn').addEventListener('click', () => {
    document.getElementById('save-title').value = document.getElementById('document-title-input').value;
    document.getElementById('save-modal').classList.remove('hidden');
});

document.getElementById('save-confirm').addEventListener('click', async () => {
    const title = document.getElementById('save-title').value;
    document.getElementById('document-title-input').value = title;
    await saveDocument(true);
    document.getElementById('save-modal').classList.add('hidden');
});

document.getElementById('save-cancel').addEventListener('click', () => {
    document.getElementById('save-modal').classList.add('hidden');
});

document.getElementById('share-btn').addEventListener('click', () => {
    if (!currentDocument) {
        showNotification('Sauvegardez d\'abord le document', 'warning');
        return;
    }
    document.getElementById('share-modal').classList.remove('hidden');
});

document.getElementById('share-generate').addEventListener('click', async () => {
    const permission = document.querySelector('input[name="share-type"]:checked').value;
    
    try {
        const response = await fetch(`/api/documents/${currentDocument}/share`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                permission: permission
            })
        });

        const result = await response.json();
        
        if (result.success) {
            document.getElementById('share-link').value = result.share_url;
            document.getElementById('share-copy').classList.remove('hidden');
        }
    } catch (error) {
        console.error('Erreur partage:', error);
        showNotification('Erreur de partage', 'error');
    }
});

document.getElementById('share-copy').addEventListener('click', () => {
    const shareLink = document.getElementById('share-link');
    shareLink.select();
    document.execCommand('copy');
    showNotification('Lien copié !', 'success');
});

document.getElementById('share-cancel').addEventListener('click', () => {
    document.getElementById('share-modal').classList.add('hidden');
    document.getElementById('share-link').value = '';
    document.getElementById('share-copy').classList.add('hidden');
});

// Fonctions utilitaires
function insertAtCursor(text) {
    const selection = window.getSelection();
    if (selection.rangeCount > 0) {
        const range = selection.getRangeAt(0);
        const textNode = document.createTextNode(text);
        range.insertNode(textNode);
        range.setStartAfter(textNode);
        range.setEndAfter(textNode);
        selection.removeAllRanges();
        selection.addRange(range);
    } else {
        editor.appendChild(document.createTextNode(text));
    }
    editor.focus();
}

function showLoading(type) {
    const button = type === 'claude' ? 
        document.getElementById('claude-submit') : 
        document.getElementById('search-submit');
    button.disabled = true;
    button.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Chargement...';
}

function hideLoading(type) {
    const button = type === 'claude' ? 
        document.getElementById('claude-submit') : 
        document.getElementById('search-submit');
    button.disabled = false;
    button.innerHTML = type === 'claude' ? 
        'Envoyer à Claude' : 
        '<i class="fas fa-search mr-2"></i>Rechercher';
}

function showNotification(message, type = 'info') {
    const colors = {
        success: 'bg-green-500',
        error: 'bg-red-500',
        warning: 'bg-yellow-500',
        info: 'bg-blue-500'
    };
    
    const notification = document.createElement('div');
    notification.className = `fixed top-4 right-4 ${colors[type]} text-white px-4 py-2 rounded-lg shadow-lg z-50`;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.remove();
    }, 3000);
}

// Démarrer l'auto-save
startAutoSave();

// Cleanup à la fermeture
window.addEventListener('beforeunload', () => {
    if (autoSaveInterval) {
        clearInterval(autoSaveInterval);
    }
});
</script>
</body>
</html>