<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>IS_Engine</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Urbanist:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://use.fontawesome.com/releases/v6.4.0/css/all.css">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <style>
        body { font-family: 'Urbanist', sans-serif; }
        .editor-content { min-height: 70vh; }
        .editor-content:empty:before {
            content: attr(data-placeholder);
            color: #9ca3af;
        }
    </style>
</head>
<body class="bg-white text-black">
    <!-- Header -->
    <header class="border-b border-gray-200 px-6 py-4">
        <div class="flex justify-between items-center max-w-7xl mx-auto">
            <div class="flex items-center space-x-4">
                <a href="/workspace" class="text-gray-600 hover:text-black text-lg">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="text-xl font-medium">IS_Engine</h1>
            </div>
            <div class="flex items-center space-x-3">
                <button id="save-btn" class="px-4 py-2 bg-black text-white rounded-md hover:bg-gray-800 text-sm">
                    Save
                </button>
                <button id="share-btn" class="px-4 py-2 border border-gray-300 rounded-md hover:bg-gray-50 text-sm">
                    Share
                </button>
                <span class="text-sm text-gray-600">Welcome to you, {{ session('username') }}</span>
            </div>
        </div>
    </header>

    <!-- Main Content -->
    <div class="max-w-7xl mx-auto px-6 py-8">
        <div class="grid grid-cols-4 gap-8">
            
            <!-- Editor -->
            <div class="col-span-3">
                <div class="bg-white">
                    <!-- Title -->
                    <input type="text" id="document-title-input" 
                           placeholder="Untitled Document" 
                           class="w-full text-3xl font-light border-none outline-none mb-6 placeholder-gray-400">
                    
                    <!-- Toolbar -->
                    <div class="flex items-center space-x-4 mb-6 pb-4 border-b border-gray-200">
                        <button class="p-2 hover:bg-gray-100 rounded" data-action="bold">
                            <i class="fas fa-bold text-sm"></i>
                        </button>
                        <button class="p-2 hover:bg-gray-100 rounded" data-action="italic">
                            <i class="fas fa-italic text-sm"></i>
                        </button>
                        <button class="p-2 hover:bg-gray-100 rounded" data-action="underline">
                            <i class="fas fa-underline text-sm"></i>
                        </button>
                        <div class="w-px h-4 bg-gray-300"></div>
                        <button class="px-3 py-1 hover:bg-gray-100 rounded text-sm" data-action="h1">H1</button>
                        <button class="px-3 py-1 hover:bg-gray-100 rounded text-sm" data-action="h2">H2</button>
                        <button class="px-3 py-1 hover:bg-gray-100 rounded text-sm" data-action="h3">H3</button>
                        <div class="w-px h-4 bg-gray-300"></div>
                        <button class="p-2 hover:bg-gray-100 rounded" data-action="bullet-list">
                            <i class="fas fa-list-ul text-sm"></i>
                        </button>
                        <button class="p-2 hover:bg-gray-100 rounded" data-action="number-list">
                            <i class="fas fa-list-ol text-sm"></i>
                        </button>
                    </div>
                    
                    <!-- Editor -->
                    <div id="editor" 
                         class="editor-content outline-none leading-relaxed text-gray-800" 
                         contenteditable="true" 
                         data-placeholder="Start writing...">
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="col-span-1">
                <div class="bg-gray-50 p-5 rounded-lg">
                    <h3 class="font-medium mb-4">Writing tool</h3>
                    
                    <!-- Tabs -->
                    <div class="flex mb-4 bg-white rounded">
                        <button id="tab-claude" class="flex-1 py-2 px-3 text-sm bg-black text-white rounded">
                            Texts
                        </button>
                        <button id="tab-perplexity" class="flex-1 py-2 px-3 text-sm text-gray-600">
                            Check
                        </button>
                    </div>

                    <!-- Claude Panel -->
                    <div id="claude-panel" class="space-y-3">
                        <button data-claude-action="grammar" class="w-full text-left p-3 bg-white rounded border text-sm hover:bg-gray-50">
                            Fix Grammar
                        </button>
                        <button data-claude-action="expand" class="w-full text-left p-3 bg-white rounded border text-sm hover:bg-gray-50">
                            Expand Text
                        </button>
                        <button data-claude-action="summarize" class="w-full text-left p-3 bg-white rounded border text-sm hover:bg-gray-50">
                            Summarize
                        </button>
                        <button data-claude-action="rephrase" class="w-full text-left p-3 bg-white rounded border text-sm hover:bg-gray-50">
                            Rephrase
                        </button>
                        
                        <div class="pt-3 border-t">
                            <textarea id="claude-prompt" 
                                      placeholder="Ask the assistant..." 
                                      class="w-full p-3 border rounded text-sm resize-none" 
                                      rows="3"></textarea>
                            <button id="claude-submit" class="w-full mt-2 py-2 bg-black text-white rounded text-sm hover:bg-gray-800">
                                Send
                            </button>
                        </div>

                        <div id="claude-response" class="hidden p-3 bg-white border rounded">
                            <div class="flex justify-between items-center mb-2">
                                <span class="font-medium text-sm">Response</span>
                                <button id="claude-close" class="text-gray-400">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                            <div id="claude-content" class="text-sm text-gray-700 mb-2"></div>
                            <button id="claude-apply" class="w-full py-2 bg-black text-white rounded text-sm">
                                Apply
                            </button>
                        </div>
                    </div>

                    <!-- Search Panel -->
                    <div id="perplexity-panel" class="hidden space-y-3">
                        <p>Check that the information you write is correctly quoted from reliable international sources.</p>
                        <input type="text" id="search-query" 
                               placeholder="Search..." 
                               class="w-full p-3 border rounded text-sm">
                        <button id="search-submit" class="w-full py-2 bg-black text-white rounded text-sm hover:bg-gray-800">
                            Search
                        </button>

                        <div id="search-results" class="hidden p-3 bg-white border rounded">
                            <div class="flex justify-between items-center mb-2">
                                <span class="font-medium text-sm">Results</span>
                                <button id="search-close" class="text-gray-400">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                            <div id="search-content" class="text-sm space-y-2 mb-2"></div>
                            <button id="search-insert" class="w-full py-2 bg-black text-white rounded text-sm">
                                Insert
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Save Modal -->
    <div id="save-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-lg p-6 w-full max-w-md">
            <h3 class="text-lg font-medium mb-4">Save Document</h3>
            <input type="text" id="save-title" placeholder="Document name" 
                   class="w-full p-3 border rounded mb-4">
            <div class="flex justify-end space-x-3">
                <button id="save-cancel" class="px-4 py-2 text-gray-600">Cancel</button>
                <button id="save-confirm" class="px-4 py-2 bg-black text-white rounded">Save</button>
            </div>
        </div>
    </div>

    <!-- Share Modal -->
    <div id="share-modal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-lg p-6 w-full max-w-md">
            <h3 class="text-lg font-medium mb-4">Share Document</h3>
            <div class="space-y-3 mb-4">
                <label class="flex items-center">
                    <input type="radio" name="share-type" value="view" class="mr-2" checked>
                    <span class="text-sm">Read only</span>
                </label>
                <label class="flex items-center">
                    <input type="radio" name="share-type" value="edit" class="mr-2">
                    <span class="text-sm">Allow editing</span>
                </label>
                <input type="text" id="share-link" readonly 
                       class="w-full p-3 border rounded bg-gray-50" 
                       placeholder="Link will appear here">
            </div>
            <div class="flex justify-end space-x-3">
                <button id="share-cancel" class="px-4 py-2 text-gray-600">Cancel</button>
                <button id="share-generate" class="px-4 py-2 bg-black text-white rounded">Generate</button>
                <button id="share-copy" class="hidden px-4 py-2 bg-black text-white rounded">Copy</button>
            </div>
        </div>
    </div>

      <div class="fixed inset-0 flex flex-col items-center justify-center bg-white text-center px-6 md:hidden">
    <p class="text-lg text-gray-700">This site is not available on mobile. Please try again from a computer.</p>
  </div>

<script>
// Variables globales
let currentDocument = {{ isset($document_id) ? $document_id : 'null' }};
let selectedText = '';
let editor = document.getElementById('editor');
let autoSaveInterval;

// Charger le document si ID fourni
if (currentDocument) {
    loadDocument(currentDocument);
}

// Fonction de chargement de document
async function loadDocument(id) {
    try {
        const response = await fetch(`/api/documents/${id}`);
        if (response.ok) {
            const document = await response.json();
            document.getElementById('document-title-input').value = document.title;
            editor.innerHTML = document.content;
        }
    } catch (error) {
        console.error('Erreur chargement document:', error);
    }
}

// Auto-save
function startAutoSave() {
    autoSaveInterval = setInterval(async () => {
        const title = document.getElementById('document-title-input').value;
        const content = editor.innerHTML;
        
        if (title.trim() !== '' || content.trim() !== '') {
            await saveDocument(false);
        }
    }, 30000);
}

async function saveDocument(showNotif = true) {
    const title = document.getElementById('document-title-input').value || 'Untitled Document';
    const content = editor.innerHTML;

    try {
        const response = await fetch('/api/documents/save', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                title: title,
                content: content,
                document_id: currentDocument
            })
        });

        const result = await response.json();
        
        if (result.success) {
            currentDocument = result.document_id;
            if (showNotif) {
                alert('Document saved!');
            }
        }
    } catch (error) {
        console.error('Erreur sauvegarde:', error);
        if (showNotif) {
            alert('Save error');
        }
    }
}

// Gestion des onglets
document.getElementById('tab-claude').addEventListener('click', () => {
    document.getElementById('tab-claude').classList.add('bg-black', 'text-white');
    document.getElementById('tab-claude').classList.remove('text-gray-600');
    document.getElementById('tab-perplexity').classList.remove('bg-black', 'text-white');
    document.getElementById('tab-perplexity').classList.add('text-gray-600');
    document.getElementById('claude-panel').classList.remove('hidden');
    document.getElementById('perplexity-panel').classList.add('hidden');
});

document.getElementById('tab-perplexity').addEventListener('click', () => {
    document.getElementById('tab-perplexity').classList.add('bg-black', 'text-white');
    document.getElementById('tab-perplexity').classList.remove('text-gray-600');
    document.getElementById('tab-claude').classList.remove('bg-black', 'text-white');
    document.getElementById('tab-claude').classList.add('text-gray-600');
    document.getElementById('perplexity-panel').classList.remove('hidden');
    document.getElementById('claude-panel').classList.add('hidden');
});

// Gestion sélection de texte
editor.addEventListener('mouseup', function() {
    selectedText = window.getSelection().toString().trim();
});

// Barre d'outils
document.querySelectorAll('[data-action]').forEach(button => {
    button.addEventListener('click', function() {
        const action = this.dataset.action;
        executeCommand(action);
    });
});

function executeCommand(action) {
    switch(action) {
        case 'bold':
            document.execCommand('bold');
            break;
        case 'italic':
            document.execCommand('italic');
            break;
        case 'underline':
            document.execCommand('underline');
            break;
        case 'h1':
            document.execCommand('formatBlock', false, 'h1');
            break;
        case 'h2':
            document.execCommand('formatBlock', false, 'h2');
            break;
        case 'h3':
            document.execCommand('formatBlock', false, 'h3');
            break;
        case 'bullet-list':
            document.execCommand('insertUnorderedList');
            break;
        case 'number-list':
            document.execCommand('insertOrderedList');
            break;
    }
    editor.focus();
}

// Claude
document.getElementById('claude-submit').addEventListener('click', () => {
    const prompt = document.getElementById('claude-prompt').value;
    if (prompt.trim()) {
        callClaudeAPI(prompt, 'custom');
    }
});

document.querySelector('[data-claude-action="grammar"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'grammar');
    } else {
        alert('Select text first');
    }
});

document.querySelector('[data-claude-action="expand"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'expand');
    } else {
        alert('Select text first');
    }
});

document.querySelector('[data-claude-action="summarize"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'summarize');
    } else {
        alert('Select text first');
    }
});

document.querySelector('[data-claude-action="rephrase"]').addEventListener('click', () => {
    if (selectedText) {
        callClaudeAPI('', 'rephrase');
    } else {
        alert('Select text first');
    }
});

async function callClaudeAPI(prompt, action) {
    try {
        const response = await fetch('/api/claude', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                prompt: prompt,
                selected_text: selectedText,
                action: action
            })
        });

        const result = await response.json();
        
        if (result.success) {
            document.getElementById('claude-content').textContent = result.content;
            document.getElementById('claude-response').classList.remove('hidden');
        } else {
            alert('Claude error');
        }
    } catch (error) {
        console.error('Erreur Claude API:', error);
        alert('Connection error');
    }
}

document.getElementById('claude-close').addEventListener('click', () => {
    document.getElementById('claude-response').classList.add('hidden');
});

document.getElementById('claude-apply').addEventListener('click', () => {
    const claudeContent = document.getElementById('claude-content').textContent;
    if (selectedText && claudeContent) {
        const selection = window.getSelection();
        if (selection.rangeCount > 0) {
            const range = selection.getRangeAt(0);
            range.deleteContents();
            range.insertNode(document.createTextNode(claudeContent));
        }
        document.getElementById('claude-response').classList.add('hidden');
        selectedText = '';
    }
});

// Search
document.getElementById('search-submit').addEventListener('click', () => {
    const query = document.getElementById('search-query').value;
    if (query.trim()) {
        callPerplexityAPI(query);
    }
});

async function callPerplexityAPI(query) {
    try {
        const response = await fetch('/api/search', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                query: query
            })
        });

        const result = await response.json();
        
        if (result.success) {
            let html = '';
            if (result.summary) {
                html += `<div class="p-2 bg-gray-50 rounded text-xs">${result.summary}</div>`;
            }
            result.results.forEach(item => {
                html += `<div class="text-xs"><strong>${item.title}</strong><br>${item.snippet}</div>`;
            });
            document.getElementById('search-content').innerHTML = html;
            document.getElementById('search-results').classList.remove('hidden');
        } else {
            alert('Search error');
        }
    } catch (error) {
        console.error('Erreur recherche:', error);
        alert('Connection error');
    }
}

document.getElementById('search-close').addEventListener('click', () => {
    document.getElementById('search-results').classList.add('hidden');
});

document.getElementById('search-insert').addEventListener('click', () => {
    const summary = document.querySelector('#search-content div').textContent;
    if (summary) {
        const selection = window.getSelection();
        if (selection.rangeCount > 0) {
            const range = selection.getRangeAt(0);
            range.insertNode(document.createTextNode(summary));
        } else {
            editor.appendChild(document.createTextNode(summary));
        }
        document.getElementById('search-results').classList.add('hidden');
        editor.focus();
    }
});

// Modals
document.getElementById('save-btn').addEventListener('click', () => {
    document.getElementById('save-title').value = document.getElementById('document-title-input').value;
    document.getElementById('save-modal').classList.remove('hidden');
});

document.getElementById('save-confirm').addEventListener('click', async () => {
    const title = document.getElementById('save-title').value;
    document.getElementById('document-title-input').value = title;
    await saveDocument(true);
    document.getElementById('save-modal').classList.add('hidden');
});

document.getElementById('save-cancel').addEventListener('click', () => {
    document.getElementById('save-modal').classList.add('hidden');
});

document.getElementById('share-btn').addEventListener('click', () => {
    if (!currentDocument) {
        alert('Save document first');
        return;
    }
    document.getElementById('share-modal').classList.remove('hidden');
});

document.getElementById('share-generate').addEventListener('click', async () => {
    const permission = document.querySelector('input[name="share-type"]:checked').value;
    
    try {
        const response = await fetch(`/api/documents/${currentDocument}/share`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify({
                permission: permission
            })
        });

        const result = await response.json();
        
        if (result.success) {
            document.getElementById('share-link').value = result.share_url;
            document.getElementById('share-copy').classList.remove('hidden');
        }
    } catch (error) {
        console.error('Erreur partage:', error);
        alert('Share error');
    }
});

document.getElementById('share-copy').addEventListener('click', () => {
    const shareLink = document.getElementById('share-link');
    shareLink.select();
    document.execCommand('copy');
    alert('Link copied!');
});

document.getElementById('share-cancel').addEventListener('click', () => {
    document.getElementById('share-modal').classList.add('hidden');
    document.getElementById('share-link').value = '';
    document.getElementById('share-copy').classList.add('hidden');
});

// Auto-save
startAutoSave();

window.addEventListener('beforeunload', () => {
    if (autoSaveInterval) {
        clearInterval(autoSaveInterval);
    }
});
</script>
</body>
</html>