<?php
require_once '../search/db.php';
session_start();

// Vérification de la connexion
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$message = '';
$error = '';

// Récupération des informations de l'utilisateur
$query = "SELECT * FROM users WHERE id = :id";
$stmt = $db->prepare($query);
$stmt->execute([':id' => $user_id]);
$user = $stmt->fetch();

// Traitement de la mise à jour du profil
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'update_profile') {
        $firstName = trim($_POST['first_name'] ?? '');
        $lastName = trim($_POST['last_name'] ?? '');
        $newPassword = trim($_POST['new_password'] ?? '');
        $confirmPassword = trim($_POST['confirm_password'] ?? '');

        if (empty($firstName) || empty($lastName)) {
            $error = 'Name fields are required.';
        } else {
            try {
                $updateFields = ['first_name' => $firstName, 'last_name' => $lastName];
                $params = [':id' => $user_id, ':first_name' => $firstName, ':last_name' => $lastName];

                // Mise à jour du mot de passe si fourni
                if (!empty($newPassword)) {
                    if ($newPassword !== $confirmPassword) {
                        $error = 'Passwords do not match.';
                    } elseif (strlen($newPassword) < 8) {
                        $error = 'Password must be at least 8 characters long.';
                    } else {
                        $updateFields['password'] = password_hash($newPassword, PASSWORD_DEFAULT);
                        $params[':password'] = $updateFields['password'];
                    }
                }

                if (empty($error)) {
                    $setClause = implode(', ', array_map(fn($field) => "$field = :$field", array_keys($updateFields)));
                    $query = "UPDATE users SET $setClause WHERE id = :id";
                    $stmt = $db->prepare($query);
                    $stmt->execute($params);

                    $message = 'Profile updated successfully.';
                    
                    // Mettre à jour les informations en session
                    $_SESSION['user_name'] = $firstName;
                }
            } catch (PDOException $e) {
                $error = 'Error updating profile.';
            }
        }
    }
    // Traitement de l'upload d'avatar
    elseif ($_POST['action'] === 'update_avatar' && isset($_FILES['avatar'])) {
        $file = $_FILES['avatar'];
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
        $max_size = 2 * 1024 * 1024; // 2MB

        if ($file['error'] === 0) {
            if ($file['size'] <= $max_size) {
                $finfo = finfo_open(FILEINFO_MIME_TYPE);
                $mime_type = finfo_file($finfo, $file['tmp_name']);

                if (in_array($mime_type, $allowed_types)) {
                    $upload_dir = 'uploads/avatars';
                    if (!file_exists($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }

                    // Supprimer l'ancien avatar s'il existe
                    if ($user['avatar_url'] && file_exists($user['avatar_url'])) {
                        unlink($user['avatar_url']);
                    }

                    $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                    $new_filename = 'avatar_' . $user_id . '_' . uniqid() . '.' . $file_extension;
                    $upload_path = $upload_dir . '/' . $new_filename;

                    if (move_uploaded_file($file['tmp_name'], $upload_path)) {
                        try {
                            $query = "UPDATE users SET avatar_url = :avatar_url WHERE id = :id";
                            $stmt = $db->prepare($query);
                            $stmt->execute([
                                ':avatar_url' => $upload_path,
                                ':id' => $user_id
                            ]);
                            $message = 'Avatar updated successfully.';
                            
                            // Mettre à jour l'avatar dans les données utilisateur
                            $user['avatar_url'] = $upload_path;
                        } catch (PDOException $e) {
                            $error = 'Error updating avatar.';
                        }
                    } else {
                        $error = 'Error uploading avatar.';
                    }
                } else {
                    $error = 'Invalid file type. Please upload a JPEG, PNG or GIF image.';
                }
            } else {
                $error = 'File is too large (max 2MB).';
            }
        } else {
            $error = 'Error uploading file.';
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en" class="h-full">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile Settings - Peerkinton</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://api.fontshare.com/v2/css?f[]=satoshi@400,500,700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Satoshi', sans-serif;
            background: #fafafa;
        }

        .nav-link {
            transition: all 0.2s ease;
        }

        .nav-link:hover {
            background-color: rgba(0, 0, 0, 0.05);
        }

        .nav-link.active {
            background-color: #f3f4f6;
            color: #111827;
        }

        .avatar-upload:hover .avatar-overlay {
            opacity: 1;
        }
    </style>
</head>
<body class="h-full">
    <?php include '../header.php'; ?>

    <div class="flex min-h-screen pt-20">
        <!-- Sidebar -->
        <aside class="w-64 border-r border-gray-200 px-4 py-6">
            <nav class="space-y-2">
                <a href="dashboard.php" class="nav-link flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"/>
                    </svg>
                    Dashboard
                </a>
                <a href="dashboard-files.php" class="nav-link flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 8h14M5 8a2 2 0 110-4h14a2 2 0 110 4M5 8v10a2 2 0 002 2h10a2 2 0 002-2V8m-9 4h4"/>
                    </svg>
                    Member records
                </a>
                <a href="dashboard-profile.php" class="nav-link active flex items-center px-4 py-2 text-sm text-gray-600 rounded-lg">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                    </svg>
                    Profil
                </a>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="flex-1 px-8 py-6">
            <h1 class="text-2xl font-medium text-gray-900 mb-8">Profile Settings</h1>

            <?php if ($message): ?>
                <div class="bg-green-50 border border-green-200 text-green-800 rounded-lg p-4 mb-6">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div class="bg-red-50 border border-red-200 text-red-800 rounded-lg p-4 mb-6">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                <!-- Avatar Section -->
                <div class="bg-white rounded-xl p-6 border border-gray-200">
                    <h2 class="text-lg font-medium text-gray-900 mb-6">Profile Picture</h2>
                    
                    <form method="POST" enctype="multipart/form-data" class="text-center">
                        <input type="hidden" name="action" value="update_avatar">
                        
                        <div class="relative inline-block mb-6 avatar-upload">
                            <?php if ($user['avatar_url']): ?>
                                <img src="<?php echo htmlspecialchars($user['avatar_url']); ?>" 
                                     alt="Profile picture" 
                                     class="w-32 h-32 rounded-full object-cover">
                            <?php else: ?>
                                <div class="w-32 h-32 rounded-full bg-gray-200 flex items-center justify-center">
                                    <span class="text-3xl font-medium text-gray-500">
                                        <?php echo strtoupper(substr($user['first_name'], 0, 1)); ?>
                                    </span>
                                </div>
                            <?php endif; ?>
                            <div class="avatar-overlay absolute inset-0 bg-black bg-opacity-50 rounded-full flex items-center justify-center opacity-0 transition-opacity duration-200">
                                <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 9a2 2 0 012-2h.93a2 2 0 001.664-.89l.812-1.22A2 2 0 0110.07 4h3.86a2 2 0 011.664.89l.812 1.22A2 2 0 0018.07 7H19a2 2 0 012 2v9a2 2 0 01-2 2H5a2 2 0 01-2-2V9z"/>
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 13a3 3 0 11-6 0 3 3 0 016 0z"/>
                                </svg>
                            </div>
                        </div>

                        <input type="file" 
                               name="avatar" 
                               id="avatar" 
                               accept="image/*" 
                               class="hidden" 
                               onchange="this.form.submit()">
                        <label for="avatar" 
                               class="inline-block px-4 py-2 bg-gray-900 text-white rounded-full text-sm cursor-pointer hover:transform hover:-translate-y-0.5 transition-all duration-200">
                            Change Picture
                        </label>
                    </form>
                </div>

                <!-- Profile Information -->
                <div class="bg-white rounded-xl p-6 border border-gray-200">
                    <h2 class="text-lg font-medium text-gray-900 mb-6">Personal Information</h2>
                    
                    <form method="POST" class="space-y-6">
                        <input type="hidden" name="action" value="update_profile">

                        <div>
                            <label for="first_name" class="block text-sm font-medium text-gray-700 mb-2">First Name</label>
                            <input 
                                type="text" 
                                id="first_name" 
                                name="first_name" 
                                value="<?php echo htmlspecialchars($user['first_name']); ?>"
                                class="w-full px-4 py-2 rounded-lg border border-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-200"
                                required
                            >
                        </div>

                        <div>
                            <label for="last_name" class="block text-sm font-medium text-gray-700 mb-2">Last Name</label>
                            <input 
                                type="text" 
                                id="last_name" 
                                name="last_name" 
                                value="<?php echo htmlspecialchars($user['last_name']); ?>"
                                class="w-full px-4 py-2 rounded-lg border border-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-200"
                                required
                            >
                        </div>

                        <div>
                            <label for="email" class="block text-sm font-medium text-gray-700 mb-2">Email</label>
                            <input 
                                type="email" 
                                id="email" 
                                value="<?php echo htmlspecialchars($user['email']); ?>"
                                class="w-full px-4 py-2 rounded-lg border border-gray-200 bg-gray-50"
                                disabled
                            >
                        </div>

                        <div class="pt-4 border-t border-gray-200">
                            <h3 class="text-sm font-medium text-gray-900 mb-4">Change Password</h3>
                            
                            <div class="space-y-4">
                                <div>
                                    <label for="new_password" class="block text-sm font-medium text-gray-700 mb-2">New Password</label>
                                    <input 
                                        type="password" 
                                        id="new_password" 
                                        name="new_password" 
                                        class="w-full px-4 py-2 rounded-lg border border-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-200"
                                        minlength="8"
                                    >
                                </div>

                                <div>
                                    <label for="confirm_password" class="block text-sm font-medium text-gray-700 mb-2">Confirm New Password</label>
                                    <input 
                                        type="password" 
                                        id="confirm_password" 
                                        name="confirm_password" 
                                        class="w-full px-4 py-2 rounded-lg border border-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-200"
                                        minlength="8"
                                    >
                                </div>
                            </div>
                        </div>

                        <div class="flex justify-end">
                            <button 
                                type="submit" 
                                class="px-6 py-2 bg-gray-900 text-white rounded-full hover:transform hover:-translate-y-0.5 transition-all duration-200"
                            >
                                Save Changes
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Account Information -->
                <div class="bg-white rounded-xl p-6 border border-gray-200">
                    <h2 class="text-lg font-medium text-gray-900 mb-6">Account Information</h2>
                    
                    <div class="space-y-4">
                        <div>
                            <p class="text-sm text-gray-600">Account Type</p>
                            <p class="text-sm font-medium text-gray-900">
                                <?php echo ucfirst($user['role']); ?>
                            </p>
                        </div>
                        
                        <div>
                            <p class="text-sm text-gray-600">Member Since</p>
                            <p class="text-sm font-medium text-gray-900">
                                <?php echo date('F j, Y', strtotime($user['created_at'])); ?>
                            </p>
                        </div>
                        
                        <div>
                            <p class="text-sm text-gray-600">Last Login</p>
                            <p class="text-sm font-medium text-gray-900">
                                <?php echo $user['last_login'] ? date('F j, Y H:i', strtotime($user['last_login'])) : 'Never'; ?>
                            </p>
                        </div>
                    </div>
                </div>

                <!-- Account Status -->
                <div class="bg-white rounded-xl p-6 border border-gray-200">
                    <h2 class="text-lg font-medium text-gray-900 mb-6">Account Status</h2>
                    
                    <div class="flex items-center">
                        <div class="flex-shrink-0">
                            <?php if ($user['status'] === 'active'): ?>
                                <span class="inline-block w-3 h-3 bg-green-400 rounded-full"></span>
                            <?php else: ?>
                                <span class="inline-block w-3 h-3 bg-red-400 rounded-full"></span>
                            <?php endif; ?>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm font-medium text-gray-900">
                                <?php echo ucfirst($user['status']); ?>
                            </p>
                            <p class="text-sm text-gray-500">
                                <?php echo $user['status'] === 'active' 
                                    ? 'Your account is active and in good standing.' 
                                    : 'Your account is currently inactive. Please contact support.'; 
                                ?>
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Preview de l'image avant upload
        document.getElementById('avatar').addEventListener('change', function(e) {
            if (e.target.files && e.target.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    document.querySelector('.avatar-upload img').src = e.target.result;
                }
                reader.readAsDataURL(e.target.files[0]);
            }
        });
    </script>
</body>
</html>