document.addEventListener('DOMContentLoaded', () => {
    // Variables globales
    let financeChart = null;
    let financeData = {
        revenues: [],
        expenses: [],
        profit: [],
        dates: []
    };
    let goals = [];
    let transactions = [];

    // Éléments DOM
    const chartPeriodSelect = document.getElementById('chart-period');
    const chartTypeSelect = document.getElementById('chart-type');
    const addTransactionBtn = document.getElementById('add-transaction-btn');
    const transactionModal = document.getElementById('transaction-modal');
    const closeTransactionBtns = document.querySelectorAll('.close-transaction-modal');
    const transactionForm = document.getElementById('transaction-form');
    const transactionTypeRadios = document.querySelectorAll('input[name="transaction-type"]');
    const addGoalBtn = document.getElementById('add-goal-btn');
    const goalModal = document.getElementById('goal-modal');
    const closeGoalBtns = document.querySelectorAll('.close-goal-modal');
    const goalForm = document.getElementById('goal-form');

    // Initialiser les dates par défaut dans les formulaires
    const today = new Date();
    const formattedDate = today.toISOString().split('T')[0]; // Format YYYY-MM-DD
    
    if (document.getElementById('transaction-date')) {
        document.getElementById('transaction-date').value = formattedDate;
    }
    
    if (document.getElementById('goal-start-date')) {
        document.getElementById('goal-start-date').value = formattedDate;
        
        // Date d'échéance par défaut = 1 mois plus tard
        const nextMonth = new Date();
        nextMonth.setMonth(nextMonth.getMonth() + 1);
        document.getElementById('goal-end-date').value = nextMonth.toISOString().split('T')[0];
    }

    // Gestionnaires d'événements pour les modes sombre/clair
    function updateChartTheme() {
        if (!financeChart) return;
        
        const isDarkMode = document.documentElement.classList.contains('dark');
        const textColor = isDarkMode ? '#E5E7EB' : '#4B5563';
        const gridColor = isDarkMode ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.1)';
        
        financeChart.options.scales.x.grid.color = gridColor;
        financeChart.options.scales.y.grid.color = gridColor;
        financeChart.options.scales.x.ticks.color = textColor;
        financeChart.options.scales.y.ticks.color = textColor;
        financeChart.update();
    }

    document.getElementById('theme-toggle')?.addEventListener('click', () => {
        setTimeout(updateChartTheme, 100);
    });

    // Charger les données financières
    async function loadFinanceData(days = 30) {
        try {
            // Simulation: À remplacer par un vrai appel API
            // const response = await fetch(`get_finance_data.php?days=${days}`);
            // const data = await response.json();
            
            // Simulation de données pour la démo
            const data = generateDemoData(days);
            
            financeData = data;
            updateFinanceSummary(data);
            updateFinanceChart(data);
            
            return data;
        } catch (error) {
            console.error('Erreur de chargement des données financières:', error);
            showNotification('Erreur de chargement des données financières', 'error');
        }
    }
    
    // Simuler des données financières pour la démonstration
    function generateDemoData(days) {
        const data = {
            revenues: [],
            expenses: [],
            profit: [],
            dates: []
        };
        
        const endDate = new Date();
        const startDate = new Date();
        startDate.setDate(startDate.getDate() - days);
        
        // Générer des données sur la période spécifiée
        for (let d = new Date(startDate); d <= endDate; d.setDate(d.getDate() + 1)) {
            // Format de date court (ex: "15 Jan")
            const formattedDate = `${d.getDate()} ${d.toLocaleString('default', { month: 'short' })}`;
            data.dates.push(formattedDate);
            
            // Générer des valeurs aléatoires réalistes
            const revenue = Math.floor(Math.random() * 5000) + 1000;
            const expense = Math.floor(Math.random() * 3000) + 500;
            const profit = revenue - expense;
            
            data.revenues.push(revenue);
            data.expenses.push(expense);
            data.profit.push(profit);
        }
        
        return data;
    }

    // Mettre à jour le résumé financier
    function updateFinanceSummary(data) {
        const totalRevenue = document.getElementById('total-revenue');
        const totalExpenses = document.getElementById('total-expenses');
        const totalProfit = document.getElementById('total-profit');
        const revenueTrend = document.getElementById('revenue-trend');
        const expensesTrend = document.getElementById('expenses-trend');
        const profitRatio = document.getElementById('profit-ratio');
        
        if (!totalRevenue || !totalExpenses || !totalProfit) return;
        
        const sumRevenue = data.revenues.reduce((sum, val) => sum + val, 0);
        const sumExpenses = data.expenses.reduce((sum, val) => sum + val, 0);
        const sumProfit = sumRevenue - sumExpenses;
        
        totalRevenue.textContent = `${sumRevenue.toLocaleString()} €`;
        totalExpenses.textContent = `${sumExpenses.toLocaleString()} €`;
        totalProfit.textContent = `${sumProfit.toLocaleString()} €`;
        
        // Calcul des tendances
        const revenueChange = calculateTrend(data.revenues);
        const expensesChange = calculateTrend(data.expenses);
        const profitPercentage = (sumRevenue > 0) ? ((sumProfit / sumRevenue) * 100).toFixed(1) : 0;
        
        revenueTrend.innerHTML = getTrendHTML(revenueChange, true);
        expensesTrend.innerHTML = getTrendHTML(expensesChange, false);
        profitRatio.textContent = `Marge bénéficiaire: ${profitPercentage}%`;
        
        // Mise à jour de la couleur du profit selon qu'il est positif ou négatif
        if (sumProfit >= 0) {
            totalProfit.classList.remove('text-red-500');
            totalProfit.classList.add('text-green-500');
        } else {
            totalProfit.classList.remove('text-green-500');
            totalProfit.classList.add('text-red-500');
        }
    }
    
    // Calculer la tendance (pourcentage d'évolution)
    function calculateTrend(data) {
        if (data.length < 2) return 0;
        
        const half = Math.floor(data.length / 2);
        const firstHalf = data.slice(0, half);
        const secondHalf = data.slice(half);
        
        const firstHalfAvg = firstHalf.reduce((sum, val) => sum + val, 0) / firstHalf.length;
        const secondHalfAvg = secondHalf.reduce((sum, val) => sum + val, 0) / secondHalf.length;
        
        if (firstHalfAvg === 0) return 0;
        return ((secondHalfAvg - firstHalfAvg) / firstHalfAvg) * 100;
    }
    
    // Générer le HTML pour afficher la tendance
    function getTrendHTML(percentage, isRevenue) {
        const fixed = percentage.toFixed(1);
        const isPositive = percentage > 0;
        
        // Pour les revenus, positif = bon ; pour les dépenses, négatif = bon
        const isGood = isRevenue ? isPositive : !isPositive;
        
        const color = isGood ? 'text-green-500' : 'text-red-500';
        const icon = isPositive ? 'fa-arrow-up' : 'fa-arrow-down';
        
        return `<span class="${color}"><i class="fas ${icon} mr-1"></i>${Math.abs(fixed)}%</span> depuis le mois dernier`;
    }

    // Initialiser et mettre à jour le graphique financier
    function updateFinanceChart(data, chartType = 'line') {
        const chartContainer = document.getElementById('finance-chart');
        if (!chartContainer) return;
        
        // Détruire le graphique existant s'il existe
        if (financeChart) {
            financeChart.destroy();
        }
        
        // Couleurs et options en fonction du thème
        const isDarkMode = document.documentElement.classList.contains('dark');
        const textColor = isDarkMode ? '#E5E7EB' : '#4B5563';
        const gridColor = isDarkMode ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.1)';
        
        // Configurer le graphique
        const ctx = chartContainer.getContext('2d');
        
        // Options spécifiques selon le type de graphique
        let config;
        
        if (chartType === 'line') {
            config = {
                type: 'line',
                data: {
                    labels: data.dates,
                    datasets: [
                        {
                            label: 'Revenus',
                            data: data.revenues,
                            borderColor: '#10B981',
                            backgroundColor: 'rgba(16, 185, 129, 0.1)',
                            tension: 0.4,
                            fill: true
                        },
                        {
                            label: 'Dépenses',
                            data: data.expenses,
                            borderColor: '#EF4444',
                            backgroundColor: 'rgba(239, 68, 68, 0.1)',
                            tension: 0.4,
                            fill: true
                        },
                        {
                            label: 'Bénéfice',
                            data: data.profit,
                            borderColor: '#3B82F6',
                            backgroundColor: 'rgba(59, 130, 246, 0.1)',
                            tension: 0.4,
                            fill: true
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        x: {
                            grid: {
                                color: gridColor
                            },
                            ticks: {
                                color: textColor
                            }
                        },
                        