<?php
session_start();
require_once '../db.php';

const MAX_ATTEMPTS = 3;
const LOCKOUT_DURATION = 300;

// Rediriger si déjà connecté
if(isset($_SESSION['user_id'])) {
    if(isset($_SESSION['andweare_member']) && $_SESSION['andweare_member'] == 1 && !empty($_SESSION['family_site_url'])) {
        header('Location: ' . $_SESSION['family_site_url']);
        exit;
    } else {
        header('Location: ../create-family.php');
        exit;
    }
}

$error = '';
$success = '';
$is_locked = false;
$remaining_time = 0;

$_SESSION['login_attempts'] ??= 0;

if (isset($_SESSION['lockout_time']) && $_SESSION['lockout_time'] > time()) {
    $is_locked = true;
    $remaining_time = $_SESSION['lockout_time'] - time();
}

if (isset($_GET['unlock']) && (!isset($_SESSION['lockout_time']) || $_SESSION['lockout_time'] <= time())) {
    unset($_SESSION['login_attempts']);
    unset($_SESSION['lockout_time']);
    $is_locked = false;
}

if(isset($_GET['verified'])) {
    $success = "Email verified! Please login to continue.";
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$is_locked) {
    $email = filter_input(INPUT_POST, 'email', FILTER_VALIDATE_EMAIL);
    $password = $_POST['password'] ?? '';
    
    if (!$email) {
        $error = "Please enter a valid email address";
    } elseif (empty($password)) {
        $error = "Please enter your password";
    } else {
        try {
            $db = new Database();
            $conn = $db->connect();

            $stmt = $conn->prepare("SELECT * FROM utilisateurs WHERE email = ? AND is_verified = 1 LIMIT 1");
            $stmt->execute([$email]);
            
            if($stmt->rowCount() > 0) {
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                if(password_verify($password, $user['password'])) {
                    $stmt = $conn->prepare("SELECT passkey_enabled FROM utilisateurs WHERE id = ? LIMIT 1");
                    $stmt->execute([$user['id']]);
                    $passkey_enabled = $stmt->fetchColumn();
                
                    if ($passkey_enabled) {
                        $_SESSION['pending_passkey_auth'] = true;
                        $_SESSION['pending_user_id'] = $user['id'];
                        $_SESSION['pending_username'] = $user['username'];
                        $_SESSION['pending_email'] = $user['email'];
                        header('Location: passkey.php');
                        exit;
                    }

                    unset($_SESSION['login_attempts']);
                    unset($_SESSION['lockout_time']);

                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $user['username'];
                    
                    // Récupérer les données AndWeare
                    $stmt = $conn->prepare("SELECT andweare_member, family_site_url, family_role FROM utilisateurs WHERE id = ? LIMIT 1");
                    $stmt->execute([$user['id']]);
                    $andweare_data = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    $_SESSION['andweare_member'] = $andweare_data['andweare_member'] ?? 0;
                    $_SESSION['family_site_url'] = $andweare_data['family_site_url'] ?? '';
                    $_SESSION['family_role'] = $andweare_data['family_role'] ?? '';
                    
                    // Enregistrer la connexion
                    logLoginActivity($conn, $user['id'], $user['email']);

                    if($_SESSION['andweare_member'] == 1 && !empty($_SESSION['family_site_url'])) {
                        header('Location: ' . $_SESSION['family_site_url']);
                    } else {
                        header('Location: ../create-family.php');
                    }
                    exit;
                } else {
                    $_SESSION['login_attempts']++;
                    
                    if ($_SESSION['login_attempts'] >= MAX_ATTEMPTS) {
                        $_SESSION['lockout_time'] = time() + LOCKOUT_DURATION;
                        $is_locked = true;
                        $error = "Too many failed attempts. Please wait 5 minutes.";
                    } else {
                        $error = "Invalid password";
                    }
                }
            } else {
                $error = "Email not found or not verified";
            }
        } catch(PDOException $e) {
            $error = "Login failed: Database error";
        }
    }
}

function logLoginActivity($conn, $user_id, $email) {
    $ip_address = $_SERVER['REMOTE_ADDR'];
    $user_agent = $_SERVER['HTTP_USER_AGENT'];
    
    // Informations sur l'appareil
    $device_info = [
        'device' => 'Unknown',
        'os' => 'Unknown',
        'browser' => 'Unknown'
    ];

    if(preg_match('/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i',$user_agent)) {
        $device_info['device'] = 'Mobile';
    } elseif(preg_match('/tablet|ipad|playbook|silk/i', $user_agent)) {
        $device_info['device'] = 'Tablet';
    } else {
        $device_info['device'] = 'Desktop';
    }

    if(preg_match('/windows/i', $user_agent)) {
        $device_info['os'] = 'Windows';
    } elseif(preg_match('/macintosh|mac os x/i', $user_agent)) {
        $device_info['os'] = 'macOS';
    } elseif(preg_match('/linux/i', $user_agent)) {
        $device_info['os'] = 'Linux';
    } elseif(preg_match('/iphone|ipad|ipod/i', $user_agent)) {
        $device_info['os'] = 'iOS';
    } elseif(preg_match('/android/i', $user_agent)) {
        $device_info['os'] = 'Android';
    }

    if(preg_match('/MSIE|Trident/i', $user_agent)) {
        $device_info['browser'] = 'Internet Explorer';
    } elseif(preg_match('/Firefox/i', $user_agent)) {
        $device_info['browser'] = 'Firefox';
    } elseif(preg_match('/Chrome/i', $user_agent)) {
        $device_info['browser'] = 'Chrome';
    } elseif(preg_match('/Safari/i', $user_agent)) {
        $device_info['browser'] = 'Safari';
    } elseif(preg_match('/Opera|OPR/i', $user_agent)) {
        $device_info['browser'] = 'Opera';
    }
    
    // Tenter d'obtenir des informations de localisation
    $location = [];
    try {
        $ip_details = json_decode(file_get_contents("http://ip-api.com/json/" . $ip_address), true);
        if($ip_details && $ip_details['status'] === 'success') {
            $location = [
                'country' => $ip_details['country'],
                'city' => $ip_details['city'],
                'region' => $ip_details['regionName'],
                'location' => "{$ip_details['city']}, {$ip_details['country']}"
            ];
        }
    } catch (Exception $e) {
        // Silencieux en cas d'échec
    }
    
    // Enregistrer la connexion dans la base de données
    try {
        $stmt = $conn->prepare("
            INSERT INTO `connection-watchguard` 
            (user_id, `ip-connected`, `identified-screen`, `hours-of-connect`, `date-of-connect`) 
            VALUES (:user_id, :ip_connected, :identified_screen, :hours_of_connect, :date_of_connect)
        ");
        
        $stmt->execute([
            ':user_id' => $user_id,
            ':ip_connected' => json_encode([
                'ip' => $ip_address,
                'location' => $location['location'] ?? 'Unknown'
            ]),
            ':identified_screen' => json_encode($device_info),
            ':hours_of_connect' => date('H:i:s'),
            ':date_of_connect' => date('Y-m-d H:i:s')
        ]);
    } catch (Exception $e) {
        // Silencieux en cas d'échec
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Login - AndWeare</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+KR:wght@300;400;500;700&display=swap" rel="stylesheet">
      <link rel="icon" type="image/png" href="../logo.png">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://unpkg.com/alpinejs@3.x.x/dist/cdn.min.js" defer></script>
    <style>
        body {
            font-family: 'Noto Sans KR', sans-serif;
        }
    </style>
</head>
<body 
    x-data="loginManager()" 
    class="bg-white text-black min-h-screen flex items-center justify-center">
    <div class="w-full max-w-md p-8">
        <div class="text-center mb-8">
    <div class="flex items-center justify-center gap-4 mb-4">
        <a href="../">
            <img src="../logo.png" alt="AndWeare Logo" style="height: 60px;">
        </a>
        <span class="text-xl font-bold">X</span>
        <a href="../">
            <img class="rounded-md" src="https://cdn.imators.com/logo.png" alt="Second Logo" style="height: 60px;">
        </a>
    </div>
    <h1 class="text-3xl font-light">Welcome to AndWeare</h1>
    <p class="text-sm text-gray-600">Login to create or access your family hub</p>
</div>

        <form 
            method="POST" 
            x-on:submit.prevent="submitForm"
            class="space-y-6"
        >
            <?php if($error): ?>
                <div 
                    x-show="showError"
                    class="bg-red-100 border border-red-400 text-red-700 p-3 rounded-lg text-center mb-4"
                >
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <?php if($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 p-3 rounded-lg text-center mb-4">
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
            <?php if($is_locked): ?>
                <div 
                    x-data="lockoutTimer(<?php echo $remaining_time; ?>)"
                    x-init="startTimer()"
                    class="bg-yellow-100 border border-yellow-400 text-yellow-800 p-4 rounded-lg text-center mb-4"
                >
                    <p>Too many login attempts</p>
                    Time remaining: <span x-text="formattedTime"></span>
                    <a 
                        x-show="canUnlock" 
                        href="?unlock" 
                        class="mt-2 inline-block bg-green-500 text-white px-4 py-2 rounded"
                    >
                        Try Again
                    </a>
                </div>
            <?php endif; ?>
            
            <div>
                <label class="block text-sm font-medium mb-2">Email</label>
                <input 
                    type="email" 
                    name="email" 
                    required 
                    <?php echo $is_locked ? 'disabled' : ''; ?>
                    x-model="email"
                    class="w-full px-4 py-2 rounded-lg border border-gray-300 focus:border-black focus:outline-none <?php echo $is_locked ? 'opacity-50 cursor-not-allowed' : ''; ?>"
                >
            </div>
            
            <div>
                <label class="block text-sm font-medium mb-2">Password</label>
                <div class="relative">
                    <input 
                        type="password" 
                        name="password" 
                        required 
                        <?php echo $is_locked ? 'disabled' : ''; ?>
                        x-model="password"
                        id="password-field"
                        class="w-full px-4 py-2 rounded-lg border border-gray-300 focus:border-black focus:outline-none <?php echo $is_locked ? 'opacity-50 cursor-not-allowed' : ''; ?>"
                    >
                    <button 
                        type="button"
                        class="absolute inset-y-0 right-0 pr-3 flex items-center"
                        onclick="togglePasswordVisibility()"
                    >
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 text-gray-400" id="eye-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                        </svg>
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 text-gray-400 hidden" id="eye-off-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.88 9.88l-3.29-3.29m7.532 7.532l3.29 3.29M3 3l18 18" />
                        </svg>
                    </button>
                </div>
                <div class="flex justify-between mt-1">
                    <div></div>
                    <a href="reset-password.php" class="text-sm text-gray-600 hover:text-black">Forgot password?</a>
                </div>
            </div>
            
            <button 
                type="submit" 
                <?php echo $is_locked ? 'disabled' : ''; ?>
                class="w-full bg-black text-white py-2 px-4 rounded-lg font-medium hover:bg-gray-800 transition-colors <?php echo $is_locked ? 'opacity-50 cursor-not-allowed' : ''; ?>"
            >
                Login
            </button>
        </form>
        
        <div class="mt-6 text-center">
            <p class="text-sm text-gray-600">
                Don't have an account? <a href="https://idsma.imators.com/register" class="text-black font-medium hover:underline">Sign up</a>
            </p>
        </div>
    </div>

    <script>
    function togglePasswordVisibility() {
        const passwordField = document.getElementById('password-field');
        const eyeIcon = document.getElementById('eye-icon');
        const eyeOffIcon = document.getElementById('eye-off-icon');
        
        if (passwordField.type === 'password') {
            passwordField.type = 'text';
            eyeIcon.classList.add('hidden');
            eyeOffIcon.classList.remove('hidden');
        } else {
            passwordField.type = 'password';
            eyeIcon.classList.remove('hidden');
            eyeOffIcon.classList.add('hidden');
        }
    }

    function loginManager() {
        return {
            email: '',
            password: '',
            showError: <?php echo $error ? 'true' : 'false'; ?>,
            submitForm() {
                if (!this.email || !this.password) {
                    this.showError = true;
                    return;
                }
                
                document.querySelector('form').submit();
            }
        };
    }

    function lockoutTimer(initialTime) {
        return {
            remainingTime: initialTime,
            canUnlock: false,
            formattedTime: '',
            startTimer() {
                const updateTimer = () => {
                    if (this.remainingTime > 0) {
                        const minutes = Math.floor(this.remainingTime / 60);
                        const seconds = this.remainingTime % 60;
                        this.formattedTime = `${minutes} min ${seconds} sec`;
                        this.remainingTime--;
                        
                        setTimeout(updateTimer, 1000);
                    } else {
                        this.canUnlock = true;
                        this.formattedTime = 'Unlocked';
                    }
                };
                
                updateTimer();
            }
        }
    }
    </script>
</body>
</html>