<?php
require_once 'db.php';

// Pagination settings
$profiles_per_page = 12;
$current_page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($current_page - 1) * $profiles_per_page;

// Search and filter parameters
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$location = isset($_GET['location']) ? trim($_GET['location']) : '';
$skill = isset($_GET['skill']) ? trim($_GET['skill']) : '';

// Build query conditions
$conditions = ['profile_visibility = "public"'];
$params = [];

if (!empty($search)) {
    $conditions[] = "(full_name LIKE :search OR bio LIKE :search OR username LIKE :search)";
    $params[':search'] = "%$search%";
}

if (!empty($location)) {
    $conditions[] = "location LIKE :location";
    $params[':location'] = "%$location%";
}

if (!empty($skill)) {
    $conditions[] = "JSON_CONTAINS(skills, :skill, '$')";
    $params[':skill'] = json_encode($skill);
}

$where_clause = !empty($conditions) ? 'WHERE ' . implode(' AND ', $conditions) : '';

// Get total profiles for pagination
$count_query = "SELECT COUNT(*) FROM data_profile $where_clause";
$stmt = $db->prepare($count_query);
$stmt->execute($params);
$total_profiles = $stmt->fetchColumn();
$total_pages = ceil($total_profiles / $profiles_per_page);

// Get profiles for current page
$query = "SELECT id, username, full_name, bio, location, company, skills, avatar_url, last_active 
          FROM data_profile 
          $where_clause 
          ORDER BY last_active DESC 
          LIMIT :offset, :limit";

$stmt = $db->prepare($query);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':limit', $profiles_per_page, PDO::PARAM_INT);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->execute();
$profiles = $stmt->fetchAll();

// Get distinct locations for filter
$locations_query = "SELECT DISTINCT location FROM data_profile WHERE location IS NOT NULL AND location != '' ORDER BY location";
$locations = $db->query($locations_query)->fetchAll(PDO::FETCH_COLUMN);

// Get common skills for filter
$skills_query = "SELECT DISTINCT JSON_UNQUOTE(JSON_EXTRACT(skills, '$[*]')) as skill 
                FROM data_profile, JSON_TABLE(skills, '$[*]' COLUMNS (skill VARCHAR(255) PATH '$')) as skills_table 
                ORDER BY skill";
$skills = $db->query($skills_query)->fetchAll(PDO::FETCH_COLUMN);
?>

<!DOCTYPE html>
<html lang="en" class="h-full">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Search Profiles - Peerkinton</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://api.fontshare.com/v2/css?f[]=satoshi@400,500,700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Satoshi', sans-serif;
            background: #fafafa;
        }

        .profile-card {
            transition: all 0.3s ease;
        }

        .profile-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }

        .search-input {
            transition: all 0.3s ease;
        }

        .search-input:focus {
            box-shadow: 0 0 0 2px rgba(0, 0, 0, 0.1);
        }

        .skill-tag {
            transition: all 0.2s ease;
        }

        .skill-tag:hover {
            background-color: #1f2937;
            color: white;
        }
    </style>
</head>
<body class="h-full">
    <?php include '../header.php'; ?>

    <main class="pt-2 px-6">
        <div class="max-w-6xl mx-auto">
            <p>
            This page contains the profiles of our students. You can search for them by name, pseudonym, place of residence or skill. There are only public accounts. Private accounts are not displayed.
            </p>
        </div>
    </main>

    <main class="pt-32 pb-16 px-6">
        <div class="max-w-6xl mx-auto">
            <!-- Filters Section -->
            <div class="mb-12">
                <form method="GET" class="space-y-6">
                    <!-- Search Input -->
                    <div class="max-w-2xl mx-auto">
                        <input 
                            type="text" 
                            name="search" 
                            value="<?php echo htmlspecialchars($search); ?>"
                            placeholder="Search by name, username, or bio..." 
                            class="search-input w-full px-6 py-4 text-gray-900 placeholder-gray-500 bg-white rounded-full border border-gray-200 focus:outline-none"
                        >
                    </div>

                    <!-- Advanced Filters -->
                    <div class="bg-white rounded-xl p-6 border border-gray-200">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <!-- Location Filter -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Location</label>
                                <select name="location" class="w-full p-3 border border-gray-200 rounded-lg focus:outline-none">
                                    <option value="">All Locations</option>
                                    <?php foreach ($locations as $loc): ?>
                                        <option value="<?php echo htmlspecialchars($loc); ?>" 
                                                <?php echo $location === $loc ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($loc); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <!-- Skills Filter -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Skill</label>
                                <select name="skill" class="w-full p-3 border border-gray-200 rounded-lg focus:outline-none">
                                    <option value="">All Skills</option>
                                    <?php foreach ($skills as $s): ?>
                                        <option value="<?php echo htmlspecialchars($s); ?>"
                                                <?php echo $skill === $s ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($s); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <div class="mt-6 flex justify-end">
                            <button type="submit" class="px-6 py-2 bg-gray-900 text-white rounded-full hover:transform hover:-translate-y-0.5 transition-all duration-200">
                                Apply Filters
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Results Count -->
            <div class="mb-8">
                <h2 class="text-xl font-medium text-gray-900">
                    <?php echo number_format($total_profiles); ?> profiles found
                </h2>
            </div>

            <!-- Profiles Grid -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                <?php foreach ($profiles as $profile): ?>
                    <a href="profile-detail.php?id=<?php echo $profile['id']; ?>" class="profile-card bg-white rounded-xl p-6 border border-gray-100">
                        <div class="flex items-start space-x-4">
                            <!-- Avatar -->
                            <div class="flex-shrink-0">
                                <?php if ($profile['avatar_url']): ?>
                                    <img src="<?php echo htmlspecialchars($profile['avatar_url']); ?>" 
                                         alt="<?php echo htmlspecialchars($profile['full_name']); ?>"
                                         class="w-12 h-12 rounded-full object-cover">
                                <?php else: ?>
                                    <div class="w-12 h-12 rounded-full bg-gray-200 flex items-center justify-center">
                                        <span class="text-xl font-medium text-gray-500">
                                            <?php echo strtoupper(substr($profile['full_name'], 0, 1)); ?>
                                        </span>
                                    </div>
                                <?php endif; ?>
                            </div>

                            <!-- Profile Info -->
                            <div class="flex-1 min-w-0">
                                <h3 class="text-lg font-medium text-gray-900 truncate">
                                    <?php echo htmlspecialchars($profile['full_name']); ?>
                                </h3>
                                <p class="text-sm text-gray-500 mb-2">
                                    @<?php echo htmlspecialchars($profile['username']); ?>
                                </p>
                                <?php if ($profile['company']): ?>
                                    <p class="text-sm text-gray-600 mb-2">
                                        <?php echo htmlspecialchars($profile['company']); ?>
                                    </p>
                                <?php endif; ?>
                                <?php if ($profile['location']): ?>
                                    <p class="text-sm text-gray-500 mb-4">
                                        <?php echo htmlspecialchars($profile['location']); ?>
                                    </p>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Bio Preview -->
                        <?php if ($profile['bio']): ?>
                            <p class="text-gray-600 text-sm mt-4 line-clamp-2">
                                <?php echo htmlspecialchars($profile['bio']); ?>
                            </p>
                        <?php endif; ?>

                        <!-- Skills -->
                        <?php if (!empty($profile['skills'])): ?>
                            <div class="mt-4 flex flex-wrap gap-2">
                                <?php foreach (json_decode($profile['skills']) as $skill): ?>
                                    <span class="skill-tag inline-flex items-center px-3 py-1 rounded-full text-sm bg-gray-100 text-gray-800">
                                        <?php echo htmlspecialchars($skill); ?>
                                    </span>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>

                        <!-- Last Active -->
                        <div class="mt-4 text-sm text-gray-500">
                            Last active: <?php echo timeAgo($profile['last_active']); ?>
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <div class="mt-12 flex justify-center space-x-2">
                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <a 
                            href="?page=<?php echo $i; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo !empty($location) ? '&location=' . urlencode($location) : ''; ?><?php echo !empty($skill) ? '&skill=' . urlencode($skill) : ''; ?>"
                            class="<?php echo $current_page === $i ? 'bg-gray-900 text-white' : 'bg-white text-gray-900 hover:bg-gray-100'; ?> px-4 py-2 rounded-lg text-sm"
                        >
                            <?php echo $i; ?>
                        </a>
                    <?php endfor; ?>
                </div>
            <?php endif; ?>
        </div>
    </main>
</body>
</html>

<?php
// Helper function for time ago
function timeAgo($datetime) {
    $time = strtotime($datetime);
    $now = time();
    $diff = $now - $time;
    
    if ($diff < 60) {
        return 'just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' minute' . ($mins > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 2592000) {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    } else {
        return date('M j, Y', $time);
    }
}
?>